package neutrino;

import neutrino.text.TextEditor;
import javax.swing.*;
import java.awt.*;
import java.util.Random;

/**
 * Frame for reusing in program for text editing. Used in Window options dialog.
 * And for saving and restoring options related to window behaviour.
 * Date: 9/11/15
 * Time: 3:55 PM
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public abstract class TextEditorWindow extends JFrame {

    public static final int SAVE_WINDOW_LOCATION = 1;
    public static final int CENTER_WINDOW_ON_SCREEN = 2;
    public static final int RANDOMIZE_WINDOW_LOCATION = 3;

    public static int DEFAULT_WINDOW_WIDTH = 1000;
    public static int DEFAULT_WINDOW_HEIGHT = 700;

    private boolean isConfirmExit = true;
    private boolean isSaveWindowSize = true;
    private int typeOfWindowLocation = CENTER_WINDOW_ON_SCREEN;
    private boolean isToolBarVisible = true;

    public TextEditorWindow() {
        super();
        setSize(getDefaultSize());
        centerWindowOnScreen();
    }

    /**
     * Get component for editing text.
     * @return TextEditor
     */
    public abstract TextEditor getTextEditor();

    /**
     * Center window on screen.
     */
    public void centerWindowOnScreen() {
        Dimension size = getSize();
        setBounds(getBoundsOnScreen(size));
    }

    /**
     * Returns bounds of the window in the current screen resolution on the center of screen.
     * @param size - dimension of the window
     * @return rectangle
     */
    private Rectangle getBoundsOnScreen(Dimension size) {
        Rectangle bounds = new Rectangle();
        Dimension screenSize = getToolkit().getScreenSize();
        bounds.width = size.width;
        bounds.height = size.height;
        bounds.x = (screenSize.width - size.width) / 2;
        bounds.y = (screenSize.height - size.height) / 2;
        return bounds;
    }

    /**
     * Randomize window location.
     */
    public void randomizeWindowLocation() {
        Dimension size = getSize();
        Dimension screenSize = getToolkit().getScreenSize();
        Random random = new Random();
        int x = random.nextInt(screenSize.width - size.width);
        int y = random.nextInt(screenSize.height - size.height);
        setLocation(x, y);
    }

    /**
     * Returns default size of window.
     * @return Dimension
     */
    public Dimension getDefaultSize() {
        return new Dimension(DEFAULT_WINDOW_WIDTH, DEFAULT_WINDOW_HEIGHT);
    }

    /**
     * Sets visibility of toolbar.
     * @param isVisible - boolean
     */
    public void setToolBarVisible(boolean isVisible) {
        this.isToolBarVisible = isVisible;
    }

    /**
     * Returns true if toolbar is visible.
     * @return boolean
     */
    public boolean isToolBarVisible() {
        return isToolBarVisible;
    }

    /**
     * Returns the class name of the current look and feel
     * @return string
     */
    public String getLookAndFeel() {
        return UIManager.getLookAndFeel().getClass().getCanonicalName();
    }

    /**
     * Changes the look and feel of the program window.
     * @param className - look and feel class name
     */
    public void setLookAndFeel(String className) {
        if (className == null) return;
        int caretPosition = getTextEditor().getTextComponent().getCaretPosition();
        try {
            UIManager.setLookAndFeel(className);
            SwingUtilities.updateComponentTreeUI(this);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (UnsupportedLookAndFeelException e) {
            e.printStackTrace();
        }
        getTextEditor().getTextComponent().setCaretPosition(caretPosition);
    }

    /**
     * Sets confirming exit when closing window
     * @param value - boolean
     */
    public void setConfirmingExit(boolean value) {
        this.isConfirmExit = value;
    }

    /**
     * Return whether confirm exit when closing window
     * @return boolean
     */
    public boolean isConfirmExit() {
        return this.isConfirmExit;
    }

    /**
     * Sets saving of window size after quitting.
     * @param value - boolean
     */
    public void setSavingWindowSize(boolean value) {
        this.isSaveWindowSize = value;
    }

    /**
     * Returns true if window size will been saved after quitting.
     * @return boolean
     */
    public boolean isSaveWindowSize() {
        return this.isSaveWindowSize;
    }

    /**
     * Sets type of window location.
     * Chacks input value.
     * @param value - integer constant
     */
    public void setTypeOfWindowLocation(int value) {
        switch (value) {
            case SAVE_WINDOW_LOCATION:
            case CENTER_WINDOW_ON_SCREEN:
            case RANDOMIZE_WINDOW_LOCATION:
                this.typeOfWindowLocation = value;
                break;
            default:
                return;
        }
    }

    /**
     * Returns type of window location.
     * @return int
     */
    public int getTypeOfWindowLocation() {
        return this.typeOfWindowLocation;
    }

}
