package neutrino.dialogs;

import neutrino.multitext.*;
import neutrino.text.TextEditor;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListDataListener;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.util.ArrayList;

import static javax.swing.JOptionPane.*;

/**
 * Dialog for managing the sessions
 * @author Oleh Radvanskyj
 * @version 1.0
 * Date: 11/22/15
 * Time: 11:59 PM
 */
public class SessionsManager {

    private class SessionsManagingDialog extends JDialog implements ActionListener {

        protected JList lSessions = new JList();
        private JButton bOpen = new JButton("Open");
        private JButton bRename = new JButton("Rename");
        private JButton bDelete = new JButton("Delete");
        private JButton bCancel = new JButton("Cancel");
        private int currentIndex = 0;
        private MultiTextComponent multiTextComponent = null;
        private ArrayList<Session> sessions = null;
        protected int option = JOptionPane.CANCEL_OPTION;
        protected ListModel listModel = new ListModel() {

            public void addListDataListener(ListDataListener l) { }

            public Object getElementAt(int index) {
                return sessions.get(index).getName();
            }

            public int getSize() {
                return sessions.size();
            }

            public void removeListDataListener(ListDataListener l) { }

        };
        protected class TextsListCellRenderer extends JLabel implements ListCellRenderer {

            public TextsListCellRenderer() {
                setOpaque(true);
            }

            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                setText(sessions.get(index).getName());
                if (sessions.get(index).equals(multiTextComponent.getCurrentSession())) setToolTipText("The session is current");
                if (isSelected) {
                    setForeground(list.getSelectionForeground());
                    setBackground(list.getSelectionBackground());
                } else {
                    setForeground(list.getForeground());
                    setBackground(list.getBackground());
                }
                return this;
            }

        }

        private void setCurrentIndex() {
            Session session = multiTextComponent.getCurrentSession();
            for (int i = 0; i < sessions.size(); i++) {
                if (sessions.get(i).equals(session)) {
                    currentIndex = i;
                    break;
                }
            }
        }

        public SessionsManagingDialog(MultiTextComponent multiTextComponent) {
            super();
            setTitle("Sessions manager");
            setModal(true);
            setResizable(false);
            getRootPane().setDefaultButton(bOpen);
            setDefaultCloseOperation(DISPOSE_ON_CLOSE);
            this.multiTextComponent = multiTextComponent;
            sessions = multiTextComponent.getSessions();
            setCurrentIndex();

            lSessions.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            lSessions.setModel(listModel);
            Dimension size = new Dimension(400, 300);
            lSessions.setMinimumSize(size);
            lSessions.setPreferredSize(size);
            lSessions.setMaximumSize(size);
            lSessions.setSelectedIndex(currentIndex);
            lSessions.setCellRenderer(new TextsListCellRenderer());

            JPanel pEditors = new JPanel();
            pEditors.setBorder(new CompoundBorder(new TitledBorder("Editor"), new EmptyBorder(0, 5, 5, 5)));
            pEditors.setLayout(new BorderLayout());
            pEditors.add(new JScrollPane(lSessions), BorderLayout.CENTER);

            bOpen.setMnemonic(KeyEvent.VK_O);
            bRename.setMnemonic(KeyEvent.VK_R);
            bDelete.setMnemonic(KeyEvent.VK_D);
            bCancel.setMnemonic(KeyEvent.VK_C);

            bOpen.addActionListener(this);
            bRename.addActionListener(this);
            bDelete.addActionListener(this);
            bCancel.addActionListener(this);

            bOpen.setToolTipText("Open the selected session");
            bRename.setToolTipText("Rename the selected session");
            bDelete.setToolTipText("Delete the selected session");
            bCancel.setToolTipText("Abort sessions managing dialog");

            GridBagLayout layout = new GridBagLayout();
            getContentPane().setLayout(layout);
            GridBagConstraints c = new GridBagConstraints();
            c.gridx = 0;
            c.gridy = 0;
            c.gridheight = 5;
            c.insets = new Insets(5, 5, 5, 0);
            getContentPane().add(pEditors, c);
            c.gridx = 1;
            c.gridy = 0;
            c.gridheight = 1;
            c.fill = GridBagConstraints.HORIZONTAL;
            c.insets = new Insets(10, 5, 0, 5);
            getContentPane().add(bOpen, c);
            c.gridy = 1;
            c.insets = new Insets(5, 5, 0, 5);
            getContentPane().add(bRename, c);
            c.gridy = 2;
            c.insets = new Insets(5, 5, 0, 5);
            getContentPane().add(bDelete, c);
            c.gridy = 3;
            getContentPane().add(bCancel, c);
            pack();
        }

        public int getOption() {
            return this.option;
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (e.getSource() == bOpen) {
                int index = lSessions.getSelectedIndex();
                Session session = sessions.get(index);
                multiTextComponent.openSession(session);
                setVisible(false);
                dispose();
            } else if (e.getSource() == bRename) {
                // get session
                int index = lSessions.getSelectedIndex();
                Session session = sessions.get(index);
                // show dialog
                JPanel panel = new JPanel();
                BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
                panel.setLayout(layout);
                panel.add(new JLabel("Enter the name of session"));
                panel.add(Box.createVerticalStrut(5));
                JTextField textField = new JTextField(20);
                textField.setText(session.getName());
                panel.add(textField);
                int option = showConfirmDialog(this, panel, "Session name", YES_NO_OPTION, QUESTION_MESSAGE);
                if (option == NO_OPTION) return;
                if (textField.getText() == null || textField.getText().equals("")) return;
                // rename
                session.setName(textField.getText());
                // update ui
                lSessions.revalidate();
                lSessions.repaint();
                lSessions.updateUI();
            } else if (e.getSource() == bDelete) {
                int index = lSessions.getSelectedIndex();
                Session session = sessions.get(index);
                if (session.equals(multiTextComponent.getCurrentSession())) multiTextComponent.setCurrentSession(new Session());
                sessions.remove(session);
                if (sessions.size() == 0) {
                    setVisible(false);
                    dispose();
                } else {
                    lSessions.revalidate();
                    lSessions.repaint();
                    lSessions.updateUI();
                }
            } if (e.getSource() == bCancel) {
                setVisible(false);
                dispose();
            }
        }

    }

    private static void establishBounds(JDialog dialog) {
        Dimension d1 = dialog.getSize();
        Dimension d2 = dialog.getToolkit().getScreenSize();
        int x = Math.max((d2.width-d1.width)/2, 0);
        int y = Math.max((d2.height-d1.height)/2, 0);
        dialog.setBounds(x + 0, y + 0, d1.width, d1.height);
    }

    public static void showSessionManagingDialog(MultiTextComponent multiTextComponent) {
        SessionsManagingDialog dialog = new SessionsManager().new SessionsManagingDialog(multiTextComponent);
        establishBounds(dialog);
        dialog.setVisible(true);
    }

}
