package neutrino.dialogs;

import com.inet.jortho.SpellChecker;
import neutrino.multitext.MultiTextComponent;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.text.JTextComponent;
import java.awt.*;
import java.awt.event.*;

/**
 * Dialog for spelling options.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class SpellingOptionsChooser {

    private static boolean isAutoSpellingMode = false;

    public static void setAutoSpellingMode(boolean value) {
        isAutoSpellingMode = value;
    }

    public static boolean isAutoSpellingMode() {
        return isAutoSpellingMode;
    }

    private class SpellingOptionsDialog  extends JDialog implements ActionListener {

        private JLabel lMenuLimit = new JLabel("Limit of suggestions in menu : ");
        private JTextField tfMenuLimit = new JTextField(4);
        private JLabel lDialogLimit = new JLabel("Limit of suggestions in dialog : ");
        private JTextField tfDialogLimit = new JTextField(4);
        private JCheckBox cbAutoSpellingMode = new JCheckBox("Auto spelling mode");
        private JCheckBox cbCaseSensitive = new JCheckBox("Case sensitive");
        private JCheckBox cbIgnoreAllCapsWords = new JCheckBox("Ignore all caps words");
        private JCheckBox cbIgnoreCapitalization = new JCheckBox("Ignore capitalization");
        private JCheckBox cbIgnoreWordsWithNumbers = new JCheckBox("Ignore words with numbers");
        private int option = JOptionPane.CANCEL_OPTION;
        private JButton bOk = new JButton("Ok");
        private JButton bDefaults = new JButton("Defaults");
        private JButton bCancel = new JButton("Cancel");
        private JTextComponent textComponent = null;
        private MultiTextComponent multiTextComponent = null;

        private void initialize() {
            tfMenuLimit.setText(Integer.toString(SpellChecker.getOptions().getSuggestionsLimitMenu()));
            tfDialogLimit.setText(Integer.toString(SpellChecker.getOptions().getSuggestionsLimitDialog()));
            cbAutoSpellingMode.setSelected(isAutoSpellingMode());
            cbCaseSensitive.setSelected(SpellChecker.getOptions().isCaseSensitive());
            cbIgnoreAllCapsWords.setSelected(SpellChecker.getOptions().isIgnoreAllCapsWords());
            cbIgnoreCapitalization.setSelected(SpellChecker.getOptions().getIgnoreCapitalization());
            cbIgnoreWordsWithNumbers.setSelected(SpellChecker.getOptions().isIgnoreWordsWithNumbers());
        }

        public SpellingOptionsDialog(JFrame owner, JTextComponent textComponent, MultiTextComponent multiTextComponent) {
            super(owner);
            this.textComponent = textComponent;
            this.multiTextComponent = multiTextComponent;
            setTitle("Spelling options");
            setResizable(false);
            setModal(true);
            setSize(700, 600);
            JPanel panel = new JPanel();
            panel.setBorder(new EmptyBorder(5, 0, 5, 5));
            panel.setLayout(new FlowLayout());
            panel.add(createMainPanel());
            setContentPane(panel);
            initialize();
            pack();
        }

        @Override
        public void actionPerformed(ActionEvent actionEvent) {
            if (actionEvent.getSource() == bOk) {
                if (validateInputs()) {
                    setValues();
                    this.option = JOptionPane.OK_OPTION;
                    setVisible(false);
                }
            } else if (actionEvent.getSource() == bCancel) {
                this.option = JOptionPane.CANCEL_OPTION;
                setVisible(false);
            } else if (actionEvent.getSource() == bDefaults) {
                selectDefaults();
            }
        }

        private boolean validateInputs() {
            try {
                int menuLimit = Integer.parseInt(tfMenuLimit.getText());
                if (menuLimit < 5) {
                    JOptionPane.showMessageDialog(this, "The limit of suggestions in menu mast be greater than 5", "Error message", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                if (menuLimit > 50) {
                    JOptionPane.showMessageDialog(this, "The limit of suggestions in menu mast be less than 50", "Error message", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
            } catch (NumberFormatException e) {
                JOptionPane.showMessageDialog(this, "The limit of suggestions in menu is not a number", "Error message", JOptionPane.ERROR_MESSAGE);
                return false;
            }
            try {
                int dialogLimit = Integer.parseInt(tfDialogLimit.getText());
                if (dialogLimit < 5) {
                    JOptionPane.showMessageDialog(this, "The limit of suggestions in dialog mast be greater than 5", "Error message", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                if (dialogLimit > 500) {
                    JOptionPane.showMessageDialog(this, "The limit of suggestions in dialog mast be less than 500", "Error message", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
            } catch (NumberFormatException e) {
                JOptionPane.showMessageDialog(this, "The limit of suggestions in dialog is not a number", "Error message", JOptionPane.ERROR_MESSAGE);
                return false;
            }
            return true;
        }

        private void setValues() {
            int menuLimit = Integer.parseInt(tfMenuLimit.getText());
            SpellChecker.getOptions().setSuggestionsLimitMenu(menuLimit);
            int dialogLimit = Integer.parseInt(tfDialogLimit.getText());
            SpellChecker.getOptions().setSuggestionsLimitDialog(dialogLimit);
            setAutoSpellingMode(cbAutoSpellingMode.isSelected());
            if (textComponent != null) SpellChecker.enableAutoSpell(textComponent, cbAutoSpellingMode.isSelected());
            if (multiTextComponent != null) multiTextComponent.getMultiTextOptions().enableAutoSpell(cbAutoSpellingMode.isSelected());
            SpellChecker.getOptions().setCaseSensitive(cbCaseSensitive.isSelected());
            SpellChecker.getOptions().setIgnoreAllCapsWords(cbIgnoreAllCapsWords.isSelected());
            SpellChecker.getOptions().setIgnoreCapitalization(cbIgnoreCapitalization.isSelected());
            SpellChecker.getOptions().setIgnoreWordsWithNumbers(cbIgnoreWordsWithNumbers.isSelected());
        }

        private void selectDefaults() {
            tfMenuLimit.setText("15");
            tfDialogLimit.setText("40");
            cbAutoSpellingMode.setSelected(true);
            cbCaseSensitive.setSelected(true);
            cbIgnoreAllCapsWords.setSelected(true);
            cbIgnoreCapitalization.setSelected(true);
            cbIgnoreWordsWithNumbers.setSelected(true);
        }

        /**
         * Return OK_OPTION or CANCEL_OPTION respectively to pressed button.
         * @return
         */
        public int getOption() {
            return this.option;
        }

        private JPanel createMainPanel() {
            JPanel panel = new JPanel();
            panel.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();
            c.insets = new Insets(0, 5, 0, 0);
            c.anchor = GridBagConstraints.NORTH;
            panel.add(createWidgetPanel(), c);
            c.gridx = 1;
            c.insets = new Insets(5, 10, 0, 0);
            panel.add(createButtonsPanel(), c);
            return panel;
        }

        private JPanel createWidgetPanel() {
            lMenuLimit.setLabelFor(tfMenuLimit);
            lMenuLimit.setDisplayedMnemonic(KeyEvent.VK_M);
            lMenuLimit.setToolTipText("The maximum count of entries for the suggestion menu");
            lDialogLimit.setLabelFor(tfDialogLimit);
            lDialogLimit.setDisplayedMnemonic(KeyEvent.VK_D);
            lDialogLimit.setToolTipText("The maximum count of entries for the suggestion list in the spell checker dialog");
            cbAutoSpellingMode.setMnemonic(KeyEvent.VK_A);
            cbAutoSpellingMode.setToolTipText("Automatic checking of the spelling in the text");
            cbCaseSensitive.setMnemonic(KeyEvent.VK_S);
            cbCaseSensitive.setToolTipText("Set whether the spell checker is case-sensitive");
            cbIgnoreAllCapsWords.setMnemonic(KeyEvent.VK_I);
            cbIgnoreAllCapsWords.setToolTipText("Set words that are entirely in uppercase should be ignored for spell checking");
            cbIgnoreCapitalization.setMnemonic(KeyEvent.VK_G);
            cbIgnoreCapitalization.setToolTipText("Set whether capitalized words should be correct if the word in the dictionary as lower-case");
            cbIgnoreWordsWithNumbers.setMnemonic(KeyEvent.VK_W);
            cbIgnoreWordsWithNumbers.setToolTipText("Set if words that include a number in any position should be ignored for spell checking");
            JPanel panel = new JPanel();
            Border border = new CompoundBorder(new TitledBorder("Spelling"), new EmptyBorder(0, 5, 5, 5));
            panel.setBorder(border);
            panel.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();

            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(0, 3, 5, 0);
            panel.add(lMenuLimit, c);
            c.gridx = 1;
            c.insets = new Insets(0, 0, 8, 0);
            panel.add(tfMenuLimit, c);
            c.gridx = 0;
            c.gridy = 1;
            c.insets = new Insets(0, 3, 5, 0);
            panel.add(lDialogLimit, c);
            c.gridx = 1;
            c.insets = new Insets(0, 0, 8, 0);
            panel.add(tfDialogLimit, c);
            c.gridx = 0;
            c.gridy = 2;
            c.gridwidth = 2;
            c.insets = new Insets(0, 0, 5, 0);
            panel.add(cbAutoSpellingMode, c);
            c.gridy = 3;
            panel.add(cbCaseSensitive, c);
            c.gridy = 4;
            panel.add(cbIgnoreAllCapsWords, c);
            c.gridy = 5;
            panel.add(cbIgnoreCapitalization, c);
            c.gridy = 6;
            panel.add(cbIgnoreWordsWithNumbers, c);

            return panel;
        }

        private JPanel createButtonsPanel() {
            getRootPane().setDefaultButton(bOk);
            bOk.setMnemonic(KeyEvent.VK_O);
            bDefaults.setMnemonic(KeyEvent.VK_D);
            bCancel.setMnemonic(KeyEvent.VK_C);
            bOk.setToolTipText("Set options");
            bDefaults.setToolTipText("Select defaults");
            bCancel.setToolTipText("Abort spelling options dialog");
            bOk.addActionListener(this);
            bDefaults.addActionListener(this);
            bCancel.addActionListener(this);
            JPanel panel = new JPanel();
            panel.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();

            c.fill = GridBagConstraints.HORIZONTAL;
            c.insets = new Insets(0, 0, 5, 0);
            panel.add(bOk, c);
            c.gridy = 1;
            c.insets = new Insets(0, 0, 5, 0);
            panel.add(bCancel, c);
            c.gridy = 2;
            panel.add(bDefaults, c);

            return panel;
        }

    }

    private static void establishBounds(JDialog dialog, Window owner) {
        Dimension d1 = dialog.getSize();
        Dimension d2 = owner.getSize();
        Dimension ds = dialog.getToolkit().getScreenSize();
        int x = Math.max((d2.width-d1.width)/2, 0);
        int y = Math.max((d2.height-d1.height)/2, 0);
        int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
        int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
        dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
    }


    /**
     * Create and show spelling chooser dialog. Returns true when settings is changed
     * @param owner - parent frame
     * @param textComponent - JTextComponent can be null
     * @param multiTextComponent - MultiTextComponent can be null
     * @return boolean
     */
    public static boolean showSpellingOptionsDialog(JFrame owner, JTextComponent textComponent, MultiTextComponent multiTextComponent) {
        SpellingOptionsChooser chooser = new SpellingOptionsChooser();
        SpellingOptionsDialog dialog = chooser.new SpellingOptionsDialog(owner, textComponent, multiTextComponent);
        establishBounds(dialog, owner);
        dialog.setVisible(true);
        boolean result = false;
        if (dialog.getOption() == JOptionPane.OK_OPTION) {
            result = true;
        }
        dialog.dispose();
        return result;
    }

}
