package neutrino.dialogs;

import neutrino.multitext.MultiTextComponent;
import neutrino.multitext.MultiTextOptions;

import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;

/**
 * The dialog for choosing the options for XML component
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class SyntaxOptionsChooser {

    private class SyntaxOptionsDialog extends JDialog {

        private JCheckBox cbShowLineNumbers = new JCheckBox("Show line numbers");
        private JCheckBox cbEnableBookmarks = new JCheckBox("Enable bookmarks");
        private JCheckBox cbEnableCodeFolding = new JCheckBox("Enable code folding");
        private JCheckBox cbHighlightCurrentLine = new JCheckBox("Highlight current line");
        private JCheckBox cbAnimateBracketMatching = new JCheckBox("Animate bracket matching");
        private JCheckBox cbMarkOccurrences = new JCheckBox("Mark occurrences");
        private JCheckBox cbPaintTabLines = new JCheckBox("Paint tab lines");

        private JButton bOk = new JButton("Ok");
        private JButton bCancel = new JButton("Cancel");
        private JButton bDefaults = new JButton("Defaults");

        private MultiTextComponent multiTextComponent;

        private void initialize() {
            MultiTextOptions options = multiTextComponent.getMultiTextOptions();
            cbShowLineNumbers.setSelected(options.isLineNumbersShowed());
            cbEnableBookmarks.setSelected(options.isBookmarksEnabled());
            cbEnableCodeFolding.setSelected(options.isCodeFoldingEnabled());
            cbHighlightCurrentLine.setSelected(options.isCurrentLineHighlighted());
            cbAnimateBracketMatching.setSelected(options.isBracketMatchingAnimated());
            cbMarkOccurrences.setSelected(options.isOccurrencesMarked());
            cbPaintTabLines.setSelected(options.isTabLinesPainted());
        }

        /**
         * Set default values.
         */
        private void defaults() {
            cbShowLineNumbers.setSelected(true);
            cbEnableBookmarks.setSelected(true);
            cbEnableCodeFolding.setSelected(true);
            cbHighlightCurrentLine.setSelected(true);
            cbAnimateBracketMatching.setSelected(true);
            cbMarkOccurrences.setSelected(true);
            cbPaintTabLines.setSelected(true);
        }

        /**
         * Establish program variables
         */
        private void establishValues() {
            MultiTextOptions options = multiTextComponent.getMultiTextOptions();
            options.setLineNumbersShowed(cbShowLineNumbers.isSelected());
            options.setBookmarksEnabled(cbEnableBookmarks.isSelected());
            options.setCodeFoldingEnabled(cbEnableCodeFolding.isSelected());
            options.setCurrentLineHighlighted(cbHighlightCurrentLine.isSelected());
            options.setBracketMatchingAnimated(cbAnimateBracketMatching.isSelected());
            options.setOccurrencesMarked(cbMarkOccurrences.isSelected());
            options.setTabLinesPainted(cbPaintTabLines.isSelected());
        }

        public SyntaxOptionsDialog(JFrame owner, MultiTextComponent multiTextComponent) {
            super(owner);
            this.multiTextComponent = multiTextComponent;
            setTitle("Syntax options");
            JPanel panel = new JPanel();
            panel.setBorder(new EmptyBorder(5, 0, 5, 5));
            panel.setLayout(new FlowLayout());
            panel.add(getMainPanel());
            getContentPane().add(panel);
            getRootPane().setDefaultButton(bOk);
            pack();
            initialize();
            setResizable(false);
            setModal(true);
            setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        }

        private JPanel getMainPanel() {
            JPanel mainPanel = new JPanel();
            GridBagLayout mainLayout = new GridBagLayout();
            GridBagConstraints constraints = new GridBagConstraints();
            mainPanel.setLayout(mainLayout);

            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTH;
            constraints.insets = new Insets(0, 5, 0, 0);
            mainPanel.add(getEditingPanel(), constraints);

            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTH;
            constraints.insets = new Insets(5, 10, 0, 0);
            mainPanel.add(getButtonsPanel(), constraints);
            return mainPanel;
        }

        private JPanel getEditingPanel() {
            cbShowLineNumbers.setToolTipText("Set whether the line numbers in the text component is visible");
            cbEnableBookmarks.setToolTipText("Set whether the bookmarks in the text component is visible");
            cbEnableCodeFolding.setToolTipText("Set whether the code in the text component can be folded");
            cbHighlightCurrentLine.setToolTipText("Set whether the current line is highlighted");
            cbAnimateBracketMatching.setToolTipText("Set whether the matched bracket is animated");
            cbMarkOccurrences.setToolTipText("Set whether the occurrences of the word under the cursor is marked");
            cbPaintTabLines.setToolTipText("Set whether the tab lines is painted");
            JPanel editingPanel = new JPanel();
            Border editingBorder = new CompoundBorder(new TitledBorder("Syntax"), new EmptyBorder(0, 5, 5, 5));
            editingPanel.setBorder(editingBorder);
            editingPanel.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();
            c.gridx = 0;
            c.gridy = 0;
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(0, 0, 5, 0);
            editingPanel.add(cbShowLineNumbers, c);
            c.gridy = 1;
            editingPanel.add(cbEnableBookmarks, c);
            c.gridy = 2;
            editingPanel.add(cbEnableCodeFolding, c);
            c.gridy = 3;
            editingPanel.add(cbHighlightCurrentLine, c);
            c.gridy = 4;
            editingPanel.add(cbAnimateBracketMatching, c);
            c.gridy = 5;
            editingPanel.add(cbMarkOccurrences, c);
            c.gridy = 6;
            editingPanel.add(cbPaintTabLines, c);
            return editingPanel;
        }

        private JPanel getButtonsPanel() {
            bOk.setToolTipText("Set options");
            bDefaults.setToolTipText("Select default values");
            bCancel.setToolTipText("Abort syntax options dialog");
            bOk.setMnemonic(KeyEvent.VK_O);
            bCancel.setMnemonic(KeyEvent.VK_C);
            bDefaults.setMnemonic(KeyEvent.VK_D);
            bOk.addActionListener(alOk);
            bCancel.addActionListener(alCancel);
            bDefaults.addActionListener(alDefaults);
            JPanel buttonsPanel = new JPanel();
            GridLayout buttonsLayout = new GridLayout(3, 1, 0, 5);
            buttonsPanel.setLayout(buttonsLayout);
            buttonsPanel.add(bOk);
            buttonsPanel.add(bCancel);
            buttonsPanel.add(bDefaults);
            return buttonsPanel;
        }

        private ActionListener alOk = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                establishValues();
                setVisible(false);
                dispose();
            }
        };

        private ActionListener alCancel = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setVisible(false);
                dispose();
            }
        };

        private ActionListener alDefaults = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                defaults();
            }
        };

    }

    private static void establishBounds(JDialog dialog, Window owner) {
        Dimension d1 = dialog.getSize();
        Dimension d2 = owner.getSize();
        Dimension ds = dialog.getToolkit().getScreenSize();
        int x = Math.max((d2.width-d1.width)/2, 0);
        int y = Math.max((d2.height-d1.height)/2, 0);
        int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
        int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
        dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
    }

    public static void showSyntaxOptionsDialog(JFrame owner, MultiTextComponent multiTextComponent) {
        SyntaxOptionsChooser chooser = new SyntaxOptionsChooser();
        SyntaxOptionsDialog dialog = chooser.new SyntaxOptionsDialog(owner, multiTextComponent);
        establishBounds(dialog, owner);
        dialog.setVisible(true);
    }

}