package neutrino.dialogs;

import neutrino.multitext.MultiTextComponent;
import neutrino.multitext.MultiTextOptions;
import neutrino.text.components.plain.PlainTextArea;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.awt.*;
import java.awt.event.*;
import static javax.swing.JOptionPane.ERROR_MESSAGE;
import static javax.swing.JOptionPane.showMessageDialog;

/**
 * The dialog for choosing the options for syntax text component
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class TextOptionsChooser {

    private abstract class TextOptionsDialog extends JDialog {

        protected JLabel lTabSize = new JLabel("Tab size :");
        protected JTextField tfTabSize = new JTextField(4);
        protected JLabel lMargin = new JLabel("Margin :");
        protected JTextField tfMargin = new JTextField(4);
        protected JCheckBox cbLineWrap = new JCheckBox("Line wrap");
        protected JRadioButton rbCharacter = new JRadioButton("Character      ");
        protected JRadioButton rbWord = new JRadioButton("Word");
        protected ButtonGroup bgWrapStyle = new ButtonGroup();
        protected JCheckBox cbAutoIndentMode = new JCheckBox("Auto indent mode");

        private JButton bOk = new JButton("Ok");
        private JButton bCancel = new JButton("Cancel");
        private JButton bDefaults = new JButton("Defaults");

        protected abstract void initialize();

        /**
         * Set default values.
         */
        private void defaults() {
            tfTabSize.setText(Integer.toString(4));
            tfMargin.setText(Integer.toString(10));
            cbLineWrap.setSelected(false);
            rbWord.setSelected(true);
            cbAutoIndentMode.setSelected(true);
        }

        /**
         * Validate entered data
         * @return true if data is valid
         */
        private boolean validateInput() {
            int tabSize;
            int margin;
            try {
                tabSize = Integer.parseInt(tfTabSize.getText());
            } catch (NumberFormatException e) {
                showMessageDialog(this, "Tab size is not a number", "Error message", ERROR_MESSAGE);
                return false;
            }
            try {
                margin = Integer.parseInt(tfMargin.getText());
            } catch (NumberFormatException e) {
                showMessageDialog(this, "Margin is not a number", "Error message", ERROR_MESSAGE);
                return false;
            }
            if (tabSize < 1 || tabSize > 99) {
                showMessageDialog(this, "Tab size mast be in interval from 1 to 99", "Error message", ERROR_MESSAGE);
                return false;
            } else if (margin < 0 || margin > 99) {
                showMessageDialog(this, "Margin mast be in inteval from 0 to 99", "Error message", ERROR_MESSAGE);
                return false;
            } else {
                return true;
            }
        }

        /**
         * Establish program variables
         */
        protected abstract void establishValues();

        public TextOptionsDialog(JFrame owner) {
            super(owner);
            setTitle("Text options");
            JPanel panel = new JPanel();
            panel.setBorder(new EmptyBorder(5, 0, 5, 5));
            panel.setLayout(new FlowLayout());
            panel.add(getMainPanel());
            getContentPane().add(panel);
            getRootPane().setDefaultButton(bOk);
            pack();
            setResizable(false);
            setModal(true);
            setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        }

        private JPanel getMainPanel() {
            JPanel mainPanel = new JPanel();
            GridBagLayout mainLayout = new GridBagLayout();
            GridBagConstraints constraints = new GridBagConstraints();
            mainPanel.setLayout(mainLayout);

            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTH;
            constraints.insets = new Insets(0, 5, 0, 0);
            mainPanel.add(getEditingPanel(), constraints);

            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTH;
            constraints.insets = new Insets(5, 10, 0, 0);
            mainPanel.add(getButtonsPanel(), constraints);
            return mainPanel;
        }

        private JPanel getEditingPanel() {
            lTabSize.setLabelFor(tfTabSize);
            lTabSize.setDisplayedMnemonic(KeyEvent.VK_T);
            lMargin.setLabelFor(tfMargin);
            lMargin.setDisplayedMnemonic(KeyEvent.VK_M);
            cbAutoIndentMode.setMnemonic(KeyEvent.VK_A);
            cbAutoIndentMode.setToolTipText("Set auto indentation mode");
            JPanel editingPanel = new JPanel();
            Border editingBorder = new CompoundBorder(new TitledBorder("Text"), new EmptyBorder(0, 5, 5, 5));
            editingPanel.setBorder(editingBorder);
            editingPanel.setLayout(new GridBagLayout());
            GridBagConstraints c = new GridBagConstraints();
            c.gridx = 0;
            c.gridy = 0;
            c.anchor = GridBagConstraints.WEST;
            c.insets = new Insets(0, 5, 5, 5);
            editingPanel.add(lTabSize, c);
            c.gridx = 1;
            c.insets = new Insets(0, 5, 5, 5);
            editingPanel.add(tfTabSize, c);
            c.gridx = 0;
            c.gridy = 1;
            c.insets = new Insets(0, 5, 5, 5);
            editingPanel.add(lMargin, c);
            c.gridx = 1;
            c.insets = new Insets(0, 5, 5, 5);
            editingPanel.add(tfMargin, c);
            c.gridx = 0;
            c.gridy = 2;
            c.gridwidth = 2;
            c.insets = new Insets(0, 0, 5, 0);
            editingPanel.add(cbAutoIndentMode, c);
            c.gridy = 3;
            c.insets = new Insets(0, 0, 0, 0);
            editingPanel.add(getWrapPanel(), c);
            return editingPanel;
        }

        private JPanel getWrapPanel() {
            cbLineWrap.setMnemonic(KeyEvent.VK_L);
            rbCharacter.setMnemonic(KeyEvent.VK_H);
            rbWord.setMnemonic(KeyEvent.VK_W);
            JPanel wrapStylePanel = new JPanel();
            Border wrapStyleBorder = new TitledBorder("Wrap style");
            BoxLayout wrapStyleLayout = new BoxLayout(wrapStylePanel, BoxLayout.Y_AXIS);
            wrapStylePanel.setBorder(wrapStyleBorder);
            wrapStylePanel.setLayout(wrapStyleLayout);
            wrapStylePanel.add(rbCharacter);
            wrapStylePanel.add(Box.createVerticalStrut(5));
            wrapStylePanel.add(rbWord);
            bgWrapStyle.add(rbCharacter);
            bgWrapStyle.add(rbWord);

            JPanel wrapPanel = new JPanel();
            Border wrapBorder = new TitledBorder("Line wrap");
            BoxLayout wrapLayout = new BoxLayout(wrapPanel, BoxLayout.Y_AXIS);
            wrapPanel.setBorder(wrapBorder);
            wrapPanel.setLayout(wrapLayout);
            wrapPanel.add(cbLineWrap);
            wrapPanel.add(Box.createVerticalStrut(5));
            wrapPanel.add(wrapStylePanel);

            return wrapPanel;
        }

        private JPanel getButtonsPanel() {
            bOk.setToolTipText("Set options");
            bDefaults.setToolTipText("Select default values");
            bCancel.setToolTipText("Abort text options dialog");
            bOk.setMnemonic(KeyEvent.VK_O);
            bCancel.setMnemonic(KeyEvent.VK_C);
            bDefaults.setMnemonic(KeyEvent.VK_D);
            bOk.addActionListener(alOk);
            bCancel.addActionListener(alCancel);
            bDefaults.addActionListener(alDefaults);
            JPanel buttonsPanel = new JPanel();
            GridLayout buttonsLayout = new GridLayout(3, 1, 0, 5);
            buttonsPanel.setLayout(buttonsLayout);
            buttonsPanel.add(bOk);
            buttonsPanel.add(bCancel);
            buttonsPanel.add(bDefaults);
            return buttonsPanel;
        }

        private ActionListener alOk = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (validateInput()) {
                    establishValues();
                    setVisible(false);
                    dispose();
                }
            }
        };

        private ActionListener alCancel = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setVisible(false);
                dispose();
            }
        };

        private ActionListener alDefaults = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                defaults();
            }
        };

    }

    private class PlainTextOptionsDialog extends TextOptionsDialog {

        private PlainTextArea textComponent;

        protected void initialize() {
            tfTabSize.setText(Integer.toString(textComponent.getTabSize()));
            tfMargin.setText(Integer.toString(textComponent.getMargin().left));
            cbAutoIndentMode.setSelected(textComponent.isAutoIndentMode());
            cbLineWrap.setSelected(textComponent.getLineWrap());
            if (textComponent.getWrapStyleWord()) {
                rbWord.setSelected(true);
            } else {
                rbCharacter.setSelected(true);
            }
            cbLineWrap.addChangeListener(new ChangeListener() {
                public void stateChanged(ChangeEvent e) {
                    boolean isEnable = cbLineWrap.isSelected();
                    rbCharacter.setEnabled(isEnable);
                    rbWord.setEnabled(isEnable);
                }
            });
            boolean isWrapStyleEnabled = cbLineWrap.isSelected();
            rbCharacter.setEnabled(isWrapStyleEnabled);
            rbWord.setEnabled(isWrapStyleEnabled);
        }

        /**
         * Establish program variables
         */
        protected void establishValues() {
            int tabSize = Integer.parseInt(tfTabSize.getText());
            int margin = Integer.parseInt(tfMargin.getText());
            textComponent.setTabSize(tabSize);
            textComponent.setMargin(new Insets(margin, margin, margin, margin));
            textComponent.setLineWrap(cbLineWrap.isSelected());
            textComponent.setWrapStyleWord(rbWord.isSelected());
            textComponent.setAutoIndentMode(cbAutoIndentMode.isSelected());
            // repaint
            textComponent.repaint();
        }

        public PlainTextOptionsDialog(JFrame owner, PlainTextArea textComponent) {
            super(owner);
            this.textComponent = textComponent;
            initialize();
        }

    }

    private class MultiTextOptionsDialog extends TextOptionsDialog {

        private MultiTextComponent multiTextComponent;

        protected void initialize() {
            MultiTextOptions options = multiTextComponent.getMultiTextOptions();
            tfTabSize.setText(Integer.toString(options.getTabSize()));
            tfMargin.setText(Integer.toString(options.getMargin()));
            cbAutoIndentMode.setSelected(options.isAutoIndentMode());
            cbLineWrap.setSelected(options.getLineWrap());
            if (options.getWrapStyleWord()) {
                rbWord.setSelected(true);
            } else {
                rbCharacter.setSelected(true);
            }
            cbLineWrap.addChangeListener(new ChangeListener() {
                public void stateChanged(ChangeEvent e) {
                    boolean isEnable = cbLineWrap.isSelected();
                    rbCharacter.setEnabled(isEnable);
                    rbWord.setEnabled(isEnable);
                }
            });
            boolean isWrapStyleEnabled = cbLineWrap.isSelected();
            rbCharacter.setEnabled(isWrapStyleEnabled);
            rbWord.setEnabled(isWrapStyleEnabled);
        }

        /**
         * Establish program variables
         */
        protected void establishValues() {
            int tabSize = Integer.parseInt(tfTabSize.getText());
            int margin = Integer.parseInt(tfMargin.getText());
            MultiTextOptions options = multiTextComponent.getMultiTextOptions();
            options.setTabSize(tabSize);
            options.setMargin(margin);
            options.setLineWrap(cbLineWrap.isSelected());
            options.setWrapStyleWord(rbWord.isSelected());
            options.setAutoIndentMode(cbAutoIndentMode.isSelected());
            // repaint
            multiTextComponent.getCurrentTextEditor().getTextComponent().repaint();
        }

        public MultiTextOptionsDialog(JFrame owner, MultiTextComponent multiTextComponent) {
            super(owner);
            this.multiTextComponent = multiTextComponent;
            initialize();
        }

    }

    private static void establishBounds(JDialog dialog, Window owner) {
        Dimension d1 = dialog.getSize();
        Dimension d2 = owner.getSize();
        Dimension ds = dialog.getToolkit().getScreenSize();
        int x = Math.max((d2.width-d1.width)/2, 0);
        int y = Math.max((d2.height-d1.height)/2, 0);
        int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
        int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
        dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
    }

    public static void showPlainTextOptionsDialog(JFrame owner, PlainTextArea textComponent) {
        TextOptionsChooser chooser = new TextOptionsChooser();
        TextOptionsDialog dialog = chooser.new PlainTextOptionsDialog(owner, textComponent);
        establishBounds(dialog, owner);
        dialog.setVisible(true);
    }

    public static void showMultiTextOptionsDialog(JFrame owner, MultiTextComponent multiTextComponent) {
        TextOptionsChooser chooser = new TextOptionsChooser();
        TextOptionsDialog dialog = chooser.new MultiTextOptionsDialog(owner, multiTextComponent);
        establishBounds(dialog, owner);
        dialog.setVisible(true);
    }

}
