package neutrino.multitext;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.util.*;
import java.util.regex.Pattern;
import javax.swing.*;
import javax.swing.event.*;
import static javax.swing.JOptionPane.*;
import javax.swing.text.JTextComponent;
import neutrino.dialogs.*;
import neutrino.multitext.adapters.plain.*;
import neutrino.text.*;
import neutrino.text.components.plain.*;
import neutrino.text.components.syntax.*;

/**
 * Component for interaction with several text components. 
 * When contains greater then one text component changes view to tabbed pane otherwise
 * shows only text component
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class MultiTextComponent extends JComponent implements SessionSupport, NavigationSupport {

	protected ArrayList<TextEditorAdapter> m_components = null;
	private JTabbedPane m_tabbedPane = null;
	private ArrayList<EditorChangeListener> editorChangeListeners = null;
	private JPopupMenu popupMenu = null;
	private JPopupMenu popupMenuForTextComponents = null;
	private File currentDirectory = null;
    private Stack<File> closingStack = new Stack<File>();
    private Hashtable<String, ArrayList<BookmarkInformation>> m_bookmarks = null;
    private MultiTextOptions multiTextOptions = null;
    private ArrayList<Session> sessions = new ArrayList<Session>();
    private Session currentSession = new Session();
    private boolean isReadOnlyMode = false;

	public MultiTextComponent() {
		super();
		m_components = new ArrayList<TextEditorAdapter>();
        multiTextOptions = new MultiTextOptions(this, m_components);
		TextEditorAdapter textEditorAdapter = new PlainTextAreaEditorAdapter(); 
		m_components.add(textEditorAdapter);
		establishSettings(textEditorAdapter);
		m_tabbedPane = new JTabbedPane();
		setLayout(new BorderLayout());
		add(textEditorAdapter, BorderLayout.CENTER);
		
		editorChangeListeners = new ArrayList<EditorChangeListener>();
		m_tabbedPane.addChangeListener(new ChangeListener() {
			@Override
			public void stateChanged(ChangeEvent e) {
				fireEditorChanged(EditorChangeEventType.SELECTED);
			}
		});
		m_tabbedPane.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (popupMenu != null && e.getButton() == e.BUTTON3) {
					popupMenu.show(m_tabbedPane, e.getX(), e.getY());
				}
			}
		});
		createDefaultPopupMenu();
        m_bookmarks = new Hashtable<String, ArrayList<BookmarkInformation>>();
	}

    /**
     * Creates a new session
     */
    public void newSession() {
        Session session = new Session();
        this.currentSession = session;

    }

    /**
     * Returns true when the current session is a new session
     * @return boolean
     */
    public boolean isNewSession() {
        return !currentSession.isNamed();
    }

    /**
     * Returns true when can create a new session
     * @return boolean
     */
    public boolean canCreateSession() {
        return !isNewSession();
    }

    /**
     * Saves the session. Asks for a name of session when necessary
     */
    public void saveSession() {
        if (!isNewSession()) {
            currentSession.setTexts(getCurrentFiles());
        } else {
            // get the name
            JPanel panel = new JPanel();
            BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
            panel.setLayout(layout);
            panel.add(new JLabel("Enter the name of session"));
            panel.add(Box.createVerticalStrut(5));
            JTextField textField = new JTextField(20);
            panel.add(textField);
            int option = showConfirmDialog(this, panel, "Session name", YES_NO_OPTION, QUESTION_MESSAGE);
            if (option == NO_OPTION) return;
            if (textField.getText() == null || textField.getText().equals("")) return;
            currentSession.setName(textField.getText());
            currentSession.setTexts(getCurrentFiles());
            if (!sessions.contains(currentSession)) sessions.add(currentSession);
        }
    }

    /**
     * Saves the current session with a different name
     */
    public void saveSessionAs() {
        JPanel panel = new JPanel();
        BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
        panel.setLayout(layout);
        panel.add(new JLabel("Enter the name of session"));
        panel.add(Box.createVerticalStrut(5));
        JTextField textField = new JTextField(20);
        panel.add(textField);
        int option = showConfirmDialog(this, panel, "Session name", YES_NO_OPTION, QUESTION_MESSAGE);
        if (option == NO_OPTION) return;
        if (textField.getText() == null || textField.getText().equals("")) return;
        Session session = new Session();
        this.sessions.add(session);
        session.setName(textField.getText());
        session.setTexts(getCurrentFiles());
        this.currentSession = session;
    }

    /**
     * Returns true when the current session can be renamed
     * @return boolean
     */
    public boolean canRenameSession() {
        return !isNewSession();
    }

    /**
     * Renames the current session
     */
    public void renameSession() {
        if (!canRenameSession()) return;
        JPanel panel = new JPanel();
        BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
        panel.setLayout(layout);
        panel.add(new JLabel("Enter the name of session"));
        panel.add(Box.createVerticalStrut(5));
        JTextField textField = new JTextField(20);
        textField.setText(currentSession.getName());
        panel.add(textField);
        int option = showConfirmDialog(this, panel, "Session name", YES_NO_OPTION, QUESTION_MESSAGE);
        if (option == NO_OPTION) return;
        if (textField.getText() == null || textField.getText().equals("")) return;
        this.currentSession.setName(textField.getText());
    }

    /**
     * Returns true when the current session can be closed
     * @return boolean
     */
    public boolean canCloseSession() {
        return this.currentSession.isNamed();
    }

    /**
     * Closes all files and the current session
     */
    public void closeSession() {
        if (!canCloseSession()) return;
        if (multiTextOptions.isConfirmClosingSession()) {
            JPanel panel = new JPanel();
            BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
            panel.setLayout(layout);
            panel.add(new JLabel("Do you want to close the current session?"));
            panel.add(Box.createVerticalStrut(5));
            JCheckBox checkBox = new JCheckBox("Always close the session without prompt");
            panel.add(checkBox);
            int option = showConfirmDialog(getParent(), panel, "Confirm closing of session", YES_NO_OPTION, QUESTION_MESSAGE);
            multiTextOptions.setConfirmationClosingSession(!checkBox.isSelected());
            if (option == NO_OPTION) return;
        }
        closeAll();
        if (!isAllClosed()) return;
        sessions.remove(currentSession);
        newSession();
    }

    /**
     * Returns trues when all editors is closed
     * @return boolean
     */
    public boolean isAllClosed() {
        return isSingleTextComponent() && !getCurrentTextEditor().canClear();
    }

    /**
     * Returns true when there is sessions to open
     * @return boolean
     */
    public boolean hasSessions() {
        return this.sessions.size() > 0;
    }

    /**
     * Opens the given session
     * @param session
     */
    public void openSession(Session session) {
        closeAll();
        if (!isAllClosed()) return;
        openFiles(session.getTexts());
        this.currentSession = session;
    }

    /**
     * Returns the current session
     * @return Session
     */
    public Session getCurrentSession() {
        return this.currentSession;
    }

    /**
     * Sets the current session
     * @param session
     */
    public void setCurrentSession(Session session) {
        if (session == null) return;
        if (!sessions.contains(session) && session.isNamed()) sessions.add(session);
        this.currentSession = session;
    }

    /**
     * Sets all given sessions
     * @param sessions - ArrayList
     */
    public void setSessions(ArrayList<Session> sessions) {
        if (sessions == null) return;
        this.sessions = sessions;
    }

    /**
     * Returns all saved sessions
     * @return array list of session
     */
    public ArrayList<Session> getSessions() {
        return this.sessions;
    }

    /**
     * Sets the bookmarks for multi text component
     * @param bookmarks - Hashtbale
     */
    public void setBookmarks(Hashtable<String, ArrayList<BookmarkInformation>> bookmarks) {
        if (bookmarks == null) return;
        m_bookmarks = bookmarks;
        for (TextEditorAdapter adapter : m_components) {
            if (!(adapter instanceof BookmarkSupport)) continue;
            if (adapter.getTextEditor().isFileLoaded()) {
                String path = adapter.getTextEditor().getFile().getAbsolutePath();
                if (bookmarks.containsKey(path)) {
                    ((BookmarkSupport) adapter).setBookmarks(bookmarks.get(path));
                }
            }
        }
    }

    /**
     * Returns the bookmarks of multi text component
     * @return Hashtable
     */
    public Hashtable<String, ArrayList<BookmarkInformation>> getBookmarks() {
        for (TextEditorAdapter adapter : m_components) {
            if (!(adapter instanceof BookmarkSupport)) continue;
            if (adapter.getTextEditor().isFileLoaded()) {
                String path = adapter.getTextEditor().getFile().getAbsolutePath();
                if (((BookmarkSupport) adapter).hasBookmarks()) {
                    m_bookmarks.put(path, ((BookmarkSupport) adapter).getBookmarks());
                } else if (m_bookmarks.containsKey(path)) m_bookmarks.remove(path);
            }
        }
        return m_bookmarks;
    }

    private void addBookmarks(TextEditorAdapter adapter) {
        if (!(adapter instanceof BookmarkSupport)) return;
        if (adapter.getTextEditor().isFileLoaded()) {
            if (m_bookmarks.containsKey(adapter.getTextEditor().getFile().getAbsolutePath())
                    && !((BookmarkSupport) adapter).hasBookmarks()) {
                m_bookmarks.remove(adapter.getTextEditor().getFile().getAbsolutePath());
            }
            if (((BookmarkSupport) adapter).hasBookmarks()) {
                m_bookmarks.put(adapter.getTextEditor().getFile().getAbsolutePath(), ((BookmarkSupport) adapter).getBookmarks());
            }
        }
    }

    /**
     * Returns true when the multi text component has a saved bookmarks for reopening
     * @return boolean
     */
    public boolean hasFavorites() {
        Enumeration<String> iterator = getBookmarks().keys();
        while (iterator.hasMoreElements()) {
            String path = iterator.nextElement();
            File file = new File(path);
            if (!file.exists() || !file.isFile()) {
                m_bookmarks.remove(path);
                continue;
            }
        }
        return m_bookmarks.size() > 0;
    }

    /**
     * Returns the options of multi text component
     * @return MultiTextOptions
     */
    public MultiTextOptions getMultiTextOptions() {
        return this.multiTextOptions;
    }

	protected void createDefaultPopupMenu() {
		// create meu items
		final JMenu pmNewEditor = EditorChooser.createEditorMenu(this);
        final JMenuItem pmiActivatePrevious = new JMenuItem("Activate Previous");
        final JMenuItem pmiActivateNext = new JMenuItem("Activate Next");
        final JMenuItem pmiMoveLeft = new JMenuItem("Move Left");
        final JMenuItem pmiMoveRight = new JMenuItem("Move Right");
		final JMenuItem pmiClose = new JMenuItem("Close");
		final JMenuItem pmiCloseOthers = new JMenuItem("Close Others");
		final JMenuItem pmiCloseAll = new JMenuItem("Close All");
		// create popup menu
		popupMenu = new JPopupMenu();
		popupMenu.addPopupMenuListener(new PopupMenuListener() {
			@Override
			public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                pmiActivatePrevious.setEnabled(canActivatePrevious());
                pmiActivateNext.setEnabled(canActivateNext());
                pmiMoveLeft.setEnabled(canMoveLeft());
                pmiMoveRight.setEnabled(canMoveRight());
				pmiClose.setEnabled(canClose());
				pmiCloseOthers.setEnabled(canCloseOthers());
				pmiCloseAll.setEnabled(canCloseAll());
			}
			@Override
			public void popupMenuWillBecomeInvisible(PopupMenuEvent e) { }
			@Override
			public void popupMenuCanceled(PopupMenuEvent e) { }
		});
		popupMenu.add(pmNewEditor);
		popupMenu.addSeparator();
        popupMenu.add(pmiActivatePrevious);
        popupMenu.add(pmiActivateNext);
        popupMenu.addSeparator();
        popupMenu.add(pmiMoveLeft);
        popupMenu.add(pmiMoveRight);
        popupMenu.addSeparator();
		popupMenu.add(pmiClose);
		popupMenu.add(pmiCloseOthers);
		popupMenu.add(pmiCloseAll);
		// build mnemonics
		pmNewEditor.setMnemonic(KeyEvent.VK_N);
        pmiActivatePrevious.setMnemonic(KeyEvent.VK_P);
        pmiActivateNext.setMnemonic(KeyEvent.VK_N);
        pmiMoveLeft.setMnemonic(KeyEvent.VK_L);
        pmiMoveRight.setMnemonic(KeyEvent.VK_R);
		pmiClose.setMnemonic(KeyEvent.VK_C);
		pmiCloseOthers.setMnemonic(KeyEvent.VK_O);
		pmiCloseAll.setMnemonic(KeyEvent.VK_A);		
		// build actions
		ActionListener listener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
                if (e.getSource() == pmiActivatePrevious) {
                    activatePrevious();
                } else if (e.getSource() == pmiActivateNext) {
					activateNext();
				} else if (e.getSource() == pmiMoveLeft) {
                    moveLeft();
                } else if (e.getSource() == pmiMoveRight) {
                    moveRight();
                } else if (e.getSource() == pmiClose) {
                    close();
                } else if (e.getSource() == pmiCloseOthers) {
					closeOthers();
				} else if (e.getSource() == pmiCloseAll) {
					closeAll();
				}
			}
		};
        pmiActivatePrevious.addActionListener(listener);
        pmiActivateNext.addActionListener(listener);
        pmiMoveLeft.addActionListener(listener);
        pmiMoveRight.addActionListener(listener);
		pmiClose.addActionListener(listener);
		pmiCloseOthers.addActionListener(listener);
		pmiCloseAll.addActionListener(listener);		
	}
	
	public void setPopupMenu(JPopupMenu popupMenu) {
		this.popupMenu = popupMenu;
	}
	
	public void setPopupMenuForTextComponents(JPopupMenu popupMenu) {
		if (popupMenu == null) return;
		popupMenuForTextComponents = popupMenu;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.getTextComponent() instanceof ITextComponent) {
                ITextComponent textComponent = (ITextComponent) textEditor.getTextComponent();
				textComponent.setPopupMenu(popupMenuForTextComponents);
			}
		}
	}
	
	private boolean isTextEmpty(JTextComponent textComponent) {
		return textComponent.getDocument().getLength() == 0;
	}
	
	/**
	 * Returns true when current tab can be closed
	 * @return boolean
	 */
	public boolean canClose() {
		TextEditor textEditor = getCurrentTextEditor();
		return !isSingleTextComponent() || textEditor.isFileLoaded() || !isTextEmpty(textEditor.getTextComponent());
	}
	
	/**
	 * Returns true when others tabs may be closed
	 * @return boolean
	 */
	public boolean canCloseOthers() {
		return !isSingleTextComponent();
	}
	
	/**
	 * Returns true when all tab may be closed
	 * @return boolean
	 */
	public boolean canCloseAll() {
		TextEditor textEditor = getCurrentTextEditor();
		return !isSingleTextComponent() || textEditor.isFileLoaded() || !isTextEmpty(textEditor.getTextComponent());
	}
	
	/**
	 * Returns the current text editor
	 * @return TextEditor
	 */
	public TextEditor getCurrentTextEditor() {
		return getCurrentEditorAdapter().getTextEditor();
	}
	
	/**
	 * Returns the current text editor adapter
	 * @return TextEditorAdapter
	 */
	public TextEditorAdapter getCurrentEditorAdapter() {
		if (isSingleTextComponent()) {
			return m_components.get(0);
		} else {
			int index = m_tabbedPane.getSelectedIndex();
			if (index == -1) return m_components.get(0);
			return (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
		}
	}

	protected void establishSettings(TextEditorAdapter textEditorAdapter) {
        TextEditor textEditor = textEditorAdapter.getTextEditor();
        int margin = multiTextOptions.getMargin();
		textEditor.getTextComponent().setMargin(new Insets(margin, margin, margin, margin));
		textEditor.getBackupManager().setEnabled(multiTextOptions.isPersistBackups());
		textEditor.setEncoding(multiTextOptions.getEncoding());
		textEditor.setCurrentDirectory(currentDirectory);
        textEditor.setConfirmationReloading(multiTextOptions.isConfirmReloading());
        textEditor.setConfirmationRemoving(multiTextOptions.isConfirmRemoving());
        textEditor.setConfirmationOverwriting(multiTextOptions.isConfirmOverwriting());
		textEditor.removeFileChangeListener(fileListener);
		textEditor.addFileChangeListener(fileListener);
		if (textEditor.getTextComponent() instanceof JTextArea) {
			JTextArea textComponent = (JTextArea) textEditor.getTextComponent();
			textComponent.setFont(multiTextOptions.getFont());
			textComponent.setForeground(multiTextOptions.getForeground());
			textComponent.setCaretColor(multiTextOptions.getForeground());
			textComponent.setSelectionColor(multiTextOptions.getForeground());
			textComponent.setBackground(multiTextOptions.getBackground());
			textComponent.setSelectedTextColor(multiTextOptions.getBackground());
			textComponent.setTabSize(multiTextOptions.getTabSize());
			textComponent.setLineWrap(multiTextOptions.getLineWrap());
			textComponent.setWrapStyleWord(multiTextOptions.getWrapStyleWord());
		}
		if (textEditor.getTextComponent() instanceof PlainTextArea) {
			PlainTextArea textComponent = (PlainTextArea) textEditor.getTextComponent();
			textComponent.setAutoIndentMode(multiTextOptions.isAutoIndentMode());
		}
		if (textEditor.getTextComponent() instanceof PlainTextComponent) {
			PlainTextComponent textComponent = (PlainTextComponent) textEditor.getTextComponent();
			textComponent.setAutoIndentMode(multiTextOptions.isAutoIndentMode());
			textComponent.setFont(multiTextOptions.getFont());
			textComponent.setForeground(multiTextOptions.getForeground());
			textComponent.setCaretColor(multiTextOptions.getForeground());
			textComponent.setSelectionColor(multiTextOptions.getForeground());
			textComponent.setBackground(multiTextOptions.getBackground());
			textComponent.setSelectedTextColor(multiTextOptions.getBackground());

		}
        if (textEditor.getTextComponent() instanceof SyntaxTextComponent) {
            SyntaxTextComponent syntaxTextComponent = (SyntaxTextComponent) textEditor.getTextComponent();
            syntaxTextComponent.setCodeFoldingEnabled(multiTextOptions.isCodeFoldingEnabled());
            syntaxTextComponent.setHighlightCurrentLine(multiTextOptions.isCurrentLineHighlighted());
            syntaxTextComponent.setBracketMatchingEnabled(multiTextOptions.isBracketMatchingAnimated());
            syntaxTextComponent.setMarkOccurrences(multiTextOptions.isOccurrencesMarked());
            syntaxTextComponent.setPaintTabLines(multiTextOptions.isTabLinesPainted());
            multiTextOptions.setThemeForTextComponent(syntaxTextComponent);
        }
		if (textEditor.getTextComponent() instanceof ITextComponent) {
            ITextComponent textComponent = (ITextComponent) textEditor.getTextComponent();
			if (popupMenuForTextComponents != null) {
				textComponent.setPopupMenu(popupMenuForTextComponents);
			}
		}
        if (textEditorAdapter instanceof BookmarkSupport) {
            BookmarkSupport syntaxTextEditorAdapter = (BookmarkSupport) textEditorAdapter;
            syntaxTextEditorAdapter.setLineNumbersEnabled(multiTextOptions.isLineNumbersShowed());
            syntaxTextEditorAdapter.setBookmarksEnabled(multiTextOptions.isBookmarksEnabled());
            BookmarkChangeListener bookmarkListener = new BookmarkChangeListener(textEditorAdapter);
            textEditorAdapter.getTextEditor().addFileChangeListener(bookmarkListener);
        }
	}

	/**
	 * Returns true when can changed current editor
	 * @return boolean
	 */
	protected boolean canChangeEditor() {
		TextEditor currentTextEditor = getCurrentTextEditor();
		return !currentTextEditor.canClear();
	}
	
	/**
	 * Replaces current text editor component with new one
	 * @param textEditorAdapter - TextEditorAdapter 
	 */
	protected void changeEditor(TextEditorAdapter textEditorAdapter) {
		TextEditor currentTextEditor = getCurrentTextEditor();
		currentTextEditor.getBackupManager().turnOff();
		establishSettings(textEditorAdapter);
		if (isSingleTextComponent()) {
			TextEditorAdapter firstComponentAdapter = m_components.get(0);
			remove(firstComponentAdapter);
			m_components.clear();
			m_components.add(textEditorAdapter);
			add(textEditorAdapter, BorderLayout.CENTER);
			textEditorAdapter.revalidate();
			textEditorAdapter.repaint();
		} else {
			TextEditorAdapter currentTextEditorAdapter = getCurrentEditorAdapter(); 
			int index = m_components.indexOf(currentTextEditorAdapter);
			m_components.set(index, textEditorAdapter);
			int selectedIndex = m_tabbedPane.getSelectedIndex();
			m_tabbedPane.removeTabAt(selectedIndex);
			m_tabbedPane.insertTab(createTitleForTextComponent(textEditorAdapter.getTextEditor()), null, textEditorAdapter, null, selectedIndex);
			m_tabbedPane.setSelectedIndex(selectedIndex);
			m_tabbedPane.revalidate();
			m_tabbedPane.repaint();
		}
	}

    /**
     * Replaces the current text editor with new one
     * precodition - text is saved
     * @param textEditorAdapter - TextEditorAdapter
     */
    public void replaceEditor(TextEditorAdapter textEditorAdapter) {
        TextEditorAdapter currentTextEditorAdapter = getCurrentEditorAdapter();
        File file = currentTextEditorAdapter.getTextEditor().getFile();
        changeEditor(textEditorAdapter);
        if (file != null) textEditorAdapter.getTextEditor().open(file);
        fireEditorChanged(EditorChangeEventType.REPLACED);
    }
	
	/**
	 * Creates new tab with empty text component
	 */
	public void newEditor() {
		newEditor(new PlainTextAreaEditorAdapter());
	}
	
	public void newEditor(TextEditorAdapter textEditorAdapter) {
		if (canChangeEditor()) {
			changeEditor(textEditorAdapter);
		} else {
			TextEditor textEditor = textEditorAdapter.getTextEditor();
			establishSettings(textEditorAdapter);
			if (isSingleTextComponent()) {
				TextEditorAdapter firstComponentAdapter = m_components.get(0);
				m_tabbedPane.addTab(createTitleForTextComponent(m_components.get(0).getTextEditor()), firstComponentAdapter);
				m_tabbedPane.addTab(createTitleForTextComponent(textEditor), textEditorAdapter);
				remove(firstComponentAdapter);
				add(m_tabbedPane, BorderLayout.CENTER);
				m_tabbedPane.setSelectedIndex(1);
			} else {
				m_tabbedPane.addTab(createTitleForTextComponent(textEditor), textEditorAdapter);		
				m_tabbedPane.setSelectedIndex(m_tabbedPane.getTabCount() - 1);
			}
			m_components.add(textEditorAdapter);
			m_tabbedPane.revalidate();
			m_tabbedPane.repaint();
		}
		fireEditorChanged(EditorChangeEventType.ADDED);
	}
	
	public static String createTitleForTextComponent(TextEditor textEditor) {
		if (!textEditor.isFileLoaded()) {
			return "noname";
		} else {
			String fname = textEditor.getFile().getName();
			final int MAX_LENGTH = 40;
			return  (fname.length() >= MAX_LENGTH) ? (fname.substring(0, MAX_LENGTH - 1) + "...") : fname;
		}
	}
	
	/**
	 * Returns true when only one text component is created
	 * @return boolean
	 */
	public boolean isSingleTextComponent() {
		return m_components.size() == 1;
	}
	
	protected void replaceTabbedPaneOnScrollPane() {
		if (m_tabbedPane.getTabCount() == 1) {
			TextEditorAdapter firstComponentAdapter = m_components.get(0);
			m_tabbedPane.removeTabAt(0);
			remove(m_tabbedPane);
			add(firstComponentAdapter, BorderLayout.CENTER);
			firstComponentAdapter.revalidate();
			firstComponentAdapter.repaint();
		} else {
			m_tabbedPane.revalidate();
			m_tabbedPane.repaint();
		}
	}
	
	/**
	 * Closes current tab. Clears text component when it's single
	 */
	public void close() {
		TextEditorAdapter textEditorAdapter = getCurrentEditorAdapter();
		TextEditor textEditor = textEditorAdapter.getTextEditor();
		if (!textEditor.isTextSaved()) return;
        closingStack.push(textEditor.getFile());
        addBookmarks(textEditorAdapter);
		if (isSingleTextComponent()) {
			textEditor.clear();
		} else {
			textEditor.getBackupManager().turnOff();
			m_components.remove(textEditorAdapter);
			int index = m_tabbedPane.getSelectedIndex();
			if (index != -1) m_tabbedPane.removeTabAt(index);
			replaceTabbedPaneOnScrollPane();
			getCurrentTextEditor().getTextComponent().grabFocus();
		}
		fireEditorChanged(EditorChangeEventType.CLOSED);
		System.gc();
	}
	
	/**
	 * Closes all tabs without current. Returns true when success
	 * @return boolean
	 */
	public boolean closeOthers() {
		if (isSingleTextComponent()) return true;
		TextEditor currentTextComponent = getCurrentTextEditor();
		for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
			TextEditorAdapter textEditorAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
			TextEditor textEditor = textEditorAdapter.getTextEditor();
			if (textEditor == currentTextComponent) continue;
			if (textEditor.conditionOfSaving()) {
				activateTab(textEditor);
				if (!textEditor.isTextSaved()) return false;
			}
			textEditor.getBackupManager().turnOff();
            closingStack.push(textEditor.getFile());
            addBookmarks(textEditorAdapter);
			m_components.remove(textEditorAdapter);
			m_tabbedPane.removeTabAt(index);
			index--;
			fireEditorChanged(EditorChangeEventType.CLOSED);
		}
		replaceTabbedPaneOnScrollPane();
		getCurrentTextEditor().getTextComponent().grabFocus();
		System.gc();
		return true;
	}
	
	/**
	 * Closes all tabs. Clears current text component
	 */
	public void closeAll() {
		if (!closeOthers()) return;
		TextEditor textEditor = getCurrentTextEditor();
		if (!textEditor.isTextSaved()) return;
        closingStack.push(textEditor.getFile());
        addBookmarks(getCurrentEditorAdapter());
		textEditor.clear();
		getCurrentTextEditor().getTextComponent().grabFocus();
		fireEditorChanged(EditorChangeEventType.CLOSED);
	}
	
	/**
	 * Returns the array of text editors in order of their appearance in the tabbed pane
	 * @return array of text component
	 */
	public TextEditor[] getTextEditors() {
		TextEditor[] array = new TextEditor[m_components.size()];
		if (isSingleTextComponent()) {
			array[0] = m_components.get(0).getTextEditor();
		} else {
			for (int i = 0; i < m_tabbedPane.getTabCount(); i++) {
				TextEditorAdapter componentAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(i);
				array[i] = componentAdapter.getTextEditor();
			}
		}
		return array;
	}

    /**
     * Returns the array of text editor adapters in order of their appearance in the tabbed pane
     * @return array of text editor adapters
     */
    public TextEditorAdapter[] getTextEditorAdapters() {
        TextEditorAdapter[] array = new TextEditorAdapter[m_components.size()];
        if (isSingleTextComponent()) {
            array[0] = m_components.get(0);
        } else {
            for (int i = 0; i < m_tabbedPane.getTabCount(); i++) {
                array[i] =  (TextEditorAdapter) m_tabbedPane.getComponentAt(i);
            }
        }
        return array;
    }

    /**
     * Returns true when the current editor can be duplicated
     * @return boolean
     */
    public boolean canDuplicate() {
        return getCurrentTextEditor().canClear();
    }

    /**
     * Creates the same editor as a current and loads the same file
     */
    public void duplicate() {
        if (!getCurrentTextEditor().isTextSaved()) return;
        try {
            String className = getCurrentEditorAdapter().getClass().getCanonicalName();
            File file = getCurrentTextEditor().getFile();
            TextEditorAdapter adapter = (TextEditorAdapter) Class.forName(className).newInstance();
            newEditor(adapter);
            getCurrentTextEditor().open(file);
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when tab with editor is closed before
     * @return boolean
     */
    public boolean canReopenClosed() {
        return !closingStack.empty();
    }

    /**
     * Reopens closed before tab with an editor
     */
    public void reopenClosed() {
        if (!canReopenClosed()) return;
        File file = closingStack.pop();
        String fname = file.getName();
        String extension = fname.substring(fname.lastIndexOf('.') + 1);
        String className =  EditorAdapterFactory.getTextEditorAdapterClassNameForFileExtension(extension);
        try {
            TextEditorAdapter adapter = (TextEditorAdapter) Class.forName(className).newInstance();
            newEditor(adapter);
            getCurrentTextEditor().open(file);
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
    }

	/**
	 * Activates tab with given text editor
	 * @param textEditor - a text component
	 */
	public void activateTab(TextEditor textEditor) {
		if (textEditor == null) return;
		for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
			TextEditorAdapter textEditorAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
			if (textEditorAdapter.getTextEditor() == textEditor) {
				m_tabbedPane.setSelectedIndex(index);
				textEditor.getTextComponent().grabFocus();
				fireEditorChanged(EditorChangeEventType.SELECTED);
				return;
			}
		}
	}
	
	/**
	 * Activates tab with given file
	 * @param file - a File
	 */
	public void activateTab(File file) {
		if (file == null && isSingleTextComponent()) return;
		for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
			TextEditorAdapter textEditorAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
			if (textEditorAdapter.getTextEditor().isFileLoaded() 
					&&  textEditorAdapter.getTextEditor().getFile().equals(file)) {
				m_tabbedPane.setSelectedIndex(index);
				textEditorAdapter.getTextEditor().getTextComponent().grabFocus();
				fireEditorChanged(EditorChangeEventType.SELECTED);
				return;
			}
		}
	}

    /**
     * Returns true when the file is opened by multi text component
     * @param file - File
     * @return boolean
     */
    public boolean canActivate(File file) {
        for (TextEditorAdapter adapter : m_components) {
            if (adapter.getTextEditor().isFileLoaded() && adapter.getTextEditor().getFile().equals(file)) return true;
        }
        return false;
    }

    /**
     * Returns true when other text editor can be activated
     * @return boolean
     */
    public boolean canActivate() {
        return !isSingleTextComponent();
    }

    /**
     * Returns true when the previous to current text editor can be activated
     * @return boolean
     */
    public boolean canActivatePrevious() {
        return m_components.indexOf(getCurrentEditorAdapter()) > 0;
    }

    /**
     * Activates the previous to current text editor
     */
    public void activatePrevious() {
        if (!canActivatePrevious()) return;
        int currentIndex = m_components.indexOf(getCurrentEditorAdapter());
        TextEditorAdapter adapter = m_components.get(currentIndex - 1);
        activateTab(adapter.getTextEditor());
    }

    /**
     * Returns true when the next to current text editor can be activated
     * @return boolean
     */
    public boolean canActivateNext() {
        return m_components.indexOf(getCurrentEditorAdapter()) < m_components.size() - 1;
    }

    /**
     * Activates the next to current text editor
     */
    public void activateNext() {
        if (!canActivateNext()) return;
        int currentIndex = m_components.indexOf(getCurrentEditorAdapter());
        TextEditorAdapter adapter = m_components.get(currentIndex + 1);
        activateTab(adapter.getTextEditor());
    }

    /**
     * Returns true when some editor with modified text exists
     * @return boolean
     */
    public boolean canActivateModified() {
        for (TextEditorAdapter adapter : m_components) {
            if (adapter.getTextEditor().getTextComponent() instanceof ITextComponent) {
                ITextComponent textComponent = (ITextComponent) adapter.getTextEditor().getTextComponent();
                if (textComponent.isTextChanged()) return true;
            }
        }
        return false;
    }

    /**
     * Activates first editor with modified text
     */
    public void activateModified() {
        if (!canActivateModified()) return;
        boolean isFoundCurrent = false;
        for (TextEditorAdapter adapter : m_components) {
            if (adapter == getCurrentEditorAdapter()) {
                isFoundCurrent = true;
                continue;
            }
            if (isFoundCurrent) {
                if (adapter.getTextEditor().getTextComponent() instanceof ITextComponent) {
                    ITextComponent textComponent = (ITextComponent) adapter.getTextEditor().getTextComponent();
                    if (textComponent.isTextChanged()) {
                        activateTab(adapter.getTextEditor());
                        return;
                    }
                }
            }
        }
        for (TextEditorAdapter adapter : m_components) {
            if (adapter == getCurrentEditorAdapter()) return;
            if (adapter.getTextEditor().getTextComponent() instanceof ITextComponent) {
                ITextComponent textComponent = (ITextComponent) adapter.getTextEditor().getTextComponent();
                if (textComponent.isTextChanged()) {
                    activateTab(adapter.getTextEditor());
                    return;
                }
            }
        }
    }

    /**
     * Returns true when exists the editor with no loaded file
     * @return boolean
     */
    public boolean canActivateUnloaded() {
        for (TextEditorAdapter adapter : m_components) {
            if (!adapter.getTextEditor().isFileLoaded()) return true;
        }
        return false;
    }

    /**
     * Activates first editor without loaded file
     */
    public void activateUnloaded() {
        if (!canActivateUnloaded()) return;
        boolean isFoundCurrent = false;
        for (TextEditorAdapter adapter : m_components) {
            if (adapter == getCurrentEditorAdapter()) {
                isFoundCurrent = true;
                continue;
            }
            if (isFoundCurrent) {
                if (!adapter.getTextEditor().isFileLoaded()) {
                    activateTab(adapter.getTextEditor());
                    return;
                }
            }
        }
        for (TextEditorAdapter adapter : m_components) {
            if (adapter == getCurrentEditorAdapter()) return;
            if (!adapter.getTextEditor().isFileLoaded()) {
                activateTab(adapter.getTextEditor());
                return;
            }
        }
    }

    /**
     * Returns true when exists the editor with changed file
     * @return boolean
     */
    public boolean canActivateReplaced() {
        for (TextEditorAdapter adapter : m_components) {
            if (adapter.getTextEditor().isFileChanged()) return true;
        }
        return false;
    }

    /**
     * Activates first editor with changed file
     */
    public void activateReplaced() {
        if (!canActivateReplaced()) return;
        boolean isFoundCurrent = false;
        for (TextEditorAdapter adapter : m_components) {
            if (adapter == getCurrentEditorAdapter()) {
                isFoundCurrent = true;
                continue;
            }
            if (isFoundCurrent) {
                if (adapter.getTextEditor().isFileChanged()) {
                    activateTab(adapter.getTextEditor());
                    return;
                }
            }
        }
        for (TextEditorAdapter adapter : m_components) {
            if (adapter == getCurrentEditorAdapter()) return;
            if (adapter.getTextEditor().isFileChanged()) {
                activateTab(adapter.getTextEditor());
                return;
            }
        }
    }

    /**
     * Returns true when the current text editor can be moved to the left
     * @return  boolean
     */
    public boolean canMoveLeft() {
        return m_components.indexOf(getCurrentEditorAdapter()) > 0;
    }

    /**
     * Moves the current text component on the one position to the left in the tabbed  panel
     */
    public void moveLeft() {
        if (!canMoveLeft()) return;
        int currentIndex = m_components.indexOf(getCurrentEditorAdapter());
        TextEditorAdapter adapter = m_components.get(currentIndex - 1);
        String title = m_tabbedPane.getTitleAt(currentIndex - 1);
        m_components.set(currentIndex - 1, m_components.get(currentIndex));
        m_components.set(currentIndex, adapter);
        m_tabbedPane.removeTabAt(currentIndex - 1);
        m_tabbedPane.insertTab(title, null, adapter, null, currentIndex);
    }

    /**
     * Returns true when the current text editor can be moved to the right
     * @return  boolean
     */
    public boolean canMoveRight() {
        return m_components.indexOf(getCurrentEditorAdapter()) < m_components.size() - 1;
    }

    /**
     * Moves the current text component on the one position to the right in the tabbed  panel
     */
    public void moveRight() {
        if (!canMoveRight()) return;
        int currentIndex = m_components.indexOf(getCurrentEditorAdapter());
        TextEditorAdapter adapter = m_components.get(currentIndex + 1);
        String title = m_tabbedPane.getTitleAt(currentIndex + 1);
        m_components.set(currentIndex + 1, m_components.get(currentIndex));
        m_components.set(currentIndex, adapter);
        m_tabbedPane.removeTabAt(currentIndex + 1);
        m_tabbedPane.insertTab(title, null, adapter, null, currentIndex);
    }

	/**
	 * Returns true when all text components is empty
	 * @return boolean
	 */
	public boolean isTextsEmpty() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (!isTextEmpty(textEditor.getTextComponent())) return false;
		}
		return true;
	}
	
	public ArrayList<TextInfo> getCurrentFiles() {
		ArrayList<TextInfo> list = new ArrayList<TextInfo>();
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditorAdapter textEditorAdapter = iterator.next();
			TextEditor textEditor = textEditorAdapter.getTextEditor();
			if (textEditor.isFileLoaded()) {
				TextInfo textInfo = new TextInfo();
				textInfo.setFile(textEditor.getFile());
				textInfo.setCaretPosition(textEditor.getTextComponent().getCaretPosition());
				textInfo.setAdapterClassName(textEditorAdapter.getClass().getCanonicalName());
				list.add(textInfo);
			}
		}
		return list;
	}
	
	public void openFiles(String[] fileNames) {
		ArrayList<TextInfo> files = new ArrayList<TextInfo>();
		for (int i = 0; i < fileNames.length; i++) {
			TextInfo text = new TextInfo();
			text.setFile(new File(fileNames[i]));
		}
		openFiles(files);
	}
	
	public void openFiles(ArrayList<TextInfo> files) {
		if (files == null || files.size() == 0) return;
		Iterator<TextInfo> iterator = files.iterator();
		while (iterator.hasNext()) {
			TextInfo text = iterator.next();
			if (text.getFile().exists() && text.getFile().isFile()) {
				try {
					TextEditorAdapter textEditorAdapter = (TextEditorAdapter) Class.forName(text.getAdapterClassName()).newInstance();
					newEditor(textEditorAdapter);
                    TextEditor currentTextEditor = getCurrentTextEditor();
					currentTextEditor.open(text.getFile());
                    try {
                        currentTextEditor.getTextComponent().setCaretPosition(text.getCaretPosition());
                    } catch (IllegalArgumentException e) {
                        // don't set caret position
                    }
				} catch (Exception e) {
					// skip file opening
				}
			}
		}
	}

    public void openFiles(File[] files) {
        ArrayList<TextInfo> infos = new ArrayList<TextInfo>();
        for (int i = 0; i < files.length; i++) {
            TextInfo info = new TextInfo();
            info.setFile(files[i]);
            String fname = files[i].getName();
            String extension = fname.substring(fname.lastIndexOf('.') + 1);
            info.setAdapterClassName(EditorAdapterFactory.getTextEditorAdapterClassNameForFileExtension(extension));
            infos.add(info);
        }
        openFiles(infos);
    }

    public void openFiles() {
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        fileChooser.setMultiSelectionEnabled(true);
        fileChooser.setAcceptAllFileFilterUsed(true);
        fileChooser.setCurrentDirectory(getCurrentDirectory());
        if (fileChooser.showOpenDialog(getParent()) == JFileChooser.CANCEL_OPTION) return;
        File[] files = fileChooser.getSelectedFiles();
        openFiles(files);
    }

    private java.io.FileFilter fileFilter = new java.io.FileFilter() {
        @Override
        public boolean accept(File file) {
            String regex = fileNamePattern;
            regex = regex.replaceAll("\\.", "\\.");
            regex = regex.replaceAll("\\*", ".*");
            regex = regex.replaceAll("\\?", ".?");
            boolean isMatched = Pattern.matches(regex, file.getName());
            if (file.exists() && file.isFile() && isMatched)
                return true;
            else
                return false;
        }
    };

    public void openDirectoryRecursively(File directory) {
        File[] files = directory.listFiles();
        for (int i = 0; i < files.length; i++) {
            if (files[i].isDirectory()) {
                openDirectoryRecursively(files[i]);
            }
        }
        openFiles(directory.listFiles(fileFilter));
    }

    /**
     * Used in fileFilter and openDirectory()
     */
    private String fileNamePattern = "";

    public void openDirectory() {
        File directory = DirectoryChooser.showDirectoryDialog(this.getParent(), getCurrentDirectory());
        if (directory == null) return;
        File[] files = directory.listFiles(fileFilter);    // (java.io.FileFilter)
        JPanel panel = new JPanel();
        BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
        panel.setLayout(layout);
        JCheckBox checkBox = new JCheckBox("Open sub directories recursively");
        panel.add(checkBox);
        panel.add(Box.createVerticalStrut(5));
        panel.add(new JLabel("Pattern to match the file name (symbols * and ? can be used)"));
        JTextField textField = new JTextField(20);
        panel.add(Box.createVerticalStrut(5));
        panel.add(textField);
        textField.setText("*.*");
        int option = showConfirmDialog(this.getParent(), panel, "Open directory", YES_NO_OPTION, QUESTION_MESSAGE);
        if (option == NO_OPTION) return;
        fileNamePattern = textField.getText();
        if (checkBox.isSelected()) {
            openDirectoryRecursively(directory);
        } else {
            openFiles(files);
        }
    }

    /**
     * Restores the texts of backups from the list of files in the multi text component
     * @param backups - array of the files
     */
	public void restoreBackups(File[] backups) {
		if (backups == null || backups.length == 0) return;
		for (int i = 0; i < backups.length; i++) {
			File file = backups[i];
			if (file.exists() && file.isFile()) {
				String className = DocumentRestorer.getTextComponentClassName(file);
				newEditor(EditorAdapterFactory.createTextEditorAdapterForTextComponent(className));
				getCurrentTextEditor().restore(file);
			}
		}		
	}
	
	/**
	 * Returns true when can save at least one text
	 * @return boolean
	 */
	public boolean canSaveAll() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.canSave()) return true;
		}
		return false;
	}
	
	/**
	 * Saves all texts. Returns true when success
	 * @return boolean
	 */
	public boolean saveAll() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.canSave()) { 
				if (!textEditor.save()) return false;
			}
		}
		return true;
	}

    /**
     * Returns true when the texts can be synchronized with a file system
     * @return boolean
     */
    public boolean canSynchronize() {
        for (TextEditorAdapter adapter : m_components) {
            TextEditor textEditor = adapter.getTextEditor();
            if (textEditor.canReload()) return true;
        }
        return false;
    }

    /**
     * Synchronizes the edited texts with a file system
     */
    public void synchronize() {
        boolean isConfirmReloading = multiTextOptions.isConfirmReloading();
        multiTextOptions.setConfirmationReloading(false);
        for (TextEditorAdapter adapter : m_components) {
            TextEditor textEditor = adapter.getTextEditor();
            if (textEditor.canReload()) textEditor.reload();
        }
        multiTextOptions.setConfirmationReloading(isConfirmReloading);
    }
	
	/**
	 * Returns list of text component that need to save
	 * @return a list of text components
	 */
	public ArrayList<TextEditor> getTextsForSaving() {
		ArrayList<TextEditor> list = new ArrayList<TextEditor>();
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			if (textEditor.conditionOfSaving()) list.add(textEditor);
		}
		return list;
	}

	/**
	 * Returns true when all texts is saved. Forces saving of text in the file
	 * @return boolean
	 */
	public boolean isTextsSaved() {
		ArrayList<TextEditor> list = getTextsForSaving();
		if (list.size() == 0) { 
			return true;
		} else if (list.size() == 1) {
			TextEditor textEditor = list.get(0);
			activateTab(textEditor); 
			return textEditor.isTextSaved();
		} else {
			return TextSaver.showSavingConfirmationDialog(this, list);	
		}
	}
	
	/**
	 * Stops creating backups for all text component
	 */
	public void shutdown() {
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			textEditor.getBackupManager().turnOff();
		}
	}

    /**
     * Returns true when can revert text changes in at least one component
     * @return a boolean
     */
    public boolean canRevertAll() {
        for (TextEditor textEditor : getTextEditors()) {
            JTextComponent textComponent = textEditor.getTextComponent();
            if (textComponent instanceof ITextComponent && ((ITextComponent) textComponent).canRevert()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Reverts text changes in all text components
     */
    public void revertAll () {
        for (TextEditor textEditor : getTextEditors()) {
            JTextComponent textComponent = textEditor.getTextComponent();
            if (textComponent instanceof ITextComponent) {
                ((ITextComponent) textComponent).revert();
            }
        }
    }

    /**
     * Return true when the text fragment at list in one text can be deselected
     * @return boolean
     */
    public boolean canDeselectAll() {
        for (TextEditor textEditor : getTextEditors()) {
            JTextComponent textComponent = textEditor.getTextComponent();
            if (textComponent instanceof ITextComponent && ((ITextComponent) textComponent).canDeselect()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Deselects the text fragment in all texts
     */
    public void deselectAll() {
        for (TextEditor textEditor : getTextEditors()) {
            JTextComponent textComponent = textEditor.getTextComponent();
            if (textComponent instanceof ITextComponent) {
                ((ITextComponent) textComponent).deselect();
            }
        }
    }

	/**
	 * Set current directory for save and open operations.
	 * @param dir - current directory
	 */
	public void setCurrentDirectory(File dir) {
		this.currentDirectory = dir;
		Iterator<TextEditorAdapter> iterator = m_components.iterator();
		while (iterator.hasNext()) {
			TextEditor textEditor = iterator.next().getTextEditor();
			textEditor.setCurrentDirectory(dir);
		}
	}
	
	public File getCurrentDirectory() {
		return this.currentDirectory;
	}

    /**
     * Returns true when the editor in read only mode
     * @return - boolean
     */
    public boolean isReadOnlyMode() {
        boolean hasReadOnlyEditor = false;
        for (TextEditorAdapter adapter : m_components) {
            JTextComponent textComponent = adapter.getTextEditor().getTextComponent();
            if (textComponent instanceof ITextComponent) {
                if (((ITextComponent) textComponent).isReadOnlyMode()) {
                    hasReadOnlyEditor = true;
                    break;
                }
            }
        }
        return hasReadOnlyEditor && this.isReadOnlyMode;
    }

    /**
     * Sets the read only mode
     * @param mode - boolean
     */
    public void setReadOnlyMode(boolean mode) {
        if (!canToggleReadOnlyMode()) return;
        this.isReadOnlyMode = mode;
        for (TextEditorAdapter adapter : m_components) {
            JTextComponent textComponent = adapter.getTextEditor().getTextComponent();
            if (textComponent instanceof ITextComponent) {
                ((ITextComponent) textComponent).setReadOnlyMode(mode);
            }
        }
    }

    /**
     * Returns true when the read only mode can be set
     * @return - boolean
     */
    public boolean canToggleReadOnlyMode() {
        for (TextEditorAdapter adapter : m_components) {
            JTextComponent textComponent = adapter.getTextEditor().getTextComponent();
            if (textComponent instanceof ITextComponent) {
                if (((ITextComponent) textComponent).canToggleReadOnlyMode()) return true;
            }
        }
        return false;
    }

    /**
     * Toggles the reading and editing modes in the text editor
     */
    public void toggleReadOnlyMode() {
        if (!canToggleReadOnlyMode()) return;
        boolean hasReadOnlyEditor = false;
        for (TextEditorAdapter adapter : m_components) {
            JTextComponent textComponent = adapter.getTextEditor().getTextComponent();
            if (textComponent instanceof ITextComponent) {
                if (((ITextComponent) textComponent).isReadOnlyMode()) {
                    hasReadOnlyEditor = true;
                    break;
                }
            }
        }
        if (!hasReadOnlyEditor) this.isReadOnlyMode = true;
        else this.isReadOnlyMode = !this.isReadOnlyMode;
        for (TextEditorAdapter adapter : m_components) {
            JTextComponent textComponent = adapter.getTextEditor().getTextComponent();
            if (textComponent instanceof ITextComponent) {
                if (((ITextComponent) textComponent).canToggleReadOnlyMode()) {
                    ((ITextComponent) textComponent).setReadOnlyMode(this.isReadOnlyMode);
                }
            }
        }
    }

    /**
     * Returns the array of read only modes in text components
     * @return array of boolean
     */
    public boolean[] getReadOnlyModes() {
        boolean[] modes = new boolean[m_components.size()];
        for (int index = 0; index < m_components.size(); index++) {
            TextEditorAdapter adapter = m_components.get(index);
            JTextComponent textComponent = adapter.getTextEditor().getTextComponent();
            if (textComponent instanceof ITextComponent) {
                modes[index] = ((ITextComponent) textComponent).isReadOnlyMode();
            } else {
                modes[index] = false;
            }
        }
        return modes;
    }

    /**
     * Sets the array of read only modes in text components
     * @param modes - array of boolean
     */
    public void setReadOnlyModes(boolean[] modes) {
        if (!canToggleReadOnlyMode()) return;
        try {
            for (int index = 0; index < m_components.size(); index++) {
                TextEditorAdapter adapter = m_components.get(index);
                JTextComponent textComponent = adapter.getTextEditor().getTextComponent();
                if (textComponent instanceof ITextComponent) {
                    ((ITextComponent) textComponent).setReadOnlyMode(modes[index]);
                }
            }
        } catch (ArrayIndexOutOfBoundsException e) {
            e.printStackTrace();
        } catch (NullPointerException e) {
            e.printStackTrace();
        }
    }

	/**
	 * Adds EditorChangeListener for notification
	 * @param listener
	 */
	public void addEditorChangeListener(EditorChangeListener listener) {
		editorChangeListeners.add(listener);
	}
	
	/**
	 * Removes EditorChangeListener from list
	 * @param listener
	 */
	public void removeEditorChangeListener(EditorChangeListener listener) {
		editorChangeListeners.remove(listener);
	}
	
	/**
	 * Notifies listeners about changing of current tab
	 */
	protected void fireEditorChanged(EditorChangeEventType type) {
		EditorChangeEvent event = new EditorChangeEvent(this, getCurrentTextEditor(), type);
		Iterator<EditorChangeListener> iterator = editorChangeListeners.iterator();
		while (iterator.hasNext()) {
			EditorChangeListener listener = iterator.next();
			listener.editorChanged(event);
		}
	}

	private FileChangeListener fileListener = new FileChangeListener() {

		public void fileChanged(FileChangeEvent event) {
            if (event.getType() == FileChangeEventType.RELOADING) multiTextOptions.setConfirmationReloading(event.getSource().isConfirmReloading());
            if (event.getType() == FileChangeEventType.REMOVING) multiTextOptions.setConfirmationRemoving(event.getSource().isConfirmRemoving());
            if (event.getType() == FileChangeEventType.COPING || event.getType() == FileChangeEventType.MOVING)
                multiTextOptions.setConfirmationOverwriting(event.getSource().isConfirmOverwriting());
			if (isSingleTextComponent()) return;
			TextEditor textEditor;
			for (int index = 0; index < m_tabbedPane.getTabCount(); index++) {
				TextEditorAdapter componentAdapter = (TextEditorAdapter) m_tabbedPane.getComponentAt(index);
				textEditor = componentAdapter.getTextEditor();
				if (textEditor == event.getSource()) {
					m_tabbedPane.setTitleAt(index, createTitleForTextComponent(textEditor));
					break;
				}
			}
		}
		
	};

    private class BookmarkChangeListener implements FileChangeListener {

        private TextEditorAdapter adapter;
        private BookmarkSupport syntaxAdapter;

        /**
         * TextEditorAdapter only instance of BookmarkSupport
         * @param adapter
         */
        public BookmarkChangeListener(TextEditorAdapter adapter) {
            this.adapter = adapter;
            if (adapter instanceof BookmarkSupport) syntaxAdapter = (BookmarkSupport) adapter;
        }

        @Override
        public void fileChanged(FileChangeEvent event) {
            if (syntaxAdapter == null) return;
            switch (event.getType()) {
                case CLEARING:
                case REMOVING:
                case RESTORING:
                    syntaxAdapter.removeAllBookmarks();
                    break;
                case OPENING:
                    syntaxAdapter.removeAllBookmarks();
                    syntaxAdapter.setBookmarks(m_bookmarks.get(event.getSource().getFile().getAbsolutePath()));
                    break;
                case COPING:
                case MOVING:
                case RENAMING:
                case SAVING:
                    addBookmarks(adapter);
                    break;
            }
        }

    }

}
