package neutrino.text.components.styled;

import com.inet.jortho.SpellChecker;
import neutrino.dialogs.SpellingOptionsChooser;
import neutrino.text.ITextComponent;
import neutrino.text.ReadModeEvent;
import neutrino.text.ReadModeListener;

import java.awt.datatransfer.DataFlavor;
import java.awt.event.*;
import java.util.ArrayList;
import java.util.Stack;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.undo.*;

/**
 * Incapsulates styled text component.
 * Also supports undo and redo operations.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class StyledTextComponent extends JTextPane implements ITextComponent {
	
	private UndoManager undoManager = null;
 	private Stack<CompoundEdit> edits = null;  	
	private int numberOfCurrentEdit = 0;	
	private JPopupMenu popupMenu = null;
	private boolean undoable = true;
    private ArrayList<ReadModeListener> readModeListeners = new ArrayList<ReadModeListener>();

    public StyledTextComponent() {
		super();
		undoManager = new UndoManager();
		undoManager.setLimit(-1);
		edits = new Stack<CompoundEdit>();
		getDocument().addUndoableEditListener(undoableEditListener);
		final StyledTextComponent instance = this;
		addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (popupMenu != null && e.getButton() == e.BUTTON3) {
					popupMenu.show(instance, e.getX(), e.getY());
				}
			}
		});
        SpellChecker.register(this, false, false, false, SpellingOptionsChooser.isAutoSpellingMode());
        createDefaultPopupMenu();
	}

    protected void createDefaultPopupMenu() {
        // create menu items
        final JMenuItem pmiUndo = new JMenuItem("Undo");
        final JMenuItem pmiRedo = new JMenuItem("Redo");
        final JMenuItem pmiCut = new JMenuItem("Cut");
        final JMenuItem pmiCopy = new JMenuItem("Copy");
        final JMenuItem pmiPaste = new JMenuItem("Paste");
        final JMenuItem pmiClear = new JMenuItem("Clear");
        final JMenu pmCheckSpelling = SpellChecker.createCheckerMenu();
        final JMenu pmLanguages = SpellChecker.createLanguagesMenu();
        final JMenuItem pmiSelectAll = new JMenuItem("Select all");
        final JMenuItem pmiDeselect = new JMenuItem("Deselect");
        // create popup menu
        popupMenu = new JPopupMenu();
        popupMenu.addPopupMenuListener(new PopupMenuListener() {
            @Override
            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                pmiUndo.setEnabled(canUndo());
                pmiRedo.setEnabled(canRedo());
                pmiCut.setEnabled(canCut());
                pmiCopy.setEnabled(canCopy());
                pmiPaste.setEnabled(canPaste());
                pmiClear.setEnabled(canClear());
                pmiSelectAll.setEnabled(canSelectAll());
                pmiDeselect.setEnabled(canDeselect());
            }

            @Override
            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
            }

            @Override
            public void popupMenuCanceled(PopupMenuEvent e) {
            }
        });
        // build popup menu
        popupMenu.add(pmiUndo);
        popupMenu.add(pmiRedo);
        popupMenu.addSeparator();
        popupMenu.add(pmiCut);
        popupMenu.add(pmiCopy);
        popupMenu.add(pmiPaste);
        popupMenu.add(pmiClear);
        popupMenu.addSeparator();
        popupMenu.add(pmCheckSpelling);
        popupMenu.add(pmLanguages);
        popupMenu.addSeparator();
        popupMenu.add(pmiSelectAll);
        popupMenu.add(pmiDeselect);
        // build mnemonics
        pmiUndo.setMnemonic(KeyEvent.VK_U);
        pmiRedo.setMnemonic(KeyEvent.VK_R);
        pmiCut.setMnemonic(KeyEvent.VK_C);
        pmiCopy.setMnemonic(KeyEvent.VK_O);
        pmiPaste.setMnemonic(KeyEvent.VK_P);
        pmiClear.setMnemonic(KeyEvent.VK_L);
        pmCheckSpelling.setMnemonic(KeyEvent.VK_H);
        pmLanguages.setMnemonic(KeyEvent.VK_N);
        pmiSelectAll.setMnemonic(KeyEvent.VK_S);
        pmiDeselect.setMnemonic(KeyEvent.VK_D);
        // build accelerators
        pmiUndo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, InputEvent.CTRL_DOWN_MASK));
        pmiRedo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y, InputEvent.CTRL_DOWN_MASK));
        pmiCut.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.CTRL_DOWN_MASK));
        pmiCopy.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_DOWN_MASK));
        pmiPaste.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, InputEvent.CTRL_DOWN_MASK));
        pmiClear.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0));
        pmiSelectAll.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_DOWN_MASK));
        pmiDeselect.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0));
        // build actions
        ActionListener listener = new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (e.getSource() == pmiUndo) {
                    undo();
                } else if (e.getSource() == pmiRedo) {
                    redo();
                } else if (e.getSource() == pmiCut) {
                    cut();
                } else if (e.getSource() == pmiCopy) {
                    copy();
                } else if (e.getSource() == pmiPaste) {
                    paste();
                } else if (e.getSource() == pmiClear) {
                    clear();
                } else if (e.getSource() == pmiSelectAll) {
                    selectAll();
                } else if (e.getSource() == pmiDeselect) {
                    deselect();
                }
            }
        };
        pmiUndo.addActionListener(listener);
        pmiRedo.addActionListener(listener);
        pmiCut.addActionListener(listener);
        pmiCopy.addActionListener(listener);
        pmiPaste.addActionListener(listener);
        pmiClear.addActionListener(listener);
        pmiSelectAll.addActionListener(listener);
        pmiDeselect.addActionListener(listener);
    }

    public AttributeSet getCharacterAttributes() {
    	if (isTextSelected()) {
    		return super.getCharacterAttributes();
    	} else {
    		return getInputAttributes();
    	}
    }
	
	/**
	 * Sets popup menu for text component
	 * @param popupMenu - a popup menu
	 */
	public void setPopupMenu(JPopupMenu popupMenu) {
		this.popupMenu = popupMenu;
	}
	
	/**
	 * Notices that text is not changed and read only mode is disabled
     * when new document is set Prepares undo manager for set document
	 */
	public void setDocument(Document doc) {
		super.setDocument(doc);
		doc.addUndoableEditListener(undoableEditListener);
		reset(true);
        setReadOnlyMode(false);
		System.gc();
	}

	/**
	 * Returns true when all text may be selected
	 * @return boolean
	 */
	public boolean canSelectAll() {
		return !isTextEmpty() && !isAllTextSelected() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true when text may be printed 
	 * @return boolean
	 */
	public boolean canPrint() {
		return !isTextEmpty();
	}
	
	/**
	 * Chack is text component is empty.
	 * @return true when text is empty.
	 */
	public boolean isTextEmpty() {
		return getDocument().getLength() == 0;
	}
	
	/**
	 * Returns true when all text is selected
	 * @return boolean
	 */
	public boolean isAllTextSelected() {
		if (!isTextSelected()) return false;
		else return (getSelectionStart() == 0) && (getSelectionLength() == getDocument().getLength());
	}

	/**
	 * Returns true if text is selected.
	 */
	public boolean isTextSelected() {
		return getSelectionStart() != getSelectionEnd();
	}
	
	/**
	 * Return length of selection or 0 if text is not selected. 
	 */
	public int getSelectionLength() {
		return getSelectionEnd() - getSelectionStart();
	}

	// undoable capability
	
	/**
	 * Sets undoable for text component
	 * value - boolean
	 */
	public void setUndoable(boolean value) {
		this.undoable = value;
	}
	
	/**
	 * Returns true when text component is undoable
	 * @return boolean
	 */
	public boolean isUndoable() {
		return this.undoable;
	}
	
	/**
	 * Chack is text is changed.
	 * @return true when text is changed.
	 */
	public boolean isTextChanged() {
		return numberOfCurrentEdit != 0;
	}	

	/**
	 * Returns true when text changes may be reverted
	 * @return boolean
	 */
	public boolean canRevert() {
		return undoable && isTextChanged() && !isReadOnlyMode();
	}	
	
	/**
	 * Returns true if the text change may be undone
	 */
	public boolean canUndo() {
		return undoable && undoManager.canUndo() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true if the text change may be redone
	 */
	public boolean canRedo() {
		return undoable && undoManager.canRedo() && !isReadOnlyMode();
	}
	
	/**
	 * Undoes the last text change if possible. 
	 */
	public void undo() {
		if (canUndo()) {
			undoManager.undo();
			numberOfCurrentEdit--;
		}
	}

	/**
	 * Redose the last text change if possible.
	 */
	public void redo() {
		if (canRedo()) {
			undoManager.redo();
			numberOfCurrentEdit++;
		}
	}
	
 	/**
 	 * Returns true when compound edit is begun
 	 * @return boolean
 	 */
	public boolean isEditBegun() {
		return !edits.isEmpty();
	}
	
	/**
	 * Marks beginning of coalesce edit operation
	 */
	public void beginEdit() {
		if (!undoable) return;
		edits.push(new CompoundEdit());
	}
	
	/**
	 * Marks end of coalesce edit operation
	 */
	public void endEdit() {
		if (undoable) {
			if (!isEditBegun()) {
				return;
			}
			CompoundEdit currentEdit = edits.pop();
			if (isEditBegun()) {
				edits.peek().addEdit(currentEdit);
			} else {
				undoManager.addEdit(currentEdit);
			}
			currentEdit.end();
		}
		numberOfCurrentEdit++;
	}
	
	public void revert() {
		if (!canRevert()) return;
		try {
			while (isTextChanged()) {
				undoManager.undo();
				numberOfCurrentEdit--;
			}
		} catch (CannotUndoException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Marks text as not changed
	 */
	public void reset(boolean discardAllEdits) {
		if (discardAllEdits) {
			if (undoManager != null) {
				undoManager.discardAllEdits();
			}
			if (edits != null) {
				edits.clear();
			}
		}
		numberOfCurrentEdit = 0;
	}
	
	private UndoableEditListener undoableEditListener = new UndoableEditListener() {
		
		public void undoableEditHappened(UndoableEditEvent e) {
			if (isEditBegun()) {
				if (undoable) edits.peek().addEdit(e.getEdit());
			} else {
				if (undoable) undoManager.addEdit(e.getEdit());
				numberOfCurrentEdit++;
			}
		}
		
	};

	/**
	 * Returns true if clipboard contain text.
	 * @throws IllegalStateException
	 * @return boolean value
	 */
	public boolean isClipboardContainText() throws IllegalStateException {
		for (DataFlavor dataFlavor : getToolkit().getSystemClipboard().getAvailableDataFlavors()) {
			if (dataFlavor.isFlavorTextType()) {
				return true;
			}
		}
		return false;
	}
	
	public void cut() {
		if (canCut()) super.cut();
	}
	
	public void copy() {
		if (canCopy()) super.copy();
	}
	
	public void paste() {
		if (canPaste()) super.paste();
	}
	
	/**
	 * Returns true when text fragment may be cut
	 * @return boolean
	 */
	public boolean canCut() {
		return isTextSelected() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true when text fragment may be copied
	 * @return boolean
	 */
	public boolean canCopy() {
		return isTextSelected() && !isReadOnlyMode();
	}
	
	/**
	 * Returns true when text fragment may be pasted
	 * @return boolean
	 */
	public boolean canPaste() {
		try {
			return isClipboardContainText() && !isReadOnlyMode();
		} catch (IllegalStateException e1) {
			return true;
		}
	}

    /**
     * Returns true when selection can be cleared
     * @return boolean
     */
    public boolean canClear() {
        return isTextSelected() && !isReadOnlyMode();
    }

    /**
     * Clears the selected text
     */
    public void clear() {
        if (!canClear()) return;
        int start = getSelectionStart();
        int end = getSelectionEnd();
        try {
            getDocument().remove(start, end - start);
        } catch (BadLocationException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns true when text can be deselected
     * @return boolean
     */
    public boolean canDeselect() {
        return isTextSelected();
    }

    /**
     * Deselects the text fragment
     */
    public void deselect() {
        int caretPosition = getSelectionEnd();
        select(caretPosition, caretPosition);
    }

    /**
     * Returns true when the editor in read only mode
     * @return - boolean
     */
    public boolean isReadOnlyMode() {
        return !isEditable();
    }

    /**
     * Sets the read only mode
     * @param mode - boolean
     */
    public void setReadOnlyMode(boolean mode) {
        if (!canToggleReadOnlyMode()) return;
        setEditable(!mode);
        fireReadModeChanged();
    }

    /**
     * Toggles the reading and editing modes in the text editor
     */
    public void toggleReadOnlyMode() {
        if (!canToggleReadOnlyMode()) return;
        setEditable(!isEditable());
        fireReadModeChanged();
    }

    /**
     * Returns true when the read only mode can be set
     * @return - boolean
     */
    public boolean canToggleReadOnlyMode() {
        return !isTextEmpty();
    }

    /**
     * Fires the read mode listeners on change of read only mode
     */
    protected void fireReadModeChanged() {
        if (readModeListeners == null || readModeListeners.size() == 0) return;
        ReadModeEvent event = new ReadModeEvent(this);
        for (ReadModeListener listener : readModeListeners) {
            listener.readModeChanged(event);
        }
    }

    /**
     * Adds the read mode listener to the list of of observes
     * @param listener - ReadModeListener
     */
    public void addReadModeListener(ReadModeListener listener) {
        readModeListeners.add(listener);
    }

    /**
     * Removes the listener from the list of observes
     * @param listener - ReadModeListener
     */
    public void removeReadModeListener(ReadModeListener listener) {
        readModeListeners.remove(listener);
    }

}
