package properties;

import gui.Application;
import neutrino.options.*;
import neutrino.text.components.plain.PlainTextArea;
import java.io.*;

/**
 * Holds the properties of the program for persisting in the file system between launches of the program.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class EditorProperties implements Serializable {

    private File directory;
    private File file;
    private int caretPosition;
    private Settings settings;
    private FileOptions fileOptions;
    private WindowOptions windowOptions;
    private SpellingOptions spellingOptions;
    private TextOptions textOptions;

    public EditorProperties() {
        settings = new Settings();
        directory = null;
        file = null;
        fileOptions = new FileOptions();
        windowOptions = new WindowOptions();
        textOptions = new TextOptions();
        spellingOptions = new SpellingOptions();
    }

    /**
     * Sets the values of properties. Turns off the backup manager
     * @param application
     */
    public void buildProperties(Application application) {
        settings = application.getSettings();
        PlainTextArea textComponent = (PlainTextArea) application.getTextEditor().getTextComponent();
        if (application.getTextEditor().isFileLoaded() && application.getSettings().isPersistCurrentFile()) {
            file = application.getTextEditor().getFile();
        } else {
            file = null;
        }
        directory = application.getTextEditor().getCurrentDirectory();
        caretPosition = textComponent.getCaretPosition();
        spellingOptions.buildProperties();
        textOptions.buildProperties((PlainTextArea) application.getTextEditor().getTextComponent());
        windowOptions.buildProperties(application);
        fileOptions.buildProperties(application.getTextEditor());
    }

    /**
     * Establishes the properties. Prepares the application for using.
     * @param application
     * @param commandLineArguments
     */
    public void establishProperties(Application application, String[] commandLineArguments) {
        // establish the option of persisting current file
        application.setSettings(settings);
        // establish file properties
        fileOptions.establishProperties(application.getTextEditor());
        // establish text properties
        textOptions.establishProperties((PlainTextArea) application.getTextEditor().getTextComponent());
        // establish current file
        if (commandLineArguments != null && commandLineArguments.length > 0) {
            File file = new File(commandLineArguments[0]);
            application.getTextEditor().open(file);
        } else {
            if (file != null && application.getSettings().isPersistCurrentFile()) {
                if (file.exists() && file.isFile()) {
                    application.getTextEditor().open(file);
                }
            }
        }
        // establish current directory
        if (directory != null && directory.exists() && directory.isDirectory()) {
            application.getTextEditor().setCurrentDirectory(directory);
        }
        // establish caret position
        try {
            if (application.getSettings().isSaveCaretPosition() && application.getSettings().isPersistCurrentFile()) {
                application.getTextEditor().getTextComponent().setCaretPosition(caretPosition);
            }
        } catch (IllegalArgumentException e) {
            // do nothing
        }
        // establish window properties
        windowOptions.establishProperties(application);
        // establish spelling options
        spellingOptions.establishProperties(application.getTextEditor().getTextComponent(), null);
    }

}
