/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JTabbedPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jnpad.config.Config;
import jnpad.text.BufferSet;
import jnpad.text.Buffer;
import jnpad.ui.icon.CompositeIcon;
import jnpad.ui.icon.IconClickTarget;

/**
 * The Class JNPadTabbedViewer.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class JNPadTabbedViewer extends JNPadViewer {
  static ImageIcon            iiTabGenericClose;
  static ImageIcon            iiTabModifiedClose;
  static ImageIcon            iiTabReadOnlyClose;
  static ImageIcon            iiTabModifiedReadOnlyClose;

  /** Logger */
  private final static Logger LOGGER           = Logger.getLogger(JNPadTabbedViewer.class.getName());

  /** UID */
  private static final long   serialVersionUID = 5946227426715682848L;

  static {
    try {
      iiTabGenericClose          = GUIUtilities.getIcon("tab-generic-close.png");          //$NON-NLS-1$
      iiTabModifiedClose         = GUIUtilities.getIcon("tab-modified-close.png");         //$NON-NLS-1$
      iiTabReadOnlyClose         = GUIUtilities.getIcon("tab-readonly-close.png");         //$NON-NLS-1$
      iiTabModifiedReadOnlyClose = GUIUtilities.getIcon("tab-modified-readonly-close.png");//$NON-NLS-1$
    }
    catch(Exception ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
    }
  }
  
  /**
   * Instantiates a new jNPad tabbed viewer.
   *
   * @param jNPad the jNPpad's frame
   * @param mWindow the window menu
   */
  public JNPadTabbedViewer(JNPadFrame jNPad, JMenu mWindow) {
    super(jNPad, mWindow);
    
    _bufferSet = new JNPadTabbedPane(this);
    _bufferSet.addChangeListener(new ChangeListener() {
      public void stateChanged(final ChangeEvent e) {
        handleStateChanged(e);
      }
    });
    setMainContent((JNPadTabbedPane) _bufferSet);
  }

  /**
   * Gets the jNPad tabbed pane.
   *
   * @return the jNPad tabbed pane
   */
  public JNPadTabbedPane getJNPadTabbedPane() {
    return (JNPadTabbedPane) _bufferSet;
  }

  /**
   * Gets the icon.
   *
   * @param buffer the buffer
   * @return the icon
   * @see jnpad.JNPadViewer#getIcon(jnpad.text.Buffer)
   */
  @Override
  protected Icon getIcon(Buffer buffer) {
    Icon icon;
    
    if (Config.TAB_CLOSE_BUTTON.getValue() && _bufferSet.getTabLayoutPolicy() == JTabbedPane.WRAP_TAB_LAYOUT) {
      List<Icon> list = new ArrayList<Icon>();
      Icon fileIcon;

      if(buffer.isDirty()) {
        fileIcon = iiFileModified;
        list.add(new IconClickTarget(buffer.isReadOnly() ? iiTabModifiedReadOnlyClose : iiTabModifiedClose));
      }
      else if(buffer.isNew()) {
        fileIcon = iiFileNew;
        list.add(new IconClickTarget(buffer.isReadOnly() ? iiTabReadOnlyClose : iiTabGenericClose));
      }
      else {
        fileIcon = iiFile;
        list.add(new IconClickTarget(buffer.isReadOnly() ? iiTabReadOnlyClose : iiTabGenericClose));
      }

      list.add(fileIcon);

      Icon[] icons = list.toArray(new Icon[list.size()]);

      CompositeIcon cicon = new CompositeIcon(icons, 2, CompositeIcon.HORIZONTAL);
      cicon.setMainIcon(fileIcon);

      icon = cicon;
    }
    else {
      icon = super.getIcon(buffer);
    }

    return icon;
  }

  /**
   * Creates the buffer set.
   *
   * @param oldBufferSet the old buffer set
   * @return the buffer set
   * @see jnpad.text.Viewer#createBufferSet(jnpad.text.BufferSet)
   */
  @Override
  protected BufferSet createBufferSet(BufferSet oldBufferSet) {
    JNPadTabbedPane newBufferSet = new JNPadTabbedPane(this);
    
    for (int i = 0; i < _bufferSet.getTabCount(); i++) {
      Buffer buffer = _bufferSet.getBufferAt(i);
      
      Buffer e = create(newBufferSet, buffer);
      
      final String title = _bufferSet.getTitleAt(i);
      final String tip   = _bufferSet.getToolTipTextAt(i);
      final Icon   icon  = _bufferSet.getIconAt(i);
      
      newBufferSet.addTab(title, icon, e, tip); 
    }

    newBufferSet.addChangeListener(new ChangeListener() {
      public void stateChanged(final ChangeEvent e) {
        handleStateChanged(e);
      }
    });
    
    return newBufferSet;
  }
  
}
