/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text;

import java.util.TreeSet;
import java.util.Vector;

import javax.swing.text.BadLocationException;

/**
 * The Class CompletionUtilities.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
final class CompletionUtilities {
  /** no instances */
  private CompletionUtilities() {
    super();
  }

  /**
   * Gets the word.
   *
   * @param textArea the text area
   * @param offset the offset
   * @return the word
   * @throws BadLocationException the bad location exception
   */
  static String getWord(JNPadTextArea textArea, int offset) throws BadLocationException {
    String line = textArea.getLineTextOfOffset(offset);
    int dot = offset - textArea.getLineStartOffset(textArea.getLineOfOffset(offset));
    if (dot == 0)
      return null;
    int wordStart = TextUtilities.findWordStart(line, dot - 1, textArea.getDelimiters());
    return line.substring(wordStart, dot);
  }

  /**
   * Gets the word.
   *
   * @param line the line
   * @param offset the offset
   * @param delimiters the delimiters
   * @return the word
   */
  static String getWord(String line, int offset, String delimiters) {
    int wordEnd = TextUtilities.findWordEnd(line, offset + 1, delimiters);
    return line.substring(offset, wordEnd);
  }

  /**
   * Gets the divergent index.
   *
   * @param str1 String
   * @param str2 String
   * @return the divergent index
   */
  static int getDivergentIndex(String str1, String str2) {
    int result = str1.length();
    if (!(str1.equals(str2))) {
      for (result = 0; result < str1.length() &&
           result < str2.length() &&
           str1.charAt(result) == str2.charAt(result); result++);
    }
    return result;
  }

  /**
   * Sets the.
   *
   * @param tp the tp
   * @param word the word
   * @param wordLen the word len
   * @param delimiters the delimiters
   * @param completions the completions
   * @throws BadLocationException the bad location exception
   */
  static void set(JNPadTextArea tp, String word, int wordLen, String delimiters, TreeSet<String> completions) throws BadLocationException {
    for (int i = 0; i < tp.getLineCount(); i++) {
      String line = tp.getLineText(i);
      if (line.startsWith(word)) {
        String _word = CompletionUtilities.getWord(line, 0, delimiters);
        if (_word.length() != wordLen) {
          completions.add(_word);
        }
      }

      int len = line.length() - word.length();
      for (int j = 0; j < len; j++) {
        char c = line.charAt(j);
        if (!Character.isLetterOrDigit(c) && delimiters.indexOf(c) == -1) {
          if (line.regionMatches(j + 1, word, 0, wordLen)) {
            String _word = CompletionUtilities.getWord(line, j + 1, delimiters);
            if (_word.length() != wordLen) {
              completions.add(_word);
            }
          }
        }
      }
    }
  }

  /**
   * Refresh list data.
   *
   * @param word String
   * @param completions String[]
   * @return the vector
   */
  static Vector<String> refreshListData(final String word, final String[] completions) {
    Vector<String> v = new Vector<String>();
    for (String item : completions) {
      if (item.startsWith(word)) {
        v.add(item);
      }
    }
    return v;
  }

}
