/**
 * $Id: ConnectionInterface.java,v 1.11 2001/10/09 00:44:04 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.awt.*;
import java.awt.event.*;
import java.util.Hashtable;
import java.util.Vector;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import javax.swing.*;
import javax.swing.border.*;

import redlight.hotline.*;
import redlight.crypto.*;
import redlight.utils.DebuggerOutput;
import redlight.utils.TimeFormat;
import redlight.graphics.Spinner;
import redlight.script.*;

/**
 * This class implements the interface for a single connection.
 * The actual connection itself is handled by a Machine.
 */ 
public class ConnectionInterface 
    implements Interface, 
	       Scriptable {

    static int x = 0, y = 0;
    static Integer lock = new Integer(0);
    public Machine rlm;
    Container contentPane;
    JButton news, 
	chat, 
	users, 
	transfers,
	createAccount,
        openAccount,
	textColor, 
	backgroundColor;
    JLabel timeConnected;
    JTextField nickField;
    JCheckBox audioMute;
    JPanel colorBorderPanel, spinnerPanel;
    JPanel userPanel, chatPanel, newsPanel, transfersPanel, filesPanel;
    JSplitPane userSplitPane;
    JTabbedPane tabbedPane, placeHolder;
    Spinner spinner;
    // set to true in windowClosing()
    boolean closing = false, hasBeenClosed = false;
    // set to true after connectSuccess()
    boolean connected = false;
    boolean swallowingComplete = false;
    Thread timer;
    long startTimeMillis = 0;
    int howManySwallowed = 0;
    ConnectInterface tripConnect;
    ChatInterface tripChat;
    NewsInterface tripNews;
    UsersInterface tripUsers;
    PostInterface tripPost;
    TransferListInterface tripTransfers;
    FilesInterface tripFiles;
    Shell myShell;
    Vector children, chatWindows;

    public ConnectionInterface(String host,
                               int port, 
                               String login, 
                               String password, 
                               String name) {

        rlm = new Machine(this);
        
        if(name != null) 
            rlm.setServerName(name);

        rlm.setServer(host);
        rlm.setPort(port);
        rlm.setLogin(login);
        rlm.setPassword(password);
        
        contentPane = new JPanel();

	Font guiFont = (Font) Main.rlo.getProperty("Font.gui");

	GridBagLayout gbl = new GridBagLayout();
      	GridBagConstraints gbc = new GridBagConstraints();
	JPanel p = new JPanel();
	JPanel p2 = new JPanel();
	JPanel p3 = new JPanel();
        JPanel nickPanel = new JPanel();
        JPanel buttonPanel = new JPanel();
	spinnerPanel = new JPanel();
	news = new JButton("News");
	chat = new JButton("Chat");
	users = new JButton("Users");
	transfers = new JButton("Transfers");
	timeConnected = new JLabel();
	audioMute = new JCheckBox("Mute audio", false);
        JLabel nickLabel = new JLabel("Nick:");
        nickField = new JTextField(new String(Main.rlo.getStringProperty("User.Nick")), 15);
	spinner = new Spinner(Main.rlo.getImageSet("Spinner"), 
			      100, 16, 16);

	p.setLayout(gbl);
	p2.setLayout(new FlowLayout(FlowLayout.LEFT));
        p3.setLayout(new BorderLayout());
        buttonPanel.setLayout(new FlowLayout());
	spinnerPanel.setLayout(new BorderLayout());
	contentPane.setLayout(new BorderLayout());
        nickPanel.setLayout(new GridBagLayout());
	news.setEnabled(false);
	transfers.setEnabled(false);
	users.setEnabled(false);
	chat.setEnabled(false);
        nickField.setToolTipText("Type a nickname and hit enter.");
	news.setFont(guiFont);
	chat.setFont(guiFont);
	transfers.setFont(guiFont);
	users.setFont(guiFont);
        nickLabel.setFont(guiFont);
        news.setActionCommand("ActivateNews");
        chat.setActionCommand("ActivateChat");
        users.setActionCommand("ActivateUsers");
        transfers.setActionCommand("ActivateTransfers");
        nickField.setActionCommand("ActivateChangeuser");        
	audioMute.setFont(guiFont);

        ActionListener actionListener = new ActionListener() {
                
                public void actionPerformed(ActionEvent e) {
                    
                    if(e.getActionCommand().equals("ActivateNews")) {
                        
                        if(tripNews == null) 
                            tripNews = new NewsInterface(rlm);
                        
                        else 
                            tripNews.show();
                        
                    } else if(e.getActionCommand().equals("ActivateChat")) {
                        
                        if(tripChat == null) 
                            tripChat = new ChatInterface(rlm);
                        
                        else 
                            tripChat.show();
                        
                    } else if(e.getActionCommand().equals("ActivateUsers")) {
                        
                        if(tripUsers == null) 		
                            tripUsers = new UsersInterface(rlm, true);
                        
                        else 
                            tripUsers.show();
                        
                    } else if(e.getActionCommand().equals("ActivateTransfers")) {
                        
                        if(tripTransfers == null)
                            tripTransfers = 
                                new TransferListInterface(rlm, false);
                        else tripTransfers.show();
                        
                    } else if(e.getActionCommand().equals("ActivateChangeuser")) {
                        
                        if(!rlm.getHLC().getNick().equals(nickField.getText())) {
                            
                            String newNick = nickField.getText();
                            
                            if(newNick.length() > 0) {
                                
                                try {
                                    
                                    rlm.getHLC().sendUserChange(newNick, Main.rlo.getIntegerProperty("User.IconNumber"));
                                    
                                } catch(IOException _e) {}
                                
                            }
                            
                        }
                        
                    }
                    
                }

            };
        
	news.addActionListener(actionListener);
	chat.addActionListener(actionListener);
	users.addActionListener(actionListener);
	transfers.addActionListener(actionListener);
        nickField.addActionListener(actionListener);
        
        audioMute.addItemListener(new ItemListener() {
                
                public void itemStateChanged(ItemEvent e) {
                    
                    if(rlm != null) {

                        if(e.getStateChange() == ItemEvent.SELECTED)
                            rlm.muteAudio(true);
                        else
                            rlm.muteAudio(false);
                        
                    }
                    
                }

            });

	if(Main.rlo.getBooleanProperty("Toggle.SwallowConnections")) {

	    userPanel = new JPanel();
	    chatPanel = new JPanel();
	    newsPanel = new JPanel();
            transfersPanel = new JPanel();
            filesPanel = new JPanel();
            userSplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
                                           Main.rlo.getBooleanProperty("Toggle.LiveResizing"));
            tabbedPane = new JTabbedPane();
            placeHolder = new JTabbedPane();
            placeHolder.addTab("Users", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), userPanel, "Logged on users [ctrl-4]");
            tabbedPane.addTab("Chat", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), chatPanel, "Public chat [ctrl-1]");
            tabbedPane.addTab("News", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), newsPanel, "News and discussion [ctrl-2]");
            tabbedPane.addTab("Files", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), filesPanel, "Upload / download files [ctrl-3]");
            tabbedPane.addTab("Transfers", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), transfersPanel, "Transfer progress [ctrl-4]");

            KeyStroke chatKey = 
                KeyStroke.getKeyStroke(KeyEvent.VK_1, Event.CTRL_MASK);
            KeyStroke newsKey = 
                KeyStroke.getKeyStroke(KeyEvent.VK_2, Event.CTRL_MASK);
            KeyStroke filesKey = 
                KeyStroke.getKeyStroke(KeyEvent.VK_3, Event.CTRL_MASK);
            KeyStroke transfersKey = 
                KeyStroke.getKeyStroke(KeyEvent.VK_4, Event.CTRL_MASK);
            KeyStroke usersKey = 
                KeyStroke.getKeyStroke(KeyEvent.VK_5, Event.CTRL_MASK);
            
            userSplitPane.registerKeyboardAction(new ActionListener() { 
                    public void actionPerformed(ActionEvent e) {
                        tabbedPane.setSelectedIndex(0);
                        tripChat.show();
                    }
                }, "", chatKey, JComponent.WHEN_IN_FOCUSED_WINDOW);
            
            userSplitPane.registerKeyboardAction(new ActionListener() { 
                    public void actionPerformed(ActionEvent e) {
                        tabbedPane.setSelectedIndex(1);
                        tripNews.show();
                    }
                }, "", newsKey, JComponent.WHEN_IN_FOCUSED_WINDOW);
            
            userSplitPane.registerKeyboardAction(new ActionListener() { 
                    public void actionPerformed(ActionEvent e) {
                        tabbedPane.setSelectedIndex(2);
                        tripFiles.show();
                    }
                }, "", filesKey, JComponent.WHEN_IN_FOCUSED_WINDOW);
            
            userSplitPane.registerKeyboardAction(new ActionListener() { 
                    public void actionPerformed(ActionEvent e) {
                        tabbedPane.setSelectedIndex(3);
                        tripTransfers.show();
                    }
                }, "", transfersKey, JComponent.WHEN_IN_FOCUSED_WINDOW);
            
            userPanel.registerKeyboardAction(new ActionListener() { 
                    public void actionPerformed(ActionEvent e) {
                        placeHolder.setSelectedIndex(0);
                        tripUsers.show();
                    }
                }, "", usersKey, JComponent.WHEN_IN_FOCUSED_WINDOW);
            
            tabbedPane.setRequestFocusEnabled(false);
            placeHolder.setRequestFocusEnabled(false);
            
            userSplitPane.setLeftComponent(tabbedPane);
            userSplitPane.setRightComponent(placeHolder);
            userSplitPane.setOneTouchExpandable(true);

	    p3.add(userSplitPane, BorderLayout.CENTER);

	}

       	spinnerPanel.add(spinner);

	if(!Main.rlo.getBooleanProperty("Toggle.SwallowConnections")) {

	    buttonPanel.add(news);
	    buttonPanel.add(chat);
	    buttonPanel.add(users);
            buttonPanel.add(transfers);

	}

        if(rlm.isAudioSupported())
            buttonPanel.add(audioMute);

        gbc.ipadx = 4;
        gbc.ipady = 4;
        gbc.gridx = 0;
        gbc.gridy = 0;
        nickPanel.add(nickLabel, gbc);
        gbc.fill = gbc.BOTH;
        gbc.gridx = 1;
        gbc.weightx = .4;
        nickPanel.add(nickField, gbc);

        p2.add(nickPanel);
        p2.add(Box.createHorizontalStrut(16));
	p2.add(buttonPanel);

        gbc.ipadx = 0;
        gbc.ipady = 0;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.fill = gbc.NONE;
	gbc.weightx = 1;
	gbc.anchor = gbc.WEST;
	p.add(p2, gbc);
	gbc.gridx = 1;
	gbc.insets = new Insets(0, 0, 0, 4);
	gbc.anchor = gbc.EAST;
	p.add(spinnerPanel, gbc);

	contentPane.add(p3, BorderLayout.CENTER);
	contentPane.add(p, BorderLayout.SOUTH);

        userSplitPane.setDividerLocation(0.65);

        myShell = Main.getActiveShell();
        myShell.addConnectionPane(rlm.getServerName(), this, contentPane);
	spinner.start(); 
	howManySwallowed = 0;
	children = new Vector();
	chatWindows = new Vector();

        new Thread(new Runnable() {

                public void run() {

                    try {

                        rlm.create();
                        rlm.connect();

                    } catch(IOException e) {
                        
                        String reason = e.getMessage();

                        if(reason == null)
                            reason = e.toString();

                        new Error(rlm.getServerName() + ": Could not connect to the server: " + reason);
                        
                        rlm.disconnect();
                        
                    }

                }

            }, "Waiting for connection " + rlm).start();
        
    }

    void timerCancel() {

	if(timer != null) {

	    timer.interrupt();

	    try {

                DebuggerOutput.debug("timerCancel: joining timer thread ...");
		timer.join();
                DebuggerOutput.debug("timerCancel: timer thread joined");

	    } catch (InterruptedException e) {

		timer.setPriority(Thread.MIN_PRIORITY);

	    }

	}

    }

    private synchronized void notifyIfSwallowingComplete() {

	howManySwallowed++;

        DebuggerOutput.debug("ConnectionInterface: swallowed = " + howManySwallowed);

	if(howManySwallowed >= 5) {

	    swallowingComplete = true;
	    notify();

	}

    }

    /**
     * Following method implements ActionListener.
     */
    /**
     * Following method implements ItemStateListener.
     */
    
    /**
     * Following method implements Runnable.
     */

    /**
     * Following methods implement Interface.
     */
    public Shell getShell() {

        return myShell;

    }

    public void registerChild(Child c) {

	children.addElement(c);
	DebuggerOutput.debug("registerChild: "+c.toString());

    }

    public void unregisterChild(Child c) {

	children.removeElement(c);
	DebuggerOutput.debug("unregisterChild: "+c.toString());

    }

    public void error(Object o) {

	if(!closing)
            new Error(rlm.getServerName() + ": " + o.toString());

    }

    public synchronized void connectSuccess() {
	final boolean swallow = 
	    Main.rlo.getBooleanProperty("Toggle.SwallowConnections");

        DebuggerOutput.debug("ConnectionInterface: entered connectSuccess");

	rlm.getScriptBroker().addTarget(this);

	if(!closing) {

	    rlm.playAudio("login");

	    if(swallow) {

		try {

		    SwingUtilities.invokeLater(new Runnable() {

			public void run() {

                            tripNews = new NewsInterface(rlm, swallow);
                            tripChat = new ChatInterface(rlm, swallow);
                            tripUsers = new UsersInterface(rlm, swallow);
                            tripTransfers = new TransferListInterface(rlm, swallow);
                            tripFiles = new FilesInterface(rlm, ":", swallow);
                            
			    userSplitPane.resetToPreferredSizes();
			    userSplitPane.setDividerLocation(userSplitPane.getMaximumDividerLocation());

			}

		    });

		    if(!swallowingComplete) {

                        DebuggerOutput.debug("ConnectionInterface: waiting for swallowing to complete ...");
			wait();
                        DebuggerOutput.debug("ConnectionInterface: swallowing complete.");

		    }

		} catch (InterruptedException e) {}

	    }

	}

	news.setEnabled(true);
	chat.setEnabled(true);
	users.setEnabled(true);
	transfers.setEnabled(true);
	spinner.stop();
	spinnerPanel.remove(spinner);
	spinnerPanel.add(timeConnected);

	/* Start clock for on-line timer. */

	startTimeMillis = System.currentTimeMillis();

	timer = new Thread(new Runnable() {

	    public void run() {

		try {

		    while(true) {

			Thread.currentThread().sleep(1000);

			if(Thread.currentThread().isInterrupted())
			    throw new InterruptedException();
		      
			timeConnected.
                            setText(TimeFormat.
                                    format((System.currentTimeMillis() - 
                                            startTimeMillis) / 1000));

		    }

		} catch(InterruptedException e) {}

	    }

	}, "Timer " + rlm.getServerName());

	timer.start();
        DebuggerOutput.debug("ConnectionInterface.connectSuccess: connected = true");
	connected = true;

    }

    public synchronized void disconnectSuccess() {

        closeUI();

        Object[] l = (Object[]) children.toArray();

	for(int i = 0; i < l.length; i++) {

            DebuggerOutput.debug("disconnectSuccess: closing child " + 
                                 l[i].toString());
            ((Child) l[i]).close();

	}

        DebuggerOutput.debug("disconnectSuccess: removing from DirOpener ...");
	DirOpener.remove(rlm);
        DebuggerOutput.debug("disconnectSuccess: DirOpener removed");

	if(tripTransfers != null) {

            DebuggerOutput.debug("disconnectSuccess: closing transfers ...");
            tripTransfers.close();
            DebuggerOutput.debug("disconnectSuccess: transfers closed");
            tripTransfers = null;

        }
	if(tripNews != null) { 

            DebuggerOutput.debug("disconnectSuccess: closing news ...");
            tripNews.close(); 
            DebuggerOutput.debug("disconnectSuccess: news closed");
            tripNews = null; 

        }
	if(tripChat != null) {

            DebuggerOutput.debug("disconnectSuccess: closing chat ...");
            tripChat.close(); 
            DebuggerOutput.debug("disconnectSuccess: chat closed");
            tripChat = null;

        }
	if(tripUsers != null) {

            DebuggerOutput.debug("disconnectSuccess: closing users ...");
            tripUsers.close(); 
            DebuggerOutput.debug("disconnectSuccess: users closed");
            tripUsers = null;

        }

        if(connected) {

            DebuggerOutput.debug("disconnectSuccess: removing self as script target ...");
            rlm.getScriptBroker().removeTarget(this);
            DebuggerOutput.debug("disconnectSuccess: self removed as script target");
            
        }

        DebuggerOutput.debug("disconnectSuccess: cancelling thread ...");
	timerCancel();
	connected = false;
        children.removeAllElements();
        chatWindows.removeAllElements();
	DebuggerOutput.debug("disconnectSuccess: finished");
    }

    public void displayPropertyChanged(String what, Object property) {

	if(what.endsWith("Color")) 
	    rlm.setSchemeColor(what.substring(0, what.length()-5), 
				       (Color) property); 

	if(tripChat != null) 
	    tripChat.displayPropertyChanged(what, property);

	if(tripUsers != null) 
	    tripUsers.displayPropertyChanged(what, property);

	if(tripNews != null) 
	    tripNews.displayPropertyChanged(what, property);

	if(tripTransfers != null) 
	    tripTransfers.displayPropertyChanged(what, property);

	if(tripPost != null) 
	    tripPost.displayPropertyChanged(what, property);

	userSplitPane.setContinuousLayout(Main.rlo.getBooleanProperty("Toggle.LiveResizing"));

        Object[] l = (Object[]) children.toArray();
        
	for(int i = 0; i < l.length; i++) 
            ((Child) l[i]).displayPropertyChanged(what, property);
        
    }

    public void close() {

        closing = true;                    
        closeUI();        
        rlm.disconnect();

    }

    private void closeUI() {

        if(!hasBeenClosed) {

            spinner.stop();
            spinner = null;

            DebuggerOutput.debug("ConnectionInterface.close: removing connection pane ...");
            myShell.removeConnectionPane(contentPane);
            contentPane = null;
            DebuggerOutput.debug("ConnectionInterface.close: connection pane removed");
            
            hasBeenClosed = true;

        }

    }
    
    /**
     * Following methods implement Scriptable.
     */
    public void gotTarget(ScriptBroker sb) {}
    public void lostTarget(ScriptBroker sb) {}
    public long getKnownMessages() {

	return Scripting.SWALLOW_REQUEST |
	    Scripting.UNSWALLOW_REQUEST |
	    Scripting.REPAINT;

    }
    public ScriptResult executeScript(ScriptObject s) 
	throws UnknownMessageException {

	Object o = s.getUserObject();
	int value = (int) s.getType();   // losing 32 bits of precision here
	switch(value) {

	case Scripting.SWALLOW_REQUEST:

	    if(o != null) {

		if(o instanceof UsersInterface) {

		    notifyIfSwallowingComplete();
		    return new ScriptResult(this, userPanel);

                } else if(o instanceof ChatInterface) {

		    notifyIfSwallowingComplete();
		    return new ScriptResult(this, chatPanel);

		} else if(o instanceof NewsInterface) {

		    notifyIfSwallowingComplete();
		    return new ScriptResult(this, newsPanel);

		} else if(o instanceof TransferListInterface) {

                    notifyIfSwallowingComplete();
                    return new ScriptResult(this, transfersPanel);

                } else if(o instanceof FilesInterface) {

                    notifyIfSwallowingComplete();
                    return new ScriptResult(this, filesPanel);

                }

	    }

	    break;

	case Scripting.UNSWALLOW_REQUEST:
	    return new ScriptResult(this, null);

	case Scripting.REPAINT:
	    return new ScriptResult(this, null);

	}

	throw new UnknownMessageException("Unknown message in ConnectionInterface");

    }



}
