/**
 * $Id: FileInfoInterface.java,v 1.5 2001/10/02 19:21:46 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.awt.*;
import java.awt.event.*;
import java.io.IOException;
import java.util.Calendar;
import java.text.SimpleDateFormat;
import java.text.NumberFormat;

import javax.swing.*;
import javax.swing.border.*;

import redlight.hotline.*;
import redlight.utils.BytesFormat;
import redlight.utils.TextUtils;
import redlight.utils.DebuggerOutput;
import redlight.graphics.Spinner;

/**
 * The file info edit interface.
 */
public class FileInfoInterface 
    implements Child,
	       Runnable {

    Machine rlm;
    Container contentPane;
    JTextField filename;
    JTextArea comment, infoHelpText, commentsHelpText;
    JFrame f;
    JPopupMenu popupMenu;
    JButton save, delete, cancel;
    JLabel type, creator, size, created, modified;
    JScrollPane scrollPane;
    Spinner spinner;
    String path;
    HLProtocol.FileInfo fileinfo;
    Thread getter;
    SimpleDateFormat date_formatter;

    public FileInfoInterface(Machine machine, 
                             String p) {
	rlm = machine;
	path = p;

	date_formatter
	    = new SimpleDateFormat ("EEE, MMM dd, yyyy, kk:mm:ss");

	String lastitem = path.toString().
	    substring(path.toString().
		      lastIndexOf(HLProtocol.DIR_SEPARATOR) + 1);
        
        ActionListener actionListener = new ActionListener() {
                
                public void actionPerformed(ActionEvent e) {
                    
                    try {
                        
                        if(e.getActionCommand().equals("Save")) {
                            
                            rlm.getHLC().requestFileInfoChange(path, filename.getText(), TextUtils.findAndReplace(comment.getText(), "\n", "\r"));
                            
                        } else if(e.getActionCommand().equals("Delete")) {
                            
                            /* Not implemented. */
                            
                        }
                        
                    } catch (IOException ex) {}
                    
                    close();
                    
                }

            };

	Font guiFont = (Font) Main.rlo.getProperty("Font.gui");

	f = new JFrame(lastitem + " (" + rlm.getServerName() + ")");
	
	contentPane = f.getContentPane();
	GridBagLayout gbl = new GridBagLayout();
	GridBagConstraints gbc = new GridBagConstraints();
	JPanel infoPanel = new JPanel();
        JPanel commentsPanel = new JPanel();
        infoHelpText = new JTextArea("Retrieving file info, please wait ...\n", 2, 20);
        commentsHelpText = new JTextArea("Retrieving comment ...\n", 2, 20);
	JLabel l1 = new JLabel("Name: ");
	JLabel l2 = new JLabel("Type: ");
	JLabel l3 = new JLabel("Creator: ");
	JLabel l4 = new JLabel("Size: ");
	JLabel l5 = new JLabel("Created: ");
	JLabel l6 = new JLabel("Modified: ");
	type = new JLabel("");
	creator = new JLabel("");
	size = new JLabel("");
	created = new JLabel("");
	modified = new JLabel("");
	JPanel buttonPanel = new JPanel();
	JPanel deleteButtonPanel = new JPanel();
	JPanel otherButtonPanel = new JPanel();
	spinner = new Spinner(Main.rlo.getImageSet("Spinner"), 
			      100, 16, 16);
	filename = new JTextField(lastitem, 20);
	comment = new JTextArea(2, 20);
	scrollPane = new JScrollPane(comment);
        cancel = new JButton("Cancel");
	save = new JButton("Save");
	delete = new JButton("Delete");

        Border emptyBorder = 
            new EmptyBorder((Insets) Main.rlo.getProperty("BorderInsets"));
        Border lineBorder = 
            BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "File info");
        Border infoBorder = 
            BorderFactory.createCompoundBorder(lineBorder, emptyBorder);
        lineBorder = 
            BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Comments");
        Border commentsBorder = 
            BorderFactory.createCompoundBorder(lineBorder, emptyBorder);

        infoPanel.setBorder(infoBorder);
        commentsPanel.setBorder(commentsBorder);

	buttonPanel.setLayout(new BorderLayout());
        buttonPanel.registerKeyboardAction(actionListener, "Cancel", KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), JComponent.WHEN_IN_FOCUSED_WINDOW);
	contentPane.setLayout(gbl);
	otherButtonPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
	infoPanel.setLayout(new GridBagLayout());
        commentsPanel.setLayout(new GridBagLayout());
        deleteButtonPanel.setLayout(new FlowLayout(FlowLayout.LEFT));

        type.setFont(guiFont);
        creator.setFont(guiFont);
        size.setFont(guiFont);
        created.setFont(guiFont);
        modified.setFont(guiFont);
        type.setForeground(infoHelpText.getForeground());
        creator.setForeground(infoHelpText.getForeground());
        size.setForeground(infoHelpText.getForeground());
        created.setForeground(infoHelpText.getForeground());
        modified.setForeground(infoHelpText.getForeground());

        comment.setLineWrap(true);
	comment.setWrapStyleWord(true);

	comment.setBackground(rlm.getSchemeColor("background"));
	comment.setForeground(rlm.getSchemeColor("foreground"));
	filename.setBackground(rlm.getSchemeColor("background"));
	filename.setForeground(rlm.getSchemeColor("foreground"));

	cancel.addActionListener(actionListener);
	save.addActionListener(actionListener);
	delete.addActionListener(actionListener);
        cancel.setMnemonic('C');
        save.setMnemonic('S');
        infoHelpText.setLineWrap(true);
        infoHelpText.setWrapStyleWord(true);
        infoHelpText.setOpaque(false);
        infoHelpText.setEditable(false);
        commentsHelpText.setLineWrap(true);
        commentsHelpText.setWrapStyleWord(true);
        commentsHelpText.setOpaque(false);
        commentsHelpText.setEditable(false);

        filename.setEnabled(false);
        type.setEnabled(false);
        creator.setEnabled(false);
        size.setEnabled(false);
        created.setEnabled(false);
        modified.setEnabled(false);
        comment.setEnabled(false);
        save.setEnabled(false);

        /* File info panel. */

        gbc.gridx = 0;
	gbc.gridy = 0;
	gbc.fill = gbc.HORIZONTAL;
	gbc.anchor = gbc.NORTHEAST;
	gbc.weightx = 0;
        gbc.ipadx = 4;
        gbc.ipady = 4;
        gbc.gridwidth = 2;
        Insets oldInsets = gbc.insets;
        gbc.insets = new Insets(0, 0, 8, 0);
        infoPanel.add(infoHelpText, gbc);
        gbc.insets = oldInsets;

        gbc.gridwidth = 1;
        gbc.gridy++;
	infoPanel.add(l1, gbc);	
        gbc.gridy++;
	infoPanel.add(l2, gbc);
        gbc.gridy++;
	infoPanel.add(l3, gbc);
        gbc.gridy++;
	infoPanel.add(l4, gbc);
        gbc.gridy++;
	infoPanel.add(l5, gbc);
        gbc.gridy++;
	infoPanel.add(l6, gbc);
	gbc.weightx = 1;
	gbc.gridwidth = gbc.REMAINDER;
	gbc.gridx = 1;
	gbc.fill = gbc.BOTH;
	gbc.gridy = 1;
	infoPanel.add(filename, gbc);
	gbc.gridy = gbc.RELATIVE;
	infoPanel.add(type, gbc);
	infoPanel.add(creator, gbc);
	infoPanel.add(size, gbc);
	infoPanel.add(created, gbc);
	infoPanel.add(modified, gbc);

        /* Comments panel. */

        gbc.anchor = gbc.NORTHEAST;
	gbc.gridx = 0;
	gbc.gridy = 0;
        gbc.fill = gbc.HORIZONTAL;
        gbc.weightx = 1;
        gbc.weighty = 0;
        gbc.insets = new Insets(0, 0, 8, 0);
        commentsPanel.add(commentsHelpText, gbc);
        gbc.insets = oldInsets;
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.fill = gbc.BOTH;
	gbc.weighty = 1;
	gbc.weightx = 1;
	commentsPanel.add(scrollPane, gbc);

        /* Button panel. */

	otherButtonPanel.add(cancel);
	otherButtonPanel.add(save);
	buttonPanel.add(deleteButtonPanel, BorderLayout.WEST);
	buttonPanel.add(otherButtonPanel, BorderLayout.EAST);

        /* Add panels to contentpane. */

	gbc.anchor = gbc.EAST;
	gbc.fill = gbc.BOTH;
	gbc.gridx = 0;
	gbc.gridy = 0;
	gbc.weightx = 1;
	gbc.weighty = 0;
	gbc.gridwidth = gbc.REMAINDER;
	gbc.insets = 
	    (Insets) Main.rlo.getProperty("BorderInsets");
	contentPane.add(infoPanel, gbc);
        gbc.gridy++;
        gbc.weighty = 1;
        contentPane.add(commentsPanel, gbc);
        gbc.gridy++;
	gbc.weightx = 1;
	gbc.weighty = 0;
	gbc.fill = gbc.BOTH;
	gbc.anchor = gbc.SOUTH;
	gbc.gridwidth = gbc.REMAINDER;
        gbc.insets = new Insets(0, 0, 0, 8); /* keep a little to the right */
	contentPane.add(buttonPanel, gbc);

        SwingUtilities.getRootPane(f).setDefaultButton(save);

	f.pack();

	Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension w = f.getSize();

        f.setLocation((d.width / 2) - (w.width / 2), 
                      (d.height / 2) - (w.height / 2));

	f.addWindowListener(new WindowAdapter() {

                public void windowClosing(WindowEvent e) {

                    close();

                }

            });

	rlm.getInterface().registerChild(this);

	getter = new Thread(this);
	getter.start();

        f.setVisible(true);

    }
    
    /**
     * The following methods implement Runnable.
     */
    public void run() {

	spinner.start();

	try {

	    fileinfo = rlm.getHLC().getFileInfo(path);

	    if(fileinfo != null) {

                infoHelpText.setText("To rename this file, type the new name and hit Save.");
                commentsHelpText.setText("To change this comment, type the new comment and hit Save.");
		filename.setText(fileinfo.name);
		type.setText(fileinfo.type);
		creator.setText(fileinfo.creator);
		size.setText(BytesFormat.format(fileinfo.size) +
			     " (" + 
			     NumberFormat.getInstance().format(fileinfo.size) +
			     " bytes)");
		created.setText(date_formatter.
				format(fileinfo.created.getTime()));
		modified.setText(date_formatter.
				 format(fileinfo.modified.getTime()));
		comment.setText(TextUtils.findAndReplace(fileinfo.comment, 
							 "\r", "\n"));

                filename.setEnabled(true);
                type.setEnabled(true);
                creator.setEnabled(true);
                size.setEnabled(true);
                created.setEnabled(true);
                modified.setEnabled(true);
                comment.setEnabled(true);
                save.setEnabled(true);

	    } else {

		close();

	    }
	    
	} catch (InterruptedException e) {
	} catch (HLException e) {

            close();

	} catch (IOException e) {

            close();

	}

	spinner.stop();

    }

    /**
     * The following methods implement Child.
     */
    public void close() {

	DebuggerOutput.debug(this.getClass().getName() + 
			     ": close()");
	rlm.getInterface().unregisterChild(this);
        getter.interrupt();
	f.dispose();

    }

    public void displayPropertyChanged(String what, Object property) {

	if(what.equals("listfont")) { 
	    
	    
	} else {

	    comment.setBackground(rlm.getSchemeColor("background"));
	    comment.setForeground(rlm.getSchemeColor("foreground"));
	    filename.setBackground(rlm.getSchemeColor("background"));
	    filename.setForeground(rlm.getSchemeColor("foreground"));
	    
	}

	f.repaint();
    }
    
}
