/**
 * $Id: FilesInterface.java,v 1.20 2001/10/09 02:45:41 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Date;
import java.util.Stack;

import java.awt.datatransfer.*;
import java.awt.dnd.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.SwingUtilities;

import redlight.hotline.*;
import redlight.utils.BytesFormat;
import redlight.utils.TextUtils;
import redlight.utils.DebuggerOutput;
import redlight.graphics.Spinner;
import redlight.script.*;

/**
 * Renderer for the tree cells.
 */

class FileNameTreeCellRenderer extends DefaultTreeCellRenderer {

    static Color idleColor = new Color(0xababab);

    FileNameTreeCellRenderer() {

	setOpaque(true);

    }

    public Component getTreeCellRendererComponent(JTree tree,
						  Object value,
						  boolean sel,
						  boolean expanded,
						  boolean leaf,
						  int row,
						  boolean hasFocus) {

	DefaultMutableTreeNode tn = (DefaultMutableTreeNode) value;
        Object label = value;
        HLProtocol.FileListComponent file = null;

        if(tn.getUserObject() instanceof HLProtocol.FileListComponent) {

            file = (HLProtocol.FileListComponent) tn.getUserObject();

            if(file.fileType.equals("fldr"))
                leaf = false;

            label = file.fileName;

        }

	super.getTreeCellRendererComponent(tree, 
					   label,
					   sel, 
					   expanded, 
					   leaf, 
					   row, 
					   hasFocus);
	setBackground(tree.getBackground());

        if(file != null && file.fileType.equals("alis"))
            setForeground(idleColor);
        else
            setForeground(tree.getForeground());
        
        if(Main.iconFile != null) 
            setIcon(Main.iconFile.getIconForFile(file));

	return this;

    }

}

/**
 * TreeTableModel for files.
 */

class FileTreeTableModel extends AbstractTreeTableModel {

    /* Names of the columns. */

    static protected String[]  cNames = { "Name", "Size" };

    /* Types of the columns. */

    static protected Class[]  cTypes = { TreeTableModel.class,
					 String.class };
    
    public FileTreeTableModel(TreeNode root) {

	super(root);

    }

    /**
     * Returns the number of columns.
     */
    public int getColumnCount() {

	return cNames.length;

    }

    /**
     * Returns the name for a particular column.
     */
    public String getColumnName(int column) {

	return cNames[column];

    }

    /**
     * Returns the class for the particular column.
     */
    public Class getColumnClass(int column) {

	return cTypes[column];

    }

    /**
     * Returns the value of the specified column.
     */
    public Object getValueAt(Object node, int column) {
	DefaultMutableTreeNode tn = (DefaultMutableTreeNode) node;

        /* Shortcut for String values, independant of column. */

        if(tn.getUserObject() instanceof String) {

            if(tn.getUserObject().equals("?") && column == 0)
                return "Please wait ...";
            else
                return tn.getUserObject();

        }

        HLProtocol.FileListComponent file = (HLProtocol.FileListComponent) tn.getUserObject();

        switch(column) {

        case 0:

            /* File name. */

            return file.fileName;

        case 1:

            /* File (or directory) size. */

            if(file.fileType.equals("fldr"))
                return "(" + new Long(file.fileSize).toString() + ")";
            else if(file.fileType.equals("alis"))
                return "?";
            else
                return BytesFormat.format(file.fileSize);                
            
        default:

            return null;

        }

    }

}

/**
 * The files interface.
 */
public class FilesInterface 
    implements ActionListener, 
               Child {

    Machine rlm;
    Container contentPane;
    JTreeTable list;
    JFrame f;
    JScrollPane scrollPane;
    JPopupMenu popupMenu;
    JMenuItem infoItems, deleteItems, downloadItems, uploadItem, openInNewWindow, viewItem;
    JButton reload, download, upload, delete, mkdir, search, getinfo, newWindow;
    JLabel info;
    Spinner spinner;
    HLProtocol.FileListComponent[] fileList;
    boolean threadsInterrupted;
    String path, previousDirectory;
    Hashtable expandedPaths;
    int previousItemState = ItemEvent.DESELECTED, fileGettersActive;
    FileTreeTableModel treeModel;
    DefaultMutableTreeNode rootNode;
    TreePath openNewWindowPrimedPath;
    boolean swallow;

    public FilesInterface(Machine machine, String pa, boolean sw) {
	rlm = machine;
	path = pa;
        swallow = sw;

	Font guiFont = (Font) Main.rlo.getProperty("Font.gui");

	if(swallow == false) {

            f = new JFrame("Files (" + rlm.getServerName() + ")");
            contentPane = f.getContentPane();

	} else {

	    try {

		ScriptResult[] sr = rlm.getScriptBroker().
		    executeScript(new ScriptObject(Scripting.
						   SWALLOW_REQUEST, 
						   this));
                contentPane = (Container) sr[0].getUserObject();

	    } catch (UnknownMessageException e) {

		e.printStackTrace();

	    }

	}

	GridBagLayout gbl = new GridBagLayout();
	GridBagConstraints gbc = new GridBagConstraints();
	JToolBar toolbar = new JToolBar();
	reload = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.REFRESH_ICON]));
        newWindow = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.FILES_ICON]));
	getinfo = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.INFO_ICON]));
	download = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.DOWNLOAD_ICON]));
	upload = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.UPLOAD_ICON]));
	mkdir = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.FOLDER_ICON]));
	delete = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.TRASH_ICON]));
	search = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.FIND_ICON]));
	spinner = new Spinner(Main.rlo.getImageSet("Spinner"), 
			      100, 16, 16);
	JPanel spinnerPanel = new JPanel();
	rootNode = new DefaultMutableTreeNode(path);
	treeModel = new FileTreeTableModel(rootNode);
	list = new JTreeTable(rlm, path, treeModel);
	JScrollPane scrollPane = new JScrollPane(list);
        scrollPane.getViewport().setBackground(list.getBackground());
	info = new JLabel();
	popupMenu = new JPopupMenu();
	infoItems = new JMenuItem("Get info on selected items");
	downloadItems = new JMenuItem("Download selected items");
	openInNewWindow = new JMenuItem("Open in new window");
	deleteItems = new JMenuItem("Delete selected items");
        uploadItem = new JMenuItem("Upload into selected directory");
        viewItem = new JMenuItem("View selected items");
	FileTCTableCellRenderer ftcTableCellRenderer = 
	    new FileTCTableCellRenderer();
	
	contentPane.setLayout(gbl);
        toolbar.setFloatable(false);
	spinnerPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
        reload.setMnemonic('R');
	reload.setActionCommand("Reload");
        newWindow.setActionCommand("ActivateNewWindow");
	download.setActionCommand("Download");
	upload.setActionCommand("Upload");
	getinfo.setActionCommand("Info");
	mkdir.setActionCommand("MkDir");
	delete.setActionCommand("Delete");
	search.setActionCommand("Search");
	reload.setFont(guiFont);
        newWindow.setFont(guiFont);
	download.setFont(guiFont);
	upload.setFont(guiFont);
	mkdir.setFont(guiFont);
	delete.setFont(guiFont);
	search.setFont(guiFont);
	getinfo.setFont(guiFont);
	popupMenu.setFont(guiFont);
	infoItems.setFont(guiFont);
	downloadItems.setFont(guiFont);
	openInNewWindow.setFont(guiFont);
	deleteItems.setFont(guiFont);
        uploadItem.setFont(guiFont);
        viewItem.setFont(guiFont);
	getinfo.setEnabled(false);
	delete.setEnabled(false);
	download.setEnabled(false);
        
        reload.setToolTipText("Refresh the list of files");
        newWindow.setToolTipText("Open in separate window");
        search.setToolTipText("Search for files on the server");
	download.setToolTipText("Download selected files");
	upload.setToolTipText("Upload a file");
	mkdir.setToolTipText("Create new directory");
	getinfo.setToolTipText("Get info on selected items");
	delete.setToolTipText("Delete selected items");
	info.setFont((Font) Main.rlo.getProperty("Font.info"));
	list.setBackground(rlm.getSchemeColor("background"));
	list.setForeground(rlm.getSchemeColor("foreground"));
	list.getTree().setBackground(rlm.getSchemeColor("background"));
	list.getTree().setForeground(rlm.getSchemeColor("foreground"));
	list.setFont((Font) Main.rlo.getProperty("Font.list"));
	list.getColumnModel().getColumn(0).setPreferredWidth(750);
	list.getColumnModel().getColumn(1).setPreferredWidth(175);
        //	list.getColumnModel().getColumn(2).setPreferredWidth(150);
        //	list.getColumnModel().getColumn(3).setPreferredWidth(150);
	list.getTree().setCellRenderer(new FileNameTreeCellRenderer());
	list.getColumn("Size").setCellRenderer(new FileSizeTableCellRenderer());
        //	list.getColumn("Type").setCellRenderer(ftcTableCellRenderer);
        //	list.getColumn("Creator").setCellRenderer(ftcTableCellRenderer);
	list.getTree().setFont((Font) Main.rlo.getProperty("Font.list"));
	list.getTree().addTreeWillExpandListener(new TreeWillExpandListener() {

                public void treeWillCollapse(TreeExpansionEvent e) {}
                public void treeWillExpand(TreeExpansionEvent e) {
                    
                    TreePath tp = e.getPath();
                    
                    if(!expandedPaths.containsKey(tp)) {
                        
                        if(tp.getPathCount() > 1) {
                            
                            expandedPaths.put(tp, treePathToFilePath(path, tp));
                            new GetFiles(treePathToFilePath(path, tp), (DefaultMutableTreeNode) tp.getLastPathComponent());
                            
                        }
                        
                    }
                    
                }

            });

	list.getTree().addTreeExpansionListener(new TreeExpansionListener() {
                
                public void treeCollapsed(TreeExpansionEvent e) {
                    
                    updateInfo();
                    
                }
                
                public void treeExpanded(TreeExpansionEvent e) { 
                    
                    updateInfo();	
                    
                }
                
            });

	scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

        viewItem.addActionListener(this);
        uploadItem.addActionListener(this);
	downloadItems.addActionListener(this);
	deleteItems.addActionListener(this);
	openInNewWindow.addActionListener(this);
	infoItems.addActionListener(this);
	reload.addActionListener(this);
        newWindow.addActionListener(this);
	download.addActionListener(this);
	upload.addActionListener(this);
	delete.addActionListener(this);
	mkdir.addActionListener(this);
	search.addActionListener(this);
	getinfo.addActionListener(this);
	list.addMouseListener(new MouseAdapter() {

                public void mousePressed(MouseEvent e) {
                    
                    mouseReleased(e);
                    
                }
                
                public void mouseReleased(MouseEvent e) {
                    
                    if(e.isPopupTrigger()) {
                        
                        int[] selectedItems = list.getTree().getSelectionRows();
                        
                        /* Configure the popup menu. */
                        
                        deleteItems.setEnabled(selectedItems == null ? 
                                               false : true);
                        downloadItems.setEnabled(selectedItems == null ? 
                                                 false : true);
                        openInNewWindow.setEnabled(false);
                        uploadItem.setEnabled(false);
                        viewItem.setEnabled(selectedItems == null ? 
                                            false : true);
                        
                        int selRow = list.getTree().getClosestRowForLocation(e.getX(), e.getY());
                        TreePath selPath = list.getTree().getPathForRow(selRow);
                        
                        if(selRow != -1) {
                            
                            HLProtocol.FileListComponent files = (HLProtocol.FileListComponent) ((DefaultMutableTreeNode) selPath.getLastPathComponent()).getUserObject();
                            
                            if(files.fileType.equals("fldr")) {
                                
                                if(selectedItems != null && selectedItems.length == 1) {
                                    
                                    openInNewWindow.setEnabled(true);
                                    uploadItem.setEnabled(true);
                                    
                                }
                                
                                openNewWindowPrimedPath = selPath;
                                
                            }
                            
                        }
                        
                        popupMenu.show(list, e.getX(), e.getY());
                        
                    }
                    
                }
                
                public void mouseClicked(MouseEvent e) {
                    int selRow = list.getTree().getClosestRowForLocation(e.getX(), e.getY());
                    
                    TreePath selPath = list.getTree().getPathForRow(selRow);
                    
                    if(selRow != -1) {
                        
                        if(((DefaultMutableTreeNode) selPath.getLastPathComponent()).getUserObject() instanceof HLProtocol.FileListComponent) {
                            
                            HLProtocol.FileListComponent files = (HLProtocol.FileListComponent) ((DefaultMutableTreeNode) selPath.getLastPathComponent()).getUserObject();
                            
                            if(e.getClickCount() == 2) 
                                if(!files.fileType.equals("fldr")) 
                                    itemActivated(e, selPath);
                            
                        }
                        
                    }
                    
                }
                
            });
        
        list.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
                
                public void valueChanged(ListSelectionEvent e) {
                    
                    if(!e.getValueIsAdjusting()) {
                        
                        delete.setEnabled(e.getFirstIndex() == -1 ?
                                          false : true);
                        getinfo.setEnabled(e.getFirstIndex() == -1 ?
                                           false : true);
                        download.setEnabled(e.getFirstIndex() == -1 ?
                                            false : true);
                        
                    }
                    
                }
                
            });

        list.registerKeyboardAction(this, "Activate", 
                                    KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, 
                                                           0), 
                                    JComponent.WHEN_IN_FOCUSED_WINDOW);

	popupMenu.add(infoItems);
        popupMenu.add(viewItem);
	popupMenu.add(downloadItems);
        popupMenu.add(uploadItem);
	popupMenu.add(openInNewWindow);
	popupMenu.addSeparator();
	popupMenu.add(deleteItems);
	toolbar.add(reload);
        toolbar.add(newWindow);
	toolbar.add(search);	
	toolbar.add(getinfo);
	toolbar.add(download);
	toolbar.add(upload);
	toolbar.add(mkdir);
	toolbar.add(delete);
	gbc.gridwidth = gbc.RELATIVE;
	gbc.fill = GridBagConstraints.BOTH;
	contentPane.add(toolbar, gbc);
	spinnerPanel.add(spinner);
	gbc.gridx = 1;
	gbc.gridwidth = gbc.REMAINDER;
	contentPane.add(spinnerPanel, gbc);
	gbc.gridx = 0; gbc.gridy = 1;
	gbc.weightx = 1; gbc.weighty = 1;
	gbc.fill = GridBagConstraints.BOTH;
	gbc.gridwidth = gbc.REMAINDER;
        gbc.insets =
            (Insets) Main.rlo.getProperty("BorderInsets");
	contentPane.add(scrollPane, gbc);
	gbc.gridx = 0; gbc.gridy = 2;
	gbc.weightx = 0; gbc.weighty = 0;
	gbc.gridwidth = gbc.REMAINDER;
	gbc.fill = gbc.BOTH;
	gbc.anchor = gbc.EAST;
	contentPane.add(info, gbc);

        if(swallow == false) {

            f.pack();
            f.setSize(350, 375);
            
            Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
            Dimension w = f.getSize();
            
            f.setLocation((d.width / 2) - (w.width / 2), 
                          (d.height / 2) - (w.height / 2));

            f.addWindowListener(new WindowAdapter() {

                    public void windowClosing(WindowEvent e) {
                        
                        close();
                        
                    }

                });

            f.show();
            
            rlm.getInterface().registerChild(this);
            
        }

        expandedPaths = new Hashtable();
	new GetFiles(path, rootNode);

    }
   
    void updateInfo() {

	new Thread(new Runnable() {

	    public void run() {

		if(fileGettersActive != 0)
                    return;
                
                try {
                    
                    Thread.currentThread().sleep(1000);
                    
                    if(reload.isEnabled()) {
                        
                        ItemCount ic = getItemCountForTree();
                        info.setText((ic.files + ic.dirs) + 
                                     " items (" + ic.files + " files, " + 
                                     ic.dirs + " directories).");
                        info.repaint();
                        //                            show();
                        
                    }
                    
                } catch(InterruptedException e) {}
                
	    }
                
            }).start();
        
    }
    
    /**
     * This class ties a list of items to a node in
     * the tree.
     * @param f a list of files
     * @param n the node to tie the list to
     */
    class AddFilesToNode implements Runnable {

	HLProtocol.FileListComponent[] fileList;
	DefaultMutableTreeNode root;

	AddFilesToNode(HLProtocol.FileListComponent[] f, 
		       DefaultMutableTreeNode n) {

	    fileList = f;
	    root = n;

	}

	public void run() {

	    synchronized(treeModel) {

                //                list.setEnabled(false);

		for(int i=0; i < fileList.length; i++) { 

		    DefaultMutableTreeNode node = new DefaultMutableTreeNode(fileList[i], (fileList[i].fileType.equals("fldr") ? true : false));

		    if(fileList[i].fileType.equals("fldr")) 
			node.add(new DefaultMutableTreeNode("?"));

		    synchronized(root) {

			root.add(node);

		    }

		}

		if(root != rootNode) { 

		    DefaultMutableTreeNode c = 
			(DefaultMutableTreeNode) treeModel.getChild(root, 0);
		    treeModel.removeNodeFromParent((MutableTreeNode) c);

		}

		treeModel.nodeStructureChanged((TreeNode) root);
		list.getTree().setRootVisible(false);
                //                int min = list.getSelectionModel().getMinSelectionIndex();
                //                int max = list.getSelectionModel().getMaxSelectionIndex();
                //                list.getSelectionModel().setSelectionInterval(min, max);
                //                list.setEnabled(true);
		list.repaint();

	    }

	}

    }

    /**
     * This class retrieves a directory listing and ties
     * it to the specified node in the tree.
     * @param p the directory of which to retrieve a listing
     * @param n the node to tie the directory listing to
     */
    class GetFiles implements Runnable {
	String pathToGet;
	DefaultMutableTreeNode nodeToExpand;
	Thread me;

	GetFiles(String p, DefaultMutableTreeNode n) {

	    pathToGet = p;
	    nodeToExpand = n;

	    if(!threadsInterrupted) {

		me = new Thread(this, 
				"Files: " + path + "; " + 
				rlm.getServerName());
		me.start();

	    }

	}

	public void run() {

	    if(!threadsInterrupted) {

		fileGettersActive++;
		spinner.start();
		reload.setEnabled(false);
		delete.setEnabled(false);
		getinfo.setEnabled(false);
		download.setEnabled(false);

		try {

		    info.setText("Getting information...");
		    fileList = rlm.getHLC().getFileList(pathToGet);

                    /* Need to trim "Please wait" entry on error. */

                } catch(HLException _e) {

                    fileList = null;

		} catch(IOException _e) {

                    fileList = null;

		} catch(InterruptedException _e) {
                    
                    fileList = null;

		} finally {

		    if(!threadsInterrupted) {

                        if(fileList == null)
                            fileList = new HLProtocol.FileListComponent[0];

						// sort the list					
						redlight.utils.QuickSort.sort(fileList, 0, fileList.length-1);

			SwingUtilities.invokeLater(new AddFilesToNode(fileList, nodeToExpand));

                        if(swallow == false)
                            f.setTitle((path.equals("") ? "Files" : path) + " (" + rlm.getServerName() + ")");

			expandedPaths.put(pathToGet, new Integer(1));

		    }

                }

	    }

	    if(!threadsInterrupted) {

		reload.setEnabled(true);
		spinner.stop();
		fileGettersActive--;
		updateInfo();

	    }

	}

    }

    class ItemCount { 

	int files = 0;
	int dirs = 0;

    }

    ItemCount getItemCountForTree() {
	ItemCount ic = new ItemCount();
	JTree tree = list.getTree();

	for(int i=0; i<tree.getRowCount(); i++) {

	    TreePath tp = tree.getPathForRow(i);
	    Object userObject = ((DefaultMutableTreeNode) tp.getLastPathComponent()).getUserObject();

	    if(userObject instanceof String) 
                continue;

	    HLProtocol.FileListComponent file =
                (HLProtocol.FileListComponent) userObject;

	    if(!file.fileType.equals("fldr")) 
		ic.files++;
	    else
		ic.dirs ++;

	}

	return ic;

    }

    void itemDelete(TreePath[] treePaths) {
	String filePath;
	boolean okToDelete = false, deleteBatch = false;

	if(treePaths.length > 1) {

            /* When there is more than one selected file, ask
               permission to do a batch delete of all selected
               files. */

	    int r = JOptionPane.showConfirmDialog(f, "Are you sure you wish to delete ALL selected items?", "Delete?", JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);

	    if(r == JOptionPane.YES_OPTION)
                deleteBatch = true;

      	    if(!deleteBatch) 
                return;

	}

        /* Walk over all selected files and delete them one by one.
           If this is not a batch delete, ask for permission before
           deleting. */

	for(int i = 0; i < treePaths.length; i++) {

	    filePath = treePathToFilePath(path, treePaths[i]);

	    if(!deleteBatch) {

		okToDelete = false;
		int r = JOptionPane.showConfirmDialog(f, TextUtils.wordWrap("Are you sure you wish to delete " + filePath, 40), "Delete?", JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);

		if(r == JOptionPane.YES_OPTION) 
                    okToDelete = true;

	    }

	    if(deleteBatch || okToDelete) {

		try {

		    rlm.getHLC().requestFileDelete(filePath);
                    Thread.currentThread().sleep(20);

		} catch(IOException ex) {
                } catch(InterruptedException ex) {}

	    }

	}

    }

    void itemGetInfo(TreePath[] treePaths) {
	String filePath;

	for(int i=0; i<treePaths.length; i++) {

	    filePath = treePathToFilePath(path, treePaths[i]);
	    new FileInfoInterface(rlm, filePath);

	}

    }
    
    void itemUpload(TreePath[] treePaths) {
        String target = null;
        
        if(treePaths != null) {
            
            if(treePaths.length == 1) {
                
                TreePath tp = treePaths[0];
                HLProtocol.FileListComponent files =
                    (HLProtocol.FileListComponent) 
                    ((DefaultMutableTreeNode) tp.getLastPathComponent()).getUserObject();	
                
                String filePath = treePathToFilePath(path, tp);
                
                if(files.fileType.equals("fldr"))
                    target = filePath;
                
            }

        }

        if(target == null)
            target = path;

        if(target != null) {
            
            FileDialog fd = new FileDialog(Main.getActiveShell().f, "Upload file", FileDialog.LOAD);
            
            if(previousDirectory != null)
                fd.setDirectory(previousDirectory);
            
            fd.show();
            
            if(fd.getDirectory() != null)
                previousDirectory = fd.getDirectory();
            
            if(fd.getFile() != null) {

                File sourceFile = new File(fd.getDirectory(), fd.getFile());
                new UploadInterface(rlm, 
                                    target, 
                                    sourceFile);

            }                
            
        }

    }
    
    void itemView(TreePath[] treePaths) {

	for(int i=0; i < treePaths.length; i++) {

	    TreePath tp = treePaths[i];
	    String filePath = treePathToFilePath(path, tp);
            Object o = ((DefaultMutableTreeNode) tp.getLastPathComponent()).getUserObject();

            if(o instanceof HLProtocol.FileListComponent) {

                HLProtocol.FileListComponent files = (HLProtocol.FileListComponent) o;	
                
                if(files.fileType.equals("fldr"))
                    DebuggerOutput.debug(filePath + ": Can't view directories");
                else
                    new ViewInterface(rlm,
                                      filePath,
                                      files,
                                      null);

            }

	}

    }

    void itemDownload(TreePath[] treePaths) {

	for(int i=0; i<treePaths.length; i++) {

	    TreePath tp = treePaths[i];
	    HLProtocol.FileListComponent files =
		(HLProtocol.FileListComponent) ((DefaultMutableTreeNode) tp.getLastPathComponent()).getUserObject();	
	    String filePath = treePathToFilePath(path, tp);

	    if(files.fileType.equals("fldr"))
                new FolderDownloadInterface(rlm,
                                            filePath,
                                            files);
	    else
		new DownloadInterface(rlm, 
                                      filePath,
                                      files);
	}

    }

    void itemActivated(InputEvent e, TreePath tp) {
	HLProtocol.FileListComponent files =
	    (HLProtocol.FileListComponent) ((DefaultMutableTreeNode) tp.getLastPathComponent()).getUserObject();	
	String filePath = treePathToFilePath(path, tp);

 	if(files.fileType.equals("fldr"))
	    DirOpener.open(rlm, filePath);
	else
	    new DownloadInterface(rlm, 
                                  filePath,
                                  files);

    }

    /**
     * Converts a tree path to a Hotline file path.
     */
    static String treePathToFilePath(String rootPath, TreePath tp) {
	Object[] components = tp.getPath();
        String thePath = new String();

        thePath = rootPath.equals(":") ? "" : rootPath;

        for(int i = 1; i < components.length; i ++) {

            DefaultMutableTreeNode mtn = (DefaultMutableTreeNode) components[i];
            String label = mtn.getUserObject().toString();

            if(mtn.getUserObject() instanceof HLProtocol.FileListComponent)
                label = ((HLProtocol.FileListComponent) mtn.getUserObject()).fileName;

	    thePath +=  HLProtocol.DIR_SEPARATOR + label;

        }

        return thePath;

    }

    /**
     * Cancels all threads (eventually) by setting the appropriate
     * flag.
     */
    void cancelAllThreads() {

	threadsInterrupted = true;

    }

    /**
     * May be called by DirOpener.
     */
    protected void show() {

        if(swallow == false)
            f.toFront();

        list.requestFocus();

    }
    
    /**
     * The following methods implement ActionListener.
     */
    public void actionPerformed(ActionEvent e) {

        TreePath[] treePaths = list.getTree().getSelectionPaths();

	if(e.getActionCommand().equals("Reload")) {

	    /* Reset everything and reload the root node. */

	    rootNode = new DefaultMutableTreeNode(path);
	    expandedPaths.clear();

	    synchronized(treeModel) {

		treeModel.setRoot(rootNode);
		treeModel.reload();

	    }

	    new GetFiles(path, rootNode);
	    reload.setEnabled(false);

        } else if(e.getActionCommand().equals("ActivateNewWindow")) {

	    DirOpener.open(rlm, path);

        } else if(e.getActionCommand().equals("MkDir")) {

	    new DirCreateInterface(rlm, path);

	} else if(e.getActionCommand().equals("Search")) {
            
	    new FileSeeker(rlm, path);
            
	} else if(e.getSource() == openInNewWindow) {
            
	    itemActivated(null, openNewWindowPrimedPath);

	} else if(e.getActionCommand().equals("Upload")) {
                
            if(treePaths == null)
                itemUpload(null);
            else if(treePaths.length == 1)
                itemUpload(treePaths);
            
        } else {
            
            if(treePaths == null)
                return;
            
            if(e.getActionCommand().equals("Info")) {
                
                itemGetInfo(treePaths);
                
            } else if(e.getActionCommand().equals("Delete")) {
                
                itemDelete(treePaths);
                
            } else if(e.getActionCommand().equals("Download")) {
                
                itemDownload(treePaths);
                
            } else if(e.getSource() == deleteItems) {
                
                itemDelete(treePaths);
                
            } else if(e.getSource() == downloadItems) {
                
                itemDownload(treePaths);
                
            } else if(e.getSource() == infoItems) {
                
                itemGetInfo(treePaths);
                
            } else if(e.getSource() == uploadItem) {
                
                itemUpload(treePaths);
                
            } else if(e.getSource() == viewItem) {
                
                itemView(treePaths);
                
            } else if(e.getActionCommand().equals("Activate")) {
                
                if(treePaths.length != 1)
                    return;
                
                TreePath selPath = treePaths[0];
                Object o =
                    ((DefaultMutableTreeNode) 
                     selPath.getLastPathComponent()).getUserObject();
                
                if(!(o instanceof HLProtocol.FileListComponent))
                    return;
                
                HLProtocol.FileListComponent files =
                    (HLProtocol.FileListComponent) o;
                
                /* Download file. */
                
                if(!files.fileType.equals("fldr")) {
                    
                    itemActivated(null, selPath);
                    return;
                    
                }
                
                /* Collapse directories. */
                
                if(list.getTree().isExpanded(selPath)) {
                    
                    list.getTree().collapsePath(selPath);
                    return;
                    
                }
                
                /* Expand. */
                
                //                int min = 
                //                    list.getSelectionModel().getMinSelectionIndex();
                //                int max = 
                //                    list.getSelectionModel().getMaxSelectionIndex();
                list.getTree().expandPath(selPath);
                //                list.getSelectionModel().setSelectionInterval(min, max);
                
            }
            
        }
        
    }
    
    /**
     * The following methods implement Child.
     */
    public void close() {

	rlm.getInterface().unregisterChild(this);
	cancelAllThreads();
	DirOpener.close(rlm, path);

        if(swallow == false) {

            f.setVisible(false);
            f.dispose();

        } else {
            
            try {
                
                rlm.getScriptBroker().
                    executeScript(new ScriptObject(Scripting.
                                                   UNSWALLOW_REQUEST, 
                                                   contentPane));
                
            } catch (UnknownMessageException e) {
                
                e.printStackTrace();
                
            }
            
        }

    }

    public void displayPropertyChanged(String what, final Object property) {

	if(what.equals("listfont")) { 

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        list.setFont((Font) property);
                        list.getTree().setFont((Font) property);

                    }

                });

	} else {

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        list.setBackground(rlm.getSchemeColor("background"));
                        list.setForeground(rlm.getSchemeColor("foreground"));
                        list.getTree().setBackground(rlm.getSchemeColor("background"));
                        list.getTree().setForeground(rlm.getSchemeColor("foreground"));
                        
                    }

                });
	    
	}

	list.repaint();

    }
	
}

