/**
 * $Id: FolderDownloadInterface.java,v 1.6 2001/10/08 22:03:28 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Vector;
import java.util.Enumeration;
import java.util.StringTokenizer;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import redlight.hotline.*;
import redlight.utils.TextUtils;
import redlight.utils.FilenameUtils;
import redlight.utils.DebuggerOutput;
import redlight.utils.BytesFormat;
import redlight.utils.TimeFormat;
import redlight.macfiles.*;
import redlight.utils.Meter;
import redlight.utils.MeterSource;

/**
 * Interface for downloading folders.
 */
public class FolderDownloadInterface extends TransferInterface 
    implements Runnable {
    String path;
    HLProtocol.FileListComponent file;
    long totalSize = 0;
    long folderProgress = 0, previousFolderProgress = 0;
    long previousTimeMillis = 0;
    int filesLeft = 0;
    int totalFiles = 0;
    int folderErrors = 0;
    DepthFirstFileSearch fileSearch;
    Thread downloadThread;
    int maxFolderErrors = Main.rlo.getIntegerProperty("Auto.MaxFolderErrors");

    public FolderDownloadInterface(Machine machine, 
                                   String path, 
                                   HLProtocol.FileListComponent file) {

	super(machine, file.fileName);
	this.path = path;
	this.file = file;

        if(Main.iconFile != null)
            info.setIcon(Main.iconFile.getIconForFile(file));
        
	/* When doRequest() succeeds, granted() is called by the
           superclass. */

	doRequest();

    }

    /**
     * Invoked by the superclass when this transfer is ready to be
     * started.
     */
    public synchronized void granted() {

        downloadThread = new Thread(this);
        downloadThread.setName("FolderDownload " + path + file.fileName);
        downloadThread.start();

    }

    public void run() {
        
        final Vector list = new Vector();
        fileSearch = new DepthFirstFileSearch();

        try {

            /* Build a list of files. */
            
            fileSearch.descend(rlm.getHLC(), 
                               path + HLProtocol.DIR_SEPARATOR, 
                               0, 
                               0, 
                               new FileSearchAction() {
                                       
                                       public void fileFound(String path, HLProtocol.FileListComponent file) {

                                           list.addElement(new Location(path, file));
                                           info.setText("Calculating total size ... " + path);
                                           repaintStatus();
                                           
                                       }
                                       
                                   });
            
            /* Collect the total directory size. */

            Location l = null;

            for(Enumeration en = list.elements(); 
                en.hasMoreElements(); 
                l = (Location) en.nextElement()) {

                /* FIXME: l == null first time around, this loop is
                   dumb. */

                if(l == null) 
                    continue;

                if(!l.file.fileType.equals("fldr")) {
                    
                    totalSize += l.file.fileSize;
                    totalFiles++;
                    
                }

            }

            long oldFolderProgress = 0;
            filesLeft = totalFiles;
            progressBar.setMaximum((int) (totalSize / 100));
            repaintStatus();

            /* Get a constructor object for the MacFile's we're
               creating or fail. */

            Constructor cs = null;

            try {

                String macFileMethod = 
                    Main.rlo.getProperty("MacFileMethod") + "MacFile";
                Class c = Class.forName("redlight.macfiles." + macFileMethod);
                Class[] paramTypes = new Class[] { File.class, Integer.class };
                cs = c.getConstructor(paramTypes);

            } catch(Exception e) {

                rlm.getInterface().error(filename + ": " + 
                                                 e);
                DebuggerOutput.stackTrace(e);

                if(!interrupted)
                    close();

                return;

            }

            /* Download all the files. */

            for(Enumeration en = list.elements(); 
                en.hasMoreElements(); 
                l = (Location) en.nextElement()) {

                /* FIXME: l == null first time around, this loop is
                   dumb. */

                if(l == null || l.file.fileType.equals("fldr"))
                    continue;

                /* Exit loop when the number of errors exceeds the
                   limit or when our thread was interrupted. */

                if(Thread.currentThread().isInterrupted() || 
                   folderErrors > maxFolderErrors) {

                    DebuggerOutput.debug("FolderDownloadInterface.run: was interrupted");
                    break;

                }

                oldFolderProgress = folderProgress;

                try {
                    
                    String localFolder = (String) Main.rlo.getProperty("Path.Downloads").toString(); 
                    StringTokenizer st = new StringTokenizer(l.path, String.valueOf(HLProtocol.DIR_SEPARATOR));

                    /* Strip illegal chars from destination directory
                       names. */

                    while(st.hasMoreTokens())
                        localFolder += System.getProperty("file.separator") + FilenameUtils.qualify(st.nextToken());

                    /* Create the destination or flag an error. */

                    File destination = new File(localFolder);

                    if(!destination.exists())
                        destination.mkdirs();
                    
                    if(!destination.exists()) {
                        
                        folderErrors++;
                        rlm.getInterface().error(filename + ": " + "Cannot create destination folder " + destination);
                        continue;

                    }

                    /* Create the MacFile object to contain the file
                       we're about to download and request the file. */
                    
                    Object[] params =  new Object[] { new File(destination, FilenameUtils.qualify(l.file.fileName)), new Integer(MacFile.READ_PERM | MacFile.WRITE_PERM) };
                    localFile = (MacFile) cs.newInstance(params);
                    
                    setTransferTask(rlm.getHLC().requestFileDownload(l.path + l.file.fileName, localFile, this, false));
                    rlm.getHLC().waitFor(transferTask);

                    /* Sleep a bit or some servers may ban us for
                       flooding. */

                    Thread.currentThread().sleep(20);

                } catch(InterruptedException e) {

                    break;

                } catch(Exception e) {
                    
                    folderErrors++;

                    String error = e.getMessage() == null ? 
                        "Error " + e.toString() : e.getMessage();
                    rlm.getInterface().error(filename + ": " + 
                                             error);
                    DebuggerOutput.stackTrace(e);
                    
                }

                /* The idea here is to advance the progress bar by the
                   size of the file that was transferred, so that
                   resume's and stuff do not confuse the progress
                   meter. For some reason either this, or the
                   totalSize calculation are not very accurate
                   however, and folderProgress can become bigger than
                   totalSize. */

                folderProgress = l.file.fileSize + oldFolderProgress; filesLeft--;
                
            }
            
        } catch(InterruptedException e) {

        } catch(Exception e) {
         
            folderErrors++;

            if(!(e instanceof HLTaskErrorException)) {
                
                String error = e.getMessage() == null ? 
                    "Error " + e.toString() : e.getMessage();
                rlm.getInterface().error(filename + ": " + 
                                                 error);
                DebuggerOutput.stackTrace(e);

            }

        }

        /* The superclass will close us if we were interrupted. */

        if(!interrupted)
            close();

        DebuggerOutput.debug("FolderDownloadInterface.run: exiting");

    }

    /**
     * Following functions implement Meter.
     */
    public void startMeter(MeterSource ms, int ref, String f, final long size) {
        filename = f;
        transferrer = ms;
        xfref = ref;
        previousProgress = 0;
        previousTimeMillis = System.currentTimeMillis();
        DebuggerOutput.debug("FolderDownloadInterface.startMeter[" + transferTask + "]: total = " + total);
        progressSampleCounter = 0;

        SwingUtilities.invokeLater(new Runnable() {
                            
                public void run() {

                    info.setText("[" + filesLeft + "] " + filename + " (" + 
                                 BytesFormat.format(folderProgress) + " / " + 
                                 BytesFormat.format(totalSize) + ")");
                    progressBar.setValue((int) (folderProgress / 100));
                    repaintStatus();

                }

            });

    }

    long bytes_sec = 0;
    long secs_left = 0;

    public void progressMeter(int ref, final long progress) {

        folderProgress += (progress - previousProgress);

        long time_d = System.currentTimeMillis() - previousTimeMillis;
        long bytes_d = folderProgress - previousFolderProgress;

        if(progressSampleCounter % 5 == 0) {

            previousTimeMillis = System.currentTimeMillis();
            previousFolderProgress = folderProgress;
            bytes_sec = bytes_d / ((time_d / 1000) + 1);
            secs_left = (totalSize - folderProgress) / 
                (bytes_sec == 0 ? 1 : bytes_sec);

        }

        progressSampleCounter++;

        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    String text = "[" + filesLeft + "] " + filename + " (" + 
                        BytesFormat.format(folderProgress) + " / " +
                        BytesFormat.format(totalSize);

                    if(bytes_sec != 0) {

                        text += " " + 
                            BytesFormat.format(bytes_sec) + "/s ETA: " +
                            TimeFormat.format(secs_left) + ")";
                    
                    } else {

                        text += ")";

                    }

                    info.setText(text);
                    progressBar.setValue((int) (folderProgress / 100));
                    repaintStatus();

                }

            });

	previousProgress = progress;

    }

    public void stopMeter(int ref) {

        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    info.setText("[" + filesLeft + "] " + filename + " (" + 
                                 BytesFormat.format(folderProgress) + " / " + 
                                 BytesFormat.format(totalSize) + ") [finishing]");
                    progressBar.setValue((int) (folderProgress / 100));
                    repaintStatus();

                }

            });

	rlm.playAudio("filedone");

    }

    public void stopMeterWithError(int ref, Throwable t) {

        DebuggerOutput.stackTrace((Exception) t);
        DebuggerOutput.debug("TransferInterface.stopMeterWithError[" + transferTask + "]: error = " + t.toString());
        
        folderErrors++;
            
	if(t instanceof Exception) {

	    if(t.getMessage() == null) {

		rlm.getInterface().error(filename + ": " + "An error occurred during transfer. Perhaps the server disconnected.");

            } else {

		rlm.getInterface().error(filename + ": " +
                                                 t.getMessage() + "");

            }

            localFile = null;

	}

    }

    public void handleTaskError(int task, String error) {

        DebuggerOutput.debug("FolderDownloadInterface.handleTaskError[" + task + "]: " + error);

    }

    /**
     * Invoked from the superclass close() method.
     */
    protected synchronized void interrupt() {

        super.interrupt();

        if(fileSearch != null)
            fileSearch.interrupt();

        if(downloadThread != null) {

            downloadThread.interrupt();
            
            try {
                
                downloadThread.join();
                
            } catch(InterruptedException e) {}

        }

    }

    class Location {

        String path;
        HLProtocol.FileListComponent file;

        Location(String path, HLProtocol.FileListComponent file) {

            this.path = path;
            this.file = file;

        }

        public String toString() {

            return path + HLProtocol.DIR_SEPARATOR + file.toString();
            
        }

    }

}

