/**
 * $Id: HLServer.java,v 1.20 2001/10/09 21:36:37 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.hotline;

import java.net.ServerSocket;
import java.net.Socket;
import java.net.InetAddress;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Random;
import java.util.Date;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.Serializable;
import java.io.File;

import redlight.utils.BytesFormat;
import redlight.utils.TimeFormat;
import redlight.utils.InterruptableInputStream;
import redlight.utils.DebuggerOutput;
import redlight.utils.ToArrayConverters;
import redlight.utils.QuickSort;
import redlight.utils.TextUtils;
import redlight.macfiles.MacFile;
import redlight.macfiles.Transferrer;

/**
 * Extensible base-class for building Hotline server applications.
 */
public class HLServer {

    /* Server configuration. */

    int port = 5500;
    int backlog = 10;
    int maximumUsers = 1000;
    int maximumProtocolErrors = 10;
    int maximumConcurrentDownloads = 10, maximumConcurrentUploads = 10, maximumDownloadSpots;
    long banSeconds = 60 * 10;
    InetAddress address = null;
    HLServerAgreement agreement;
    HLServerAccountsTable accountsTable;
    HLServerBanTable banTable;
    HLServerTrackerTable trackerTable;
    HLServerFlatnews flatnews;
    HLServerMonitor hsm;
    String serverName, serverDescription, administratorPassword;
    File homeDirectory, trashDirectory;
    int maximumDownloadsPerUser = 1, maximumUploadsPerUser = 2;
    int maximumDownloads = 10, maximumUploads = 20;
    int maximumDownloadQueueSpots = 25, maximumUploadQueueSpots = 50;

    /* Connection state. */

    ServerSocket socket = null, transferSocket = null;
    Vector clients = null;
    boolean serverIsRunning = false;
    short[] socketInUse = null;
    short socketCounter = 1;

    /* Read-only values. */

    public int downloadsInProgress = 0, 
        uploadsInProgress = 0, 
        downloadsInQueue = 0,
        uploadsInQueue = 0,
        downloadCounter = 0,
        uploadCounter = 0, 
        connectionCounter = 0, 
        connectionPeak = 0;
    public long uptime = 0;

    /* We need an instance of HLProtocol to instantiate a number of
       inner classes from it. */

    static HLProtocol hlp = new HLProtocol();
    HLServerListener hsl;
    HLServerPolicy hsp;
    boolean hslSet = false, hspSet = false;
    HLTransferServer transferServer = null;
    Random randomGenerator;
    RealmTable realmTable;
    TransferQueue transferQueue;

    /**
     * Creates a HLServer with default settings.
     */
    public HLServer() {

        clients = new Vector();
        hsl = new HLServerListener();
        hsp = new HLServerPolicy(this);
        agreement = new HLServerAgreement(this);
        accountsTable = new HLServerAccountsTable(this);
        banTable = new HLServerBanTable(this);
        trackerTable = new HLServerTrackerTable(this);
        flatnews = new HLServerFlatnews(this);        
        realmTable = new RealmTable(this);
        transferQueue = new TransferQueue(this);
        serverName = "Red Light";
        serverDescription = "Red Light advanced server.";
        homeDirectory = new File(System.getProperty("user.dir"));
        trashDirectory = new File(System.getProperty("java.io.tmpdir"));
        randomGenerator = new Random(System.currentTimeMillis());
        administratorPassword = null;

    }

    /**
     * Sets the server name, used to identify this server on Hotline
     * trackers. At most 255 characters in the name are sent to the 
     * tracker.
     * @param newName the new description.
     * @throws IllegalArgumentException if newName is null.
     */
    public void setServerName(String newName) {

        if(newName == null) {

            throw new IllegalArgumentException("newName == null");

        } else {

            serverName = newName;

        }

    }

    /**
     * Returns the server name.
     * @return the servers name.
     */
    public String getServerName() {

        return serverName;

    }

    /**
     * Sets the server description, used to describe this server on 
     * Hotline trackers. At most 255 characters in the description are
     * sent to the tracker.
     * @param newDescription the new description.
     * @throws IllegalArgumentException if newDescription is null.
     */
    public void setServerDescription(String newDescription) {

        if(newDescription == null) {

            throw new IllegalArgumentException("newDescription == null");

        } else {

            serverDescription = newDescription;

        }

    }

    /**
     * Returns the server description.
     * @return the servers description.
     */
    public String getServerDescription() {

        return serverDescription;

    }

    /**
     * Sets the server home directory (the root from which files are
     * served). The default is the current working directory.
     * @param newRoot the new home directory.
     * @throws IllegalArgumentException if the newRoot is null,
     * does not exist, or is not a directory.
     */
    public void setHomeDirectory(File newRoot) {

        if(newRoot == null)
            throw new IllegalArgumentException("newRoot == null");
        
        if(!newRoot.exists())
            throw new IllegalArgumentException(newRoot + " does not exist");
        
        if(!newRoot.isDirectory())
            throw new IllegalArgumentException(newRoot + " is not a directory");

        homeDirectory = newRoot;

    }

    /**
     * Returns the server home directory.
     * @return the servers home directory.
     */
    public File getHomeDirectory() {

        return homeDirectory;

    }

    /**
     * Sets the server trash directory (where files end up once
     * deleted), by default in the location pointed to by the
     * "java.tmpdir" property.
     * @param newRoot the new home directory.
     * @throws IllegalArgumentException if the newRoot is null,
     * does not exist, or is not a directory.  
     */
    public void setTrashDirectory(File newRoot) {

        if(newRoot == null)
            throw new IllegalArgumentException("newRoot == null");
        
        if(!newRoot.exists())
            throw new IllegalArgumentException(newRoot + " does not exist");
        
        if(!newRoot.isDirectory())
            throw new IllegalArgumentException(newRoot + " is not a directory");

        trashDirectory = newRoot;

    }

    /**
     * Returns the server trash directory.
     * @return the servers trash directory.
     */
    public File getTrashDirectory() {

        return trashDirectory;

    }

    /**
     * Prepends some text to the flat news and broadcasts it
     * to all connected clients.
     * @param newPost the text to prepend.
     * @throws IllegalArgumentException if newPost is null.
     */
    public void postFlatnews(String newPost) {

        String oldNews;

        if(newPost != null) {

            oldNews = flatnews.get();
            flatnews.set(newPost + oldNews);

            /* Construct and broadcast the news post as a
               Hotline packet. */
            
            HLProtocol.DataComponent[] dataComponents = 
                new HLProtocol.DataComponent[] { 
                    
                    hlp.new DataComponent(HLProtocol.HTLC_DATA_NEWS_POST, newPost.getBytes()),
                    
                };
            
            realmTable.broadcastPacket(hlp.new Packet(HLProtocol.HTLS_HDR_NEWS_POST, 0, dataComponents));
                
        } else {
            
            throw new IllegalArgumentException("newPost == null");
            
        }
        
    }

    /**
     * Replaces the flatnews component. Clients are not notified of
     * this change.
     * @param newNews the new flatnews component.
     * @throws IllegalArgumentException when newNews is null.  
     */
    public void setFlatnews(HLServerFlatnews newFlatnews) {

        if(newFlatnews != null) {
            
            flatnews = newFlatnews;

        } else {

            throw new IllegalArgumentException("newFlatnews == null");

        }

    }

    /**
     * Returns the current flat news.
     * @return the news text.
     */
    public HLServerFlatnews getFlatnews() {

        return flatnews;

    }

    /**
     * Sets a new account table for this server.
     * @param newAccounts the new account table.
     * @throws IllegalArgumentException when newAccounts is null.  
     */
    public void setAccountsTable(HLServerAccountsTable newAccounts) {

        if(newAccounts != null) {

            accountsTable = newAccounts;

        } else {

            throw new IllegalArgumentException("newAccounts == null");

        }

    }

    /**
     * Returns the accountstable.
     * @return the accountstable.
     */
    public HLServerAccountsTable getAccountsTable() {
        
        return accountsTable;

    }

    /**
     * Returns the bantable for this server.
     * @return the bantable.
     */
    public HLServerBanTable getBanTable() {

        return banTable;

    }

    /**
     * Sets a bantable for the server.
     * @param newBanTable the new bantable.
     * @throws IllegalArgumentException if newBanTable is null.
     */
    public void setBanTable(HLServerBanTable newBanTable) {

        if(newBanTable != null) {

            banTable = newBanTable;
            
        } else {

            throw new IllegalArgumentException("newBanTable == null");

        }

    }

    /**
     * Returns the trackertable for this server.
     * @return the trackertable.
     */
    public HLServerTrackerTable getTrackerTable() {

        return trackerTable;

    }

    /**
     * Sets a trackertable for the server.
     * @param newTrackerTable the new trackertable.
     * @throws IllegalArgumentException if newTrackerTable is null.
     */
    public void setTrackerTable(HLServerTrackerTable newTrackerTable) {

        if(newTrackerTable != null) {

            trackerTable = newTrackerTable;
            
        } else {

            throw new IllegalArgumentException("newTrackerTable == null");

        }

    }

    /**
     * Sets the policy for this server.
     * This method can be called only once. If it is never called,
     * a default implementation is used, which handles login and
     * privilege checking.
     * @param p the policy.
     * @throws RuntimeException if called twice.
     * @throws IllegalArgumentException if p is null.
     */
    public void setHLServerPolicy(HLServerPolicy p) {

        if(hspSet)
            throw new RuntimeException("attempt to call setHLServerPolicy twice");

        if(p == null)
            throw new IllegalArgumentException("p == null");

        hspSet = true;
        hsp = p;

    }
        
    /**
     * Returns the policy object for this server.
     * @return the HLServerPolicy object.
     */
    public HLServerPolicy getHLServerPolicy() {

        return hsp;

    }

    /**
     * Sets the listener for server events for this server.
     * This method can be called only once. If it is never called,
     * a very sparse default implementation is used.
     * @param l the listener.
     * @throws RuntimeException if called twice.
     * @throws IllegalArgumentException if l is null.
     */
    public void setHLServerListener(HLServerListener l) {

        if(hslSet)
            throw new RuntimeException("attempt to call setHLServerListener twice");

        if(l == null)
            throw new IllegalArgumentException("l == null");

        hslSet = true;
        hsl = l;

    }

    public void setAdministratorPassword(String pw) {

        administratorPassword = pw;

    }

    String getAdministratorPassword() {

        return administratorPassword;

    }

    /**
     * Sets the number of consecutive protocol errors that the server
     * will tolerate from a client before disconnecting it. The
     * default value is 10. Protocol errors are conditions where the
     * input from the client appears suspicious, but where
     * disconnecting the client immediately is too drastic. Protocol
     * errors may arise because of slightly different expectations in
     * the implementations of various Hotline clients and this server,
     * the use of unknown or unimplemented features, or just plain bad
     * input. Protocol errors often involve some situation where a
     * request is missing some vital data (such as a news posting
     * without content) and there is no particularly sensible default.
     * @param newMaximumProtocolErrors the new maximum number of
     * consecutive protocol errors that the server will allow.  
     */
    public void setMaximumProtocolErrors(int newMaximumProtocolErrors) {
        
        maximumProtocolErrors = newMaximumProtocolErrors;

    }

    /** 
     * Returns the maximum number of consecutive protocol errors
     * before a client is disconnected.
     * @return maximum number of consecutive protocol errors.  
     */
    public int getMaximumProtocolErrors() {

        return maximumProtocolErrors;

    }

    /**
     * Sets the maximum number of users allowed on the server.  It
     * defaults to 100. There is no way to set an "unlimited" number
     * of users, and there is no way to change the maximum number of
     * users to more than the initially set number of users while the
     * server is running. Also, this method does not allow you to
     * specify less than 1 or more than 31999 users.  It's best to
     * keep this number as low as possible.
     * @param newMaximumUsers the maximum number of users.
     * @throws IllegalArgumentException when the argument smaller than 1 or
     * greater than 31999.  
     */
    public void setMaximumUsers(int newMaximumUsers) {

        if(!(newMaximumUsers > 0 && newMaximumUsers < 32000))
            throw new IllegalArgumentException("maximum users must be 0 ... 32000 exclusive");

        if(socketInUse != null && newMaximumUsers > socketInUse.length)
            throw new IllegalArgumentException("sorry, cannot increase maximum users beyond initial setting");

        maximumUsers = newMaximumUsers;

       
    }

    /**
     * Returns the maximum number of users that the server will allow
     * to connect.
     * @return the maximum number of users (between 0 ... 32000 exclusive).
     */
    public int getMaximumUsers() {

        return maximumUsers;

    }

    public void setMaximumDownloads(int newMaximumDownloads) {

        maximumDownloads = newMaximumDownloads;

    }

    public int getMaximumDownloads() {

        return maximumDownloads;

    }

    public void setMaximumDownloadsPerUser(int newMaxDownloadsPerUser) {

        maximumDownloadsPerUser = newMaxDownloadsPerUser;
    }

    public int getMaximumDownloadsPerUser() {

        return maximumDownloadsPerUser;
    }

    public void setMaximumDownloadQueueSpots(int newDLQSpots) {

        maximumDownloadQueueSpots = newDLQSpots;

    }

    public int getMaximumDownloadQueueSpots() {

        return maximumDownloadQueueSpots;

    }

    public void setMaximumUploads(int newMaximumUploads) {

        maximumUploads = newMaximumUploads;

    }

    public int getMaximumUploads() {

        return maximumUploads;

    }

    public void setMaximumUploadsPerUser(int newMaxUploadsPerUser) {

        maximumUploadsPerUser = newMaxUploadsPerUser;
    }

    public int getMaximumUploadsPerUser() {

        return maximumUploadsPerUser;
    }

    public void setMaximumUploadQueueSpots(int newULQSpots) {

        maximumUploadQueueSpots = newULQSpots;

    }

    public int getUploadQueueSpots() {

        return maximumUploadQueueSpots;

    }

    /**
     * Returns the number of seconds that a ban remains in effect.
     * @return number of seconds.
     */
    public long getBanSeconds() {

        return banSeconds;

    }

    /**
     * Sets the number of seconds that a ban remains in effect. A value
     * of 0 means to ban forever.
     * @param seconds the number of seconds that a ban remains in effect.
     */
    public void setBanSeconds(long seconds) {

        banSeconds = seconds;

    }

    /**
     * Sets the port to listen on. You can call this method anytime,
     * but any changes do not take effect on a running server until
     * after you've called {@link #shutdown} and {@link #listen} on 
     * it again.
     * @param newPort the port to listen on.
     * @throws IllegalArgumentException if the port is negative or 0.
     */
    public void setPort(int newPort) {

        if(newPort >= 0)
            port = newPort;
        else
            throw new IllegalArgumentException("port must be positive");

    }

    /**
     * Returns the port that the server is (or will be) listening on.
     * @return the port.
     */
    public int getPort() {

        return port;

    }

    /**
     * Sets the address to listen on. You can call this method
     * anytime, but any changes do not take effect on a running server
     * until after you've called {@link #shutdown} and {@link #listen}
     * on it again. Setting the address to null means to listen
     * on all addresses.
     * @param newAddress the address to listen on.
     */
    public void setAddress(InetAddress newAddress) {

        address = newAddress;

    }

    /**
     * Returns the address that the server is (or will be) listening on.
     * This may be null, signifying all addresses.
     * @return the address
     */
    public InetAddress getAddress() {

        return address;

    }

    /**
     * Sets the agreement.
     * @param newAgreement the agreement text.  
     */
    public void setAgreement(HLServerAgreement newAgreement) {

        agreement = newAgreement;

    }

    /**
     * Returns the agreement.
     * @return agreement text. 
     */
    public HLServerAgreement getAgreement() {
        
        return agreement;

    }

    /**
     * Shuts the server down. If the argument is non-null, this will
     * perform a polite shutdown by informing all connected users and
     * giving them some time to comply (not implemented yet). If
     * the argument is null, then all connected users are immediately
     * disconnected. This function blocks until all client threads
     * have died, which may take a few seconds.
     * @param message a message to broadcast to all connected users
     * or null to shut down ASAP.
     */
    public void shutdown(String message) {

        log("Server shutting down" + (message == null ? "" : (": " + message)));

        if(message != null) {

            HLProtocol.DataComponent[] dataComponents = 
                new HLProtocol.DataComponent[1];
            dataComponents[0] = hlp.new DataComponent(HLProtocol.HTLS_DATA_MSG, message.getBytes());
            getRealmTable().broadcastPacket(hlp.new Packet(HLProtocol.HTLS_HDR_MSG, 0, dataComponents));
            
            try {

                Thread.currentThread().sleep(1000);

            } catch(InterruptedException e) {}

        }

        serverIsRunning = false;
        
        try {
            
            if(socket != null) {

                socket.close();
                socket = null;

            }

        } catch(Exception e) {}

        try {
            
            if(transferSocket != null) {
                
                transferSocket.close();
                transferSocket = null;

            }

        } catch(IOException e) {}

        try {

            if(hsm != null) {

                hsm.interrupt();
                hsm.join();
                hsm = null;

            }

        } catch(InterruptedException e) {

            DebuggerOutput.stackTrace(e);

        }

        try {

            transferServer.interrupt();
            DebuggerOutput.debug("joining transfer server");
            transferServer.join();

        } catch(InterruptedException e) {}
            
        HLServerDispatcher clients[] = getClients();

        for(int i = 0; i < clients.length; i++) {
            
            HLServerDispatcher hsd = clients[i];
            DebuggerOutput.debug("SHUTDOWN: Killing dispatcher " + hsd);
            hsd.disconnect();

            try {
                
                DebuggerOutput.debug("SHUTDOWN: Waiting for dispatcher to die ... " + hsd);
                hsd.join();
                
            } catch(InterruptedException _e) {}
            
        }

        log("Server shut down complete.");

    }

    /**
     * Starts the server. This method blocks until you call {@link
     * #shutdown} to take the server down or an error occurs. In both
     * cases, an exception is thrown. It is an error to invoke this
     * method twice without calling {@link #shutdown} in between,
     * which will generate a RuntimeException.  
     */
    public void listen() throws Exception {
        
        uptime = System.currentTimeMillis();
        connectionCounter = 0;
        connectionPeak = 0;
        uploadCounter = 0;
        downloadCounter = 0;
        
        try {

            /* Initialize socket list. */

            socketInUse = new short[maximumUsers];

            for(int i = 0; i < socketInUse.length; i++)
                socketInUse[i] = -1;

            if(!serverIsRunning) {
                
                /* Create server sockets. */

                socket = new ServerSocket(port, backlog, address);
                transferSocket = new ServerSocket(port + 1, backlog, address);

                serverIsRunning = true;
                
                log("Server listening on " + (address == null ? "all interfaces" : address.toString()) + ", port " + port + " (+2), " + maximumUsers + " maximum users.");
                log("Home directory " + homeDirectory);
                log("Max.          uploads /          downloads: " + 
                    TextUtils.prepadCharacter(new Integer(maximumUploads).toString(), 3, ' ') + " / " +
                    TextUtils.prepadCharacter(new Integer(maximumDownloads).toString(), 3, ' '));
                log("Max. uploads per user / downloads per user: " + 
                    TextUtils.prepadCharacter(new Integer(maximumUploadsPerUser).toString(), 3, ' ') + " / " + 
                    TextUtils.prepadCharacter(new Integer(maximumDownloadsPerUser).toString(), 3, ' '));
                log("Max.     upload queue /     download queue: " + 
                    TextUtils.prepadCharacter(new Integer(maximumUploadQueueSpots).toString(), 3, ' ') + " / " + 
                    TextUtils.prepadCharacter(new Integer(maximumDownloadQueueSpots).toString(), 3, ' '));

                /* Engage file transfer thread. */
                
                transferServer = new HLTransferServer(this, transferSocket);

                /* Engage monitor (cleanup) thread. */

                hsm = new HLServerMonitor(this);

                /* Let listener know we're ready. */
              
                hsl.serverReady();

                while(serverIsRunning) {
                    
                    try {

                        /* Listen for client connections. */
                        
                        socket.setSoTimeout(5000);
                        Socket client = socket.accept();
                        client.setSoTimeout(2500);
                        client.setSoLinger(true, 5);

                        if(serverIsRunning) {
                            
                            banTable.expire();
                            
                            /* Check whether client is on the bantable. If
                               so, just disconnect immediately */
                            
                            if(banTable.contains(client.getInetAddress())) {
                                
                                log("Client " + client.getInetAddress().toString() + " is on bantable, disconnecting.");
                                
                                try {
                                    
                                    client.close();
                                    
                                } catch(IOException _e) {}
                                
                            } else {
                                
                                /* If we're still supposed to be running,
                                   fire off a dispatcher to service this
                                   client. */
                                
                                HLServerDispatcher hsd = new HLServerDispatcher(this, client);
                                
                                synchronized(clients) {
                                    
                                    connectionCounter++;

                                    clients.addElement(hsd);

                                    if(clients.size() > connectionPeak)
                                        connectionPeak = clients.size();
                                    
                                }
                                
                                hsd.start();
                                
                            }
                        
                        } else {
                            
                            /* If we are to shut down, don't accept this
                               connection. */
                            
                            client.close();
                            
                        }

                    } catch(InterruptedIOException _e) {}               
                    
                }
                
            } else {
                
                throw new RuntimeException("listen() called twice without calling shutdown() in between");
                
            }

        } catch(Exception e) {

            DebuggerOutput.stackTrace(e);
            //            shutdown(e.getMessage() + ".");
            throw e;

        }

    }

    /**
     * Returns a copy of the array of HLServerDispatchers which represents
     * every connected client.
     * @return array of {@link HLServerDispatcher}s.  
     */
    public HLServerDispatcher[] getClients() {

        HLServerDispatcher[] clientArray;

        synchronized(clients) {

            clientArray = new HLServerDispatcher[clients.size()];
            clients.copyInto(clientArray);

        }

        return clientArray;

    }

    /**
     * May be called by a HLServerDispatcher linked to this server
     * to indicate that the particular client no longer exists.
     * @param client the HLServerDispatcher to unlink.
     */
    protected void removeClient(HLServerDispatcher client) {

        synchronized(clients) {

            clients.removeElement(client);

        }

        transferQueue.stopTransfersForClient(client);

        HLProtocol.UserListComponent user = client.getUser();

        if(user.sock != -1) {

            releaseSocketID(user.sock);
            hsl.userLeft(client);

        }

    }
    /**
     * Allocates a unique socket ID and returns it. A socket ID is a
     * number between 0 ... 32000 exclusive.
     * @throws OutOfSocketIDsException when there are no free socket IDs left.
     */
    protected synchronized short allocateSocketID() throws OutOfSocketIDsException {
        short availableSocket = -1;

        synchronized(clients) {

            if(clients.size() > maximumUsers)
                throw new OutOfSocketIDsException("too many connected users");

        }

        DebuggerOutput.debug("HLServer.allocateSocketID: starting ...");

        DebuggerOutput.debug("HLServer.allocateSocketID: sorting ...");

        /* First sort the socket list so that unallocated sockets (-1)
           get on top. */

        QuickSort.sort(socketInUse, 0, socketInUse.length - 1);

        DebuggerOutput.debug("HLServer.allocateSocketID: sorted.");

        if(socketInUse[0] == -1) {

            /* OK, there is at least one empty socket. Because the
               maximum number of available sockets is always smaller
               than or equal to the number of possible socket ID's,
               this means we can always find a successor socket ID
               that has not already been allocated. */

            /* Skip the as yet unconnected clients. */

            int i = 0;
            for(i = 0; i < socketInUse.length; i++)
                if(socketInUse[i] != -1)
                    break;
            
            if(i == socketInUse.length) {

                /* If there are no socket ID's in use, short-circuit and just 
                   use the current socket counter whatever it is. */

                socketInUse[0] = socketCounter;

            } else {

                do {
                
                    /* We need to find two socket ID's N and M in the
                       socket list such that N <= socket counter <=
                       M. If there is no N or M for which this holds,
                       we can use the current value of the socket
                       counter. If we do find an N and an M, and it is
                       possible to construct a X such that N < X < M,
                       then the value of the socket counter becomes
                       X. */

                       /* First try to find N. */
                    
                    int n_position = -1;
                    
                    for(int n = i; socketInUse.length < n && socketInUse[n] <= socketCounter; n++)
                        if(socketInUse[n] <= socketCounter)
                            n_position = n;
                    
                    if(n_position == -1) {
                        
                        /* Did not find N, so we are done. */
                        
                        socketInUse[0] = socketCounter;
                        
                    } else {
                        
                        /* Try to find M. */
                        
                        int m_position = -1;
                        
                        for(int m = socketInUse.length - 1; m >= i && socketCounter <= socketInUse[m]; m--)
                            if(socketCounter <= socketInUse[m])
                                m_position = m;
                        
                        if(m_position == i - 1) {
                            
                            /* Did not find M, so we are done. */
                            
                            socketInUse[0] = socketCounter;
                            
                        } else {
                            
                            /* At this point, N <= M. In addition, if
                               N + 1 < M, we are done by setting the
                               socket counter to N + 1. If N + 1 = M,
                               then we increase the socket counter and
                               repeat the procedure. */
                            
                            if((short) (socketInUse[n_position] + 1) < socketInUse[m_position]) {
                                
                                socketCounter = (short) (socketInUse[n_position] + 1);
                                socketInUse[0] = socketCounter;
                                
                            } else {
                                
                                /* Repeat the procedure. */

                            }
                            
                        }
                        
                    }

                    /* Continue as long as the socket has not been
                       allocated. */
                    
                } while(socketInUse[0] == -1);
                
            }

            /* Increase the socket counter. */

            socketCounter++;

            /* Wrap around the socket counter as it reaches 32000. */

            if(socketCounter >= 32000)
                socketCounter = 1;

            /* Add the user to the public realm and return the
               allocated socket. */

            realmTable.addUser("Public", socketInUse[0]);

            return socketInUse[0];
            
        }

        throw new OutOfSocketIDsException("maximum number of sockets already in use: " + maximumUsers);

    }

    /**
     * Releases the specified socket ID for future use.
     * @param sock the socket ID to release.
     * @throws IllegalArgumentException when the socket ID is out of
     * range or not in use.
     */
    protected synchronized void releaseSocketID(int sock) {

        if(sock < 1 || sock >= 32000)
            throw new IllegalArgumentException("socket ID " + sock + " is out of range");

        for(int i = 0; i < socketInUse.length; i++) {
            
            if(socketInUse[i] == sock) {
                
                realmTable.removeUser(sock);
                socketInUse[i] = -1;
                return;
                
            }
            
        }

    }

    /**
     * Resolves a socket ID into a reference to the dispatcher
     * thread for that client.
     * @param sock the socket ID to resolve.  
     * @return a reference to the dispatcher thread or null 
     * (for example because the client no longer exists).
     * @throws IllegalArgumentException if the socket ID is out of
     * range.  
     */
    HLServerDispatcher resolveSocketID(int sock) {
        
        if(sock < 1 || sock >= 32000)
            throw new IllegalArgumentException("socket ID " + sock + " is out of range");

        synchronized(clients) {
        
            /* Loop through all connected clients and find one whose
               socket ID matches the requested socket id. */
            
            for(Enumeration en = clients.elements(); en.hasMoreElements(); ) {
                
                HLServerDispatcher client = 
                    (HLServerDispatcher) en.nextElement();

                if(client.sock == sock)
                    return client;

            }

        }

        return null;

    }
    
    /**
     * Changes the user with the specified socket ID.
     */
    public void changeUserExcept(HLServerDispatcher client,
                                 short sock,
                                 String nick,
                                 short icon,
                                 short color) {

        try {

            changeUser(client,
                       sock,
                       nick,
                       icon, 
                       color, 
                       "Public",
                       HLProtocol.HTLS_HDR_USER_CHANGE,
                       sock);

        } catch(InvalidRealmException e) {

            /* Can never happen because the public realm
               always exists. */

        }

    }

    /**
     * Changes the user with the specified socket ID.
     */
    public void changeUser(HLServerDispatcher client,
                           short sock,
                           String nick,
                           short icon,
                           short color) {

        try {

            changeUser(client,
                       sock,
                       nick,
                       icon, 
                       color, 
                       "Public",
                       HLProtocol.HTLS_HDR_USER_CHANGE,
                       -1);

        } catch(InvalidRealmException e) {

            /* Can never happen because the public realm
               always exists. */

        }

    }

    /**
     * Changes the user with the specified socket ID. Also
     * used to change users in private chat realms.
     */
    public void changeUser(HLServerDispatcher client,
                           short sock,
                           String nick,
                           short icon,
                           short color,
                           Object refObject,
                           int type,
                           int exceptSock) throws InvalidRealmException {
        
        if(client == null || nick == null)
            throw new IllegalArgumentException("null argument");
        
        HLProtocol.DataComponent[] dataComponents;
            
        synchronized(client.userStateLock) {
            
            client.color = color;
            client.sock = sock;
            client.icon = icon;
            client.nick = nick;

            dataComponents = new HLProtocol.DataComponent[refObject.toString().equals("Public") ? 4 : 5];

            dataComponents[0] = hlp.new DataComponent(HLProtocol.HTLS_DATA_SOCKET, ToArrayConverters.intToByteArray(sock));
            dataComponents[1] = hlp.new DataComponent(HLProtocol.HTLS_DATA_ICON, ToArrayConverters.intToByteArray(icon));
            dataComponents[2] = hlp.new DataComponent(HLProtocol.HTLS_DATA_NICK, nick.getBytes());
            dataComponents[3] = hlp.new DataComponent(HLProtocol.HTLS_DATA_COLOUR, ToArrayConverters.intToByteArray(color));

            if(!refObject.toString().equals("Public"))
                dataComponents[4] = hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(((Integer) refObject).intValue()));
            
        }

        realmTable.broadcastPacketExcept(refObject, hlp.new Packet(type, 0, dataComponents), exceptSock);

        log(client.client.getInetAddress(), "now known as " + nick);

    }    

    /**
     * Gets info on the specified user. Returns either some freeform
     * descriptive String or null if the user is no longer connected
     * to the server.
     * @param sock the user to get info on.
     * @return null or String.
     */
    public String getUserInfo(int sock) {

        HLServerDispatcher client = resolveSocketID(sock);

        if(client == null) 
            return null;

        String lSep = System.getProperty("line.separator");
        String info;

        synchronized(client.userStateLock) {

            info = 
                "Nick:    " + client.nick + lSep +
                "Login:   " + client.login + lSep +
                "From:    " + client.client.getInetAddress().getHostAddress() + lSep +
                "Since:   " + new Date(client.firstTransmissionTime) + lSep +
                "Elapsed: " + TimeFormat.format((System.currentTimeMillis() - client.firstTransmissionTime) / 1000) + lSep + lSep;
                            
        }

        String uploadsInProgress = "", downloadsInProgress = "",
            uploadsInQueue = "", downloadsInQueue = "";

        TransferRequest tr;
        
        Enumeration en = getTransferRequests();

        while(en.hasMoreElements()) {

            tr = (TransferRequest) en.nextElement();

            if(tr.client.sock == sock) {

                if(tr.queuePosition == 0) {

                    /* In progress or wait... */

                    String inProgress;

                    if(!tr.lock) {

                        /* Wait... */

                        inProgress = " waiting ...     ";
                        
                    } else {
                        
                        /* In progress... */
                        
                        inProgress = 
                            TextUtils.prepad(BytesFormat.format(tr.totalSize), 7) + " (";
                        
                        if(tr.progressDone > 0) 
                            inProgress += TextUtils.prepad("" + (int) (100 * ((double) tr.progressDone / tr.totalSize)), 2);
                        else
                            inProgress += "  0";
                        
                        
                        inProgress += "%)  ";
                        
                    }

                    inProgress += tr.local.getFile().getName() + lSep;
                        
                    if(tr.type == TransferRequest.FILE_UPLOAD) 
                        uploadsInProgress += inProgress;
                    else
                        downloadsInProgress += inProgress;

                } else {

                    /* In queue... */

                    String inQueue = 
                        "#" + TextUtils.prepadCharacter(new Integer(tr.queuePosition).toString(), 2, '0') + "  " +
                        tr.local.getFile().getName() + lSep;

                    if(tr.type == TransferRequest.FILE_UPLOAD) 
                        uploadsInQueue += inQueue;
                    else 
                        downloadsInQueue += inQueue;
                    
                }
                
            }

        }

        info += "--- Downloads in progress" + lSep + lSep + 
            downloadsInProgress + lSep;
        info += "--- Uploads in progress" + lSep + lSep +
            uploadsInProgress + lSep;
        info += "--- Downloads in queue" + lSep + lSep + 
            downloadsInQueue + lSep;
        info += "--- Uploads in queue" + lSep + lSep +
            uploadsInQueue;

        return info;

    }

    /**
     * Kicks the user with the specified socket. If the additional ban
     * parameter is set to true, disallows the user to log in for the
     * amount of time specified with {@link #setBanPeriod}.  
     * @param sock the user to disconnect.
     * @param ban true if the user should be banned for a period of time.
     * @return true if the user was disconnected, or false if the user
     * cannot be disconnected (either because the user has {@link
     * HLProtocol.AccountInfo.CANNOT_BE_DISCONNECTED} privileges or
     * because the user is not connected to the server).  
     * @throws IllegalArgumentException if socket ID is out of range.
     */
    public boolean kickUser(int sock, boolean ban) {

        if(sock < 1 || sock >= 32000)
            throw new IllegalArgumentException("socket ID " + sock + " is out of range");

        HLServerDispatcher[] clients = getClients();
        
        /* Loop through all connected clients and find one whose
           socket ID matches that of the user to kick. */
        
        for(int i = 0; i < clients.length; i++) {
            
            if(clients[i].getUser().sock == sock) {
                
                /* We found the user to disconnect, but check
                   whether the user can be disconnected. */
                
                if((clients[i].privileges & 
                    HLProtocol.AccountInfo.CANNOT_BE_DISCONNECTED) != 
                   HLProtocol.AccountInfo.CANNOT_BE_DISCONNECTED) {
                    
                    if(ban) {
                        
                        if(banSeconds == 0) {
                            
                            banTable.put(clients[i].client.getInetAddress(), 0);
                            
                        } else {
                            
                            banTable.put(clients[i].client.getInetAddress(), banSeconds + (System.currentTimeMillis() / 1000));
                            
                        }
                        
                    }
                    
                    clients[i].disconnect();
                    return true;
                    
                } else {
                    
                    return false;
                    
                }
                
            }
            
        }
        
        return false;

    }

    /**
     * Sends a private message to the specified user.
     * @param source the client where the message originated.
     * @param sock the socket to send the message to.
     * @param msg the message.
     * @return true if the message could be sent or false if the 
     * message could not be sent (because the user is no longer 
     * connected to the server).
     * @throws IllegalArgumentException if source or msg are null or
     * if sock is smaller than or equal to 0.
     */
    public boolean sendPrivateMessage(HLServerDispatcher source, 
                                      int sock, 
                                      String msg) {

        if(!(msg != null && source != null && sock > 0))
            throw new IllegalArgumentException("incorrect parameter");
        
        HLServerDispatcher client = resolveSocketID(sock);
        
        if(client != null) {
            
            /* If we found the recipient, construct a private
               message packet and send it. */
            
            HLProtocol.UserListComponent sender = source.getUser();
            
            HLProtocol.DataComponent[] dataComponents = 
                new HLProtocol.DataComponent[] { 
                    
                    hlp.new DataComponent(HLProtocol.HTLC_DATA_MSG, msg.getBytes()),
                    hlp.new DataComponent(HLProtocol.HTLC_DATA_SOCKET, ToArrayConverters.intToByteArray(sender.sock)),
                    hlp.new DataComponent(HLProtocol.HTLC_DATA_NICK, sender.nick.getBytes()),
                    
                };
            
            client.send(hlp.new Packet(HLProtocol.HTLS_HDR_MSG, 0, dataComponents));
            return true;
            
        }

        return false;

    }

    /**
     * Returns the realm table for this server.
     */
    RealmTable getRealmTable() {

        return realmTable;

    }

    /**
     * Returns the transfer queue for this server.
     */
    public TransferQueue getTransferQueue() {

        return transferQueue;

    }

    /**
     * Returns an enumeration of outstanding transfer requests
     * (HLServer.TransferRequest). 
     */
    public Enumeration getTransferRequests() {

        return transferQueue.getTransferRequests();

    }

    /**
     * Convenience method around broadcastPacket(), constructs a
     * "administrator message" packet and broadcasts it to all
     * connected clients.
     * @param msg the message to broadcast.  
     */
    public void broadcastAdministratorMessage(String msg) {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[1];
        dataComponents[0] = hlp.new DataComponent(HLProtocol.HTLS_DATA_MSG, msg.getBytes());
        getRealmTable().broadcastPacket(hlp.new Packet(HLProtocol.HTLS_HDR_MSG, 0, dataComponents));
        
    }

    /**
     * Appends a line to the log.
     * @param address the source address where this event emanated.
     * @param msg the log message.
     */
    public void log(InetAddress address, String msg) {

        log("[" + TextUtils.prepad(address.getHostAddress(), 14) + "] " + msg);

    }

    /**
     * Appends a line to the log.
     * @param msg the log message.
     */
    public void log(String msg) {

        hsl.logLine(msg);

    }

    /**
     * The transfer queue manages file transfers.
     */
    public class TransferQueue {
        
        Vector requests;
        Vector downloadQueueSpots, uploadQueueSpots;
        HLServer hls;

        TransferQueue(HLServer h) {

            hls = h;
            requests = new Vector();
            downloadQueueSpots = new Vector();
            uploadQueueSpots = new Vector();

        }

        /**
         * Creates a transfer request for a (client-side) upload.
         * @param local the file to receive.
         * @return an ID for this transfer request.
         */
        protected TransferRequest createUpload(HLServerDispatcher client, MacFile local) throws TooManyTransfersException {
            
            return create(client, TransferRequest.FILE_UPLOAD, local, null, HLProtocol.DL_WHOLE_FILE);
            
        }
        
        /**
         * Creates a transfer request for a (client-side) download.
         * @param local the file to receive.
         * @return an ID for this transfer request.
         */
        protected TransferRequest createDownload(HLServerDispatcher client, MacFile local, HLProtocol.ResumeTransferComponent rflt, int forkOrFile) throws TooManyTransfersException {
            
            return create(client, TransferRequest.FILE_DOWNLOAD, local, rflt, forkOrFile);
            
        }
        
        /**
         * Creates a transfer request.
         * @param type the type (TransferRequest.FILE_UPLOAD or 
         * TransferRequest.FILE_DOWNLOAD) of transfer.
         * @param local the local file.
         * @param rflt if type == FILE_DOWNLOAD, an object specifiying
         * resumption positions.
         * @return an ID for this transfer request.
         * @throws IllegalArgumentException if one of the parameters is 
         * incorrect.
         */
        private synchronized TransferRequest create(HLServerDispatcher client, byte type, MacFile local, HLProtocol.ResumeTransferComponent rflt, int forkOrFile) throws TooManyTransfersException {
            
            if(local == null || 
               !(type == TransferRequest.FILE_UPLOAD || 
                 type == TransferRequest.FILE_DOWNLOAD))
                throw new IllegalArgumentException("incorrect parameter");
            
            String what;
            int inProgress, maximum, inProgressForUser, maximumPerUser, maximumQueueSpots, inQueue, inQueueForUser;
            Vector queueSpots;

            if(type == TransferRequest.FILE_UPLOAD) {
                
                inProgress = hls.uploadsInProgress;
                inQueue = hls.uploadsInQueue;
                maximum = hls.maximumUploads;
                inProgressForUser = client.uploadsInProgress;
                inQueueForUser = client.uploadsInQueue;
                maximumPerUser = hls.maximumUploadsPerUser;
                queueSpots = uploadQueueSpots;
                maximumQueueSpots = hls.maximumUploadQueueSpots;
                what = "upload";
                
            } else {
                
                inProgress = hls.downloadsInProgress;
                inQueue = hls.downloadsInQueue;
                maximum = hls.maximumDownloads;
                inProgressForUser = client.downloadsInProgress;
                inQueueForUser = client.downloadsInQueue;
                maximumPerUser = hls.maximumDownloadsPerUser;
                queueSpots = downloadQueueSpots;
                maximumQueueSpots = hls.maximumDownloadQueueSpots;
                what = "download";
                
            }

            TransferRequest transferRequest = new TransferRequest();

            if(inProgress >= maximum ||
               inProgressForUser >= maximumPerUser) {

                /* Server can't transfer more files right now. Try to
                   put this request in the queue. */

                synchronized(queueSpots) {

                    if(queueSpots.size() < maximumQueueSpots) {
                        
                        queueSpots.addElement(transferRequest);
                        transferRequest.queuePosition = queueSpots.size();

                    } else {
                        
                        /* Queue is full as well. */

                        throw new TooManyTransfersException("The maximum number of " + what + "s for this server has been reached, and the queue is full. Try again later.");
                                
                    }

                }

            }
            
            
            /* Get a unique transfer ID. */
            
            int id = randomGenerator.nextInt();
            boolean notUnique = true;
            
            while(notUnique) {
                
                notUnique = false;
                
                for(Enumeration en = requests.elements(); en.hasMoreElements(); ) {
                    
                    TransferRequest tr = (TransferRequest) en.nextElement();
                    
                    if(tr.ref == id) {
                        
                        notUnique = true;
                        break;
                        
                    }
                    
                }
                
            }

            if(type == TransferRequest.FILE_UPLOAD) {

                uploadCounter++;

                if(transferRequest.queuePosition == 0) {

                    hls.uploadsInProgress++;
                    client.uploadsInProgress++;
                    
                } else {

                    hls.uploadsInQueue++;
                    client.uploadsInQueue++;
                    hls.hsl.transferQueueStart(transferRequest);

                }

            } else {

                downloadCounter++;

                if(transferRequest.queuePosition == 0) {

                    hls.downloadsInProgress++;
                    client.downloadsInProgress++;

                } else {
                    
                    hls.downloadsInQueue++;
                    client.downloadsInQueue++;
                    
                }
                
            }

            transferRequest.transferThread = null;
            transferRequest.transferThreadNotified = false;
            transferRequest.client = client;
            transferRequest.local = local;
            transferRequest.rflt = rflt;
            transferRequest.forkOrFile = forkOrFile;
            transferRequest.ref = id;
            transferRequest.type = type;
            requests.addElement(transferRequest);

            if(transferRequest.queuePosition == 0)
                hls.hsl.transferProgressStart(transferRequest);
            else
                hls.hsl.transferQueueStart(transferRequest);

            return transferRequest;
            
        }
        
        /**
         * Returns an object describing the requested transfer request.
         * @param ref the transfer request to look up.
         * @return an object describing the transfer request.
         * @throws IllegalArgumentException if ref does not refer
         * to any transfer request.
         */
        protected synchronized TransferRequest get(int ref) {
            
            for(Enumeration en = requests.elements(); en.hasMoreElements(); ) {
                
                TransferRequest tr = (TransferRequest) en.nextElement();
                
                if(tr.ref == ref) {
                    
                    if(tr.lock)
                        throw new IllegalArgumentException("cannot process transfer ID " + ref + " because it is already being processed");
                    
                    tr.lock = true;
                    return tr;
                    
                }
                
            }
            
            throw new IllegalArgumentException(ref + " is not a valid transfer ID");
            
        }
        
        /**
         * Destroys a transfer request, releasing the reference for
         * future reuse.
         * @param ref the reference to release.
         * @throws IllegalArgumentException if ref does not refer
         * to any transfer request.
         */
        public synchronized void destroy(int ref) {
            
            for(Enumeration en = requests.elements(); en.hasMoreElements(); ) {
                
                TransferRequest tr = (TransferRequest) en.nextElement();

                DebuggerOutput.debug("TransferQueue.destroy: looking for " + ref + " in " + tr);

                if(tr.ref == ref) {

                    DebuggerOutput.debug("TransferQueue.destroy: destroying " + tr.local.toString());

                    Vector queueSpots;

                    if(tr.type == TransferRequest.FILE_UPLOAD) {

                        queueSpots = uploadQueueSpots;
                        
                        if(tr.queuePosition == 0) {
                            
                            hls.uploadsInProgress--;
                            tr.client.uploadsInProgress--;
                            hls.hsl.transferProgressStop(tr);

                        } else {
                            
                            hls.uploadsInQueue--;
                            tr.client.uploadsInQueue--;
                            hls.hsl.transferQueueStop(tr);
                            
                        }

                    } else {

                        queueSpots = downloadQueueSpots;

                        if(tr.queuePosition == 0) {
                            
                            hls.downloadsInProgress--;
                            tr.client.downloadsInProgress--;
                            hls.hsl.transferProgressStop(tr);

                        } else {

                            hls.downloadsInQueue--;
                            tr.client.downloadsInQueue--;
                            hls.hsl.transferQueueStop(tr);

                        }

                    }

                    /* Somebody dropped out / was kicked out of the
                       queue. */
                    
                    if(tr.queuePosition != 0)
                        queueSpots.removeElement(tr);
                    
                    if(queueSpots.size() > 0) {
                        
                        DebuggerOutput.debug("TransferQueue.destroy: reordering queue (size = " + queueSpots.size() + ")");
                       
                        for(int i = 0; i < queueSpots.size(); i++) {

                            TransferRequest queueRequest = 
                                (TransferRequest) queueSpots.elementAt(i);

                            queueRequest.queuePosition = i + 1;
                            queueRequest.client.sendQueuePosition(queueRequest.ref, queueRequest.queuePosition);

                        }
                        
                        /* Check whether we can remove the top element
                           from the queue. */

                        TransferRequest eligibleRequest = 
                            (TransferRequest) queueSpots.elementAt(0);
                        
                        int inProgress, maximum, inProgressForUser, maximumPerUser;

                        if(eligibleRequest.type == TransferRequest.FILE_UPLOAD) {
                
                            inProgress = hls.uploadsInProgress;
                            maximum = hls.maximumUploads;
                            inProgressForUser = eligibleRequest.client.uploadsInProgress;
                            maximumPerUser = hls.maximumUploadsPerUser;
                
                        } else {
                
                            inProgress = hls.downloadsInProgress;
                            maximum = hls.maximumDownloads;
                            inProgressForUser = eligibleRequest.client.downloadsInProgress;
                            maximumPerUser = hls.maximumDownloadsPerUser;
                
                        }

                        if(inProgress < maximum &&
                           inProgressForUser < maximumPerUser) {
                            
                            /* There is space for this request,
                               do it. */
                            
                            DebuggerOutput.debug("TransferQueue.destroy: eligible request is " + eligibleRequest.local.toString() + ", removing from queue");
                            
                            /* Reorder queue again. */
                            
                            queueSpots.removeElementAt(0);

                            for(int i = 0; i < queueSpots.size(); i++) {
                                
                                TransferRequest queueRequest = 
                                    (TransferRequest) queueSpots.elementAt(i);
                                
                                queueRequest.queuePosition = i + 1;
                                queueRequest.client.sendQueuePosition(queueRequest.ref, queueRequest.queuePosition);
                            
                            }

                            if(!eligibleRequest.cancelled) {
                                
                                /* Notify eligible request and update
                                   stats. */
                                
                                eligibleRequest.queuePosition = 0;
                                
                                if(eligibleRequest.type == 
                                   TransferRequest.FILE_UPLOAD) {
                                    
                                    hls.uploadsInQueue--;
                                    hls.uploadsInProgress++;
                                    eligibleRequest.client.uploadsInQueue--;
                                    eligibleRequest.client.uploadsInProgress++;
                                    hls.hsl.transferQueueStop(eligibleRequest);
                                    hls.hsl.transferProgressStart(eligibleRequest);
                                } else {
                                    
                                    hls.downloadsInQueue--;
                                    hls.downloadsInProgress++;
                                    eligibleRequest.client.downloadsInQueue--;
                                    eligibleRequest.client.downloadsInProgress++;
                                    hls.hsl.transferQueueStop(eligibleRequest);
                                    hls.hsl.transferProgressStart(eligibleRequest);
                                    
                                }
                                
                                if(eligibleRequest.transferThread != null) 
                                    eligibleRequest.transferThread.nextInLine();

                            }

                        }
                        
                    }

                    try {
                        
                        if(tr.local != null) {

                            tr.local.cleanup();
                            tr.local.close();
                        
                        }

                    } catch(IOException _e) {}

                    DebuggerOutput.debug("TransferQueue.destroy: removing " + tr + " from transfer list.");
                    tr.client = null;
                    tr.transferThread = null;
                    tr.cancel();
                    requests.removeElement(tr);
                    
                    return;
                    
                }
                
            }
            
            DebuggerOutput.debug("TransferQueue.destroy: " + ref + " is not a valid transfer ID");
            
        }

        /**
         * Returns an enumeration for all the transfer requests.
         */
        public Enumeration getTransferRequests() {

            return requests.elements();

        }

        /**
         * Expires transfer requests that are older than 1 minute
         * and not being serviced.
         */
        protected synchronized void expire() {
            
            for(Enumeration en = requests.elements(); en.hasMoreElements(); ) {
                
                TransferRequest tr = (TransferRequest) en.nextElement();
                
                if(tr.cancelled || 
                   (!tr.lock && System.currentTimeMillis() - tr.when > 60000)) {
                    
                    log("Expired transfer request " + tr);
                    destroy(tr.ref);
                    
                }
                
            }

        }

        protected synchronized void stopTransfersForClient(HLServerDispatcher client) {
            
            Vector v = new Vector();

            /* Can't do destroy in this loop, destroy may perturb the
               remove an element from the Enumeration that we're
               stepping. */

            for(Enumeration en = requests.elements(); en.hasMoreElements(); ) {
                
                TransferRequest tr = (TransferRequest) en.nextElement();
                
                if(tr.client == client)
                    v.addElement(new Integer(tr.ref));
                
            }
            
            for(Enumeration en = v.elements(); en.hasMoreElements(); )
                destroy(((Integer) en.nextElement()).intValue());
            
            v.removeAllElements();

        }
     
    }
    
    /**
     * A transfer request is created after the client requests a
     * download / upload. 
     */
    public class TransferRequest {
        
        /**
         * Value of the type field if this is an upload 
         * (from the user's point of view).
         */
        public static final byte FILE_UPLOAD = 1;

        /**
         * Value of the type field if this is a download.
         * (from the user's point of view).
         */
        public static final byte FILE_DOWNLOAD = 2;

        /**
         * Client for this transfer (may be null).
         */
        public HLServerDispatcher client;

        /**
         * Whether this is an up or download.
         */
        public byte type;

        /**
         * Unique key for this transfer.
         */
        public int ref;

        /**
         * Local file for this transfer.
         */
        public MacFile local;

        /**
         * True if a TransferThread has claimed ownership
         * (i.e. when the transfer is in progress).
         */
        public boolean lock;

        /**
         * Meaningless until lock is true.
         */
        public long totalSize;

        /**
         * Meaningless until lock is true.
         */
        public long progressDone;

        /**
         * Becomes 0 when ready to be serviced.
         */
        public int queuePosition;
        
        /**
         * Whether this transfer has been cancelled.
         */
        public boolean cancelled;

        /**
         * The TransferThread that has set lock to true.
         */
        TransferThread transferThread;

        /**
         * When this transfer was created. We periodically delete old
         * transfer requests that linger forever in wait state.  
         */
        long when;
        boolean transferThreadNotified;
        HLProtocol.ResumeTransferComponent rflt;
        int forkOrFile;
        
        TransferRequest() {
            
            /* Keep track of when this transfer request was created,
               so we don't drag around stale transfer requests
               forever. */
            
            when = System.currentTimeMillis();
            lock = false;
            queuePosition = 0;
            cancelled = false;
            
        }
        
        public void cancel() {

            cancelled = true;
            
            if(transferThread != null)
                transferThread.disconnect();
                
        }

        public String toString() {
            
            return "TransferRequest[ref = " + ref + ", local = " + local  + ", queuePos = " + queuePosition + ", type = " + type + ", lock = " + lock + ", total = " + totalSize + ", done = " + progressDone + ", cancelled = " + cancelled + "]";
            
        }
        
    }
        
    /**
     * The RealmTable manages a list of Realms and their users.
     */
    class RealmTable {
        HLServer hls;
        Hashtable realms;

        /**
         * Creates the realm table and initializes the public realm
         * with no users. 
         */
        RealmTable(HLServer h) {

            hls = h;
            realms = new Hashtable();
            realms.put("Public", new Realm("Public", "No subject."));

        }

        /**
         * Creates a new realm.
         * @return a unique ID for the new realm.
         */
        synchronized Object createRealm(short sock) {
            Object refObject = null;
            boolean unique = false;

            if(sock < 1 || sock >= 32000)
                throw new IllegalArgumentException("socket ID " + sock + " is out of range");
                        
            /* Get a unique reference. */
            
            while(!unique) {
                
                refObject = new Integer(randomGenerator.nextInt());
                unique = !realms.containsKey(refObject);
                
            }
            
            realms.put(refObject, new Realm(refObject, "No subject."));
            addUser(refObject, sock);

            return refObject;

        }

        /**
         * Adds a user to the given realm.
         * @param refObject the realm to add the user to.
         * @param sock the user to add.
         * @throws IllegalArgumentException if the realm does not exist
         * or when the socket ID is non-existant or out of range.
         */
        synchronized boolean addUser(Object refObject, int sock) {

            if(!realms.containsKey(refObject))
                return false;

            ((Realm) realms.get(refObject)).addUser(sock);

            return true;

        }

        /** 
         * Returns a list of users for the given realm.
         * @param refObject the realm to get users from.
         * @return an array of UserListComponent objects.
         */
        synchronized HLProtocol.UserListComponent[] getUsers(Object refObject) throws InvalidRealmException {

            if(!realms.containsKey(refObject))
                throw new InvalidRealmException("no such realm: " + refObject);

            Realm realm = (Realm) realms.get(refObject);
            HLProtocol.UserListComponent[] userList = 
                new HLProtocol.UserListComponent[realm.users.size()];

            for(int i = 0; i < userList.length; i++) 
                userList[i] = 
                    hls.resolveSocketID(((Integer) realm.users.elementAt(i)).
                                        intValue()).getUser();

            return userList;

        }

        /**
         * Removes a user from the given realm. If the realm is not
         * the public realm, and the realm becomes empty after
         * removing the specified user, then the entire realm is
         * destroyed.
         * @param sock the socket ID of the user to remove.
         * @throws IllegalArgumentException if the socket ID is out of
         * range or not a member of the realm.  
         */
        synchronized boolean removeUser(Object refObject, int sock) {

            if(!realms.containsKey(refObject))
                return false;
            
            Realm realm = (Realm) realms.get(refObject);
            realm.removeUser(sock);
            
            if(refObject.toString().equals("Public")) {

                /* This is the public realm, so send a user leave
                   message to all connected clients. */
                
                HLProtocol.DataComponent[] dataComponents = 
                    new HLProtocol.DataComponent[] { 
                        
                        HLServer.hlp.new DataComponent(HLProtocol.HTLS_DATA_SOCKET, ToArrayConverters.intToByteArray(sock)),
                        
                    };

                broadcastPacket(HLServer.hlp.new Packet(HLProtocol.HTLS_HDR_USER_LEAVE, 0, dataComponents));

            } else {

                try {

                    /* This is not the public realm, so it may need to
                       be destroyed in the case where we just removed
                       the last user. */
                    
                    if(realm.users.size() == 0) {
                        
                        destroyRealm(refObject);
                        realm.refObject = null;
                        
                    } else {
                    
                        /* Inform the users in this realm that a comrade
                           has left. */
                        
                        HLProtocol.DataComponent[] dataComponents = 
                            new HLProtocol.DataComponent[] { 
                                
                                hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(((Integer) refObject).intValue())),
                                hlp.new DataComponent(HLProtocol.HTLS_DATA_SOCKET, ToArrayConverters.intToByteArray(sock)),
                                
                            };
                        
                        broadcastPacket(refObject, hlp.new Packet(HLProtocol.HTLS_HDR_CHAT_USER_LEAVE, 0, dataComponents));
                        
                    }
                    
                } catch(InvalidRealmException e) {}
                
            }

            return true;

        }

        /**
         * Removes the given user from all realms.
         * @param sock the user to remove.
         * @throws IllegalArgumentException if the socket ID is invalid.
         */
        synchronized void removeUser(int sock) {

            Integer socketID = new Integer(sock);

            for(Enumeration en = realms.elements(); en.hasMoreElements(); ) {

                Realm realm = (Realm) en.nextElement();

                if(realm.users.contains(socketID))
                    removeUser(realm.refObject, sock);
                
            }

        }

        /**
         * Destroys the realm specified by the given reference.
         * @param refObject the reference to destroy (usually an
         * Integer object).
         */
        private synchronized void destroyRealm(Object refObject) {

            if(!realms.containsKey(refObject))
                throw new IllegalArgumentException("no such realm: " + refObject);

            Realm realm = (Realm) realms.get(refObject);

            if(realm.users.size() > 0)
                throw new IllegalArgumentException("realm " + refObject + " cannot be destroyed because it still contains users.");

            realms.remove(refObject);

        }

        /**
         * Sends a packet to all users in the given realm.
         * @param refObject the realm to broadcast the packet to.
         * @param packet the packet to broadcast.
         */
        synchronized void broadcastPacket(Object refObject,
                                          HLProtocol.Packet packet) throws InvalidRealmException {

            _broadcastPacket(refObject, packet, -1);

        }

        /**
         * Sends a packet to all users in the given realm.
         * @param refObject the realm to broadcast the packet to.
         * @param packet the packet to broadcast.
         */
        synchronized void broadcastPacketExcept(Object refObject,
                                                HLProtocol.Packet packet,
                                                int exceptSock) throws InvalidRealmException {

            _broadcastPacket(refObject, packet, exceptSock);

        }

        /**
         * Sends a packet to all users in the public realm.
         * @param packet the packet to broadcast.
         */
        void broadcastPacket(HLProtocol.Packet packet) {

            try {

                _broadcastPacket("Public", packet, -1);

            } catch(InvalidRealmException e) {

                /* Never happens because the public realm always
                   exists. */

            }

        }

        /**
         * Sends a packet to all users in the public realm.
         * @param packet the packet to broadcast.
         */
        void broadcastPacketExcept(HLProtocol.Packet packet,
                                   int exceptSock) {

            try {

                _broadcastPacket("Public", packet, exceptSock);

            } catch(InvalidRealmException e) {

                /* Never happens because the public realm always
                   exists. */

            }

        }

        /**
         * Internal method for broadcasting packet (so we can avoid
         * synchronization for broadcasts to the public realm).
         * @param refObject the realm to broadcast the packet to.
         * @param packet the packet to broadcast.
         */
        private void _broadcastPacket(Object refObject,
                                      HLProtocol.Packet packet,
                                      int exceptSock) throws InvalidRealmException {

            if(!realms.containsKey(refObject))
                throw new InvalidRealmException("no such realm: " + refObject);

            Realm realm = (Realm) realms.get(refObject);

            /* (Shouldn't broadcast packets without adjusting their
               transaction ID first.)  */
            
            for(Enumeration en = realm.users.elements(); en.hasMoreElements(); ) {
                
                HLServerDispatcher client = resolveSocketID(((Integer) en.nextElement()).intValue());
                
                if(client != null && client.sock != exceptSock) {

                    packet.header.trans = client.nextTrans();
                    client.send(packet);

                }

            }

        }

        /**
         * A realm is a group of users sharing a common reference.
         * Realms are used to implement private chat.
         */
        class Realm {
            Object refObject;
            Vector users;
            String subject;
            
            /**
             * Creates a new realm with the specified ID and subject.
             * @param refObject the unique ID of this realm.
             * @param subject the subject of this realm.
             */
            private Realm(Object refObject, 
                          String subject) {
                
                this.refObject = refObject;
                this.subject = subject;
                users = new Vector();
                
            }
            
            /**
             * Adds a user to this realm.
             * @param sock the socket ID of the user to add.
             * @throws IllegalArgumentException if the socket ID already
             * exists or if it is out of range.
             */
            public void addUser(int sock) {
                
                if(sock < 1 || sock >= 32000)
                    throw new IllegalArgumentException("socket ID " + sock + " is out of range");
                
                if(users.contains(new Integer(sock)))
                    throw new IllegalArgumentException("socket ID " + sock + " already in realm " + refObject);
            
                users.addElement(new Integer(sock));

            }

            /**
             * Removes a user from this realm.
             * @param sock the socket ID of the user to remove.
             * @throws IllegalArgumentException if the socket ID is out of
             * range or not a member of the realm.  
             */
            public void removeUser(int sock) {
                
                if(sock < 1 || sock >= 32000)
                    throw new IllegalArgumentException("socket ID " + sock + " is out of range");
                
                if(!users.contains(new Integer(sock)))
                    throw new IllegalArgumentException("socket ID " + sock + " is not a member of this realm");
                
                users.removeElement(new Integer(sock));
                
            }
            
        }

    }

}

/**
 * Monitor thread; periodically idles user and notifies trackers on
 * trackertable.  
 */
class HLServerMonitor extends Thread {

    HLServer hls;

    HLServerMonitor(HLServer h) {

        hls = h;
        setName("HLServerMonitor " + h);

        start();
    }

    public synchronized void run() {

        DebuggerOutput.debug("HLServerMonitor starting ...");

        try {

            while(!isInterrupted()) {
                    
                /* Tell trackers that we're alive. */

                hls.getTrackerTable().broadcast();

                /* Find idle / ghost users and adjust their state /
                   disconnect them. */

                idleUsersAndDisconnectGhosts();

                /* Expire transfer requests when they're older than 1
                   minute and nobody has come to collect them. */

                hls.getTransferQueue().expire();

                /* Sever ghost file transfer threads. */

                synchronized(hls.transferServer.transferThreads) {
                    
                    for(Enumeration en = hls.transferServer.transferThreads.elements(); en.hasMoreElements(); ) {
                        
                        TransferThread t = (TransferThread) en.nextElement();
                        
                        if(!t.isAlive && (System.currentTimeMillis() - t.creationTime > 60000)) {

                            hls.log(t.client.getInetAddress(), "Disconnecting ghost file transfer connection.");

                            try {
                                
                                t.client.getInputStream().close();
                                t.client.getOutputStream().close();
                                t.client.close();
                                t.interrupt();
                                t.join();
                                
                            } catch(IOException e) {
                                
                                if(DebuggerOutput.on)
                                    e.printStackTrace();
                                
                            } catch(InterruptedException e) {
                                
                                if(DebuggerOutput.on)
                                    e.printStackTrace();
                                
                            }

                            hls.transferServer.transferThreads.removeElement(t);
                        
                        }
                        
                    }
                    
                } 
                
                /* One minute is nice, two minutes and you start
                   dropping off of trackers. */
                
                wait(60000);
                
            }
            
        } catch(InterruptedException _e) {
            
            if(DebuggerOutput.on)
                _e.printStackTrace();
            
        }
        
        DebuggerOutput.debug("HLServerMonitor exiting");

    }

    void idleUsersAndDisconnectGhosts() {
        
        HLServerDispatcher[] clients = hls.getClients();
        
        for(int i = 0; i < clients.length; i++) {
            
            synchronized(clients[i].userStateLock) {
                
                if(clients[i].sock == -1) {

                    /* Disconnect clients who connect but don't log in 
                       after a minute. */

                    DebuggerOutput.debug("ghost last tranmission = " + clients[i].lastTransmissionTime + ", current = " + System.currentTimeMillis() + ", delta = " + (System.currentTimeMillis() - clients[i].lastTransmissionTime));

                    if(System.currentTimeMillis() - clients[i].lastTransmissionTime > 60000) {

                        clients[i].disconnect();

                        hls.log(clients[i].client.getInetAddress(),
                                "Disconnected ghost client.");

                    }

                } else {

                    /* For connected clients ... */
                    
                    if((clients[i].color & 
                        HLProtocol.UserListComponent.HAS_BEEN_IDLE) !=
                       HLProtocol.UserListComponent.HAS_BEEN_IDLE) {
                        
                        /* Idle users after ten minutes, if they're
                           not already idle. */
                        
                        long deltaTime = System.currentTimeMillis() - clients[i].lastTransmissionTime;
                        
                        if(deltaTime > 60000 * 10)
                            hls.changeUser(clients[i], clients[i].sock, clients[i].nick, clients[i].icon, (short) (clients[i].color | HLProtocol.UserListComponent.HAS_BEEN_IDLE));
                        
                    }
                    
                }
                
            }
            
        }

    }

}
