/**
 * $Id: HLServerAccountsTable.java,v 1.2 2001/09/10 22:09:30 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.util.Hashtable;
import java.util.Enumeration;

import redlight.hotline.HLProtocol;

/**      
 * Accesses and modifications to the accounts table must be
 * couched by {@link #AccountsTable#lock} and {@link
 * #AccountsTable#unlock} calls. For example:
 * <pre>
 *   HLServer hls = new HLServer();
 *   ... do something ...
 *   HLServer.AccountsTable accountsTable = hls.getAccountsTable();
 *   accountsTable.lock();
 *   if(accountsTable.exists("guest"))
 *     System.out.println("Account 'guest' exists");
 *     if((({@link HLProtocol.AccountInfo}) accountsTable.get(login)).password.equals(""))
 *       System.out.println("The password for guest is empty");
 *   accountsTable.set("guest", "guest");
 *   System.out.println("Set the password for guest to guest");
 *   hls.unlockAccounts();
 *   ... continue ...
 * </pre>
 * @return the accounts table.  
 */
public class HLServerAccountsTable extends HLServerComponent {
    
    /* The account table lock. */
    
    boolean accountLock = false;
    
    /* Tracks the accounts. */
    
    protected Hashtable table;

    /**
     * Creates an empty accounts table.
     */
    public HLServerAccountsTable(HLServer h) {
        
        super(h);
        table = new Hashtable();
        
    }
    
    /**
     * Checks whether the given login exists. Always call this
     * method before getting an account.
     * @param login the login to check for.
     */
    public boolean exists(String login) {

        checkModified();
        return table.containsKey(login);
        
    }
    
    /**
     * Returns the account info for the given login. Before you
     * get an account, use the {@link #exists} method to determine
     * whether the account exists.
     * @param login the login to query.
     * @return the account info for the given login.
     */
    public HLProtocol.AccountInfo get(String login) {

        /* Don't check modified, that's done in exists(). */

        return (HLProtocol.AccountInfo) table.get(login);
        
    }
    
    /**
     * Inserts or replaces an account.
     * @param login the login to insert or replace.
     * @param newAccount the account info.
     */
    public void put(String login, 
                    HLProtocol.AccountInfo newAccount) {
        
        table.put(login, newAccount);
        hasChanged();
        
    }
    
    /**
     * Removes the given account from the accounts table.
     * @param login the account to remove.
     */
    public void remove(String login) {
        
        table.remove(login);
        hasChanged();
        
    }

    /**
     * Returns a list of all accounts.
     * @return all accounts.
     */
    public HLProtocol.AccountInfo[] list() {

        checkModified();

        HLProtocol.AccountInfo[] list = 
            new HLProtocol.AccountInfo[table.size()];

        Enumeration en = table.elements();
        for(int i = 0; en.hasMoreElements(); i++)
            list[i] = (HLProtocol.AccountInfo) en.nextElement();

        return list;

    }

    /**
     * Locks the account table. This method must be called 
     * before querying / modifying the accounts table.
     */
    public synchronized void lock() throws InterruptedException {
        
        while(accountLock == true)
            wait();
        
        accountLock = true;
        
    }
    
    /**
     * Unlocks the accounts table. This method must be called after
     * querying / modifying te accounts table.
     * @throws IllegalArgumentException when the accounts table was
     * not previously locked.
     */
    public synchronized void unlock() {
        
        if(accountLock == true) {
            
            accountLock = false;
            notify();
            
        } else {
            
            throw new RuntimeException("attempt to unlock accounts, but they were never locked");
            
        }
        
    }

    /**
     * This method is invoked whenever the accounts table changes.
     * Subclasses can override this method to e.g. save the accounts
     * table to disk.
     */
    public void hasChanged() {}

    /** 
     * Called whenever the latest version of the agreement is required.
     * Can be overridden by subclasses to implement persistent storage.
     */    
    public void checkModified() {}

}
