/**
 * $Id: HLServerComponent.java,v 1.2 2001/10/06 14:31:30 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;

/**
 * This abstract class provides utility methods that help to load and
 * save data to and from files. Subclasses must implement methods
 * to load and store with InputStream and OutputStream arguments.
 */
public abstract class HLServerComponent {

    public HLServer hls;

    HLServerComponent(HLServer h) {

        if(h == null)
            throw new IllegalArgumentException("h == null");

        hls = h;

    }

    public abstract void hasChanged();

    public abstract void checkModified();

    /**
     * Tries to open a FileInputStream on the given file and then
     * invokes {@link #load(InputStream)} on that stream, for which
     * subclasses are supposed to provide a sensible implementation.  
     */
    public void load(File file) {

        FileInputStream fis = null;

        try {

            fis = new FileInputStream(file);
            load(fis);

        } catch(IOException e) {

        } finally {

            if(fis != null) {

                try {

                    fis.close();

                } catch(IOException _e) {}

            }

        }

    }

    /**
     * Stub implementation. Subclasses must override this method to
     * implement persistent storage.
     * @param is the InputStream to load from.
     */
    public void load(InputStream is) {}

    /**
     * Tries to open a FileOutputStream on the given file and then
     * invokes {@link #load(OutputStream)} on that stream, for which
     * subclasses are supposed to provide a sensible implementation.  
     */
    public void write(File file) {

        FileOutputStream fos = null;

        try {

            fos = new FileOutputStream(file);
            write(fos);
            fos.flush();
            
        } catch(IOException e) {

        } finally {

            if(fos != null) {

                try {

                    fos.close();

                } catch(IOException _e) {}

            }

        }

    }
    
    /**
     * Stub implementation. Subclasses must override this method to
     * implement persistent storage.
     * @param is the OutputStream to write to.
     */
    public void write(OutputStream os) {}

}
