/**
 * $Id: MacFile.java,v 1.11 2001/10/07 04:43:39 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.macfiles;

import java.io.*;
import java.util.Calendar;
import java.util.Hashtable;
import java.util.Date;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import redlight.utils.Meter;
import redlight.utils.MeterSource;
import redlight.utils.DebuggerOutput;
import redlight.utils.FilenameUtils;

/**
 * A MacFile provides mechanisms to preserve some or all
 * of the Mac specific attributes of a file, most notably
 * it's resource fork. Different methods to do so are 
 * implemented by subclasses of MacFile.<P>
 *
 * The available methods are:<P>
 * <DL>
 * <DT><B>Native</B>
 * <DD>This method relies on functions native to the Macintosh 
 * platform to store files. This method preserves the most 
 * information and is the preferred method, if it is supported.
 * <DT><B>MacBinary</B>
 * <DD>This method stores all information in a MacBinary file. This 
 * retains almost all of the information in the original file
 * but requires that the file be decoded by a decoder program
 * before it can be used.
 * <DT><B>Split</B>
 * <DD>This method stores the data fork and the resource fork 
 * in separate files. Some additional meta information such
 * as type/creator and finder flags is also stored, in a 
 * separate .info file.
 * </DL>
 * To use one of these methods, you must instantiate a 
 * <I>method</I>MacFile class, where <I>method</I> is the method you
 * want to use. Before instantiation you should check, however,
 * if the method you want to use is available and supported
 * on the host platform.<P>
 * 
 * <I>Important:</I><P> It is not possible to mix reads/writes to the
 * different forks of a MacFile. You need to first read/write the data
 * fork, then read/write the resource fork, because both forks may be
 * stored sequentially in the same file. You must call the
 * setDataSize() and setResourceSize() methods after writing to the
 * data fork and the resource fork respectively. Finally, you must
 * always close() the data fork after writing to it, and / or before
 * using the resource fork. <P>
 * 
 * For example, to write some data to a MacBinaryMacFile: <P>
 * <BLOCKQUOTE>
 * <PRE>
 * MacFile mf = new MacBinaryMacFile("filename"); // or new NativeMacFile, new SplitMacFile
 * mf.writeHeader(...);
 * DataOutput dataForkOutput = mf.getDataFork().getDataOutput();
 * ... write data for the data fork ...
 * xf.local.setDataSize(... number of bytes written ...);
 * xf.local.getDataFork.close(); // close the data fork, very important!

 * DataOutput resourceForkOutput = mf.getResourceFork().getResourceOutput();
 * ... write data for the resource fork ...
 * xf.local.setResourceSize(... number of bytes written ...);
 * xf.local.getResourceFork.close();
 * xf.local.cleanup(); // also very important, adjusts header information.
 * xf.local.close();
 * </PRE>
 * </BLOCKQUOTE> 
 */
public abstract class MacFile {
    File file;
    Fork dataFork, resourceFork;
    long dataSize, resourceSize;
    Integer permissions;
    String comment, type, creator;
    int finderFlags;

    static private String[] macFileMethods = 
    { "Native", "Split", "MacBinary" };
    static private int currentMacFileMethod;
    static int NATIVE = 0;
    static int SPLIT = 1;
    static int MACBINARY = 2;
    static public int READ_PERM = 1;
    static public int WRITE_PERM = 2;
    static public boolean mappingsInitialized = false;
    static public Hashtable typeCodeMapping = null;
    static public Hashtable creatorCodeMapping = null;

    public static final void createTypeCreatorCodeMapping() {

        typeCodeMapping = new Hashtable();
        typeCodeMapping.put("ChangeLog", "TEXT");
        typeCodeMapping.put("TODO", "TEXT");
        typeCodeMapping.put("AUTHORS", "TEXT");
        typeCodeMapping.put("", "TEXT");
        typeCodeMapping.put(".c", "TEXT");
        typeCodeMapping.put(".cpp", "TEXT");
        typeCodeMapping.put(".hpf", "HTft");
        typeCodeMapping.put(".gif", "GIFf");
        typeCodeMapping.put(".jpg", "JPEG");
        typeCodeMapping.put(".html", "TEXT");
        typeCodeMapping.put(".txt", "TEXT");
        typeCodeMapping.put(".mov", "MooV");
        typeCodeMapping.put(".rm", "PNRA");
        typeCodeMapping.put(".ra", "PNRA");
        typeCodeMapping.put(".ram", "PNst");
        typeCodeMapping.put(".sit", "SITD");
        typeCodeMapping.put(".zip", "ZIP ");
        typeCodeMapping.put(".smi", "APPL");
        typeCodeMapping.put(".gz", "Gzip");
        typeCodeMapping.put(".rsrc", "rsrc");
        typeCodeMapping.put(".aif", "AIFF");
        typeCodeMapping.put(".aiff", "AIFF");
        typeCodeMapping.put(".avi", "VfW ");
        typeCodeMapping.put(".asf", "VfW ");
        typeCodeMapping.put(".mpg", "MPEG");
        typeCodeMapping.put(".mpeg", "MPEG");
        typeCodeMapping.put(".mp3", "MP3 ");
        typeCodeMapping.put(".wav", "WAVE");
        typeCodeMapping.put("DEFAULTMAPPING", "TEXT");

        creatorCodeMapping = new Hashtable();
        creatorCodeMapping.put(".java", "R*ch");
        creatorCodeMapping.put(".c", "R*ch");
        creatorCodeMapping.put(".cpp", "R*ch");
        creatorCodeMapping.put(".txt", "ttxt");
        creatorCodeMapping.put(".hpf", "HTLC");
        creatorCodeMapping.put(".sit", "SIT!");
        creatorCodeMapping.put(".rm", "PNst");
        creatorCodeMapping.put(".ra", "PNst");
        creatorCodeMapping.put(".ram", "PNst");
        creatorCodeMapping.put(".smi", "oneb");
        creatorCodeMapping.put(".gz", "Gzip");
        creatorCodeMapping.put(".rsrc", "RSED");
        creatorCodeMapping.put(".avi", "TVOD");
        creatorCodeMapping.put(".mpg", "TVOD");
        creatorCodeMapping.put(".asf", "TVOD");
        creatorCodeMapping.put(".mpeg", "TVOD");
        creatorCodeMapping.put(".mp3", "MAmp");
        creatorCodeMapping.put(".wav", "TVOD");
        creatorCodeMapping.put("DEFAULTMAPPING", "ttxt");

        mappingsInitialized = true;

    }

    /**
     * Constructs a MacFile out of a given File.
     * @param f a File to operate on as a MacFile.
     * @param p permissions: MacFile.READ_PERM | Macfile.WRITE_PERM
     */
    public MacFile(File f, Integer p) {

	file = new File(f.getParent(), FilenameUtils.qualify(f.getName()));

        if(!mappingsInitialized)
            createTypeCreatorCodeMapping();

        permissions = p;

    }

    /**
     * Returns all available methods for storing Macintosh
     * files.
     * @return a list of all available methods
     */
    public static String[] getMacFileMethods() {
	return macFileMethods;
    }

    /**
     * Returns all supported methods for storing Macintosh
     * files. You should use this method if you want to dynamically
     * determine which subclass of MacFile you should
     * instantiate.
     * @return a list of all supported methods.
     */
    public static String[] getSupportedMacFileMethods() {
	String[] allmodes = getMacFileMethods();
	String[] filter = new String[allmodes.length];
	int count = 0;
	for(int i=0; i<allmodes.length; i++)
	    if(isSupportedMacFileMethod(allmodes[i]))
		filter[count++] = allmodes[i];
	    
	String[] modes = new String[count];
	for(int i=0; i<count; i++) 
	    modes[i] = filter[i];
	return modes;
    }

    /**
     * Queries whether a particular method is supported
     * on the host platform.
     * @param s the method name
     * @return true if the method is supported
     */
    public static boolean isSupportedMacFileMethod(String s) {
        if(s.equals("Native")) {
            if(System.getProperty("os.name").startsWith("Mac") &&
               !System.getProperty("os.name").equals("Mac OS X"))
                return true;
            
	    return false;
	}
	return true;
    }

    /**
     * Returns an object representing the resource fork of the file.
     * @return Fork object representing resource fork.
     */
    public Fork getResourceFork() {

	return resourceFork;

    }

    /**
     * Returns an object representing the data fork of the file.
     * @return Fork object representing data fork.
     */
    public Fork getDataFork() {

	return dataFork;

    }

    /**
     * Returns the <TT>File</TT> that best describes this <TT>MacFile</TT>.
     * The information is for informational purposes only and cannot be
     * assumed to be complete.
     * @return <TT>File</TT> object
     */
    public File getFile() {
	return file;
    }

    /**
     * Called to write a file header. Any or all of these
     * parameters may be ignored by a particular implementation of MacFile.
     * @param name the filename
     * @param type the file type (Mac OS)
     * @param creator the file creator (Mac OS)
     * @param comments the file comments (Mac OS)
     * @param created file creation date
     * @param modified file modification date
     * @param finderFlags the Finder flags
     */
    public abstract void writeHeader(String name, 
				     String type, 
				     String creator,
				     String comments, 
				     Calendar created, 
				     Calendar modifed, 
				     int finderFlags) throws IOException;
   
    /**
     * Called to read header information.
     */
    public abstract void readHeader() throws IOException;

    /**
     * Returns the size of the data fork.
     * @return the size of the data fork.
     */
    public long getDataSize() {
	return dataSize;
    }

    /**
     * Returns the size of the resource fork.
     * @return the size of the resource fork.
     */
    public long getResourceSize() {
	return resourceSize;
    }

    /**
     * Sets the size of the data fork.
     * @param s the size of the data fork.
     */
    public void setDataSize(long s) {
	dataSize = s;
    }

    /**
     * Sets the size of the resource fork.
     * @param s the size of the resource fork.
     */
    public void setResourceSize(long s) {
	resourceSize = s;
    }

    /**
     * Called before the file is closed to allow any necessary cleanup 
     */
    public abstract void cleanup() throws IOException;

    /**
     * Closes all files related to the <TT>MacFile</TT>.
     */
    public abstract void close() throws IOException;

    /**
     * Deletes all files related to the <TT>MacFile</TT>.
     * @return true if deleting the <TT>MacFile</TT> was successfull.
     */
    public abstract boolean delete();

    /**
     * Returns true if this <TT>MacFile</TT> exists.
     * @return true if the <TT>MacFile</TT> exists.
     */
    public abstract boolean exists();

    public boolean isDirectory() {
        
        return getFile().isDirectory();

    }

    public long length() {

        return getDataSize() + getResourceSize();

    }

    public String[] list() {

        return getFile().list();

    }

    /**
     * Returns the Finder flags for this file.
     * @return Finder flags.
     */
    public int getFinderFlags() {

        return finderFlags;

    }

    /**
     * Returns the 4 byte type code for this file. If the file name
     * ends with .hpf, returns "HTft". If the .info file contained a
     * valid type code, returns that. Otherwise maps the filename
     * onto a predefined set of type codes.
     * @return 4 byte type code. 
     */
    public String getType() {

        if(getFile().getName().endsWith(".hpf"))
            return "HTft";

        if(type != null)
            return type;

        if(getFile().isDirectory())
            return "fldr";

        return getMapping(typeCodeMapping);

    }

    /**
     * Returns the 4 byte creator code for this file. If the file name
     * ends with .hpf, returns "HTLC". If the .info file contained a
     * valid creator code, returns that. Otherwise maps the filename
     * onto a predefined set of creator codes.
     * @return 4 byte type code. 
     */
    public String getCreator() {

        if(getFile().getName().endsWith(".hpf"))
            return "HTLC";

        if(creator != null)
            return creator;

        if(getFile().isDirectory())
            return "n/a ";

        return getMapping(creatorCodeMapping);

    }

    private String getMapping(Hashtable mapTable) {

        String name = file.getName();

        if(name.indexOf('.') > 0) {

            int dotpos = name.lastIndexOf('.');
            String extension = name.substring(dotpos).toLowerCase();
            String type = (String) mapTable.get(extension);

            if(mapTable.containsKey(extension))
                return type;
            else
                return (String) mapTable.get("DEFAULTMAPPING");

        } else {

            return (String) mapTable.get("DEFAULTMAPPING");

        }

    }

    public String getComment() {

        return comment;

    }

    public Calendar getCreationDate() {

        /* Argh, this sucks. There is just no way to get reliable
           creation / last modified dates for files in Java. */
            
        return getModificationDate();

    }

    public Calendar getModificationDate() {

        Calendar c = Calendar.getInstance();
        c.setTime(new Date(file.lastModified()));
        return c;

    }

    /**
     * Renames this <TT>MacFile</TT>.
     * @return true if renaming the <TT>MacFile</TT> was successfull.
     */
    public boolean renameTo(File f) {

        DebuggerOutput.debug("Renaming " + file.toString() + " to " + f.toString());
        return false;

    }

    public String toString() {

	return getClass().getName() + "[" + 
            (file.isDirectory() ? file.toString() + " (directory)]" : 
             resourceFork.toString() + ", " + dataFork.toString() + "]");

    }

    /**
     * Factory method, returns a MacFile object for the
     * given parameters. 
     * @return MacFile object.
     */
    static public MacFile createMacFile(String method, File file, int permissions) throws InvocationTargetException, ClassNotFoundException, InstantiationException {

        try {

            /* Create the proper MacFile object to contain the file
               we're about to download. */
            
            String macFileMethod = method + "MacFile";
            Class c = Class.forName("redlight.macfiles." + macFileMethod);
            Class[] paramTypes = new Class[] { File.class, Integer.class };
            Constructor cs = c.getConstructor(paramTypes);
            Object[] params =  new Object[] { file, new Integer(permissions) };
            return (MacFile) cs.newInstance(params);

        } catch(NoSuchMethodException e) {

            DebuggerOutput.stackTrace(e);

        } catch(IllegalAccessException e) {

            DebuggerOutput.stackTrace(e);

        }

        System.out.println("MacFile.createMacFile: FIXME: returning null");
        return null;

    }

}
