/**
 * $Id: BanTable.java,v 1.1.1.1 2001/07/22 02:44:49 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.server;

import java.io.File;
import java.io.LineNumberReader;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.net.InetAddress;
import java.util.Enumeration;

import redlight.hotline.HLServer;
import redlight.hotline.HLServerBanTable;
import redlight.utils.DebuggerOutput;

/**
 * This class implements persistent bantable storage in a
 * human-readable file.  
 */
public class BanTable extends HLServerBanTable {

    File banTableFile;
    long lastModified;

    /**
     * Create the bantable.
     * @param file the file to use for persistent storage.
     */
    public BanTable(HLServer h, File file) {

        super(h);
        banTableFile = file;

    }

    /**
     * Called whenever the bantable changes.
     */
    public void hasChanged() {

        write();

    }

    /**
     * Load the component from disk if it has changed.
     */
    public void checkModified() {

        if(banTableFile.lastModified() != lastModified) {

            Main.logLine("Ban table " + banTableFile.toString() + " changed, reloading.");
            load();

        }

    }

    /**
     * Loads the bantable from persistent storage.
     */
    public void load() {

        load(banTableFile);
        lastModified = banTableFile.lastModified();

    }

    /**
     * Loads the bantable from the given InputStream.
     * @param in the InputStream to load from.
     */
    public void load(InputStream in) {
        
        LineNumberReader lr = null;
        Hashtable tempTable = new Hashtable();
        
        try {
            
            lr = new LineNumberReader(new InputStreamReader(in));
            
            for(String line = lr.readLine(); line != null; line = lr.readLine()) {

                if(line != null) {

                    if(line.indexOf(" ") != -1) {

                        InetAddress ip = InetAddress.getByName(line.substring(0, line.indexOf(" ")));
                        long expires = Long.parseLong(line.substring(line.indexOf(" ") + 1));
                        BanRecord b = this.new BanRecord(ip, expires);
                        tempTable.put(ip.getHostAddress(), b);

                    } else {

                        throw new RuntimeException("missing space");

                    }
                    
                }

            }
            
        } catch(Exception e) {

            tempTable = null;

            String error = "Error reading bantable " + banTableFile.toString();

            if(lr != null)
                error += " at line " + lr.getLineNumber();

            error += " (" + e.toString() + ").";

            Main.logLine(error);

        }

        if(tempTable != null) {

            table.clear();
            table = tempTable;

        }
        
    }

    /**
     * Writes the bantable to persistent storage.
     */
    public void write() {

        write(banTableFile);

    }

    /**
     * Writes the bantable to the specified OutputStream.
     * @param os the OutputStream to write to.
     */
    public void write(OutputStream os) {

        try {
            
            for(Enumeration en = table.elements(); en.hasMoreElements(); ) {

                String line = new String();
                BanRecord b = (BanRecord) en.nextElement();
                line += b.address.getHostAddress() + " ";
                line += b.expires + System.getProperty("line.separator");
                os.write(line.getBytes());

            }
            
        } catch(IOException e) {

            Main.logLine("Error writing bantable file " + banTableFile.toString() + " (" + e.toString() + ").");
            
        }
        
    }

}

