/**
 * $Id: Main.java,v 1.14 2001/10/25 14:13:41 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.server;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.io.IOException;
import java.io.FileInputStream;
import java.io.LineNumberReader;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.Hashtable;
import java.util.Enumeration;
import java.io.FileWriter;
import java.text.DateFormat;

import redlight.hotline.HLServer;
import redlight.hotline.HLProtocol;
import redlight.utils.DebuggerOutput;
import redlight.utils.TimeFormat;
import redlight.crypto.UnixCrypt;

/**
 * Main entry point for the commandline oriented Red Light server.
 */
public class Main {

    static InetAddress address = null;
    static int port = 5500;
    static File accountsFile, logFile, flatnewsFile, agreementFile, banTableFile, trackerTableFile, homeDirectory, trashDirectory, serverRoot, initFile;
    static Agreement agreement;
    static AccountsTable accountsTable;
    static Flatnews flatnews;
    static BanTable banTable;
    static TrackerTable trackerTable;
    static Listener isl;
    static Policy isp;
    static Console console = null;
    static boolean shutdownInitiated = false;
    static HLServer hls = null;
    static HLProtocol hlp;
    static FileWriter logWriter = null;
    static String lSep = System.getProperty("line.separator");
    static SymbolTable symbolTable;
    static String copyright = lSep + 
        "rld (C) 1998 - 2001 Pascal Haakmat <groomed@users.sourceforge.net>" + lSep +
        "rld is free software, covered by the GNU General Public License." + lSep +
        "Modification and redistribution are encouraged. No warranty." + lSep + lSep;
    static boolean DebugOutputEnabled = false;

    /* Exit codes. */

    static final int EXIT_ARG_INVALID = 1;
    static final int EXIT_ACCOUNTS_INVALID = 2;
    static final int EXIT_LOGFILE_ACCESS = 3;

    /**
     * Parse commandline arguments. May exit the program with error code
     * {@link #EXIT_ARG_INVALID}.
     */
    static void parseArgs(String argv[]) {

        int i = 0;

        try {

            for(i = 0; i < argv.length; i++) {
                
                if(argv[i].equals("-v") || argv[i].equals("--version")) {
                    
                    printVersion();
                    System.exit(0);
                    
                } else if(argv[i].equals("-h") || argv[i].equals("--help")) {
                    
                    printUsage();
                    System.exit(0);
                    
                } else if(argv[i].equals("-n") || argv[i].equals("--name")) {

                    symbolTable.internalPut("server.name", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("-d") || argv[i].equals("--description")) {

                    symbolTable.internalPut("server.description", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("-p") || argv[i].equals("--port")) {
                    
                    port = Integer.parseInt(argv[i + 1]);
                    if(port < 0 || port > 65535)
                        throw new IllegalArgumentException(argv[i] + " requires a number: 0 ... 65535");
                    i++;
                    
                } else if(argv[i].equals("--maxusers")) {
                    
                    int maximumUsers = Integer.parseInt(argv[i + 1]);
                    if(maximumUsers < 1 || maximumUsers > 31999)
                        if(maximumUsers < 0 || maximumUsers > 65535)
                            throw new IllegalArgumentException(argv[i] + " requires a number: 1 ... 31999");
                    symbolTable.internalPut("server.maxusers", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--maxul")) {
                    
                    int maxUploads = Integer.parseInt(argv[i + 1]);
                    symbolTable.internalPut("server.maxul", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--maxulperuser")) {
                    
                    int maxUploadsPerUser = Integer.parseInt(argv[i + 1]);
                    symbolTable.internalPut("server.maxulperuser", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--maxulqueue")) {
                    
                    int maxULQueue = Integer.parseInt(argv[i + 1]);
                    symbolTable.internalPut("server.maxulqueue", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--maxdl")) {
                    
                    int maxDownloads = Integer.parseInt(argv[i + 1]);
                    symbolTable.internalPut("server.maxdl", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--maxdlperuser")) {
                    
                    int maxDownloadsPerUser = Integer.parseInt(argv[i + 1]);
                    symbolTable.internalPut("server.maxdlperuser", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--maxdlqueue")) {
                    
                    int maxDLQueue = Integer.parseInt(argv[i + 1]);
                    symbolTable.internalPut("server.maxdlqueue", argv[i + 1]);
                    i++;

                } else if(argv[i].equals("-a") || argv[i].equals("--address")) {
                    
                    address = InetAddress.getByName(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--debug")) {
                    
                    Main.DebugOutputEnabled = true;
                    DebuggerOutput.setEnabled(true);
                    
                } else if(argv[i].equals("--trackertablefile")) {

                    trackerTableFile = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--bantablefile")) {

                    banTableFile = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--agreementfile")) {

                    agreementFile = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--flatnewsfile")) {

                    flatnewsFile = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--accountsfile")) {

                    accountsFile = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--logfile")) {

                    logFile = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--homedir")) {

                    homeDirectory = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--trashdir")) {

                    trashDirectory = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--rootdir")) {

                    serverRoot = new File(argv[i + 1]);
                    i++;

                } else if(argv[i].equals("--crypt")) {

                    System.out.println(UnixCrypt.crypt("aa", argv[i + 1]));
                    System.exit(0);
                    i++;

                } else {
                    
                    printUsage();
                    System.exit(EXIT_ARG_INVALID);
                    
                }
                
            }

        } catch(UnknownHostException e) {

            System.out.println("Invalid address: " + e.getMessage() + ".  Try --help.");
            System.exit(EXIT_ARG_INVALID);

        } catch(ArrayIndexOutOfBoundsException e) {
          
            System.out.println(argv[i] + " requires parameter.  Try --help.");
            System.exit(EXIT_ARG_INVALID);

        } catch(Exception e) {
            
            System.out.println(e.getMessage() + ".  Try --help.");
            System.exit(EXIT_ARG_INVALID);
            
        }  
        
    }

    /**
     * Print usage. 
     */
    static void printUsage() {

	printVersion();
	String s = 
	    copyright + 
	    "Usage: rld [options]" + lSep + lSep +
	    "Options:" + lSep + lSep +
	    " -v, --version                display version information and exit" + lSep +
	    " -h, --help                   show this message and exit" + lSep +
            " -n, --name NAME              set server name to NAME" + lSep +
            " -d, --description DESC       set server description to DESC" + lSep +
            " -p, --port PORT              listen on port PORT (0 ... 65535)" + lSep +
            " -a, --address ADDR           listen on address ADDR" + lSep +
            "     --maxusers NUM           maximum allowed users (1 ... 31999)" + lSep +
            "     --maxul NUM              maximum simultaneous uploads" + lSep +
            "     --maxdl NUM              maximum simultaneous downloads" + lSep +
            "     --maxulperuser NUM       maximum simultaneous uploads per user" + lSep +
            "     --maxdlperuser NUM       maximum simultaneous downloads per user" + lSep +
            "     --maxulqueue NUM         maximum uploads in queue" + lSep +
            "     --maxdlqueue NUM         maximum downloads in queue" + lSep +
            "     --rootdir DIR            use DIR as the server root" + lSep +
            "                              (default: " + serverRoot + ")" + lSep +
            "     --homedir DIR            use DIR as default home directory" + lSep +
            "                              (default: " + serverRoot + System.getProperty("file.separator") + "files)" + lSep +
            "     --trashdir DIR           use DIR as trash directory" + lSep +
            "                              (default: " + serverRoot + System.getProperty("file.separator") + "trash)" + lSep +
            "     --agreementfile FILE     use FILE as agreement file" + lSep + 
            "     --flatnewsfile FILE      use FILE as flat news file" + lSep + 
            "     --accountsfile FILE      use FILE as accounts database" + lSep +
            "     --bantablefile FILE      use FILE as bantable file" + lSep +
            "     --trackertablefile FILE  use FILE as trackertable file" + lSep + 
            "     --logfile FILE           use FILE as logfile" + lSep +            
            "     --crypt PWD              encrypt PWD for use in accounts file and exit" + lSep +
            "     --debug                  output lots of debugging information" + lSep;
	System.out.print(s);

    }
    
    /**
     * Print version.
     */
    static void printVersion() {

	System.out.print(getVersionString());
	
    }

    static String getVersionString() {

        return "rld (experimental) / " + 
            redlight.Version.BUILD_DATE + lSep +
	    "Options: " + redlight.Version.OPTIONS + lSep;
        
    }

    /**
     * Opens the logfile. If the logfile cannot be opened, the program
     * terminates.
     */
    static void openLogFile() {

        try {
            
            logWriter = new FileWriter(logFile.toString(), true);
            logWriter.write(System.getProperty("line.separator"));
 
        } catch(IOException e) {

            System.out.println("Cannot open logfile for writing (" + logFile.toString() + ")");
            System.exit(EXIT_LOGFILE_ACCESS);

        }

    }

    /**
     * Writes a line to the logfile (if it's open) or to stdout.
     * @param line the line to write.
     */
    static void logLine(String line) {

        String msg = TimeFormat.formatCurrentDateTime(DateFormat.SHORT, DateFormat.MEDIUM) + ": " + line + lSep;

        if(logWriter == null) {

            System.out.println(msg);

        } else {

            try {
                
                logWriter.write(msg);
                
                synchronized(logWriter) {

                    logWriter.flush();

                }

            } catch(NullPointerException e) {

                System.out.println(msg);

            } catch(IOException e) {
                
                e.printStackTrace();
                System.out.println("Fatal: cannot write to logfile.  Shutting down. Last message that could not be written:" + lSep + lSep + msg);
                logWriter = null;

                hls.shutdown("Fatal error: cannot write logfile.");
                
            }

        }

    }

    /**
     * Perform required initialization (read arguments, load config
     * files, die if errors occur).
     */
    public static void init(String[] args) {

        symbolTable = new SymbolTable(hls);
        
        DebuggerOutput.setEnabled(false);
        
        serverRoot = new File(System.getProperty("user.home"), ".redlight");
        
        /* Parse arguments. */

        parseArgs(args);

        if(!serverRoot.exists())
            if(!serverRoot.mkdirs())
                System.out.println("Cannot create server root '" + serverRoot + "', exiting." + System.getProperty("line.separator") + "Check the access privileges for the directory, or use" + System.getProperty("line.separator") + "the --rootdir option to specify an alternative server root," + System.getProperty("line.separator") + "or try --help.");

        String fSep = System.getProperty("file.separator");

        /* Reinitialize paths with possibly new values of server
           root. */

        symbolTable.internalPut("server.rootdir", 
                        serverRoot.toString());
        symbolTable.internalPut("server.initfile", 
                        serverRoot.toString() + fSep + "rld.conf");
        symbolTable.internalPut("server.agreementfile", 
                        serverRoot.toString() + fSep + "agreement");
        symbolTable.internalPut("server.flatnewsfile", 
                        serverRoot.toString() + fSep + "flatnews");
        symbolTable.internalPut("server.accountsfile", 
                        serverRoot.toString() + fSep + "accounts");
        symbolTable.internalPut("server.logfile", 
                        serverRoot.toString() + fSep + "rld.log");
        symbolTable.internalPut("server.bantablefile", 
                        serverRoot.toString() + fSep + "bantable");
        symbolTable.internalPut("server.trackertablefile", 
                        serverRoot.toString() + fSep + "trackertable");
        symbolTable.internalPut("server.homedir", 
                        serverRoot.toString() + fSep + "files");
        symbolTable.internalPut("server.trashdir", 
                        serverRoot.toString() + fSep + "trash");

        /* Load values given on commandline. */

        if(initFile != null)
            symbolTable.internalPut("server.initfile", 
                                    initFile.toString());
        if(agreementFile != null)
            symbolTable.internalPut("server.agreementfile", 
                                    agreementFile.toString());
        if(flatnewsFile != null)
            symbolTable.internalPut("server.flatnewsfile", 
                                    flatnewsFile.toString());
        if(accountsFile != null)
            symbolTable.internalPut("server.accountsfile", 
                                    accountsFile.toString());
        if(logFile != null)
            symbolTable.internalPut("server.logfile", 
                                    logFile.toString());
        if(banTableFile != null)
            symbolTable.internalPut("server.bantablefile", 
                                    banTableFile.toString());
        if(trackerTableFile != null)
            symbolTable.internalPut("server.trackertablefile", 
                                    trackerTableFile.toString());
        if(homeDirectory != null)
            symbolTable.internalPut("server.homedir", 
                                    homeDirectory.toString());
        if(trashDirectory != null)
            symbolTable.internalPut("server.trashdir", 
                                    trashDirectory.toString());

        /* Get back merged set. */

        initFile = new File(symbolTable.get("server.initfile"));
        agreementFile = new File(symbolTable.get("server.agreementfile"));
        flatnewsFile = new File(symbolTable.get("server.flatnewsfile"));
        accountsFile = new File(symbolTable.get("server.accountsfile"));
        logFile = new File(symbolTable.get("server.logfile"));
        banTableFile = new File(symbolTable.get("server.bantablefile"));
        trackerTableFile = new File(symbolTable.get("server.trackertablefile"));
        homeDirectory = new File(symbolTable.get("server.homedir"));
        trashDirectory = new File(symbolTable.get("server.trashdir"));

        if(!homeDirectory.exists())
            if(!homeDirectory.mkdirs())
                System.out.println("Cannot create home directory '" + homeDirectory + "', exiting." + System.getProperty("line.separator") + "Check the access privileges for the directory, or use" + System.getProperty("line.separator") + "the --homedir option to specify an alternative home directory," + System.getProperty("line.separator") + "or try --help.");

        if(!trashDirectory.exists())
            if(!trashDirectory.mkdirs())
                System.out.println("Cannot create trash directory '" + trashDirectory + "', exiting." + System.getProperty("line.separator") + "Check the access privileges for the directory, or use" + System.getProperty("line.separator") + "the --trashdir option to specify an alternative trash directory," + System.getProperty("line.separator") + "or try --help.");

        /* Open the logfile. */ 

        openLogFile();

        /* Create some important objects. */

        hlp = new HLProtocol();
        isl = new Listener(hls);
        isp = new Policy(hls);
        accountsTable = new AccountsTable(hls, accountsFile);
        flatnews = new Flatnews(hls, flatnewsFile);
        agreement = new Agreement(hls, agreementFile);
        banTable = new BanTable(hls, banTableFile);
        trackerTable = new TrackerTable(hls, trackerTableFile);

        accountsTable.load();        
        flatnews.load();
        agreement.load();
        banTable.load();
        trackerTable.load();

        logLine("rld (experimental) / " + redlight.Version.BUILD_DATE + ".");
        logLine("Bug reports to <groomed@users.sourceforge.net>.");
        logLine("rld server root is " + serverRoot);
        
    }

    /**
     * Main.
     */
    public static void main(String[] args) {

        hls = new HLServer();

        /* Initialize or die. */

        init(args);

        /* Configure the server, then just run it until it dies. */

        try {

            hls.setHLServerPolicy(isp);
            hls.setHLServerListener(isl);
            hls.setAccountsTable(accountsTable);
            hls.setTrackerTable(trackerTable);
            hls.setBanTable(banTable);
            hls.setFlatnews(flatnews);
            hls.setAgreement(agreement);
            hls.setHomeDirectory(homeDirectory);
            hls.setTrashDirectory(trashDirectory);
            hls.setAddress(address);
            hls.setPort(port);
            hls.setServerName(symbolTable.get("server.name"));
            hls.setServerDescription(symbolTable.get("server.description"));
            hls.setBanSeconds(Integer.parseInt(symbolTable.get("server.maxbanseconds")));
            hls.setMaximumUsers(Integer.parseInt(symbolTable.get("server.maxusers")));
            hls.setMaximumUploads(Integer.parseInt(symbolTable.get("server.maxul")));
            hls.setMaximumUploadsPerUser(Integer.parseInt(symbolTable.get("server.maxulperuser")));
            hls.setMaximumUploadQueueSpots(Integer.parseInt(symbolTable.get("server.maxulqueue")));
            hls.setMaximumDownloads(Integer.parseInt(symbolTable.get("server.maxdl")));
            hls.setMaximumDownloadsPerUser(Integer.parseInt(symbolTable.get("server.maxdlperuser")));
            hls.setMaximumDownloadQueueSpots(Integer.parseInt(symbolTable.get("server.maxdlqueue")));

            hls.listen();

        } catch(Exception e) {

            String reason = e.toString();

            if(e.getMessage() != null)
                reason = e.getMessage();

            Main.logLine("Server shut down: " + reason);
            System.out.println("Server shut down (" + reason + ")");
            DebuggerOutput.stackTrace(e);
            
        } finally {

            if(console != null) {

                console.interrupt();
                
                try {
                    
                    console.join();
                    
                } catch(InterruptedException _e) {}
                
            }

	   try {
	      
	      Thread.currentThread().sleep(1000);
	      
	   } catch(InterruptedException e) {}
	      
	   System.exit(0);
	   
        }

    }

}

/**
 * The server parameters and some defaults.
 */
class SymbolTable {

    Hashtable symbols, descriptions;
    HLServer hls;

    SymbolTable(HLServer hls) {

        this.hls = hls;
        symbols = new Hashtable();
        descriptions = new Hashtable();

        String userHome = System.getProperty("user.home");
        String fSep = System.getProperty("file.separator");

        internalPut("admin.console", 
                    "stdin", "Admin console method");
        internalPut("server.name", 
                    "Red Light", "Server name");
        internalPut("server.description", 
                    "Red Light server.", 
                    "Server description");
        internalPut("server.maxusers", "1000", "Maximum users");
        internalPut("server.maxdl", "3", "Maximum downloads");
        internalPut("server.maxul", "5", "Maximum uploads");
        internalPut("server.maxdlperuser", "1", "Maximum d/ls per user");
        internalPut("server.maxulperuser", "1", "Maximum u/ls per user");
        internalPut("server.maxdlqueue", "10", "Maximum d/l queue");
        internalPut("server.maxulqueue", "20", "Maximum u/l queue");
        internalPut("server.maxbanseconds", "180", "Maximum ban seconds");
        internalPut("server.rootdir", userHome + fSep + ".redlight",
                    "The server root directory");
        internalPut("server.homedir", 
                    get("server.rootdir") + fSep + "files",
                    "The home directory");
        internalPut("server.trashdir", 
                    get("server.rootdir") + fSep + "trash",
                    "The trash directory");
        internalPut("server.initfile", 
                    get("server.rootdir") + fSep + "rld.conf",
                    "The init file");
        internalPut("server.agreementfile", 
                    get("server.rootdir") + fSep + "agreement",
                    "The agreement file");
        internalPut("server.flatnewsfile", 
                    get("server.rootdir") + fSep + "flatnews",
                    "The flat news file");
        internalPut("server.accountsfile", 
                    get("server.rootdir") + fSep + "accounts",
                    "The accounts file");
        internalPut("server.logfile",
                    get("server.rootdir") + fSep + "rld.log",
                    "The log file");
        internalPut("server.bantablefile", 
                    get("server.rootdir") + fSep + "bantable",
                    "The ban table file");
        internalPut("server.trackertablefile", 
                    get("server.rootdir") + fSep + "trackertable",
                    "The tracker table file");

    }

    /**
     * Bypass all the checks ...
     */
    void internalPut(String symbol, String value) {

        symbols.put(symbol, value);

    }

    /**
     * Initialize descriptions.
     */
    private void internalPut(String symbol, String value, String description) {

        symbols.put(symbol, value);
        descriptions.put(symbol, description);

    }

    /**
     * Returns all symbols.
     */
    Enumeration getSymbols() {

        return symbols.keys();

    }

    /**
     * Returns the description for a given symbol.
     */
    String getDescription(String symbol) {

        if(descriptions.containsKey(symbol))
            return (String) descriptions.get(symbol);
        
        return "[not documented]";

    }

    /**
     * Returns the value for a given symbol.
     */
    String get(String symbol) {

        return (String) symbols.get(symbol);

    }

    /**
     * Sets the value for a given symbol, subject to validation
     * rules. Returns a text message indicating whether or not
     * the value was set.
     */
    String put(String symbol, String value) {
        
        if(value == null) {
            
            if(symbol.startsWith("server."))
                return "NAMEs in server. hierarchy cannot be null.";
            
            symbols.remove(symbol);

            return "Set " + symbol + " to null.";

        }
            
        if(symbol.equals("server.rootdir") ||
           symbol.equals("server.logfile"))
            return "Sorry, cannot change symbol " + symbol;
            
        int intValue = 0;
            
        if(symbol.startsWith("server.max")) {
            
            try {
                
                intValue = Integer.parseInt(value);
                
            } catch(NumberFormatException e) {
                
                return "Not a number: " + e.getMessage();
                
            }
            
        }
        
        try {
            
            if(symbol.equals("server.homedir"))             
                hls.setHomeDirectory(new File(value));            
            else if(symbol.equals("server.trashdir"))                
                hls.setTrashDirectory(new File(value));
            else if(symbol.equals("server.maxusers"))
                hls.setMaximumUsers(intValue);
            
        } catch(IllegalArgumentException e) {
            
            return "Illegal argument: " + e.getMessage();

        }

        if(symbol.startsWith("server."))
            if(!symbols.containsKey(symbol))
                return "Cannot create NAMEs in server. hierarchy (NAME does not exist)";
        
        symbols.put(symbol, value);
        
        if(symbol.startsWith("server.")) {

            if(symbol.startsWith("server.max")) {
                
                if(symbol.equals("server.maxdl"))
                    hls.setMaximumDownloads(intValue);
                else if(symbol.equals("server.maxul"))
                    hls.setMaximumUploads(intValue);
                else if(symbol.equals("server.maxdlperuser"))
                    hls.setMaximumDownloadsPerUser(intValue);
                else if(symbol.equals("server.maxulperuser"))
                    hls.setMaximumUploadsPerUser(intValue);
                else if(symbol.equals("server.maxdlqueue"))
                    hls.setMaximumDownloadQueueSpots(intValue);
                else if(symbol.equals("server.maxulqueue"))
                    hls.setMaximumUploadQueueSpots(intValue);
                else if(symbol.equals("server.maxbanseconds"))
                    hls.setBanSeconds(intValue);

            } else {
                
                if(symbol.equals("server.name"))
                    hls.setServerName(value);
                else if(symbol.equals("server.description"))
                    hls.setServerDescription(value);
                else if(symbol.equals("server.agreementfile")) {

                    ((Agreement)
                     hls.getAgreement()).agreementFile = new File(value);
                    ((Agreement) hls.getAgreement()).load();

                } else if(symbol.equals("server.flatnewsfile")) {

                    ((Flatnews) 
                     hls.getFlatnews()).flatnewsFile = new File(value);
                    ((Flatnews) hls.getFlatnews()).load();

                } else if(symbol.equals("server.accountsfile")) {

                    ((AccountsTable) 
                     hls.getAccountsTable()).accountsFile = new File(value);
                    ((AccountsTable) hls.getAccountsTable()).load();

                } else if(symbol.equals("server.bantablefile")) {

                    ((BanTable) 
                     hls.getBanTable()).banTableFile = new File(value);
                    ((BanTable) hls.getBanTable()).load();

                } else if(symbol.equals("server.trackertablefile")) {

                    ((TrackerTable) 
                     hls.getTrackerTable()).trackerTableFile = new File(value);
                    ((TrackerTable) hls.getTrackerTable()).load();

                }
            
            }

        }

        return "Set " + symbol + " to " + value + ".";
            
    }

}
