/**
 * $Id: TrackerTable.java,v 1.2 2001/10/07 04:43:39 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.server;

import java.io.File;
import java.io.LineNumberReader;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.net.InetAddress;
import java.util.Enumeration;

import redlight.hotline.HLServer;
import redlight.hotline.HLServerTrackerTable;
import redlight.utils.DebuggerOutput;

/**
 * This class implements persistent trackertable storage in a
 * human-readable file.  
 */
public class TrackerTable extends HLServerTrackerTable {

    File trackerTableFile;
    long lastModified;

    /**
     * Create the trackertable.
     * @param file the file to use for persistent storage.
     */
    public TrackerTable(HLServer h, File file) {

        super(h);
        trackerTableFile = file;

    }

    /**
     * Called whenever the trackertable changes.
     */
    public void hasChanged() {

        write();

    }

    /**
     * Load the component from disk if it has changed.
     */
    public void checkModified() {

        if(trackerTableFile.lastModified() != lastModified) {

            hls.log("Tracker table " + trackerTableFile.toString() + " changed, reloading.");
            load();

        }

    }

    /**
     * Loads the trackertable from persistent storage.
     */
    public void load() {

        load(trackerTableFile);
        lastModified = trackerTableFile.lastModified();

    }

    /**
     * Loads the trackertable from the given InputStream. A trackertable
     * is a file with plain text lines, each line looking like: 
     * HOST [PASSWORD]
     * @param in the InputStream to load from.
     */
    public void load(InputStream in) {
        
        LineNumberReader lr = null;
        Hashtable tempTable = new Hashtable();
        
        try {
            
            lr = new LineNumberReader(new InputStreamReader(in));
            
            for(String line = lr.readLine(); line != null; line = lr.readLine()) {

                if(line != null) {

                    InetAddress ip = null;
                    String password = null;

                    if(line.indexOf(" ") == -1) {

                        /* No password. */

                        ip = InetAddress.getByName(line);

                    } else {

                        /* Includes password. */

                        ip = InetAddress.getByName(line.substring(0, line.indexOf(" ")));
                        password = line.substring(line.indexOf(" ") + 1);

                    }

                    TrackerRecord t = this.new TrackerRecord(ip, password);
                    tempTable.put(ip.getHostAddress(), t);

                }

            }
            
        } catch(IOException e) {

            tempTable = null;

            String error = "Error reading trackertable " + trackerTableFile.toString();

            if(lr != null)
                error += " at line " + lr.getLineNumber();

            error += " (" + e.toString() + ").";

            hls.log(error);

        }

        if(tempTable != null) {

            table.clear();
            table = tempTable;

        }
        
    }

    /**
     * Writes the trackertable to persistent storage.
     */
    public void write() {

        write(trackerTableFile);

    }

    /**
     * Writes the trackertable to the specified OutputStream.
     * @param os the OutputStream to write to.
     */
    public void write(OutputStream os) {

        try {
            
            for(Enumeration en = table.elements(); en.hasMoreElements(); ) {

                String line = new String();
                TrackerRecord t = (TrackerRecord) en.nextElement();
                line += t.address.getHostName();

                if(t.password != null)
                    line += " " + t.password;

                line += System.getProperty("line.separator");
                os.write(line.getBytes());

            }
            
        } catch(IOException e) {

            hls.log("Error writing trackertable file " + trackerTableFile.toString() + " (" + e.toString() + ").");
            
        }
        
    }

}

