/**
 * $Id: QueueThread.java,v 1.4 2001/10/09 04:11:05 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
package redlight.utils;

import java.util.Vector;

import redlight.utils.DebuggerOutput;

/**
 * A QueueThread schedules Queueable objects on a first-in,
 * first-out basis. It invokes the Queueable that is
 * ready to be executed by calling it's granted() method.
 * @see Queueable
 */
public class QueueThread {
    Vector queue;
    Thread thread;
    boolean elementWaiting = true;
    Queueable monitor;
    Integer monitorLock;
    public int sleepTime = 0;

    /**
     * Constructs a QueueThread and starts scheduling.
     */
    public QueueThread() {

        monitor = null;
        monitorLock = new Integer(1);
	queue = new Vector();
	thread = new Thread(runner, "QueueThread");
	start();

    }

    /**
     * Adds an element to the queue. If the queue is empty,
     * the element is never added but it's granted() method
     * immediately invoked.
     * @see Queueable#granted
     * @param ft the element to add
     * @return true if the element has been queued,
     * false if the element was granted immediately.
     */
    public synchronized boolean add(Queueable ft) {
	boolean queued = true;

        synchronized(queue) {

            queued = queue.size() > 0 ? true : false;
            queue.addElement(ft);

        }

        notify();

	return queued;
    }

    /**
     * Removes an element from the queue. Each element must be
     * removed from the queue before the next one is granted.
     * @see Queueable#granted
     * @param ft the element to remove
     */
    public synchronized void remove(Queueable ft) {

        synchronized(monitorLock) {

            if(ft == monitor)
                monitor = null;

        }

        synchronized(queue) {

            queue.removeElement(ft);

        }

        notify();

    }

    /**
     * Returns the firstmost element in the queue, that is
     * the element that has most recently had it's granted() 
     * method called.
     * @see Queueable#granted
     * @return a Queueable element or null
     */
    public Queueable getMonitor() {

        return monitor;

    }

    /**
     * Starts scheduling. You only need to call this method
     * after having stopped scheduling by invoking the @see stop
     * method.
     */
    public void start() {

	if(!thread.isAlive())
	    thread.start();

    }

    /**
     * Stops scheduling elements. You may restart scheduling
     * by invoking the @see start method.
     */
    public synchronized void stop() {

	thread.interrupt();
        notify();
        queue.removeAllElements();

    }

    Runnable runner = new Runnable() {
            
            public synchronized void run() {
                
                while(!thread.isInterrupted()) {
                    
                    synchronized(monitorLock) {
                        
                        if(!thread.isInterrupted() && monitor == null) {
                            
                            synchronized(queue) {
                                
                                if(queue.size() > 0) {
                                    
                                    monitor = (Queueable) queue.elementAt(0);
                                    
                                } else {
                                    
                                    monitor = null;
                                    
                                }
                                
                            }
                            
                            if(monitor != null) {

                                monitor.granted();

                                if(sleepTime > 0) {
                                    
                                    try {
                                        
                                        Thread.currentThread().sleep(sleepTime);
                                        
                                    } catch(InterruptedException e) {}
                                    
                                }
                                
                            }
                            
                        }
                        
                    }
                    
                    try {
                
                        wait(3000);
                
                    } catch(InterruptedException e) {}
                    
                }
                
            }
            
        };
    
}
    
