/**
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.utils;

/**
 * Converts primitive types from and to byte arrays.
 */
public class ToArrayConverters {

    /**
     * Converts a long to an 8 byte array.
     * @param l a long
     * @return an 8 byte array
     */
    public static byte[] longToByteArray(long l) {
	byte[] b = new byte[8];
	b[0] = (byte) ((l >>> 56) & 0xff);
	b[1] = (byte) ((l >>> 48) & 0xff);
	b[2] = (byte) ((l >>> 40) & 0xff);
	b[3] = (byte) ((l >>> 32) & 0xff);
	b[4] = (byte) ((l >>> 24) & 0xff);
	b[5] = (byte) ((l >>> 16) & 0xff);
	b[6] = (byte) ((l >>>  8) & 0xff);
	b[7] = (byte) ( l        & 0xff);
	return b;
    }

    /**
     * Converts an integer to a 4 char array.
     * @param l the integer to convert
     * @return a 4 char array
     */
    public static char[] intToCharArray(int l) {
	char[] c = new char[4];
	c[0] = (char) ((l >> 24) & 0xff);
	c[1] = (char) ((l >> 16) & 0xff);
	c[2] = (char) ((l >> 8)  & 0xff);
	c[3] = (char) ( l        & 0xff);
	return c;
    }

    /**
     * Converts an integer to a 4 byte array.
     * @param l the integer to convert
     * @return a 4 byte array
     */
    public static byte[] intToByteArray(int l) {
	byte[] b = new byte[4];
	b[0] = (byte) ((l >> 24) & 0xff);
	b[1] = (byte) ((l >> 16) & 0xff);
	b[2] = (byte) ((l >>  8) & 0xff);
	b[3] = (byte) ( l        & 0xff);
	return b;
    }

    /**
     * Converts a short to a 2 byte array.
     * @param l the short to convert
     * @return a 2 byte array
     */
    public static byte[] shortToByteArray(int l) {
	byte[] b = new byte[2];
	b[0] = (byte) ((l >>  8) & 0xff);
	b[1] = (byte) ( l        & 0xff);
	return b;
    }

    /**
     * Converts a char to a 2 byte array.
     * @param l the char to convert
     * @return a 2 byte array
     */
    public static byte[] charToByteArray(char l) {
	byte[] b = new byte[2];
	b[0] = (byte) ((l >>  8) & 0xff);
	b[1] = (byte) ( l        & 0xff);
	return b;
    }

    /**
     * Converts a byte array to a long.
     * @param b a byte array
     * @return long
     * @exception IllegalArgumentException when b is a byte array
     * with more than 8 elements
     */
    public static long byteArrayToLong(byte[] b) {
	long mask = 0xff;
	
	if(b.length < 9) {
	    
	    long val = 0;
	    int shift = 0;
	    int j = b.length;

	    for(int i = b.length; i > 0; i--) {
		val |= (long) (byte) b[i - 1] << shift & mask;
		shift += 8;
		mask <<= 8;
	    }
	    	    
	    return val;
	}
	throw new IllegalArgumentException("byteArrayToLong requires <9 byte arrays");
    }

    /**
     * Converts a byte array to an integer.
     * @param b byte array
     * @return integer
     * @exception IllegalArgumentException when b is a byte array
     * with more than 4 elements
     */
    public static int byteArrayToInt(byte[] b) {
	int mask = 0xff;
	
	if(b.length < 5) {
	    
	    int val = 0;
	    int shift = 0;
	    int j = b.length;

	    for(int i = b.length; i > 0; i--) {
		val |= (byte) b[i - 1] << shift & mask;
		shift += 8;
		mask <<= 8;
	    }
	    	    
	    return val;
	}
	throw new IllegalArgumentException("byteArrayToLong requires <5 byte arrays");
    }

    /**
     * Converts a 2 byte array into a short.
     * @param b 2 byte array
     * @return short
     * @exception IllegalArgumentException when b is not a 2 byte array
     */
    public static short byteArrayToShort(byte[] b) {
	if(b.length == 2) {
	    short s;
	    s = (short) 
		(((short)  b[1]       & 0xff) | 
		 ((short) (b[0] <<  8 & 0xff00)));
	    return s;
	}
	throw new IllegalArgumentException("byte_array_to_short requires 2 byte array");
    }

    /**
     * Swaps the bytes in a byte array.
     * @param b the byte array
     * @return byte swapped byte array
     */
    public static byte[] swapByteArray(byte[] b) {
	byte[] b2 = new byte[b.length];
	for(int i=0; i<b.length; i++) 
	    b2[(b.length-1)-i] = b[i];
	return b2;
    }
}








