package heuser.multilang;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;

/**
 * Enables applications to switch language during runtime.<BR>
 * Only languages that have corresponding files (eg. "GER.Ml") are supported.
 * <BR><BR>
 * <B>Format of .Ml files:</B><BR>
 * ID (eg. "ml09112004")<BR>
 * Words or phrases (eg. "Open image...")<BR>
 * <BR>
 * Both, ID and words, without quotation marks. Simply followed by a new line.
 *
 * @author Jens Heuser, heuserjens@users.sourceforge.net
 */
public class Ml {

    /**
     * The language in which the strings to be translated are paraphrased.
     */
    private String refLang;

    /**
     * The target language.
     */
    private String targetLang;

    /**
     * <code>true</code> if any errors occured.
     */
    public boolean error = false;

    private String errorMsg = "";


    /**
     * Sets the language which has to be translated.
     * @param refLang The language to be translated from.
     * @return <code>true</code> if this language is available.
     */
    public boolean setOriginalLang(String refLang){
        this.refLang = refLang;
        return langAvailable(refLang);
    }


    /**
     * Sets the language in which the text should be translated into.
     * @param targetLang The language to translate into.
     * @return <code>true</code> if the language is available and the file ID matches the one of the other language.
     */
    public boolean setLang(String targetLang){
        this.targetLang = targetLang;
        return langAvailable(targetLang) && compareIDs();
    }


    /**
     * Translates the string.
     * @param original The string to be translated.
     * @return The translation of that string.
     */
    public String pr(String original){
        String fertig = translate(original);
        return fertig;
    }


    /**
     * Translates the string into the first one that matches. Useful for translating longer texts - only the beginning is needed.<BR>
     * <B>Currently not implemented!</B>
     * @param original The beginning of a text to be translated.
     * @return The translation of first string matching the beginning of <code>original</code>.
     */
    public String prl(String original){
        return "";
    }


    /**
     * Returns the error messages.
     * @return String of error messages that (might) have occured.
     */
    public String getError(){
        return errorMsg;
    }


    /**
     * Returns the language in which the strings are translated.
     * @return The target language. Eg. "GER"
     */
    public String getTargetLang(){
        return targetLang;
    }


    private boolean langAvailable(String lang){
        File language = new File(lang + ".ml");
        return language.exists();
    }


    private String translate(String original){
        InputStream org = this.getClass().getResourceAsStream(refLang + ".ml");
        InputStream tol = this.getClass().getResourceAsStream(targetLang + ".ml");


        //File org = new File (refLang + ".Ml");
        //File tol = new File (targetLang + ".Ml");

        int position = 0;
        String trans = "";

        try{
            position = searchFor(org, original);
            trans = findIn(tol, position);
        }catch(Exception e) {
            logError("Translation error: " + e.toString());
            return original;
        }
        if (position == 0) {
            if (!original.startsWith("No translation for '"))
                logError("No translation for '" + original + "' found.");
            return original;
        }
        return trans;
    }


    private int searchFor(InputStream org, String original) throws Exception{
        BufferedReader in = new BufferedReader(new InputStreamReader(org));
        String input;
        int pos = 0;

        while((input = in.readLine()) != null){
            if (input.equals(original)) return pos;
            pos++;
        }
        in.close();
        return 0;
    }


    private String findIn(InputStream tol, int pos) throws Exception{
        BufferedReader in = new BufferedReader(new InputStreamReader(tol));
        String translated = "";

        while(pos > 0) {
            in.readLine();
            pos--;
        }
        translated = in.readLine();
        in.close();

        return translated;
    }


    private void logError(String anError){
        error = true;

        if (!errorMsg.endsWith(anError + "\n"))
            errorMsg += anError + "\n";
    }


    public void clearErrors(){
        error = false;
        errorMsg = "";
    }


    private boolean compareIDs(){
        InputStream org = this.getClass().getResourceAsStream(refLang + ".ml");
        InputStream tol = this.getClass().getResourceAsStream(targetLang + ".ml");;

        boolean match = false;

        try{
            BufferedReader in1 = new BufferedReader(new InputStreamReader(org));
            BufferedReader in2 = new BufferedReader(new InputStreamReader(tol));

            if (in1.readLine().equals(in2.readLine())) match = true;
            in1.close();
            in2.close();
        }catch(Exception e){
            error = true;
            logError("Missing " + refLang + ".ml or/and " + targetLang + ".ml!");
        }
        return match;
    }
}
