package retypar;

import heuser.simpleLogger.SimpleLogger;
import retypar.converter.Converter;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.LinkedList;

/**
 * Runs retypar in command line (batch) mode. Enabled whenever arguments are given.
 */
public class batchMode extends SimpleLogger{

    private String[] args;

    private boolean AZ = true;
    private boolean az = true;
    private boolean num = true;
    private boolean spec = false;
    private boolean col = true;
    private String format = "";
    private String defaultFileExtension = "";

    private Color bgColor = Color.WHITE;
    private boolean isTransparent = false;

    private int outputFormat = retypar.TYPE_JPEG;
    private int compression = 0;
    private int scalefactor = 100;
    private Converter conversionAlgo;
    private int threshold = 5;
    private File outputFile;
    private LinkedList inputFiles = new LinkedList();


    /**
     * Verifies arguments, configures retypar and starts the conversion.<br>
     * For details see <code>java -jar retypar_0.9.3.jar -h</code>
     * @param args Arguments from the command line.
     */
    public batchMode(String args[]){
        setConsoleOutput(true);
        logMsg("\r\n--- " + retypar.appName + " " + retypar.version + " (Jens Heuser, 2005) ---");
        logMsg("=== a tool to convert images into ASCII ===\r\n");
        setConsoleOutput(true);
        this.args = args;

        retypar.readINI();
        retypar.loadConverters();
        
        if (!verifyArguments()) {
            System.exit(1);
        }
        runConversion();
    }

    private boolean verifyArguments(){
        try{
            parseArguments();
        }catch(Exception e){
            logError("Wrong arguments. Forgot a parameter value?\n");
        }
        return success();
    }


    private void runConversion(){
        BufferedImage converted;
        File inputFile;
        String outputFileName = "";

        if (inputFiles.size() == 0) logError("Must specify at least one input file!");
        if (format.equals("")) setOutputFormat("j");

        if (conversionAlgo == null) setAlgorithm("mr");


        logMsg("Using " + ((AZ)?    "A-Z, "     : "")
                        + ((az)?    "a-z, "     : "")
                        + ((num)?   "0-9, "     : "")
                        + ((spec)?  "special, " : "")
                        + ((col)?   "colors"    : ""));
        logMsg("Algorithm: " + conversionAlgo.toString());
        if (conversionAlgo.usesThreshold())
                logMsg("Threshold: " + threshold);
        logMsg("Output format: " + format);
        logMsg("Compression ratio: " + compression);
        logMsg("Scalefactor: " + scalefactor + "%");
        logMsg("Background color: #" + Integer.toHexString(bgColor.getRGB()).substring(2));

        logMsg("\r\nConverting: ");
        for (int i = 0; i < inputFiles.size(); i++){
            inputFile = (File)inputFiles.get(i);

            outputFileName = "ASCII of " + inputFile.getName().substring(0, inputFile.getName().lastIndexOf(".") + 1) + defaultFileExtension;

            logMsg("\t\"" + inputFile.getName() + "\" to \"" + ((outputFile==null)? outputFileName : outputFile.getName()) + "\"");

            converted = retypar.convertImage((File)inputFiles.get(i), AZ, az, num, spec, col, Color.WHITE, scalefactor, 0, 0, 0, conversionAlgo, threshold);
            
            retypar.stopTimer();
            if (!retypar.storeImage((outputFile == null) ? new File(outputFileName) : outputFile,
                                    converted,
                                    compression,
                                    outputFormat,
                                    col,
                                    Color.WHITE,
                                    conversionAlgo.getASCII(),
                                    conversionAlgo.getColors(),
                                    isTransparent)){
                logError("Could not write to file!");
                System.exit(1);
            }

            logMsgAppend(", " + retypar.getDuration());
        }
    }

    private void parseArguments(){
        for (int i = 0; i < args.length; i++){
            if (args[i].startsWith("-")){
                switch(args[i].charAt(1)){
                    case 'g':
                        setGranularity(args[i + 1]);
                        break;
                    case 'o':
                        setOutputFile(args[i + 1]);
                        break;
                    case 'x':
                        setOutputFormat(args[i + 1]);
                        break;
                    case 'c':
                        setCompression(args[i + 1]);
                        break;
                    case 's':
                        setScalefactor(args[i + 1]);
                        break;
                    case 'a':
                        setAlgorithm(args[i + 1]);
                        break;
                    case 't':
                        setThreshold(args[i + 1]);
                        break;
                    case 'q':
                        disableLogging();
                        i--;
                        break;
                    case 'l':
                        setLogFile(args[i + 1]);
                        break;
                    case 'b':
                        setBgColor(args[i + 1]);
                        break;
                    case 'p':
                        isTransparent = true;
                        break;
                    case 'h':
                        printHelp();
                        System.exit(0);
                        break;
                    case 'e':
                        printAlgos();
                        System.exit(0);
                        break;
                    case '?':
                        printHelp();
                        System.exit(0);
                        break;
                    default:
                        logError("Unknown switch: " + args[i]);
                }
                i++;
            }
            else setInputFile(args[i]);
        }
    }


    private void setGranularity(String g){
        AZ = false;
        az = false;
        num = false;
        spec = false;
        col = false;

        while(g.length() > 0){
            switch(g.charAt(0)){
                case 'A':
                    AZ = true;
                    break;
                case 'a':
                    az = true;
                    break;
                case '0':
                    num = true;
                    break;
                case 's':
                    spec = true;
                    break;
                case 'c':
                    col = true;
                    break;
                default:
                    logError("Wrong argument in switch '-g': " + g);
            }
            g = g.substring(1);
        }
    }


    private void setInputFile(String in){
        inputFiles.add(new File(in));
        if (!((File)inputFiles.getLast()).exists()) logError("No such file: " + in);
    }


    private void setOutputFile(String out){
        outputFile = new File(out);
    }


    private void setOutputFormat(String f){
        switch(f.charAt(0)){
            case 'j':
                outputFormat = retypar.TYPE_JPEG;
                format = "JPEG";
                defaultFileExtension = retypar.EXTENSIONS_TYPE_JPEG[0];
                break;
            case 'g':
                outputFormat = retypar.TYPE_GIF;
                format = "GIF";
                defaultFileExtension = retypar.EXTENSIONS_TYPE_GIF[0];
                break;
            case 't':
                outputFormat = retypar.TYPE_TEXT;
                format = "Text";
                defaultFileExtension = retypar.EXTENSIONS_TYPE_TEXT[0];
                break;
            case 'h':
                outputFormat = retypar.TYPE_HTML;
                format = "HTML";
                defaultFileExtension = retypar.EXTENSIONS_TYPE_HTML[0];
                break;
            case 'v':
                outputFormat = retypar.TYPE_TVID;
                format = "Text video";
                if (outputFile == null)
                    setOutputFile("textvideo.txv");
                break;
            default:
                logError("Wrong output format: " + f);
        }
    }


    private void setCompression(String c){
        try{
            compression = Integer.parseInt(c);
        }catch(Exception e){
            logError("Wrong compression ratio: " + c);
        }
        if (compression < 0 || compression > 100) logError("Wrong Compression ratio. Has to be between 0 and 100.");
    }

    private void setScalefactor(String s){
        try{
            scalefactor = Integer.parseInt(s);
        }catch(Exception e){
            logError("Wrong scalefactor: " + s);
        }
        if(scalefactor < 0) logError("Wrong scale factor. Must not be less than 0.");
    }


    private void disableLogging(){
        setConsoleOutput(false);
    }

    private void setLogFile(String lf){
        try{
            logToFile(new File(lf), true);
        }catch(Exception e){
            logError("Error logging to File '" + lf + "'");
        }
    }

    private void setBgColor(String bgColor){
        if (!bgColor.startsWith("#")) logError("Wrong color. Format (hexadecimal): #rrggbb");
        this.bgColor = new Color(Integer.parseInt(bgColor.substring(1, bgColor.length()).toLowerCase(), 16));
    }


    private void setAlgorithm(String rm){
        if (retypar.converters.isEmpty()) logError("(FATAL) No conversion algorithms installed!");
        boolean valid = false;

        for (int i = 0; i < retypar.converters.size(); i++){
            if (((Converter)retypar.converters.get(i)).getShortName().equals(rm)){
                conversionAlgo = (Converter)retypar.converters.get(i);
                valid = true;
            }
        }
        if (!valid) logError("Algorithm not found! Check 'java -jar " + retypar.appName + "_" + retypar.version + ".jar -e'");
    }


    private void setThreshold(String th){
        try{
            threshold = Integer.parseInt(th);
        }catch(Exception e){
            logError("Wrong threshold specified. Use a number!");
        }
        //if (threshold < 0 || threshold > 255) logError("Wrong threshold specified. Should be a value between 0 and 255.");
    }


    private void printAlgos(){
        System.out.println(printAllLog());
        logMsg("Installed algorithms:");

        for (int i = 0; i < retypar.converters.size(); i++){
            logMsg(((Converter)retypar.converters.get(i)).getShortName() +
                    ": " +
                    ((Converter)retypar.converters.get(i)).toString() +
                    ", by " +
                    ((Converter)retypar.converters.get(i)).getAuthor() +
                    " (" +
                    ((Converter)retypar.converters.get(i)).getWebsite() +
                    ")");
        }
    }


    private void printHelp(){
        System.out.println(printAllLog());       //TODO besser machen...
        logMsg("Usage:" +
                "\njava -jar retypar.jar input_file {input_file} -g Aa0sc -o output_file -m j|t|h -c compression_ratio -s scale_factor -q\n" +
                "\n-g: Use uppercase letters (A), lowercase letters (a), numbers (0), special characters (s) and colors (c). Default is 'Aa0c'." +
                "\n-o: The output file to write to. If none given, \"ASCII of (input_file)\" is used." +
                "\n-x: Export result as JPEG (j), text (t), HTML (h) or text video (v). Only one allowed. Defaults to 'j'." +
                "\n-c: Compression ratio of the resulting JPEG file. Default is '0'." +
                "\n-s: Scalefactor in percent. Resizes the image before converting when <> '100'. Default is '100'." +
                "\n-b: Background color. Hexadecimal, that is '#rrggbb'. Defaults to white (#ffffff)." +
                "\n-g: make background color transparent (only with GIF images)" +
                "\n-a: Specify the algorithm to be used. Default is 'mr'. Check 'java -jar " + retypar.appName + "_" + retypar.version + ".jar -e'" +
                "\n-e: List all installed conversion algorithms." +
                "\n-t: The rendering threshold. Not supported by all algorithms." +
                "\n-l: Log to a file. Console output is not disabled, to disable use also '-q'" +
                "\n-q: Quiet. Disables output to console." +
                "\n-h: This help screen." +
                "\n-?: Guess what? Again, this help screen." +
                "\n\nAll switches are optional. Only a filename has to be given. Wildcars allowed." +
                "\nVisit http://www.retypar.de.vu for more information.");
    }

    /**
     *
     * @return e.g. "retypar 0.9.1"
     */
    public String toString(){
        String about = retypar.appName + " " + retypar.version;
        return about;
    }
}
