package dks.src.textureEditor;

import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.ObjectStreamException;
import java.io.Serializable;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.gradientEditor.CGradient;
import dks.src.utils.XML.XMLWritable;
import dks.src.utils.XML.XMLWriter;
import dks.src.utils.listener.CChangeListenerDelegate;
import dks.src.utils.listener.Changeable;

/**
 * This class implement a layer used for represent a texture's layer which can be saved in a XML Format <br> date : 4 sept. 07
 * @author   DarK Sidious
 */
public class CLayer implements XMLWritable, Serializable, Changeable {

	private static final long serialVersionUID = 1714705869371314793L;

	protected static final String ERROR_LAYER_FILENAME_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : Le nom du fichier image du calque n'a pas t trouv";
	protected static final String ERROR_LAYER_COLOR_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : La couleur du calque n'a pas t trouve";
	protected static final String ERROR_LAYER_TYPE_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : le type de calque n'a pas t trouv";

	protected static final String XML_COLOR_PROPERTY = "color";
	protected static final String XML_GRADIENT_PROPERTY = "gradient";
	protected static final String XML_TYPE_LAYER_PROPERTY = "typeLayer";
	protected static final String XML_IMAGE_PROPERTY = "image";

	protected Color _color;
	protected String _imageFileName = "";
	protected CGradient _gradient;
	protected ETypeLayer _type;

	protected transient BufferedImage _image;
	protected transient CChangeListenerDelegate _changeListenerDelegate;
	protected transient Change _changeListener;

	/**
	 * @param color the color of the layer
	 */
	public CLayer(Color color) {
		_changeListener = new Change();
		_changeListenerDelegate = new CChangeListenerDelegate();
		_color = color;
		_type = ETypeLayer.ColoredLayer;
	}

	/**
	 * @param image the image of the layer
	 */
	public CLayer(BufferedImage image) {
		_image = image;
		_type = ETypeLayer.ImageLayer;
		_changeListenerDelegate = new CChangeListenerDelegate();
	}

	/**
	 * @param gradient the gradient of the layer
	 */
	public CLayer(CGradient gradient) {
		_gradient = gradient;
		_changeListenerDelegate = new CChangeListenerDelegate();
		_changeListener = new Change();
		_gradient.addChangeListener(_changeListener);
		_type = ETypeLayer.GradientLayer;
	}

	/**
	 * @param changeListener the ChangeListener to add
	 */
	public void addChangeListener(ChangeListener changeListener) {
		_changeListenerDelegate.addListener(changeListener);
	}

	/**
	 * @param changeListener the ChangeListener to remove
	 */
	public void removeChangeListener(ChangeListener changeListener) {
		_changeListenerDelegate.removeListener(changeListener);
	}

	/**
	 * @return the color of the layer
	 */
	public Color getColor() {
		return _color;
	}

	/**
	 * @param color the color of the layer
	 */
	public void setColor(Color color) {
		if (_color == null || !_color.equals(color)) {
			_color = color;
			_changeListenerDelegate.notifyChanged();
		}
	}

	/**
	 * @return the gradient of the layer
	 */
	public CGradient getGradient() {
		return _gradient;
	}

	/**
	 * @param gradient the layer of the gradient
	 */
	public void setGradient(CGradient gradient) {
		if ((_gradient == null) || !_gradient.equals(gradient)) {
			_gradient = gradient;
			_gradient.addChangeListener(_changeListener);
			_changeListenerDelegate.notifyChanged();
		}
	}

	/**
	 * @return the image of the layer
	 */
	public BufferedImage getImage() {
		return _image;
	}

	/**
	 * @param image the image of the layer
	 */
	public void setImage(BufferedImage image) {
		if ((_image == null) || !_image.equals(image)) {
			_image = image;
			_changeListenerDelegate.notifyChanged();
		}
	}

	/**
	 * @return the type of the layer
	 */
	public ETypeLayer getType() {
		return _type;
	}

	/**
	 * @param type the type of the layer
	 */
	public void setType(ETypeLayer type) {
		if (!_type.equals(type)) {
			_type = type;
			_changeListenerDelegate.notifyChanged();
		}
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLload(org.jdom.Element)
	 * @param root the XML DOM Element used to load the properties of the layer
	 */
	public void XMLload(Element root) throws JDOMException {
		final String type = root.getAttributeValue(XML_TYPE_LAYER_PROPERTY);
		if (type == null) {
			throw new JDOMException(ERROR_LAYER_TYPE_NOT_FOUND_DESCRIPTION);
		}

		_type = ETypeLayer.valueOf(type);
		switch (_type) {
		case ColoredLayer:
			_color = XMLWriter.getColorElement(XML_COLOR_PROPERTY, root);
			if (_color == null) {
				throw new JDOMException(ERROR_LAYER_COLOR_NOT_FOUND_DESCRIPTION);
			}
			break;
		case GradientLayer:
			_gradient = new CGradient();
			_gradient.XMLload(root.getChild(XML_GRADIENT_PROPERTY));
			break;
		case ImageLayer:
			_imageFileName = null;
			_imageFileName = root.getChildText(XML_IMAGE_PROPERTY);
			if (_imageFileName == null) {
				throw new JDOMException(ERROR_LAYER_FILENAME_NOT_FOUND_DESCRIPTION);
			}
			break;
		default:
			break;
		}
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLsave(org.jdom.Element)
	 * @param root the XML DOM Element used to save the properties of the layer
	 */
	public void XMLsave(Element root) {
		root.setAttribute(XML_TYPE_LAYER_PROPERTY, _type.toString());
		switch (_type) {
		case ImageLayer:
			root.addContent(XMLWriter.createElement(XML_IMAGE_PROPERTY, _imageFileName));
			break;
		case GradientLayer:
			final Element gradient = new Element(XML_GRADIENT_PROPERTY);
			root.addContent(gradient);
			_gradient.XMLsave(gradient);
			break;
		case ColoredLayer:
		default:
			root.addContent(XMLWriter.createColorElement(XML_COLOR_PROPERTY, _color));
			break;
		}
	}

	/**
	 * @return the fileName of the image
	 */
	public String getImageFileName() {
		return _imageFileName;
	}

	/**
	 * @param imageFileName the fileName of the image
	 */
	public void setImageFileName(String imageFileName) {
		if (!_imageFileName.equals(imageFileName)) {
			_imageFileName = imageFileName;
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((_color == null) ? 0 : _color.hashCode());
		result = prime * result
				+ ((_gradient == null) ? 0 : _gradient.hashCode());
		result = prime * result
				+ ((_imageFileName == null) ? 0 : _imageFileName.hashCode());
		result = prime * result + ((_type == null) ? 0 : _type.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final CLayer other = (CLayer) obj;
		if (_color == null) {
			if (other._color != null)
				return false;
		} else if (!_color.equals(other._color))
			return false;
		if (_gradient == null) {
			if (other._gradient != null)
				return false;
		} else if (!_gradient.equals(other._gradient))
			return false;
		if (_imageFileName == null) {
			if (other._imageFileName != null)
				return false;
		} else if (!_imageFileName.equals(other._imageFileName))
			return false;
		if (_type == null) {
			if (other._type != null)
				return false;
		} else if (!_type.equals(other._type))
			return false;
		return true;
	}

	public String toString() {
		return "dks.src.textureEditor.CLayer[color=" + _color + ";imageFileName=" + _imageFileName + ";gradient=" + _gradient + ";type=" + _type + "]";
	}

	protected class Change implements ChangeListener {
		public void stateChanged(ChangeEvent arg0) {
			_changeListenerDelegate.notifyChanged();
		}
	}

	protected Object readResolve() throws ObjectStreamException {
		_changeListener = new Change();
		_changeListenerDelegate = new CChangeListenerDelegate();
		return this;
	}
}
