package dks.src.utils.XML;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

public class XMLWriter {

	/**
	 * The name of the red component of the color XML Property
	 */
	public static final String XML_COLOR_RED_PROPERTY = "red";

	/**
	 * The name of the green component of the color XML Property
	 */
	public static final String XML_COLOR_GREEN_PROPERTY = "green";

	/**
	 * The name of the blue component of the color XML Property
	 */
	public static final String XML_COLOR_BLUE_PROPERTY = "blue";

	/**
	 * The name of the alpha component of the color XML Property
	 */
	public static final String XML_COLOR_ALPHA_PROPERTY = "alpha";

	/**
	 * The name of the fontname of the font XML Property
	 */
	public static final String XML_FONT_FONTNAME_PROPERTY = "fontName";

	/**
	 * The name of the size of the font XML Property
	 */
	public static final String XML_FONT_SIZE_PROPERTY = "size";

	/**
	 * The name of the bold style of the font XML Property
	 */
	public static final String XML_FONT_BOLD_PROPERTY = "bold";

	/**
	 * The name of the italic style of the font XML Property
	 */
	public static final String XML_FONT_ITALIC_PROPERTY = "italic";

	/**
	 * The name of the width of the dimension XML Property
	 */
	public static final String XML_DIMENSION_WIDTH_PROPERTY = "width";

	/**
	 * The name of the height of the dimension XML Property
	 */
	public static final String XML_DIMENSION_HEIGHT_PROPERTY = "height";

	/**
	 * @param title the name of the element
	 * @param value the value of the element
	 * @return the XML DOM Element created
	 */
	public static Element createElement(String title, String value) {
		final Element element = new Element(title);
		element.setText(value);
		return element;
	}

	/**
	 * @param title the name of the element
	 * @param color the color of the element
	 * @return the XML DOM Element which contains the color properties
	 */
	public static Element createColorElement(String title, Color color) {
		final Element element = new Element(title);
		element.setAttribute(XML_COLOR_RED_PROPERTY, Integer.valueOf(color.getRed()).toString());
		element.setAttribute(XML_COLOR_GREEN_PROPERTY, Integer.valueOf(color.getGreen()).toString());
		element.setAttribute(XML_COLOR_BLUE_PROPERTY, Integer.valueOf(color.getBlue()).toString());
		element.setAttribute(XML_COLOR_ALPHA_PROPERTY, Integer.valueOf(color.getAlpha()).toString());
		return element;
	}

	/**
	 * @param title the name of the element
	 * @param root the element which contains the colors properties
	 * @return the color
	 */
	public static Color getColorElement(String title, Element root) {
		final Element colorElement = root.getChild(title);
		Color color = null;
		if (colorElement != null) {
			final int red = Integer.parseInt(colorElement.getAttributeValue(XML_COLOR_RED_PROPERTY));
			final int green = Integer.parseInt(colorElement.getAttributeValue(XML_COLOR_GREEN_PROPERTY));
			final int blue = Integer.parseInt(colorElement.getAttributeValue(XML_COLOR_BLUE_PROPERTY));
			final int alpha = Integer.parseInt(colorElement.getAttributeValue(XML_COLOR_ALPHA_PROPERTY));
			color = new Color(red, green, blue, alpha);
		}
		return color;
	}

	/**
	 * @param title the name of the element
	 * @param font the font of the element
	 * @return the XML DOM Element which contains the font properties
	 */
	public static Element createFontElement(String title, Font font) {
		final Element oElement = new Element(title);
		oElement.setAttribute(XML_FONT_FONTNAME_PROPERTY, font.getFamily());
		oElement.setAttribute(XML_FONT_SIZE_PROPERTY, Integer.valueOf(font.getSize()).toString());
		oElement.setAttribute(XML_FONT_BOLD_PROPERTY, Boolean.valueOf(font.isBold()).toString());
		oElement.setAttribute(XML_FONT_ITALIC_PROPERTY, Boolean.valueOf(font.isItalic()).toString());
		return oElement;
	}

	/**
	 * @param title the name of the element
	 * @param root the element which contains the font properties
	 * @return the font
	 */
	public static Font getFontElement(String title, Element root) {
		final Element fontElement = root.getChild(title);
		Font font = null;
		if (fontElement != null) {
			final String fontName = fontElement.getAttributeValue(XML_FONT_FONTNAME_PROPERTY);
			final int size = Integer.parseInt(fontElement.getAttributeValue(XML_FONT_SIZE_PROPERTY));
			final boolean bold = Boolean.parseBoolean(fontElement.getAttributeValue(XML_FONT_BOLD_PROPERTY));
			final boolean italic = Boolean.parseBoolean(fontElement.getAttributeValue(XML_FONT_ITALIC_PROPERTY));
			int style = Font.PLAIN;
			if (bold) {
				style += Font.BOLD;
			}
			if (italic) {
				style += Font.ITALIC;
			}
			font = new Font(fontName, style, size);
		}
		return font;
	}

	/**
	 * Create an XML Element for a dimension
	 * @param title the name of the element
	 * @param dimension the dimension of the element
	 * @return the XML DOM Element which contains the dimension properties
	 */
	public static Element createDimensionElement(String title, Dimension dimension) {
		final Element oElement = new Element(title);
		oElement.setAttribute(XML_DIMENSION_WIDTH_PROPERTY, new Double(dimension.getWidth()).toString());
		oElement.setAttribute(XML_DIMENSION_HEIGHT_PROPERTY, new Double(dimension.getHeight()).toString());
		return oElement;
	}

	/**
	 * Read the properties of a dimension element and return an instance of the Dimension
	 * @param title the name of the element
	 * @param root the element which contains the dimension properties
	 * @return the Dimension instance
	 */
	public static Dimension getDimensionElement(String title, Element root) {
		final Element dimensionElement = root.getChild(title);
		Dimension dimension = null;
		if (dimensionElement != null) {
			final int width = (int) (Double.parseDouble(dimensionElement.getAttributeValue(XML_DIMENSION_WIDTH_PROPERTY)));
			final int height = (int) (Double.parseDouble(dimensionElement.getAttributeValue(XML_DIMENSION_HEIGHT_PROPERTY)));
			dimension = new Dimension(width, height);
		}
		return dimension;
	}

	/**
	 * Save a XML DOM Element into a file
	 * @param file the file used to save the XML DOM Document
	 * @param root the XML DOM Element to save
	 * @throws IOException
	 */
	public static void saveDocument(File file, Element root) throws IOException{
		final Document document = new Document(root);
		final XMLOutputter fileOut = new XMLOutputter(Format.getPrettyFormat());
		fileOut.output(document, new FileOutputStream(file));
	}

	/**
	 * Load a file into a XML DOM Element
	 * @param file the file used to load the XML DOM Element
	 * @return the XML DOM Element loaded
	 * @throws IOException
	 * @throws JDOMException
	 */
	public static Element openDocument(File file) throws IOException, JDOMException {
		final SAXBuilder oParser = new SAXBuilder();
		final Document oDocument;
		oDocument = oParser.build(file);
		return oDocument.getRootElement();
	}
}
