package typecontrol.controls;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.ChangeListener;

import typecontrol.CTypeControl;
import typecontrol.IdksTypeControl;

public class JdksFileChooser extends JComponent implements IdksTypeControl {

	private static final long serialVersionUID = 1L;

	private Boolean _autoSelect = true;
	
	private GUI _gui;
	private Action _action; 
	private Focus _focus;
	private Keyboard _keyboardListener;
	private CTypeControl _typeControl; 
	
	public JdksFileChooser(String label, String info) {
		super();
		initialize(label, info);
	}
	
	public JdksFileChooser(String label) {
		super();
		initialize(label, "");
	}
	
	private void initialize(String label, String info) {
		_typeControl = new CTypeControl();
		_typeControl.setFocusBackground(CTypeControl.FOCUS_COLOR);
		_typeControl.setLabel(label);
		_typeControl.setInfo(info);
		_action = new Action();
		_focus = new Focus();
		_keyboardListener = new Keyboard();
		_gui = new GUI();
	}
	
	public String getValue() {
		return _gui.getTxtControl().getText();
	}

	public void setValue(String value) {
		_gui.getTxtControl().setText(value);
		raiseChangeListener(this);
	}
	
	public void setInfo(String info) {
		_typeControl.setInfo(info);
		_gui.rebuildGUI();
	}
	
	public void setLabel(String label) {
		_typeControl.setLabel(label);
		_gui.rebuildGUI();
	}

	public void setNormalBackground(Color background) {
		_gui.getTxtControl().setBackground(background);
	}
	
	public void setAutoSelect(Boolean autoSelect) {
		_autoSelect = autoSelect;
	}
	
	public void setEnabled(Boolean enabled) {
		_gui.getTxtControl().setEnabled(enabled);
	}
	
	public boolean isEnabled() {
		if (_gui != null && _gui.getTxtControl() != null) {
			return _gui.getTxtControl().isEnabled();
		} else {
			return true;
		}
	}
	
	private void btnFile_Click() {
		JFileChooser file = _gui.getFileChooser();
		int result = file.showOpenDialog(_gui.getBtnFile());
		if (result == JFileChooser.APPROVE_OPTION) {
			File filePath = file.getSelectedFile();
			_gui.getTxtControl().setText(filePath.getPath());
		}
		raiseChangeListener(this);
	}
	
	private void txtControl_Change() {
		raiseChangeListener(this);
	}
	
	private class GUI {
		protected static final String BTN_FILE_NAME = "btnFile";
		protected static final String TXT_CONTROL = "txtControl";
		
		private final String BTN_FILE_CAPTION = "..."; 
		
		private JTextField _txtControl;
		private JButton _btnFile;
		private JPanel _panButtons;
		
		private JFileChooser _filFileChooser;
		
		public GUI() {
			rebuildGUI();
		}
		
		protected void rebuildGUI() {
			setLayout(new BorderLayout());
			removeAll();
			if (isLabelled()) {
				add(getLabLabel(), BorderLayout.WEST);
			}
			
			add(getTxtControl(), BorderLayout.CENTER);
			getTxtControl().addFocusListener(_focus);
			_focus.setNormalBackground(getTxtControl().getBackground());
			
			_panButtons = null;
			add(getPanButtons(), BorderLayout.EAST);
			updateUI();
		}
				
		protected JPanel getPanButtons() {
			if (_panButtons == null) {
				_panButtons = new JPanel();
				_panButtons.setLayout(new BorderLayout());
				_panButtons.add(getBtnFile(), BorderLayout.WEST);
				if (_typeControl.getInfo() != null && !_typeControl.getInfo().equals("")) {
					_panButtons.add(getBtnInfo(), BorderLayout.EAST);
				}
			}
			return _panButtons;
		}

		protected JTextField getTxtControl() {
			if (_txtControl == null) {
				_txtControl = new JTextField();
				_txtControl.setName(TXT_CONTROL);
				_txtControl.addKeyListener(_keyboardListener);
			}
			return _txtControl;
		}
	
		protected JButton getBtnFile() {
			if (_btnFile == null) {
				_btnFile = new JButton(BTN_FILE_CAPTION);
				_btnFile.setName(BTN_FILE_NAME);
				_btnFile.addActionListener(_action);
			}
			return _btnFile;
		}
	
		protected JFileChooser getFileChooser() {
			if (_filFileChooser == null) {
				_filFileChooser = new JFileChooser();
			}
			return _filFileChooser;
		}
	}
	
	private class Action implements ActionListener {
		public void actionPerformed(ActionEvent arg0) {
			String name = ((JComponent) arg0.getSource()).getName();
			if (name.equals(GUI.BTN_FILE_NAME)) {
				btnFile_Click();
			}
		}
	}

	private class Focus implements FocusListener {
		private Color _normalBackground;
		
		public Focus() {
			super();
		}
		
		public void setNormalBackground(Color normalBackground) {
			_normalBackground = normalBackground;
		}
		
		public void focusGained(FocusEvent arg0) {
			if (_typeControl.getFocusBackground() != null) {
				_gui.getTxtControl().setBackground(_typeControl.getFocusBackground());
			}
			if (_autoSelect) {
				_gui.getTxtControl().setSelectionStart(0);
				_gui.getTxtControl().setSelectionEnd(_gui.getTxtControl().getText().length());
			}
		}

		public void focusLost(FocusEvent arg0) {
			if (_typeControl.getFocusBackground() != null) {
				_gui.getTxtControl().setBackground(_normalBackground);
			}
		}
	}

	private class Keyboard extends KeyAdapter {
		public void keyTyped(KeyEvent arg0) {
			String name = ((JComponent) arg0.getSource()).getName();
			if (name.equals(GUI.TXT_CONTROL)) {
				txtControl_Change();
			}
		}
	}

	public void addChangeListener(ChangeListener changeListener) {
		_typeControl.addChangeListener(changeListener);
	}

	public JButton getBtnInfo() {
		return _typeControl.getBtnInfo();
	}

	public Color getFocusBackground() {
		return _typeControl.getFocusBackground();
	}

	public String getInfo() {
		return _typeControl.getInfo();
	}

	public String getLabel() {
		return _typeControl.getLabel();
	}

	public JLabel getLabLabel() {
		return _typeControl.getLabLabel();
	}

	public boolean isLabelled() {
		return _typeControl.isLabelled();
	}

	public void raiseChangeListener(IdksTypeControl source) {
		_typeControl.raiseChangeListener(source);
	}

	public void removeChangeListener(ChangeListener changeListener) {
		_typeControl.removeChangeListener(changeListener);
	}

	public void setFocusBackground(Color focusBackground) {
		_typeControl.setFocusBackground(focusBackground);
	}

	public void setLabelAlignment(int alignment) {
		_typeControl.setLabelAlignment(alignment);
	}

	public void setLabelWidth(int width) {
		_typeControl.setLabelWidth(width);
	}
}
