/* (C) 1999-2000 Samuel Audet <guardia@cam.org>

Profitable use of this source code based on its execution or sale
excluding the cost of the media, shipping, manwork or supporting
hardware is not allowed unless granted by the author himself.  Any
modifications or inclusion of this code in other non-profitable programs
must contain this message and the original author's name. Programs based
on any of this source code must therefore contain the original or
modified source code files.  Use of this source code in a commercial
program will require permission from the author.  No more than 50% of
the original size of the source code from Disk Indexer can be used in a
non-commercial program, unless granted by the author.

*/

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.table.*;
import javax.swing.event.*;
import java.io.*;
import java.util.*;
import java.text.*;

public class SearchDialog extends JFrame implements ActionListener, Runnable, ListSelectionListener, PropertiesStickListener, PropertiesChangeListener
{
   JLabel message = new JLabel("Enter Wildcard Search String:");
   JTextField searchField;
   JCheckBox statusButton = new JCheckBox("Real Time Status");
   JButton searchButton = new JButton("Search");
   JButton stopButton = new JButton("Stop");
   JPanel buttonPane = new JPanel();
   JLabel status = new JLabel(" ");

   HDataTableModel tableModelOrg;
   TableSorter tableModel;
   JTable foundTable;
   JScrollPane scrollFoundTable;

   TreePath aTreePath;
   String path;

   Thread searchThread = null;

   HDatabase database;
   JFrame mainFrame;
   Properties options;
   FileManagerPanel fileManagerPanel;

   JPopupMenu popMenu;
   MyJMenuItem openPop, proPop, lookupPop;
   Vector nodesForAction = new Vector(10);

   Vector stickyPropertiesDialogs = new Vector(5);

public SearchDialog(JFrame frame, TreePath aTreePath, HDatabase database, Properties options, FileManagerPanel fileManagerPanel)
{
   super();
   this.database = database;
   this.aTreePath = aTreePath;
   mainFrame = frame;
   this.options = options;
   this.fileManagerPanel = fileManagerPanel;

   path = new String();
   for(int i = 0; i < aTreePath.getPathCount(); i++)
      path += "/" + aTreePath.getPathComponent(i);

   this.setTitle("Searching in " + path);

   tableModelOrg = new HDataTableModel(HDataTableModel.PATH,null,database);
   tableModel = new TableSorter(tableModelOrg);
   foundTable = new JTable(tableModel);
   foundTable.getSelectionModel().addListSelectionListener(this);
   tableModel.addMouseListenerToHeaderInTable(foundTable);
   scrollFoundTable = new JScrollPane(foundTable);

   message.setAlignmentX(LEFT_ALIGNMENT);

   foundTable.setShowVerticalLines(false);
   foundTable.setShowHorizontalLines(false);
   scrollFoundTable.getViewport().setBackingStoreEnabled(true);
   scrollFoundTable.getViewport().putClientProperty("EnableWindowBlit", Boolean.TRUE);

   int total = 0;
   for (int i = 0; i < 5; i++)
   {
      TableColumn column = foundTable.getColumnModel().getColumn(i);
      switch(i)
      {
         case 0: column.setPreferredWidth(UIManager.getIcon("Tree.closedIcon").getIconWidth()+15);
                 column.setMaxWidth(column.getPreferredWidth());
                 total += column.getPreferredWidth(); break;
         case 1: column.setPreferredWidth(125); total += 150; break;
         case 2: column.setPreferredWidth(175); total += 150; break;
         case 3: column.setPreferredWidth(75); total += 75; break;
         case 4: column.setPreferredWidth(100); total += 100; break;
      }
   }
   foundTable.setPreferredScrollableViewportSize(new Dimension(total,3*total/4));
   foundTable.revalidate();

   foundTable.setDefaultRenderer(Date.class, new DefaultTableCellRenderer()
   {
      public Component getTableCellRendererComponent(JTable table, Object date,
                       boolean isSelected, boolean hasFocus, int row, int column)
      {
         setHorizontalAlignment(CENTER);
         return super.getTableCellRendererComponent(table, DateFormat.getInstance().format((Date)date), isSelected, hasFocus, row, column);
      }
   });

   foundTable.addMouseListener(new TableMouseListener());
   foundTable.addKeyListener(new TableKeyListener());

   searchField = new JTextField(options.getProperty("search.pattern"));
   searchField.setMaximumSize(new Dimension(Short.MAX_VALUE,searchField.getPreferredSize().height+5));
   searchField.setAlignmentX(LEFT_ALIGNMENT);


   searchButton.setAlignmentX(LEFT_ALIGNMENT);

   buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.X_AXIS));
   buttonPane.add(searchField);
   buttonPane.add(Box.createRigidArea(new Dimension(5,0)));
   buttonPane.add(searchButton);
   buttonPane.add(Box.createRigidArea(new Dimension(5,0)));
   buttonPane.add(stopButton);
   buttonPane.add(Box.createRigidArea(new Dimension(5,0)));
   buttonPane.add(statusButton);
   buttonPane.setAlignmentX(LEFT_ALIGNMENT);
   searchButton.addActionListener(this);
   stopButton.addActionListener(this);
   statusButton.setSelected(Boolean.valueOf(options.getProperty("search.realTimeStatus")).booleanValue());
   searchButton.setMnemonic(KeyEvent.VK_S);
   statusButton.setMnemonic(KeyEvent.VK_R);
   stopButton.setMnemonic(KeyEvent.VK_T);
   stopButton.setEnabled(false);

   JPanel dialogPane = new JPanel();
   dialogPane.setLayout(new BoxLayout(dialogPane, BoxLayout.Y_AXIS));
   dialogPane.add(message);
   dialogPane.add(Box.createRigidArea(new Dimension(0,5)));
   dialogPane.add(buttonPane);
   dialogPane.add(Box.createRigidArea(new Dimension(0,5)));

   dialogPane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

   this.getContentPane().add(dialogPane,"North");
   this.getContentPane().add(scrollFoundTable,"Center");
   this.getContentPane().add(status,"South");

   addWindowListener(new WindowAdapter()
   {
      public void windowClosing(WindowEvent e)
      {
         SearchDialog.this.options.put("search.x",String.valueOf(SearchDialog.this.getLocation().x));
         SearchDialog.this.options.put("search.y",String.valueOf(SearchDialog.this.getLocation().y));
         SearchDialog.this.options.put("search.width",String.valueOf(SearchDialog.this.getSize().width));
         SearchDialog.this.options.put("search.height",String.valueOf(SearchDialog.this.getSize().height));

         SearchDialog.this.options.put("search.pattern", searchField.getText());
         SearchDialog.this.options.put("search.realTimeStatus", String.valueOf(statusButton.isSelected()));

         DiskIndexer.saveProperties();

         mainFrame.requestFocus();
      }
      boolean focusSet = false;
      public void windowActivated(WindowEvent e)
      {
         if(!focusSet)
            searchField.requestFocus();
         focusSet = true;
      }
   });

   searchField.setRequestFocusEnabled(true);
   searchField.requestFocus();
   searchField.grabFocus();
   this.pack();
//   setLocationRelativeTo(frame);

   String width  = options.getProperty("search.width"),
          height = options.getProperty("search.height");
   if(width != null && height != null)
      setSize(Integer.parseInt(width), Integer.parseInt(height));

   setLocation(Integer.parseInt(options.getProperty("search.x")),
               Integer.parseInt(options.getProperty("search.y")));

   // initialization popup menu
   popMenu = new JPopupMenu();
   openPop = new MyJMenuItem("Open",'o'); openPop.addActionListener(this);
   proPop = new MyJMenuItem("Properties",'p'); proPop.addActionListener(this);
   lookupPop = new MyJMenuItem("Lookup",'l'); lookupPop.addActionListener(this);
   /* add */
   popMenu.add(openPop);
   popMenu.add(proPop);
   popMenu.addSeparator();
   popMenu.add(lookupPop);
}

   // these are run in the event thread for Swing
   class UpdateStatus implements Runnable
   {
      String path = null;

      public void run()
      {
         if(path == "")
            status.setText("Searching...");
         else
            status.setText("Searching in " + path);
      }
   }

   class AddNodes implements Runnable
   {
      Vector nodes = new Vector(10,0);

      public void run()
      {
         tableModel.addNodes(nodes);
      }
   }

   // only one instance to minimize object creation
   UpdateStatus updateStatus = new UpdateStatus();
   AddNodes addNodes = new AddNodes();

protected void searchFiles(TreePath aTreePath, final String path, FilenameMatch fileNameMatch, Thread myThread) throws IOException
{
   DynamicTreeNode parentTreeNode = (DynamicTreeNode) aTreePath.getLastPathComponent();
   HDataNode parent = (HDataNode) parentTreeNode.getUserObject();

   short count = database.childCount(parent);

   if(statusButton.isSelected())
   {
      updateStatus.path = path;
      SwingUtilities.invokeLater(updateStatus);
   }

   addNodes.nodes.setSize(0);
   addNodes.nodes.ensureCapacity(count);

   Vector subDirs = new Vector(10,0);

   for(int i=0; i < count && myThread == searchThread; i++)
   {
      HDataNode anode = database.read(parent, i);
      if(fileNameMatch.match(anode.fileName))
      {
         anode.treePath = aTreePath; // used by open and properties
         anode.path = path; // used by tableModel(PATH);
         addNodes.nodes.addElement(anode);
      }
      if(anode.childPointer > 0)
         subDirs.addElement(anode);
   }

   try
   {
      if(!addNodes.nodes.isEmpty())
         SwingUtilities.invokeAndWait(addNodes);
   }
   catch(InterruptedException e) { }
   catch(java.lang.reflect.InvocationTargetException e) { }

   for(int i = 0; i < subDirs.size() && myThread == searchThread; i++)
   {
      HDataNode aNode = (HDataNode) subDirs.elementAt(i);
      String newpath = path.endsWith("/") ? path + aNode : path + "/" + aNode;
      DynamicTreeNode fakeTreeNode = new DynamicTreeNode(aNode, database);
      searchFiles(aTreePath.pathByAddingChild(fakeTreeNode), newpath, fileNameMatch, myThread);
   }
}


public void actionPerformed(ActionEvent e)
{
   String command = e.getActionCommand();

   if(command == "Search")
   {
      searchButton.setEnabled(false);
      stopButton.setEnabled(true);

      if(searchThread == null)
      {
         tableModel.clearTable();
         searchThread = new Thread(this,"Searching");
         searchThread.start();
      }
   }
   else if(command == "Stop")
   {
      searchThread = null;
   }
   else if(command == "Open")
   {
      for(int i = 0; i < nodesForAction.size(); i++)
      {
         HDataNode aNode = (HDataNode)nodesForAction.elementAt(i);
         DiskIndexer.executeAssoc(aNode, aNode.treePath, this);
      }
   }
   else if(command == "Properties")
   {
      for(int i = 0; i < nodesForAction.size(); i++)
      {
         HDataNode aNode = (HDataNode)nodesForAction.elementAt(i);
//         DynamicTreeNode aTreeNode = tableModel.getAssocTreeNode(aNode);

         PropertiesDialog.showDialog(this, aNode, aNode.treePath, database, this, this);
      }
   }
   else if(command == "Lookup")
   {
      for(int i = 0; i < nodesForAction.size(); i++)
      {
         HDataNode aNode = (HDataNode) nodesForAction.elementAt(i);
         lookup(aNode);
      }
   }
}


public void nodeChanged(HDataNode aNode, HDataNode newNode, TreePath aTreePath)
{
   if(newNode == aNode)
      return; // nothing changed

   try
   {
      database.update(aNode,newNode);

      int count = tableModel.getRowCount();
      for(int i = 0; i < count; i++)
      {
         HDataNode anotherNode = tableModel.getNode(i);
         if(aNode.treePath.equals(anotherNode.treePath))
         {
            if(aNode.equals(anotherNode))
            {
               tableModel.setNode(i, newNode);
            }
         }

         anotherNode = null;
      }
   }
   catch(IOException e)
   {
      JOptionPane.showMessageDialog(this,"Database corrupted, could not Update \"" + aNode + "\" node.","Error",JOptionPane.ERROR_MESSAGE);
   }
}

TreePath prevTreePath = null;

void lookup(HDataNode aNode)
{
   Object[] fakePath = aNode.treePath.getPath();
   TreePath realTreePath = new TreePath(fileManagerPanel.treeModel.getRoot());
   // if root actually has any children
   if(realTreePath.getPathComponent(0) != null)
   {
      int j;
      boolean found = true;
      // fakePath[0] always equals the root
      for(j = 1; j < fakePath.length && found; j++)
      {
         DynamicTreeNode parent = (DynamicTreeNode) realTreePath.getPathComponent(j-1);
         Enumeration allChildren = parent.children();
         found = false;
         while(allChildren.hasMoreElements())
         {
            DynamicTreeNode aTreeNode = (DynamicTreeNode) allChildren.nextElement();
            HDataNode dataNodeFromTree = (HDataNode)aTreeNode.getUserObject();
            HDataNode dataNodeFromSearch = (HDataNode)((DynamicTreeNode)fakePath[j]).getUserObject();
            if(dataNodeFromTree.equals(dataNodeFromSearch))
            {
               realTreePath = realTreePath.pathByAddingChild(aTreeNode);
               found = true;
               break;
            }
         }
      }
      // if it's a directory, container or drive, open it instead
      found = false;
      DynamicTreeNode parent = (DynamicTreeNode) realTreePath.getPathComponent(j-1);
      Enumeration allChildren = parent.children();
      while(allChildren.hasMoreElements())
      {
         DynamicTreeNode aTreeNode = (DynamicTreeNode) allChildren.nextElement();
         HDataNode dataNodeFromTree = (HDataNode)aTreeNode.getUserObject();
         if(dataNodeFromTree.equals(aNode))
         {
            realTreePath = realTreePath.pathByAddingChild(aTreeNode);
            found = true;
            break;
         }
      }

      fileManagerPanel.dirTree.setSelectionPath(realTreePath);
      fileManagerPanel.dirTree.scrollPathToVisible(realTreePath);

      // must be somewhere in the table then
      if(!found)
      {
         int last = -1;
         fileManagerPanel.listTable.clearSelection();

         for(j = 0; j < fileManagerPanel.tableModel.getRowCount(); j++)
         {
            HDataNode realNode = fileManagerPanel.tableModel.getNode(j);
            if(realNode.equals(aNode))
            {
               // does multiple selection if the right table is already opened
               if(prevTreePath != null && !realTreePath.equals(prevTreePath))
                  fileManagerPanel.listTable.clearSelection();

               fileManagerPanel.listTable.addRowSelectionInterval(j,j);
               last = j;
               break;
            }
         }
         if(last != -1)
            fileManagerPanel.listTable.scrollRectToVisible(fileManagerPanel.listTable.getCellRect(j,0,false));
      }
   }
   prevTreePath = realTreePath;

}

public void run()
{
   try
   {
      Thread myThread = Thread.currentThread();

      updateStatus.path = "";
      SwingUtilities.invokeLater(updateStatus);

      searchFiles(aTreePath, "/", new FilenameMatch(searchField.getText()), myThread);
   }
   catch(IOException ee)
   {
      JOptionPane.showMessageDialog(this,"Database corrupted","Error",JOptionPane.ERROR_MESSAGE);
   }
   finally
   {
      searchThread = null;
      SwingUtilities.invokeLater(new Runnable()
      {
         public void run()
         {
            searchButton.setEnabled(true);
            stopButton.setEnabled(false);
            status.setText("Found " + tableModel.getRowCount() + " matches");
         }
      });
   }
}

public void valueChanged(ListSelectionEvent e)
{
   if(!stickyPropertiesDialogs.isEmpty())
   {
      HDataNode[] nodes = tableModel.getNodes(foundTable.getSelectedRows());
      for(int i = 0; i < nodes.length && i < stickyPropertiesDialogs.size(); i++)
         ((PropertiesDialog) stickyPropertiesDialogs.elementAt(i)).loadNode(nodes[i],tableModel.getPathToTable());
   }
}

public void addStickDialog(PropertiesDialog aDialog)
{
   stickyPropertiesDialogs.addElement(aDialog);
}

public void removeStickDialog(PropertiesDialog aDialog)
{
   stickyPropertiesDialogs.removeElement(aDialog);
}

protected void finalize() throws Throwable
{
   tableModelOrg = null;
   tableModel = null;
   aTreePath = null;
   path = null;

   searchThread = null;

   database = null;
   nodesForAction = null;

   stickyPropertiesDialogs = null;

   super.finalize();
}


   public class TableMouseListener extends MouseAdapter
   {
      public void mousePressed(MouseEvent e)
      {
         popthis(e);
      }
      public void mouseReleased(MouseEvent e)
      {
         popthis(e);
      }
      public void popthis(MouseEvent e)
      {
         if(e.isPopupTrigger())
         {
            JTable aTable = (JTable) e.getComponent();
            TableSorter aTableModel = (TableSorter) aTable.getModel();
            int row = aTable.rowAtPoint(new Point(e.getX(),e.getY()));
            if(row != -1)
            {
               nodesForAction.removeAllElements();

               int[] selectedRows;
               if(aTable.isRowSelected(row))
                  selectedRows = aTable.getSelectedRows();
               else
                  selectedRows = new int[] {row};

               nodesForAction.ensureCapacity(selectedRows.length);
               for(int i = 0; i < selectedRows.length; i++)
               {
                  HDataNode aNode = (HDataNode) aTableModel.getNode(selectedRows[i]);
                  nodesForAction.addElement(aNode);
               }

               popMenu.show(aTable,e.getX(),e.getY());
            }
         }
      }

      public void mouseClicked(MouseEvent e)
      {
         if(e.getClickCount() == 2 && SwingUtilities.isLeftMouseButton(e))
         {
            JTable aTable = (JTable) e.getComponent();
            TableSorter aTableModel = (TableSorter) aTable.getModel();
            int row = aTable.rowAtPoint(new Point(e.getX(),e.getY()));
            if(row != -1)
            {
               HDataNode aNode = (HDataNode) aTableModel.getNode(row);
               if(aNode.type == aNode.f)
                  DiskIndexer.executeAssoc(aNode, aNode.treePath, SearchDialog.this);
               else
                  lookup(aNode);
            }
         }
      }


   }

   public class TableKeyListener extends KeyAdapter
   {
      public void keyPressed(KeyEvent e)
      {
         if(e.getKeyCode() == e.VK_ENTER)
         {
            JTable aTable = (JTable) e.getComponent();
            TableSorter aTableModel = (TableSorter) aTable.getModel();
            int row = aTable.getSelectedRow();
            HDataNode aNode = aTableModel.getNode(row);

            if(aNode.type == aNode.f)
               DiskIndexer.executeAssoc(aNode, aTableModel.getPathToTable(), SearchDialog.this);
            else
               lookup(aNode);
            e.consume();
         }
      }
   }

}

