/**
 * A sorter for TableModels. The sorter has a model (conforming to TableModel)
 * and itself implements TableModel. TableSorter does not store or copy
 * the data in the TableModel, instead it maintains an array of
 * integers which it keeps the same size as the number of rows in its
 * model. When the model changes it notifies the sorter that something
 * has changed eg. "rowsAdded" so that its internal array of integers
 * can be reallocated. As requests are made of the sorter (like
 * getValueAt(row, col) it redirects them to its model via the mapping
 * array. That way the TableSorter appears to hold another copy of the table
 * with the rows in a different order. The sorting algorthm used is stable
 * which means that it does not move around rows when its comparison
 * function returns 0 to denote that they are equivalent.
 *
 * @version 1.5 12/17/97
 * @author Philip Milne
 *
 * Modified it to support my own implemention of TableModel,  HDataTableModel
 * Maybe it should just be made as a subclass of HDataTableModel, to avoid
 * retyping all the new function names
 * - Samuel Audet <guardia@cam.org>
 */

import java.util.*;

import javax.swing.table.TableModel;
import javax.swing.event.TableModelEvent;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.tree.*;
import java.io.*;

// Imports for picking up mouse events from the JTable.

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.InputEvent;
import javax.swing.JTable;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;

public class TableSorter extends TableMap implements ShuttleSortCompare {
    int[]    indexes;
    int[]    sortingColumns = new int[] {0};
    boolean  ascending = true;

    public TableSorter() {
        indexes = new int[0]; // for consistency
    }

    public TableSorter(TableModel model) {
        setModel(model);
    }

    public void setModel(TableModel model) {
        super.setModel(model);
        reallocateIndexes();
    }

    public int compareRowsByColumn(int row1, int row2, int column) {
        Class type = model.getColumnClass(column);
        TableModel data = model;

        // Check for nulls.

        Object o1 = data.getValueAt(row1, column);
        Object o2 = data.getValueAt(row2, column);

        // If both values are null, return 0.
        if (o1 == null && o2 == null) {
            return 0;
        } else if (o1 == null) { // Define null less than everything.
            return -1;
        } else if (o2 == null) {
            return 1;
        }

        /*
         * We copy all returned values from the getValue call in case
         * an optimised model is reusing one object to return many
         * values.  The Number subclasses in the JDK are immutable and
         * so will not be used in this way but other subclasses of
         * Number might want to do this to save space and avoid
         * unnecessary heap allocation.
         */

        if (type.isAssignableFrom(Number.class)) {
            Number n1 = (Number)data.getValueAt(row1, column);
            double d1 = n1.doubleValue();
            Number n2 = (Number)data.getValueAt(row2, column);
            double d2 = n2.doubleValue();

            if (d1 < d2) {
                return -1;
            } else if (d1 > d2) {
                return 1;
            } else {
                return 0;
            }
        } else if (type.isAssignableFrom(Date.class)) {
            Date d1 = (Date)data.getValueAt(row1, column);
            long n1 = d1.getTime();
            Date d2 = (Date)data.getValueAt(row2, column);
            long n2 = d2.getTime();

            if (n1 < n2) {
                return -1;
            } else if (n1 > n2) {
                return 1;
            } else {
                return 0;
            }
        } else if (type.isAssignableFrom(String.class)) {
            String s1 = (String)data.getValueAt(row1, column);
            String s2 = (String)data.getValueAt(row2, column);
            int result = s1.compareTo(s2);

            if (result < 0) {
                return -1;
            } else if (result > 0) {
                return 1;
            } else {
                return 0;
            }
        } else if (type.isAssignableFrom(Boolean.class)) {
            Boolean bool1 = (Boolean)data.getValueAt(row1, column);
            boolean b1 = bool1.booleanValue();
            Boolean bool2 = (Boolean)data.getValueAt(row2, column);
            boolean b2 = bool2.booleanValue();

            if (b1 == b2) {
                return 0;
            } else if (b1) { // Define false < true
                return 1;
            } else {
                return -1;
            }
        } else if (type.isAssignableFrom(ImageIcon.class)) {
            HDataNode node1 = ((HDataTableModel) model).getNode(row1),
                      node2 = ((HDataTableModel) model).getNode(row2);

            int prio1 = 0, prio2 = 0;

            switch(node1.type)
            {
               case node1.c: prio1 = 1; break;
               case node1.v: prio1 = 2; break;
               case node1.d: prio1 = 3; break;
               case node1.z: prio1 = 4; break;
               case node1.f: prio1 = 5; break;
            }

            switch(node2.type)
            {
               case node2.c: prio2 = 1; break;
               case node2.v: prio2 = 2; break;
               case node2.d: prio2 = 3; break;
               case node2.z: prio2 = 4; break;
               case node2.f: prio2 = 5; break;
            }

            if (prio1 == prio2) {
                return 0;
            } else if (prio1 > prio2) {
                return 1;
            } else {
                return -1;
            }
        } else {
            Object v1 = data.getValueAt(row1, column);
            String s1 = v1.toString();
            Object v2 = data.getValueAt(row2, column);
            String s2 = v2.toString();
            int result = s1.compareTo(s2);

            if (result < 0) {
                return -1;
            } else if (result > 0) {
                return 1;
            } else {
         return 0;
            }
        }
    }

    // implementation of ShuttleSortCompare

    public int compare(int row1, int row2) {
        for (int level = 0; level < sortingColumns.length; level++) {
            int column = sortingColumns[level];
            int result = compareRowsByColumn(row1, row2, column);
            if (result != 0) {
                return ascending ? result : -result;
            }
        }
        return 0;
    }

    public void reallocateIndexes() {
        int rowCount = model.getRowCount();

        // Set up a new array of indexes with the right number of elements
        // for the new data model.
        indexes = new int[rowCount];

        // Initialise with the identity mapping.
        for (int row = 0; row < rowCount; row++) {
            indexes[row] = row;
        }
    }

    public void tableChanged(TableModelEvent e) {
        //System.out.println("Sorter: tableChanged");
        reallocateIndexes();
        // should do some fine tuning not to completely destroy the sorting

        super.tableChanged(e);
    }

    public void checkModel() {
        if (indexes.length != model.getRowCount()) {
            System.err.println("Sorter not informed of a change in model.");
        }
    }

    public void sort()
    {
        checkModel();
        // keep ".." indicator at the beginning of the listing
        if( ((HDataTableModel) model).getRowCount() > 0 &&
            ((HDataTableModel) model).getNode(0).fileName == "..")
           ShuttleSort.sort((int[])indexes.clone(), indexes, 1, indexes.length, this);
        else
           ShuttleSort.sort((int[])indexes.clone(), indexes, 0, indexes.length, this);
        super.tableChanged(new TableModelEvent(this));
    }

    public void sortByColumn(int column, boolean ascending) {
        this.ascending = ascending;
        if(model.getColumnClass(column).isAssignableFrom(ImageIcon.class))
           sortingColumns = new int[] {column, column+1};
        else
           sortingColumns = new int[] {column};

        sort();
    }


    // The mapping only affects the contents of the data rows.
    // Pass all requests to these rows through the mapping array: "indexes".

    public void clearTable()
    {
       checkModel();
       ((HDataTableModel) model).clearTable();
    }

    public void addNode(HDataNode aNode)
    {
       checkModel();
       ((HDataTableModel) model).addNode(aNode);
    }

    public void addNodes(Vector nodes)
    {
       checkModel();
       ((HDataTableModel) model).addNodes(nodes);
    }


    public void removeNode(HDataNode aNode) {
        checkModel();
        ((HDataTableModel) model).removeNode(aNode);
    }

    public Object getValueAt(int aRow, int aColumn) {
        checkModel();
        return model.getValueAt(indexes[aRow], aColumn);
    }

    public void setValueAt(Object aValue, int aRow, int aColumn) {
        checkModel();
        model.setValueAt(aValue, indexes[aRow], aColumn);
    }

    public HDataNode getNode(int row) {
       checkModel();
       return ((HDataTableModel) model).getNode(indexes[row]);
    }

    public void setNode(int row, HDataNode aNode) {
       checkModel();
       ((HDataTableModel) model).setNode(indexes[row],aNode);
    }

    public HDataNode[] getNodes(int rows[]) {
       checkModel();
       int[] realrows = new int[rows.length];
       for(int i = 0; i < rows.length; i++)
          realrows[i] = indexes[rows[i]];
       return ((HDataTableModel) model).getNodes(realrows);
    }

    public DynamicTreeNode getAssocTreeNode(HDataNode node) {
       checkModel();
       return ((HDataTableModel) model).getAssocTreeNode(node);
    }

    public DynamicTreeNode getAssocTreeNode(int index) {
       checkModel();
       return ((HDataTableModel) model).getAssocTreeNode(indexes[index]);
    }

    public TreePath getAssocTreePath(int index) {
       checkModel();
       return ((HDataTableModel) model).getAssocTreePath(indexes[index]);
    }

    public TreePath getPathToTable() {
       checkModel();
       return ((HDataTableModel) model).getPathToTable();
    }

    public void setValueAt(HDataNode aNode, int row) {
       checkModel();
       ((HDataTableModel) model).setValueAt(aNode,row);
    }

    public void loadTable(DynamicTreeNode aTreeNode) throws IOException {
       ((HDataTableModel) model).loadTable(aTreeNode);
       sortByColumn(sortingColumns[0], ascending);
    }

    // There is no-where else to put this.
    // Add a mouse listener to the Table to trigger a table sort
    // when a column heading is clicked in the JTable.
    public void addMouseListenerToHeaderInTable(JTable table) {
        final TableSorter sorter = this;
        final JTable tableView = table;
        tableView.setColumnSelectionAllowed(false);
        MouseAdapter listMouseListener = new MouseAdapter() {
            public void mouseClicked(MouseEvent e) {
                TableColumnModel columnModel = tableView.getColumnModel();
                int viewColumn = columnModel.getColumnIndexAtX(e.getX());
                int column = tableView.convertColumnIndexToModel(viewColumn);
                if (e.getClickCount() == 1 && column != -1) {
                    //System.out.println("Sorting ...");
                    int shiftPressed = e.getModifiers()&InputEvent.SHIFT_MASK;
                    boolean ascending = (shiftPressed == 0);
                    sorter.sortByColumn(column, ascending);
                }
            }
        };
        JTableHeader th = tableView.getTableHeader();
        th.addMouseListener(listMouseListener);
    }

}

