/* *******************************************************
 * 
 * name:          CreateTones.java
 * 
 * description:   StereoModule which creates different base tones
 *
 * Author:        Florian Bomers
 *
 * (c) by         JaWavedit-Team
 *                JaWavedit@bome.com
 *
 ********************************************************* */

  import java.lang.Math;
  import java.awt.*;
  import java.awt.event.*;

  public class CreateTones extends Dialog implements 
                                        StereoModule,
                                        ActionListener,
                                        AdjustmentListener,
                                        ItemListener  {
    
    public static int MAXIMUM_FREQ=4000;

    WaveFile wf;
    Selection sel;
	  double initialFreq;
		double finalFreq;
    int createFreq;
    int createLength;
      
    private static final int SBMin=1;
    private static final int SBRange=500;
    private static final int SBVisible=70;
    private Label LInitialFreq;
    private Label LFinalFreq;
    private Label LCreateLength;
    private Label LCLength;
    private Label LCreateFreq;
    private Label LCFreq;
    private Scrollbar SBInitialFreq;
    private Scrollbar SBFinalFreq;
    private Scrollbar SBCreateLength;
    private Scrollbar SBCreateFreq;
    private Checkbox CBSine;
    private Checkbox CBSquare;
    private Checkbox CBSaw;
    private Checkbox CBTriangle;
    private Checkbox CBCreate;
    private Button BOK;
    //private Button BApply;
    private Button BCancel;
    
    private static int[] Freqs={8000,11025,22050,44100,48000};
    
    private void makeSine(short Wave[], int waveFreq, 
                          double initialFreq, double finalFreq,
                          int start, int end) {
  		// here it goes...
  		double freq=initialFreq;
  		double freqFactor=(finalFreq-initialFreq)/(double) (end-start+1);
  		double periodFactor=(double) 2 * Math.PI / (double) waveFreq;
	  	double period=0;

  		for (int i=start; i<end; i++, period+=freq*periodFactor) {
  		  Wave[i]=(short)(Math.sin(period)*32000);
  		  freq=initialFreq+(double) i * freqFactor;
  		}
		}

    private void makeSquare(short Wave[], int waveFreq, 
                          double initialFreq, double finalFreq,
                          int start, int end) {
  		// here it goes...
  		double freq=initialFreq;
  		double freqFactor=(finalFreq-initialFreq)/(double) (end-start+1);
	  	int period=0;
	  	short sign=1;

  		for (int i=start; i<end; i++, period++) {
  		  Wave[i]=(short) (sign* 30000);
  		  freq=initialFreq+(double) i * freqFactor;
  		  if (period>(double) waveFreq / freq) {
  		    sign=(short) -sign;
  		    period=0;
  		  }
  		}
		}
    
    private void makeSaw(short Wave[], int waveFreq, 
                          double initialFreq, double finalFreq,
                          int start, int end) {
  		// here it goes...
  		double freq=initialFreq;
  		double freqFactor=(finalFreq-initialFreq)/(double) (end-start+1);
	  	double step=-1;

  		for (int i=start; i<end; i++, step+=freq/(double) waveFreq) {
  		  if (step>1) {
  		    step=-1;
  		  }
  		  Wave[i]=(short) (step* 30000);
  		  freq=initialFreq+(double) i * freqFactor;
  		}
		}

    private void makeTriangle(short Wave[], int waveFreq, 
                          double initialFreq, double finalFreq,
                          int start, int end) {
  		// here it goes...
  		double freq=initialFreq;
  		double freqFactor=(finalFreq-initialFreq)/(double) (end-start+1);
	  	double step=-1;
	  	double sign=2;

  		for (int i=start; i<end; i++, step+=sign*freq/(double) waveFreq) {
  		  if (Math.abs(step)>1) {
  		    step=(double)(sign/2);
  		    sign=(double) -sign;
  		  }
  		  Wave[i]=(short) (step* 30000);
  		  freq=initialFreq+(double) i * freqFactor;
  		}
		}

    public void apply(WaveFile wf,Selection sel){
		  this.wf=wf;
		  this.sel=sel;
      updateScrollbars();
      updateSBLabels();
      updateEnabled();
      setVisible(true);
  	}
  	
  /**
  * constructor - creates the interface
  */	
  public CreateTones(Frame parent) {
    super (parent,true);
try {
    setTitle ("Create nice sounds...");
    initialFreq=555;
    finalFreq=1111;
    createFreq=22050;
    createLength=5;
    
    setLayout(new StripeLayout());
    
    add(new Label("Initial Frequency"));        
    Panel p = new Panel();
    p.setLayout (new BorderLayout());
    p.add("Center",SBInitialFreq=new Scrollbar(Scrollbar.HORIZONTAL,SBMin,SBVisible,SBMin,SBRange+SBVisible));
    p.add("East",LInitialFreq=new Label("48000 Hertz......"));
    add(p);
    add(new Label("Final Frequency"));        
    p = new Panel();
    p.setLayout (new BorderLayout());
    p.add("Center",SBFinalFreq=new Scrollbar(Scrollbar.HORIZONTAL,SBMin,SBVisible,SBMin,SBRange+SBVisible));
    p.add("East",LFinalFreq=new Label("48000 Hertz......"));
    add(p);
    add(new HoriLine());
    
    CheckboxGroup cbg=new CheckboxGroup();
    p=new Panel();
    p.setLayout(new GridLayout(1,2));
    p.add(CBSine=new Checkbox("Sine",cbg,true));
    p.add(CBSquare=new Checkbox("Square",cbg,false));
    add(p);
    p=new Panel();
    p.setLayout(new GridLayout(1,2));
    p.add(CBSaw=new Checkbox("Saw",cbg,false));
    p.add(CBTriangle=new Checkbox("Triangle",cbg,false));
    add(p);
    add(new HoriLine());

    add(CBCreate=new Checkbox("Create new Wave",false));
    add(LCLength=new Label("Length of the new Wave"));        
    p = new Panel();
    p.setLayout(new BorderLayout());
    p.add("Center",SBCreateLength=new Scrollbar(Scrollbar.HORIZONTAL,5,30,1,100+30));
    p.add("East",LCreateLength=new Label("5000 seconds...."));
    add(p);
    add(LCFreq=new Label("Playback rate of the new Wave"));        
    p = new Panel();
    p.setLayout(new BorderLayout());
    p.add("Center",SBCreateFreq=new Scrollbar(Scrollbar.HORIZONTAL,Freqs.length-2,1,0,Freqs.length));
    p.add("East",LCreateFreq=new Label("44100 Hz...."));
    add(p);
    add(new HoriLine());
        
    p=new Panel();
    p.setLayout(new GridLayout(1,4));
    p.add(new Panel());
    p.add(BOK=new Button("OK"));
    p.add(BCancel=new Button("Cancel"));
    add(p);
    p.add(new Panel());
    
    CBCreate.addItemListener(this);
    BOK.addActionListener(this);
    BCancel.addActionListener(this);
    SBInitialFreq.addAdjustmentListener(this);
    SBFinalFreq.addAdjustmentListener(this);
    SBCreateLength.addAdjustmentListener(this);
    SBCreateFreq.addAdjustmentListener(this);
      pack();
} catch (Throwable t) {
    System.out.println(t.getMessage());
    t.printStackTrace();
    throw new RuntimeException();
  }    
  }

  private void updateSBLabels() {
    updateLFreq(true);
    updateLFreq(false);
    updateLCreateFreq();
    updateLCreateLength();
  }
  private void updateScrollbars() {
    updateSBFreq(true);
    updateSBFreq(false);
    updateSBCreateFreq();
    updateSBCreateLength();
  }

  private void updateLCreateFreq() {
    LCreateFreq.setText(String.valueOf(createFreq)+" Hz");
  }
  private void updateSBCreateFreq() {
    for (int i=0; i<Freqs.length; i++)
      if (Freqs[i]==createFreq) {
        SBCreateFreq.setValue(i);
        break;
      }
  }
  private void setCreateFreq(int SBValue) {
    createFreq=Freqs[Math.max(0,Math.min(SBValue,Freqs.length-1))];
  }

  private void updateLCreateLength() {
    LCreateLength.setText(String.valueOf(createLength)+" sec");
  }
  private void updateSBCreateLength() {
    SBCreateLength.setValue(createLength);
  }
  private void setCreateLength(int SBValue) {
    createLength=Math.max(0,Math.min(SBValue,100));
  }

  private void updateLFreq(boolean initial) {
    if (initial)
      LInitialFreq.setText(String.valueOf(initialFreq)+" Hz");
    else
      LFinalFreq.setText(String.valueOf(finalFreq)+" Hz");
  }
  private void updateSBFreq(boolean initial) {
    double freq=(initial?initialFreq:finalFreq);
    int SBValue=(int) (freq*SBRange/MAXIMUM_FREQ);
    if (initial)
      SBInitialFreq.setValue(SBValue);
    else
      SBFinalFreq.setValue(SBValue);
  }
  private void setFreq(int SBValue, boolean initial) {
    double newFreq=Math.min(SBValue*(MAXIMUM_FREQ/SBRange),MAXIMUM_FREQ);
    if (initial)
      initialFreq=newFreq;
    else
      finalFreq=newFreq;
  }
  
  private void addFreq(int offset, boolean initial) {
    double newFreq=(int) (initial?initialFreq:finalFreq);
    newFreq=Math.max(Math.min(newFreq+offset,MAXIMUM_FREQ),1);
    if (initial)
      initialFreq=newFreq;
    else
      finalFreq=newFreq;
  }

  public void adjustmentValueChanged(AdjustmentEvent e) {
    if (e.getAdjustable()==SBInitialFreq || e.getAdjustable()==SBFinalFreq) {
      if (e.getAdjustmentType()==AdjustmentEvent.UNIT_DECREMENT) {
        addFreq(-1, e.getAdjustable()==SBInitialFreq);
        updateSBFreq(e.getAdjustable()==SBInitialFreq);
      }
      else
      if (e.getAdjustmentType()==AdjustmentEvent.UNIT_INCREMENT) {
        addFreq(1, e.getAdjustable()==SBInitialFreq);
        updateSBFreq(e.getAdjustable()==SBInitialFreq);
      }
      else
        setFreq(e.getValue(), e.getAdjustable()==SBInitialFreq);
      updateLFreq(e.getAdjustable()==SBInitialFreq);
    }
    else
    if (e.getAdjustable()==SBCreateLength) {
      setCreateLength(e.getValue());
      updateLCreateLength();
    }
    else
    if (e.getAdjustable()==SBCreateFreq) {
      setCreateFreq(e.getValue());
      updateLCreateFreq();
    }
  }
  
  private void updateEnabled() {
    if (wf.getSize()==0) {
      CBCreate.setState(true);
      CBCreate.setEnabled(false);
    }
     else
      CBCreate.setEnabled(true);
    boolean enabled=CBCreate.getState();
    SBCreateLength.setEnabled(enabled);
    SBCreateFreq.setEnabled(enabled);
    LCFreq.setEnabled(enabled);
    LCLength.setEnabled(enabled);
    LCreateFreq.setEnabled(enabled);
    LCreateLength.setEnabled(enabled);
  }

public void itemStateChanged(ItemEvent e) {
  updateEnabled();
}

  public void actionPerformed (ActionEvent e) {
    if (e.getSource()==BOK) {
 	    short WaveL[]=wf.getLeftWave();
      short WaveR[]=wf.getRightWave();
      int start=sel.start;
      int end=sel.end;
      int waveFreq=wf.getFrequency();
      if (sel.onePoint()) {
        start=0;
        end=WaveL.length-1;
      }
      if (CBCreate.getState()) {
       	start=0;
     	  end=(int) (createLength*createFreq);
     	  WaveL=new short[end+1];
       	WaveR=new short[end+1];
       	waveFreq=createFreq;
      }
      if (CBSine.getState())
        makeSine(WaveL,waveFreq,initialFreq,finalFreq, start, end);
      else
      if (CBSquare.getState())
        makeSquare(WaveL,waveFreq,initialFreq,finalFreq, start, end);
      else
      if (CBSaw.getState())
        makeSaw(WaveL,waveFreq,initialFreq,finalFreq, start, end);
      if (CBTriangle.getState())
        makeTriangle(WaveL,waveFreq,initialFreq,finalFreq, start, end);
      System.arraycopy(WaveL,start,WaveR,start,end-start+1);
      wf.setWaves(WaveL,WaveR);
     	if (CBCreate.getState()) {
       	wf.setName("Sine.wav");
    		wf.setChannelCount((short) 1);
	      wf.setFrequency(waveFreq);
    	 	wf.setResolution((short) 16);
    	}
    }
    setVisible(false);
  }
}


