/*
 * $Id: AbstractBinding.java,v 1.3 2004/11/12 15:19:41 kleopatra Exp $
 *
 * Copyright 2004 Sun Microsystems, Inc., 4150 Network Circle,
 * Santa Clara, California 95054, U.S.A. All rights reserved.
 */

package org.jdesktop.swing.binding;

import org.jdesktop.swing.data.ConversionException;
import org.jdesktop.swing.data.Converters;
import org.jdesktop.swing.data.Converter;
import org.jdesktop.swing.data.DataModel;
import org.jdesktop.swing.data.MetaData;
import org.jdesktop.swing.data.Validator;
import org.jdesktop.swing.data.ValueChangeEvent;
import org.jdesktop.swing.data.ValueChangeListener;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import java.util.ArrayList;
import java.util.List;

import javax.swing.InputVerifier;
import javax.swing.JComponent;

/**
 * Abstract base class which implements a default mechanism for binding
 * user-interface components to elements in a data model.
 *
 * @author Amy Fowler
 * @version 1.0
 */

public abstract class AbstractBinding implements Binding {

    protected DataModel dataModel;
    protected MetaData metaData;
    protected Object cachedValue;
    protected ArrayList errorList;
    protected boolean modified = false;
    protected int validState = UNVALIDATED;
    protected boolean pulling = false;
    protected boolean pushing = false;

    private PropertyChangeSupport pcs;
    private int validationPolicy;

    protected AbstractBinding(JComponent component,
                              DataModel dataModel, String fieldName,
                              int validationPolicy) {
        setComponent(component);
        this.dataModel = dataModel;
        this.pcs = new PropertyChangeSupport(this);
        setValidationPolicy(validationPolicy);
        if (dataModel != null) {
            metaData = dataModel.getMetaData(fieldName);
            dataModel.addValueChangeListener(new ValueChangeListener() {
                public void valueChanged(ValueChangeEvent e) {
                    if (e.getFieldName().equals(metaData.getName()) &&
                          !pushing) {
                        pull();
                    }
                }
            });
        }
    }

    public DataModel getDataModel() {
        return dataModel;
    }

    public String getFieldName() {
        return metaData.getName();
    }

    public void setValidationPolicy(int policy) {
        int oldValidationPolicy = this.validationPolicy;
        this.validationPolicy = policy;
        if (policy != oldValidationPolicy) {
            firePropertyChange("validationPolicy",
                                new Integer(oldValidationPolicy),
                                new Integer(policy));
        }
        getComponent().setInputVerifier(new InputVerifier() {
            public boolean verify(JComponent input) {
                if (validationPolicy != AUTO_VALIDATE_NONE) {
                    boolean isValid = isValid();
                    if (!isValid && validationPolicy == AUTO_VALIDATE_STRICT) {
                        return false;
                    }
                    return true;
                }
                return true;
            }
        });
    }

    public int getValidationPolicy() {
        return validationPolicy;
    }

    public boolean pull() {
        pulling = true;
        cachedValue = dataModel.getValue(metaData.getName());
        setComponentValue(cachedValue);
        setModified(false);
        setValidState(UNVALIDATED);
        pulling = false;
        return true;
    }

    public boolean isModified() {
        return modified;
    }

    protected void setModified(boolean modified) {
        // JW: commented to fix issue #78
//        if (pulling) {
//            return;
//        }
        boolean oldModified = this.modified;
        this.modified = modified;
        if (modified) {
            cachedValue = null;
            setValidState(UNVALIDATED);
        }
        if (oldModified != modified) {
            firePropertyChange("modified",
                                Boolean.valueOf(oldModified),
                                Boolean.valueOf(modified));
        }
    }

    public boolean isValid() {
        if (validState != UNVALIDATED) {
            return validState == VALID;
        }
        // need to validate
        clearValidationErrors();
        Object componentValue = getComponentValue();

        // step 1: ensure a required element has non-null value
        boolean ok = checkRequired(componentValue);

        // step 2: if necessary, convert value from component to data type
        //         appropriate for model
        Object convertedValue = null;
        if (ok) {
            try {
                convertedValue = convertToModelType(componentValue);
            } catch (Exception e) {
                ok = false;
                /**@todo aim: very nerdy message */
                addError("value must be type " + metaData.getElementClass().getName());
            }
        }

        // step 3: run any registered element-level validators
        if (ok) {
            ok = executeValidators(convertedValue);
        }

        if (ok) {
            cachedValue = convertedValue;
        }
        setValidState(ok? VALID : INVALID);

        return validState == VALID;

    }

    public int getValidState() {
        return validState;
    }

    private void setValidState(int validState) {
        int oldValidState = this.validState;
        this.validState = validState;
        if (oldValidState != validState &&
            validState == UNVALIDATED) {
            clearValidationErrors();
        }
        if (validState != oldValidState) {
            firePropertyChange("validState",
                               new Integer(oldValidState),
                               new Integer(validState));
        }
    }

    private boolean checkRequired(Object componentValue) {
        if (metaData.isRequired() &&
            (componentValue == null ||
             (componentValue instanceof String && ((String)componentValue).equals("")))) {
            addError("requires a value");
            return false;
        }
        return true;
    }

    protected Object convertToModelType(Object componentValue) throws ConversionException {
        Object convertedValue = null;
        // if the element is not required and the value is null, then it's
        // okay to skip conversion
        if (componentValue == null ||
            (componentValue instanceof String && componentValue.equals(""))) {
            return convertedValue;
        }
        Class elementClass = metaData.getElementClass();
        if (componentValue instanceof String) {
            String stringValue = (String) componentValue;
            Converter converter = metaData.getConverter();
            if (converter != null) {
                convertedValue = converter.decode(stringValue,
                                                      metaData.getDecodeFormat());
            } else if (metaData.getElementClass() == String.class) {
                convertedValue = componentValue;
            }
        }
        else {
            if (!elementClass.isAssignableFrom(componentValue.getClass())) {
                throw new ConversionException("cannot assign component value");
            } else {
                convertedValue = componentValue;
            }
        }
        return convertedValue;
    }

    protected String convertFromModelType(Object modelValue) {
        if (modelValue != null) {
            try {
                Converter converter = metaData.getConverter();
                return converter.encode(modelValue, metaData.getEncodeFormat());
            }
            catch (Exception e) {
                /**@todo aim: how to handle conversion failure? */
                return modelValue.toString();
            }
        }
        return "";
    }

    protected boolean executeValidators(Object value) {
        Validator validators[] = metaData.getValidators();
        boolean isValid = true;
        for (int i = 0; i < validators.length; i++) {
            String error[] = new String[1];
            boolean passed = validators[i].validate(value, null, error);
            if (!passed) {
                String errorMessage = error[0];
                if (errorMessage != null) {
                    addError(errorMessage);
                }
                isValid = false;
            }
        }
        return isValid;
    }

    public String[] getValidationErrors() {
        if (errorList != null) {
            return (String[])errorList.toArray(new String[1]);
        }
        return new String[0];
    }

    public void clearValidationErrors() {
        if (errorList != null) {
            errorList.clear();
        }
    }

    public abstract JComponent getComponent();
    protected abstract void setComponent(JComponent component);
    protected abstract Object getComponentValue();
    protected abstract void setComponentValue(Object value);

    protected void addError(String error) {
        if (errorList == null) {
            errorList = new ArrayList();
        }
        errorList.add(error);
    }

    public boolean push() {
        if (isValid()) {
            pushing = true;
            dataModel.setValue(metaData.getName(), cachedValue);
            setModified(false);
            pushing = false;
            return true;
        }
        return false;
    }

    /**
     * Adds the specified property change listener to this binding object.
     * @param pcl PropertyChangeListener object to receive events when binding
     *        properties change
     */
    public void addPropertyChangeListener(PropertyChangeListener pcl) {
        pcs.addPropertyChangeListener(pcl);
    }

    /**
     * Removes the specified property change listener from this binding object.
     * @param pcl PropertyChangeListener object to receive events when binding
     *        properties change
     */
    public void removePropertyChangeListener(PropertyChangeListener pcl) {
        pcs.removePropertyChangeListener(pcl);
    }

    /**
     *
     * @return array containing the PropertyChangeListener objects registered
     *         on this binding object
     */
    public PropertyChangeListener[] getPropertyChangeListeners() {
        return pcs.getPropertyChangeListeners();
    }

    protected void firePropertyChange(String propertyName,
                                      Object oldValue, Object newValue) {
        pcs.firePropertyChange(propertyName, oldValue, newValue);
    }

}
