// Circuit.java (c) 2005,2008 by Paul Falstad, www.falstad.com

import java.io.InputStream;
import java.awt.*;
import java.awt.image.*;
import java.applet.Applet;
import java.util.Vector;
import java.io.File;
import java.util.Random;
import java.util.Arrays;
import java.lang.Math;
import java.net.URL;
import java.awt.event.*;
import java.io.FilterInputStream;
import java.io.ByteArrayOutputStream;
import java.util.StringTokenizer;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;

class CircuitCanvas extends Canvas {
    CircuitFrame pg;
    CircuitCanvas(CircuitFrame p) {
	pg = p;
    }
    public Dimension getPreferredSize() {
	return new Dimension(300,400);
    }
    public void update(Graphics g) {
	pg.updateCircuit(g);
    }
    public void paint(Graphics g) {
	pg.updateCircuit(g);
    }
};

class CircuitLayout implements LayoutManager {
    public CircuitLayout() {}
    public void addLayoutComponent(String name, Component c) {}
    public void removeLayoutComponent(Component c) {}
    public Dimension preferredLayoutSize(Container target) {
	return new Dimension(500, 500);
    }
    public Dimension minimumLayoutSize(Container target) {
	return new Dimension(100,100);
    }
    public void layoutContainer(Container target) {
	Insets insets = target.insets();
	int targetw = target.size().width - insets.left - insets.right;
	int cw = targetw* 8/10;
	int targeth = target.size().height - (insets.top+insets.bottom);
	target.getComponent(0).move(insets.left, insets.top);
	target.getComponent(0).resize(cw, targeth);
	int barwidth = targetw - cw;
	cw += insets.left;
	int i;
	int h = insets.top;
	for (i = 1; i < target.getComponentCount(); i++) {
	    Component m = target.getComponent(i);
	    if (m.isVisible()) {
		Dimension d = m.getPreferredSize();
		if (m instanceof Scrollbar)
		    d.width = barwidth;
		if (m instanceof Choice && d.width > barwidth)
		    d.width = barwidth;
		if (m instanceof Label) {
		    h += d.height/5;
		    d.width = barwidth;
		}
		m.move(cw, h);
		m.resize(d.width, d.height);
		h += d.height;
	    }
	}
    }
};

public class Circuit extends Applet implements ComponentListener {
    CircuitFrame ogf;
    void destroyFrame() {
	if (ogf != null)
	    ogf.dispose();
	ogf = null;
	repaint();
    }
    boolean started = false;
    public void init() {
	addComponentListener(this);
    }
    
    void showFrame() {
	if (ogf == null) {
	    started = true;
	    ogf = new CircuitFrame(this);
	    ogf.init();
	    repaint();
	}
    }

    public void toggleSwitch(int x) { ogf.toggleSwitch(x); }
    
    public void paint(Graphics g) {
	String s = "Applet is open in a separate window.";
	if (!started)
	    s = "Applet is starting.";
	else if (ogf == null)
	    s = "Applet is finished.";
	else if (ogf.useFrame)
	    ogf.triggerShow();
	g.drawString(s, 10, 30);
    }
    
    public void componentHidden(ComponentEvent e){}
    public void componentMoved(ComponentEvent e){}
    public void componentShown(ComponentEvent e) { showFrame(); }
    public void componentResized(ComponentEvent e) {
	if (ogf != null)
	    ogf.componentResized(e);
    }
    
    public void destroy() {
	if (ogf != null)
	    ogf.dispose();
	ogf = null;
	repaint();
    }
};

class CircuitFrame extends Frame
  implements ComponentListener, ActionListener, AdjustmentListener,
             MouseMotionListener, MouseListener, ItemListener {
    
    Thread engine = null;

    Dimension winSize;
    Image dbimage;
    
    Random random;
    public static final int sourceRadius = 7;
    public static final double freqMult = 3.14159265*2*4;
    
    public String getAppletInfo() {
	return "Circuit by Paul Falstad";
    }

    Container main;
    Label titleLabel;
    Button resetButton;
    Button dumpMatrixButton;
    MenuItem exportItem, importItem, exitItem;
    Menu optionsMenu;
    Checkbox stoppedCheck;
    CheckboxMenuItem dotsCheckItem;
    CheckboxMenuItem voltsCheckItem;
    CheckboxMenuItem powerCheckItem;
    CheckboxMenuItem smallGridCheckItem;
    CheckboxMenuItem showValuesCheckItem;
    //CheckboxMenuItem conductanceCheckItem;
    CheckboxMenuItem euroResistorCheckItem;
    CheckboxMenuItem printableCheckItem;
    CheckboxMenuItem conventionCheckItem;
    Scrollbar speedBar;
    Scrollbar currentBar;
    Label powerLabel;
    Scrollbar powerBar;
    double currentMult, powerMult;
    PopupMenu elmMenu;
    MenuItem elmEditMenuItem;
    MenuItem elmDeleteMenuItem;
    MenuItem elmScopeMenuItem;
    PopupMenu scopeMenu;
    PopupMenu transScopeMenu;
    PopupMenu mainMenu;
    CheckboxMenuItem scopeVMenuItem;
    CheckboxMenuItem scopeIMenuItem;
    CheckboxMenuItem scopeMaxMenuItem;
    CheckboxMenuItem scopeFreqMenuItem;
    CheckboxMenuItem scopePowerMenuItem;
    CheckboxMenuItem scopeIbMenuItem;
    CheckboxMenuItem scopeIcMenuItem;
    CheckboxMenuItem scopeIeMenuItem;
    CheckboxMenuItem scopeVbeMenuItem;
    CheckboxMenuItem scopeVbcMenuItem;
    CheckboxMenuItem scopeVceMenuItem;
    CheckboxMenuItem scopeVIMenuItem;
    CheckboxMenuItem scopeXYMenuItem;
    CheckboxMenuItem scopeResistMenuItem;
    MenuItem scopeSelectYMenuItem;
    Class addingClass;
    int mouseMode = -1;
    String mouseModeStr = "hello";
    Font unitsFont;
    static final double pi = 3.14159265358979323846;
    static final int MODE_ADD_ELM = 0;
    static final int MODE_DRAG_ALL = 1;
    static final int MODE_DRAG_ROW = 2;
    static final int MODE_DRAG_COLUMN = 3;
    static final int MODE_DRAG_SELECTED = 4;
    static final int MODE_DRAG_POST = 5;
    static final int infoWidth = 120;
    static final int SCOPEVAL_POWER = 1;
    static final int SCOPEVAL_IB = 1;
    static final int SCOPEVAL_IC = 2;
    static final int SCOPEVAL_IE = 3;
    static final int SCOPEVAL_VBE = 4;
    static final int SCOPEVAL_VBC = 5;
    static final int SCOPEVAL_VCE = 6;
    static final int SCOPEVAL_R = 2;
    int dragX, dragY;
    int selectedSource;
    int gridSize, gridMask, gridRound;
    boolean dragging;
    boolean analyzeFlag;
    boolean dumpMatrix;
    boolean useBufferedImage;
    double t;
    int pause = 10;
    int colorScaleCount = 32;
    Color colorScale[];
    Color whiteColor, selectColor, lightGrayColor;
    int scopeSelected = -1;
    int menuScope = -1;
    int hintType = -1, hintItem1, hintItem2;
    String stopMessage;
    double timeStep;
    static final int HINT_LC = 1;
    static final int HINT_RC = 2;
    static final int HINT_3DB_C = 3;
    static final int HINT_TWINT = 4;
    static final int HINT_3DB_L = 5;
    Vector elmList;
    Vector setupList;
    CircuitElm dragElm, menuElm, mouseElm, stopElm;
    int mousePost = -1;
    CircuitElm plotXElm, plotYElm;
    int draggingPost;
    SwitchElm heldSwitchElm;
    double circuitMatrix[][], circuitRightSide[],
	origRightSide[], origMatrix[][];
    RowInfo circuitRowInfo[];
    int circuitPermute[];
    boolean circuitNonLinear;
    int voltageSourceCount;
    double voltageRange = 5;
    int circuitMatrixSize, circuitMatrixFullSize;
    boolean circuitNeedsMap;
    public boolean useFrame;
    int scopeCount;
    Scope scopes[];
    int scopeColCount[];
    EditDialog editDialog;
    ImportDialog impDialog;
    Class dumpTypes[];
    String muString = "u";
    String ohmString = "ohm";
    Rectangle circuitArea;
    Point ps1, ps2;

    int getrand(int x) {
	int q = random.nextInt();
	if (q < 0) q = -q;
	return q % x;
    }
    CircuitCanvas cv;
    Circuit applet;
    NumberFormat showFormat, shortFormat, noCommaFormat;

    CircuitFrame(Circuit a) {
	super("Circuit Simulator v1.3g");
	applet = a;
	useFrame = false;
    }

    public void init() {
	String startCircuit = null;
	String startLabel = null;
	String euroResistor = null;
	String useFrameStr = null;
	boolean printable = false;
	boolean convention = true;

	try {
	    String param = applet.getParameter("PAUSE");
	    if (param != null)
		pause = Integer.parseInt(param);
	    startCircuit = applet.getParameter("startCircuit");
	    startLabel   = applet.getParameter("startLabel");
	    euroResistor = applet.getParameter("euroResistors");
	    useFrameStr  = applet.getParameter("useFrame");
	    String x = applet.getParameter("whiteBackground");
	    if (x != null && x.equalsIgnoreCase("true"))
		printable = true;
	    x = applet.getParameter("conventionalCurrent");
	    if (x != null && x.equalsIgnoreCase("true"))
		convention = false;
	} catch (Exception e) { }
	
	if (startLabel == null)
	    startLabel = "LRC Circuit";
	if (startCircuit == null)
	    startCircuit = "lrc.txt";
	boolean euro = (euroResistor != null && euroResistor.equalsIgnoreCase("true"));
	useFrame = (useFrameStr == null || !useFrameStr.equalsIgnoreCase("false"));
	if (useFrame)
	    main = this;
	else
	    main = applet;
	
	unitsFont = new Font("SansSerif", 0, 10);
	String os = System.getProperty("os.name");
	String jv = System.getProperty("java.class.version");
	double jvf = new Double(jv).doubleValue();
	if (jvf >= 48) {
	    muString = "\u03bc";
	    ohmString = "\u03a9";
	    useBufferedImage = true;
	}
	
	dumpTypes = new Class[300];
	// these characters are reserved
	dumpTypes[(int)'o'] = Scope.class;
	dumpTypes[(int)'h'] = Scope.class;
	dumpTypes[(int)'$'] = Scope.class;

	main.setLayout(new CircuitLayout());
	cv = new CircuitCanvas(this);
	cv.addComponentListener(this);
	cv.addMouseMotionListener(this);
	cv.addMouseListener(this);
	main.add(cv);

	mainMenu = new PopupMenu();
	MenuBar mb = null;
	if (useFrame)
	    mb = new MenuBar();
	Menu m = new Menu("File");
	if (useFrame)
	    mb.add(m);
	else
	    mainMenu.add(m);
	m.add(importItem = getMenuItem("Import"));
	m.add(exportItem = getMenuItem("Export"));
	m.addSeparator();
	m.add(exitItem   = getMenuItem("Exit"));

	m = new Menu("Scope");
	if (useFrame)
	    mb.add(m);
	else
	    mainMenu.add(m);
	m.add(getMenuItem("Stack All", "stackAll"));
	m.add(getMenuItem("Unstack All", "unstackAll"));

	optionsMenu = m = new Menu("Options");
	if (useFrame)
	    mb.add(m);
	else
	    mainMenu.add(m);
	m.add(dotsCheckItem = getCheckItem("Show Current"));
	dotsCheckItem.setState(true);
	m.add(voltsCheckItem = getCheckItem("Show Voltage"));
	voltsCheckItem.setState(true);
	m.add(powerCheckItem = getCheckItem("Show Power"));
	m.add(showValuesCheckItem = getCheckItem("Show Values"));
	showValuesCheckItem.setState(true);
	//m.add(conductanceCheckItem = getCheckItem("Show Conductance"));
	m.add(smallGridCheckItem = getCheckItem("Small Grid"));
	m.add(euroResistorCheckItem = getCheckItem("European Resistors"));
	euroResistorCheckItem.setState(euro);
	m.add(printableCheckItem = getCheckItem("White Background"));
	printableCheckItem.setState(printable);
	m.add(conventionCheckItem = getCheckItem("Conventional Current Motion"));
	conventionCheckItem.setState(convention);
	
	Menu circuitsMenu = new Menu("Circuits");
	if (useFrame)
	    mb.add(circuitsMenu);
	else
	    mainMenu.add(circuitsMenu);
	
	mainMenu.add(getClassCheckItem("Add Wire", "WireElm"));
	mainMenu.add(getClassCheckItem("Add Resistor", "ResistorElm"));
	
	Menu passMenu = new Menu("Passive Components");
	mainMenu.add(passMenu);
	passMenu.add(getClassCheckItem("Add Capacitor", "CapacitorElm"));
	passMenu.add(getClassCheckItem("Add Inductor", "InductorElm"));
	passMenu.add(getClassCheckItem("Add Switch", "SwitchElm"));
	passMenu.add(getClassCheckItem("Add Push Switch", "PushSwitchElm"));
	passMenu.add(getClassCheckItem("Add DPST Switch", "Switch2Elm"));
	passMenu.add(getClassCheckItem("Add Transformer", "TransformerElm"));
	passMenu.add(getClassCheckItem("Add Tapped Transformer",
				       "TappedTransformerElm"));
	passMenu.add(getClassCheckItem("Add Transmission Line", "TransLineElm"));
	passMenu.add(getClassCheckItem("Add Memristor", "MemristorElm"));
	
	Menu inputMenu = new Menu("Inputs/Outputs");
	mainMenu.add(inputMenu);
	inputMenu.add(getClassCheckItem("Add Ground", "GroundElm"));
	inputMenu.add(getClassCheckItem("Add Voltage Source (2-terminal)", "DCVoltageElm"));
	inputMenu.add(getClassCheckItem("Add A/C Source (2-terminal)", "ACVoltageElm"));
	inputMenu.add(getClassCheckItem("Add Voltage Source (1-terminal)", "RailElm"));
	inputMenu.add(getClassCheckItem("Add A/C Source (1-terminal)", "ACRailElm"));
	inputMenu.add(getClassCheckItem("Add Square Wave (1-terminal)", "SquareRailElm"));
	inputMenu.add(getClassCheckItem("Add Analog Output", "OutputElm"));
	inputMenu.add(getClassCheckItem("Add Logic Input", "LogicInputElm"));
	inputMenu.add(getClassCheckItem("Add Logic Output", "LogicOutputElm"));
	inputMenu.add(getClassCheckItem("Add Clock", "ClockElm"));
	inputMenu.add(getClassCheckItem("Add A/C Sweep", "SweepElm"));
	inputMenu.add(getClassCheckItem("Add Var. Voltage", "VarRailElm"));
	inputMenu.add(getClassCheckItem("Add Antenna", "AntennaElm"));
	inputMenu.add(getClassCheckItem("Add Current Source", "CurrentElm"));
	inputMenu.add(getClassCheckItem("Add LED", "LEDElm"));
	
	Menu activeMenu = new Menu("Active Components");
	mainMenu.add(activeMenu);
	activeMenu.add(getClassCheckItem("Add Diode", "DiodeElm"));
	activeMenu.add(getClassCheckItem("Add Zener Diode", "ZenerElm"));
	activeMenu.add(getClassCheckItem("Add Transistor (bipolar, NPN)",
				    "NTransistorElm"));
	activeMenu.add(getClassCheckItem("Add Transistor (bipolar, PNP)",
				    "PTransistorElm"));
	activeMenu.add(getClassCheckItem("Add Op Amp (- on top)", "OpAmpElm"));
	activeMenu.add(getClassCheckItem("Add Op Amp (+ on top)",
				    "OpAmpSwapElm"));
	activeMenu.add(getClassCheckItem("Add MOSFET (n-channel)",
				    "NMosfetElm"));
	activeMenu.add(getClassCheckItem("Add MOSFET (p-channel)",
				    "PMosfetElm"));
	activeMenu.add(getClassCheckItem("Add JFET (n-channel)",
					 "NJfetElm"));
	activeMenu.add(getClassCheckItem("Add Analog Switch (SPST)", "AnalogSwitchElm"));
	activeMenu.add(getClassCheckItem("Add Analog Switch (SPDT)", "AnalogSwitch2Elm"));

	Menu gateMenu = new Menu("Logic Gates");
	mainMenu.add(gateMenu);
	gateMenu.add(getClassCheckItem("Add Inverter", "InverterElm"));
	gateMenu.add(getClassCheckItem("Add NAND Gate", "NandGateElm"));
	gateMenu.add(getClassCheckItem("Add NOR Gate", "NorGateElm"));
	gateMenu.add(getClassCheckItem("Add AND Gate", "AndGateElm"));
	gateMenu.add(getClassCheckItem("Add OR Gate", "OrGateElm"));
	gateMenu.add(getClassCheckItem("Add XOR Gate", "XorGateElm"));

	Menu chipMenu = new Menu("Chips");
	mainMenu.add(chipMenu);
	chipMenu.add(getClassCheckItem("Add D Flip-Flop", "DFlipFlopElm"));
	chipMenu.add(getClassCheckItem("Add JK Flip-Flop", "JKFlipFlopElm"));
	chipMenu.add(getClassCheckItem("Add 7 Segment LED", "SevenSegElm"));
	chipMenu.add(getClassCheckItem("Add VCO", "VCOElm"));
	chipMenu.add(getClassCheckItem("Add Phase Comparator", "PhaseCompElm"));
	chipMenu.add(getClassCheckItem("Add Counter", "CounterElm"));
	chipMenu.add(getClassCheckItem("Add Decade Counter", "DecadeElm"));
	chipMenu.add(getClassCheckItem("Add 555 Timer", "TimerElm"));
	chipMenu.add(getClassCheckItem("Add DAC", "DACElm"));
	chipMenu.add(getClassCheckItem("Add ADC", "ADCElm"));
	chipMenu.add(getClassCheckItem("Add Latch", "LatchElm"));
	
	Menu otherMenu = new Menu("Other");
	mainMenu.add(otherMenu);
	otherMenu.add(getClassCheckItem("Add Text", "TextElm"));
	otherMenu.add(getClassCheckItem("Add Scope Probe", "ProbeElm"));
	otherMenu.add(getCheckItem("Drag All", "DragAll"));
	otherMenu.add(getCheckItem("Drag Row", "DragRow"));
	otherMenu.add(getCheckItem("Drag Column", "DragColumn"));
	otherMenu.add(getCheckItem("Drag Selected", "DragSelected"));
	otherMenu.add(getCheckItem("Drag Post", "DragPost"));
	
	main.add(mainMenu);

	main.add(resetButton = new Button("Reset"));
	resetButton.addActionListener(this);
	dumpMatrixButton = new Button("Dump Matrix");
	dumpMatrixButton.addActionListener(this);
	stoppedCheck = new Checkbox("Stopped");
	stoppedCheck.addItemListener(this);
	main.add(stoppedCheck);
	
	main.add(new Label("Simulation Speed", Label.CENTER));

	// was max of 140
	main.add(speedBar = new Scrollbar(Scrollbar.HORIZONTAL, 3, 1, 0, 260));
	speedBar.addAdjustmentListener(this);

	main.add(new Label("Current Speed", Label.CENTER));
	currentBar = new Scrollbar(Scrollbar.HORIZONTAL,
				   50, 1, 1, 100);
	currentBar.addAdjustmentListener(this);
	main.add(currentBar);

	main.add(powerLabel = new Label("Power Brightness", Label.CENTER));
	main.add(powerBar = new Scrollbar(Scrollbar.HORIZONTAL,
				    50, 1, 1, 100));
	powerBar.addAdjustmentListener(this);
	powerBar.disable();
	powerLabel.disable();

	main.add(new Label("www.falstad.com"));

	if (useFrame)
	    main.add(new Label(""));
	Font f = new Font("SansSerif", 0, 10);
	Label l;
	l = new Label("Current Circuit:");
	l.setFont(f);
	titleLabel = new Label("Label");
	titleLabel.setFont(f);
	if (useFrame) {
	    main.add(l);
	    main.add(titleLabel);
	}

	setGrid();
	elmList = new Vector();
	setupList = new Vector();
	colorScale = new Color[colorScaleCount];
	int i;
	for (i = 0; i != colorScaleCount; i++) {
	    double v = i*2./colorScaleCount - 1;
	    if (v < 0) {
		int n1 = (int) (128*-v)+127;
		int n2 = (int) (127*(1+v));
		colorScale[i] = new Color(n1, n2, n2);
	    } else {
		int n1 = (int) (128*v)+127;
		int n2 = (int) (127*(1-v));
		colorScale[i] = new Color(n2, n1, n2);
	    }
	}

	scopes = new Scope[20];
	scopeColCount = new int[20];
	scopeCount = 0;
	
	random = new Random();
	ps1 = new Point();
	ps2 = new Point();
	cv.setBackground(Color.black);
	cv.setForeground(Color.lightGray);
	showFormat = DecimalFormat.getInstance();
	showFormat.setMaximumFractionDigits(2);
	shortFormat = DecimalFormat.getInstance();
	shortFormat.setMaximumFractionDigits(1);
	noCommaFormat = DecimalFormat.getInstance();
	noCommaFormat.setMaximumFractionDigits(10);
	noCommaFormat.setGroupingUsed(false);
	
	elmMenu = new PopupMenu();
	elmMenu.add(elmEditMenuItem = getMenuItem("Edit"));
	elmMenu.add(elmDeleteMenuItem = getMenuItem("Delete"));
	elmMenu.add(elmScopeMenuItem = getMenuItem("View in Scope"));
	main.add(elmMenu);
	
	scopeMenu = buildScopeMenu(false);
	transScopeMenu = buildScopeMenu(true);

	getSetupList(circuitsMenu, false);
	if (useFrame)
	    setMenuBar(mb);
	if (stopMessage == null)
	    readSetupFile(startCircuit, startLabel);

	if (useFrame) {
	    Dimension screen = getToolkit().getScreenSize();
	    resize(800, 640);
	    handleResize();
	    Dimension x = getSize();
	    setLocation((screen.width  - x.width)/2,
			(screen.height - x.height)/2);
	    show();
	} else {
	    if (!powerCheckItem.getState()) {
		main.remove(powerBar);
		main.remove(powerLabel);
		main.validate();
	    }
	    hide();
	    handleResize();
	    applet.validate();
	}
	main.requestFocus();
    }

    boolean shown = false;
    
    public void triggerShow() {
	if (!shown)
	    show();
	shown = true;
    }
    
    PopupMenu buildScopeMenu(boolean t) {
	PopupMenu m = new PopupMenu();
	m.add(getMenuItem("Remove", "remove"));
	m.add(getMenuItem("Speed 2x", "speed2"));
	m.add(getMenuItem("Speed 1/2x", "speed1/2"));
	m.add(getMenuItem("Scale 2x", "scale"));
	m.add(getMenuItem("Stack", "stack"));
	m.add(getMenuItem("Unstack", "unstack"));
	if (t) {
	    m.add(scopeIbMenuItem = getCheckItem("Show Ib"));
	    m.add(scopeIcMenuItem = getCheckItem("Show Ic"));
	    m.add(scopeIeMenuItem = getCheckItem("Show Ie"));
	    m.add(scopeVbeMenuItem = getCheckItem("Show Vbe"));
	    m.add(scopeVbcMenuItem = getCheckItem("Show Vbc"));
	    m.add(scopeVceMenuItem = getCheckItem("Show Vce"));
	} else {
	    m.add(scopeVMenuItem = getCheckItem("Show Voltage"));
	    m.add(scopeIMenuItem = getCheckItem("Show Current"));
	    m.add(scopePowerMenuItem = getCheckItem("Show Power Consumed"));
	    m.add(scopeMaxMenuItem = getCheckItem("Show Peak Value"));
	    m.add(scopeFreqMenuItem = getCheckItem("Show Frequency"));
	    m.add(scopeVIMenuItem = getCheckItem("Show V versus I"));
	    m.add(scopeXYMenuItem = getCheckItem("Plot X/Y"));
	    m.add(scopeSelectYMenuItem = getMenuItem("Select Y", "selecty"));
	    m.add(scopeResistMenuItem = getCheckItem("Show Resistance"));
	}
	main.add(m);
	return m;
    }
    
    MenuItem getMenuItem(String s) {
	MenuItem mi = new MenuItem(s);
	mi.addActionListener(this);
	return mi;
    }

    MenuItem getMenuItem(String s, String ac) {
	MenuItem mi = new MenuItem(s);
	mi.setActionCommand(ac);
	mi.addActionListener(this);
	return mi;
    }

    CheckboxMenuItem getCheckItem(String s) {
	CheckboxMenuItem mi = new CheckboxMenuItem(s);
	mi.addItemListener(this);
	mi.setActionCommand("");
	return mi;
    }

    CheckboxMenuItem getClassCheckItem(String s, String t) {
	try {
	    Class c = Class.forName("CircuitFrame$" + t);
	    CircuitElm elm = constructElement(c, 0, 0);
	    register(c, elm);
	    elm.delete();
	} catch (Exception ee) {
	    ee.printStackTrace();
	}
	return getCheckItem(s, t);
    }
    
    CheckboxMenuItem getCheckItem(String s, String t) {
	CheckboxMenuItem mi = new CheckboxMenuItem(s);
	mi.addItemListener(this);
	mi.setActionCommand(t);
	return mi;
    }

    void register(Class c, CircuitElm elm) {
	int t = elm.getDumpType();
	if (t == 0) {
	    System.out.println("no dump type: " + c);
	    return;
	}
	Class dclass = elm.getDumpClass();
	if (dumpTypes[t] == dclass)
	    return;
	if (dumpTypes[t] != null) {
	    System.out.println("dump type conflict: " + c + " " +
			       dumpTypes[t]);
	    return;
	}
	dumpTypes[t] = dclass;
    }
    
    void handleResize() {
        winSize = cv.getSize();
	if (winSize.width == 0)
	    return;
	dbimage = main.createImage(winSize.width, winSize.height);
	int h = winSize.height / 5;
	/*if (h < 128 && winSize.height > 300)
	  h = 128;*/
	circuitArea = new Rectangle(0, 0, winSize.width, winSize.height-h);
	int i;
	int minx = 1000, maxx = 0, miny = 1000, maxy = 0;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    // centered text causes problems when trying to center the circuit,
	    // so we special-case it here
	    if (!ce.isCenteredText()) {
		minx = min(ce.x, min(ce.x2, minx));
		maxx = max(ce.x, max(ce.x2, maxx));
	    }
	    miny = min(ce.y, min(ce.y2, miny));
	    maxy = max(ce.y, max(ce.y2, maxy));
	}
	// center circuit; we don't use snapGrid() because that rounds
	int dx = gridMask & ((circuitArea.width -(maxx-minx))/2-minx);
	int dy = gridMask & ((circuitArea.height-(maxy-miny))/2-miny);
	if (dx+minx < 0)
	    dx = gridMask & (-minx);
	if (dy+miny < 0)
	    dy = gridMask & (-miny);
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    ce.move(dx, dy);
	}
    }

    public boolean handleEvent(Event ev) {
        if (ev.id == Event.WINDOW_DESTROY) {
            applet.destroyFrame();
            return true;
        }
        return super.handleEvent(ev);
    }
    
    void centerString(Graphics g, String s, int y) {
	FontMetrics fm = g.getFontMetrics();
        g.drawString(s, (winSize.width-fm.stringWidth(s))/2, y);
    }

    public void paint(Graphics g) {
	cv.repaint();
    }

    static final int resct = 6;
    long lastTime = 0, lastFrameTime, lastIterTime, secTime = 0;
    int frames = 0;
    int steps = 0;
    int framerate = 0, steprate = 0;

    public void updateCircuit(Graphics realg) {
	CircuitElm realMouseElm;
	if (winSize == null || winSize.width == 0)
	    return;
	if (analyzeFlag) {
	    analyzeCircuit();
	    analyzeFlag = false;
	}
	if (editDialog != null)
	    mouseElm = editDialog.elm;
	realMouseElm = mouseElm;
	if (mouseElm == null)
	    mouseElm = stopElm;
	setupScopes();
        Graphics g = null;
	g = dbimage.getGraphics();
	selectColor = Color.cyan;
	if (printableCheckItem.getState()) {
	    whiteColor = Color.black;
	    lightGrayColor = Color.black;
	    g.setColor(Color.white);
	} else {
	    whiteColor = Color.white;
	    lightGrayColor = Color.lightGray;
	    g.setColor(Color.black);
	}
	g.fillRect(0, 0, winSize.width, winSize.height);
	if (!stoppedCheck.getState()) {
	    try {
		runCircuit();
	    } catch (Exception e) {
		e.printStackTrace();
		analyzeFlag = true;
		cv.repaint();
		return;
	    }
	}
	if (!stoppedCheck.getState()) {
	    long sysTime = System.currentTimeMillis();
	    if (lastTime != 0) {
		int inc = (int) (sysTime-lastTime);
		double c = currentBar.getValue();
		c = java.lang.Math.exp(c/3.5-14.2);
		currentMult = 1.7 * inc * c;
		if (!conventionCheckItem.getState())
		    currentMult = -currentMult;
	    }
	    if (sysTime-secTime >= 1000) {
		framerate = frames; steprate = steps;
		frames = 0; steps = 0;
		secTime = sysTime;
	    }
	    lastTime = sysTime;
	} else
	    lastTime = 0;
	powerMult = Math.exp(powerBar.getValue()/4.762-7);
	
	int i;
	Font oldfont = g.getFont();
	for (i = 0; i != elmList.size(); i++) {
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);
	    /*else if (conductanceCheckItem.getState())
	      g.setColor(Color.white);*/
	    getElm(i).draw(g);
	}
	if (mouseMode == MODE_DRAG_ROW || mouseMode == MODE_DRAG_COLUMN)
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		ce.drawPost(g, ce.x , ce.y );
		ce.drawPost(g, ce.x2, ce.y2);
	    }
	/*if (mouseElm != null) {
	    g.setFont(oldfont);
	    g.drawString("+", mouseElm.x+10, mouseElm.y);
	    }*/
	if (dragElm != null &&
	      (dragElm.x != dragElm.x2 || dragElm.y != dragElm.y2))
	    dragElm.draw(g);
	g.setFont(oldfont);
	int ct = scopeCount;
	if (stopMessage != null)
	    ct = 0;
	for (i = 0; i != ct; i++)
	    scopes[i].draw(g);
	g.setColor(whiteColor);
	int ybase = circuitArea.height;
	if (stopMessage != null) {
	    g.drawString(stopMessage, 10, ybase+15);
	} else {
	    String info[] = new String[10];
	    if (mouseElm != null) {
		if (mousePost == -1)
		    mouseElm.getInfo(info);
		else
		    info[0] = "V = " +
			getUnitText(mouseElm.getPostVoltage(mousePost), "V");
		/*
		for (i = 0; i != mouseElm.getPostCount(); i++)
		    info[0] += " " + mouseElm.nodes[i];
		if (mouseElm.getVoltageSourceCount() > 0)
		    info[0] += ";" + (mouseElm.getVoltageSource()+nodeList.size());
		*/
	    } else {
		showFormat.setMinimumFractionDigits(2);
		info[0] = "t = " + getUnitText(t, "s");
		showFormat.setMinimumFractionDigits(0);
	    }
	    if (hintType != -1) {
		for (i = 0; info[i] != null; i++)
		    ;
		String s = getHint();
		if (s == null)
		    hintType = -1;
		else
		    info[i] = s;
	    }
	    int x = 0;
	    if (ct != 0)
		x = scopes[ct-1].rightEdge() + 20;
	    x = max(x, winSize.width*2/3);
	    for (i = 0; info[i] != null; i++)
		g.drawString(info[i], x,
			     ybase+15*(i+1));
	}
	mouseElm = realMouseElm;
	frames++;
	/*g.setColor(Color.white);
	g.drawString("Framerate: " + framerate, 10, 10);
	g.drawString("Steprate: " + steprate,  10, 30);
	g.drawString("Steprate/iter: " + (steprate/getIterCount()),  10, 50);
	g.drawString("iterc: " + (getIterCount()),  10, 70);*/
	lastFrameTime = lastTime;

	realg.drawImage(dbimage, 0, 0, this);
	if (!stoppedCheck.getState() && circuitMatrix != null)
	    cv.repaint(pause);
    }

    void setupScopes() {
	int i;
	
	// check scopes to make sure the elements still exist, and remove
	// unused scopes/columns
	int pos = -1;
	for (i = 0; i < scopeCount; i++) {
	    if (locateElm(scopes[i].elm) < 0)
		scopes[i].setElm(null);
	    if (scopes[i].elm == null) {
		int j;
		for (j = i; j != scopeCount; j++)
		    scopes[j] = scopes[j+1];
		scopeCount--;
		i--;
		continue;
	    }
	    if (scopes[i].position > pos+1)
		scopes[i].position = pos+1;
	    pos = scopes[i].position;
	}
	while (scopeCount > 0 && scopes[scopeCount-1].elm == null)
	    scopeCount--;
	int h = winSize.height - circuitArea.height;
	pos = 0;
	for (i = 0; i != scopeCount; i++)
	    scopeColCount[i] = 0;
	for (i = 0; i != scopeCount; i++) {
	    pos = max(scopes[i].position, pos);
	    scopeColCount[scopes[i].position]++;
	}
	int colct = pos+1;
	int iw = infoWidth;
	if (colct <= 2)
	    iw = iw*3/2;
	int w = (winSize.width-iw) / colct;
	int marg = 10;
	if (w < marg*2)
	    w = marg*2;
	pos = -1;
	int colh = 0;
	int row = 0;
	int speed = 0;
	for (i = 0; i != scopeCount; i++) {
	    Scope s = scopes[i];
	    if (s.position > pos) {
		pos = s.position;
		colh = h / scopeColCount[pos];
		row = 0;
		speed = s.speed;
	    }
	    if (s.speed != speed) {
		s.speed = speed;
		s.resetGraph();
	    }
	    Rectangle r = new Rectangle(pos*w, winSize.height-h+colh*row,
					w-marg, colh);
	    row++;
	    if (!r.equals(s.rect))
		s.setRect(r);
	}
    }
    
    String getHint() {
	CircuitElm c1 = getElm(hintItem1);
	CircuitElm c2 = getElm(hintItem2);
	if (c1 == null || c2 == null)
	    return null;
	if (hintType == HINT_LC) {
	    if (!(c1 instanceof InductorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    InductorElm ie = (InductorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "res.f = " + getUnitText(1/(2*pi*Math.sqrt(ie.inductance*
						    ce.capacitance)), "Hz");
	}
	if (hintType == HINT_RC) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "RC = " + getUnitText(re.resistance*ce.capacitance,
					 "s");
	}
	if (hintType == HINT_3DB_C) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "f.3db = " +
		getUnitText(1/(2*pi*re.resistance*ce.capacitance), "Hz");
	}
	if (hintType == HINT_3DB_L) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof InductorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    InductorElm ie = (InductorElm) c2;
	    return "f.3db = " +
		getUnitText(re.resistance/(2*pi*ie.inductance), "Hz");
	}
	if (hintType == HINT_TWINT) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "fc = " +
		getUnitText(1/(2*pi*re.resistance*ce.capacitance), "Hz");
	}
	return null;
    }

    public void toggleSwitch(int n) {
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce instanceof SwitchElm) {
		n--;
		if (n == 0) {
		    ((SwitchElm) ce).toggle();
		    analyzeFlag = true;
		    cv.repaint();
		    return;
		}
	    }
	}
    }
    
    void needAnalyze() {
	analyzeFlag = true;
	cv.repaint();
    }
    
    class CircuitNode {
	int x, y;
	Vector links;
	boolean internal;
	CircuitNode() { links = new Vector(); }
    }
    class CircuitNodeLink {
	int num;
	CircuitElm elm;
    }
    // info about each row/column of the matrix for simplification purposes
    class RowInfo {
	static final int ROW_NORMAL = 0;  // ordinary value
	static final int ROW_CONST  = 1;  // value is constant
	static final int ROW_EQUAL  = 2;  // value is equal to another value
	int nodeEq, type, mapCol, mapRow;
	double value;
	boolean rsChanges; // row's right side changes
	boolean lsChanges; // row's left side changes
	boolean dropRow;   // row is not needed in matrix
	RowInfo() { type = ROW_NORMAL; }
    }

    Vector nodeList;
    CircuitElm voltageSources[];

    CircuitNode getCircuitNode(int n) {
	if (n >= nodeList.size())
	    return null;
	return (CircuitNode) nodeList.elementAt(n);
    }

    CircuitElm getElm(int n) {
	if (n >= elmList.size())
	    return null;
	return (CircuitElm) elmList.elementAt(n);
    }
    
    void analyzeCircuit() {
	if (elmList.isEmpty())
	    return;
	stopMessage = null;
	stopElm = null;
	int i, j;
	int vscount = 0;
	nodeList = new Vector();
	boolean gotGround = false;
	boolean gotRail = false;
	CircuitElm volt = null;

	//System.out.println("ac1");
	// look for voltage or ground element
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce instanceof GroundElm) {
		gotGround = true;
		break;
	    }
	    if (ce instanceof RailElm)
		gotRail = true;
	    if (volt == null && ce instanceof VoltageElm)
		volt = ce;
	}

	// if no ground, and no rails, then the voltage elm's first terminal
	// is ground
	if (!gotGround && volt != null && !gotRail) {
	    CircuitNode cn = new CircuitNode();
	    Point pt = volt.getPost(0);
	    cn.x = pt.x;
	    cn.y = pt.y;
	    nodeList.addElement(cn);
	} else {
	    // otherwise allocate extra node for ground
	    CircuitNode cn = new CircuitNode();
	    cn.x = cn.y = -1;
	    nodeList.addElement(cn);
	}
	//System.out.println("ac2");

	// allocate nodes and voltage sources
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    int inodes = ce.getInternalNodeCount();
	    int ivs = ce.getVoltageSourceCount();
	    int posts = ce.getPostCount();
	    
	    // allocate a node for each post and match posts to nodes
	    for (j = 0; j != posts; j++) {
		Point pt = ce.getPost(j);
		int k;
		for (k = 0; k != nodeList.size(); k++) {
		    CircuitNode cn = getCircuitNode(k);
		    if (pt.x == cn.x && pt.y == cn.y)
			break;
		}
		if (k == nodeList.size()) {
		    CircuitNode cn = new CircuitNode();
		    cn.x = pt.x;
		    cn.y = pt.y;
		    CircuitNodeLink cnl = new CircuitNodeLink();
		    cnl.num = j;
		    cnl.elm = ce;
		    cn.links.addElement(cnl);
		    ce.setNode(j, nodeList.size());
		    nodeList.addElement(cn);
		} else {
		    CircuitNodeLink cnl = new CircuitNodeLink();
		    cnl.num = j;
		    cnl.elm = ce;
		    getCircuitNode(k).links.addElement(cnl);
		    ce.setNode(j, k);
		    // if it's the ground node, make sure the node voltage is 0,
		    // cause it may not get set later
		    if (k == 0)
			ce.setNodeVoltage(j, 0);
		}
	    }
	    for (j = 0; j != inodes; j++) {
		CircuitNode cn = new CircuitNode();
		cn.x = cn.y = -1;
		cn.internal = true;
		CircuitNodeLink cnl = new CircuitNodeLink();
		cnl.num = j+posts;
		cnl.elm = ce;
		cn.links.addElement(cnl);
		ce.setNode(cnl.num, nodeList.size());
		nodeList.addElement(cn);
	    }
	    vscount += ivs;
	}
	voltageSources = new CircuitElm[vscount];
	vscount = 0;
	circuitNonLinear = false;
	//System.out.println("ac3");

	// determine if circuit is nonlinear
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.nonLinear())
		circuitNonLinear = true;
	    int ivs = ce.getVoltageSourceCount();
	    for (j = 0; j != ivs; j++) {
		voltageSources[vscount] = ce;
		ce.setVoltageSource(j, vscount++);
	    }
	}
	voltageSourceCount = vscount;

	int matrixSize = nodeList.size()-1 + vscount;
	circuitMatrix = new double[matrixSize][matrixSize];
	circuitRightSide = new double[matrixSize];
	origMatrix = new double[matrixSize][matrixSize];
	origRightSide = new double[matrixSize];
	circuitMatrixSize = circuitMatrixFullSize = matrixSize;
	circuitRowInfo = new RowInfo[matrixSize];
	circuitPermute = new int[matrixSize];
	int vs = 0;
	for (i = 0; i != matrixSize; i++)
	    circuitRowInfo[i] = new RowInfo();
	circuitNeedsMap = false;
	
	// stamp linear circuit elements
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    ce.stamp();
	}
	//System.out.println("ac4");

	// determine nodes that are unconnected
	boolean closure[] = new boolean[nodeList.size()];
	boolean tempclosure[] = new boolean[nodeList.size()];
	boolean changed = true;
	closure[0] = true;
	while (changed) {
	    changed = false;
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		// loop through all ce's nodes to see if they are connected
		// to other nodes not in closure
		for (j = 0; j < ce.getPostCount(); j++) {
		    if (!closure[ce.getNode(j)]) {
			if (ce.hasGroundConnection(j))
			    closure[ce.getNode(j)] = changed = true;
			continue;
		    }
		    int k;
		    for (k = 0; k != ce.getPostCount(); k++) {
			if (j == k)
			    continue;
			int kn = ce.getNode(k);
			if (ce.getConnection(j, k) && !closure[kn]) {
			    closure[kn] = true;
			    changed = true;
			}
		    }
		}
	    }
	    if (changed)
		continue;

	    // connect unconnected nodes
	    for (i = 0; i != nodeList.size(); i++)
		if (!closure[i] && !getCircuitNode(i).internal) {
		    System.out.println("node " + i + " unconnected");
		    stampResistor(0, i, 1e8);
		    closure[i] = true;
		    changed = true;
		    break;
		}
	}
	//System.out.println("ac5");

	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    // look for inductors with no current path
	    if (ce instanceof InductorElm) {
		FindPathInfo fpi = new FindPathInfo(FPI_INDUCT, ce,
						    ce.getNode(1));
		if (!fpi.findPath(ce.getNode(0))) {
		    System.out.println(ce + " no path");
		    ce.reset();
		}
	    }
	    // look for current sources with no current path
	    if (ce instanceof CurrentElm) {
		FindPathInfo fpi = new FindPathInfo(FPI_INDUCT, ce,
						    ce.getNode(1));
		if (!fpi.findPath(ce.getNode(0))) {
		    stop("No path for current source!", ce);
		    return;
		}
	    }
	    // look for voltage source loops
	    if (ce instanceof VoltageElm && ce.getPostCount() == 2) {
		FindPathInfo fpi = new FindPathInfo(FPI_VOLTAGE, ce,
						    ce.getNode(1));
		if (fpi.findPath(ce.getNode(0))) {
		    stop("Voltage source loop with no resistance!", ce);
		    return;
		}
	    }
	    // look for shorted caps, or caps w/ voltage but no R
	    if (ce instanceof CapacitorElm) {
		FindPathInfo fpi = new FindPathInfo(FPI_SHORT, ce,
						    ce.getNode(1));
		if (fpi.findPath(ce.getNode(0))) {
		    System.out.println(ce + " shorted");
		    ce.reset();
		} else {
		    fpi = new FindPathInfo(FPI_CAP_V, ce, ce.getNode(1));
		    if (fpi.findPath(ce.getNode(0))) {
			stop("Capacitor loop with no resistance!", ce);
			return;
		    }
		}
	    }
	}
	//System.out.println("ac6");

	// simplify the matrix; this speeds things up quite a bit
	for (i = 0; i != matrixSize; i++) {
	    int qm = -1, qp = -1;
	    double qv = 0;
	    RowInfo re = circuitRowInfo[i];
	    /*System.out.println("row " + i + " " +
	      re.lsChanges + " " + re.rsChanges);*/
	    if (re.lsChanges || re.dropRow || re.rsChanges)
		continue;
	    double rsadd = 0;

	    // look for rows that can be removed
	    for (j = 0; j != matrixSize; j++) {
		double q = circuitMatrix[i][j];
		if (circuitRowInfo[j].type == RowInfo.ROW_CONST) {
		    // keep a running total of const values that have been
		    // removed already
		    rsadd -= circuitRowInfo[j].value*q;
		    continue;
		}
		if (q == 0)
		    continue;
		if (qp == -1) {
		    qp = j;
		    qv = q;
		    continue;
		}
		if (qm == -1 && q == -qv) {
		    qm = j;
		    continue;
		}
		break;
	    }
	    //System.out.println("line " + i + " " + qp + " " + qm);
	    /*if (qp != -1 && circuitRowInfo[qp].lsChanges) {
		System.out.println("lschanges");
		continue;
	    }
	    if (qm != -1 && circuitRowInfo[qm].lsChanges) {
		System.out.println("lschanges");
		continue;
		}*/
	    if (j == matrixSize) {
		if (qp == -1) {
		    stop("Matrix error", null);
		    return;
		}
		RowInfo elt = circuitRowInfo[qp];
		if (qm == -1) {
		    // we found a row with only one nonzero entry; that value
		    // is a constant
		    int k;
		    for (k = 0; elt.type == RowInfo.ROW_EQUAL && k < 100; k++) {
			// follow the chain
			/*System.out.println("following equal chain from " +
			  i + " " + qp + " to " + elt.nodeEq);*/
			qp = elt.nodeEq;
			elt = circuitRowInfo[qp];
		    }
		    if (elt.type == RowInfo.ROW_EQUAL) {
			// break equal chains
			elt.type = RowInfo.ROW_NORMAL;
			continue;
		    }
		    if (elt.type != RowInfo.ROW_NORMAL) {
			System.out.println("type already " + elt.type + " for " + qp + "!");
			continue;
		    }
		    elt.type = RowInfo.ROW_CONST;
		    elt.value = (circuitRightSide[i]+rsadd)/qv;
		    circuitRowInfo[i].dropRow = true;
		    //System.out.println(qp + " * " + qv + " = const " + elt.value);
		    i = -1; // start over from scratch
		} else if (circuitRightSide[i]+rsadd == 0) {
		    // we found a row with only two nonzero entries, and one
		    // is the negative of the other; the values are equal
		    if (elt.type != RowInfo.ROW_NORMAL) {
			//System.out.println("swapping");
			int qq = qm;
			qm = qp; qp = qq;
			elt = circuitRowInfo[qp];
			if (elt.type != RowInfo.ROW_NORMAL) {
			    // we should follow the chain here, but this
			    // hardly ever happens so it's not worth worrying
			    // about
			    System.out.println("swap failed");
			    continue;
			}
		    }
		    elt.type = RowInfo.ROW_EQUAL;
		    elt.nodeEq = qm;
		    circuitRowInfo[i].dropRow = true;
		    //System.out.println(qp + " = " + qm);
		}
	    }
	}
	//System.out.println("ac7");

	// find size of new matrix
	int nn = 0;
	for (i = 0; i != matrixSize; i++) {
	    RowInfo elt = circuitRowInfo[i];
	    if (elt.type == RowInfo.ROW_NORMAL) {
		elt.mapCol = nn++;
		//System.out.println("col " + i + " maps to " + elt.mapCol);
		continue;
	    }
	    if (elt.type == RowInfo.ROW_EQUAL) {
		RowInfo e2 = null;
		// resolve chains of equality; 100 max steps to avoid loops
		for (j = 0; j != 100; j++) {
		    e2 = circuitRowInfo[elt.nodeEq];
		    if (e2.type != RowInfo.ROW_EQUAL)
			break;
		    if (i == e2.nodeEq)
			break;
		    elt.nodeEq = e2.nodeEq;
		}
	    }
	    if (elt.type == RowInfo.ROW_CONST)
		elt.mapCol = -1;
	}
	for (i = 0; i != matrixSize; i++) {
	    RowInfo elt = circuitRowInfo[i];
	    if (elt.type == RowInfo.ROW_EQUAL) {
		RowInfo e2 = circuitRowInfo[elt.nodeEq];
		if (e2.type == RowInfo.ROW_CONST) {
		    // if something is equal to a const, it's a const
		    elt.type = e2.type;
		    elt.value = e2.value;
		    elt.mapCol = -1;
		    //System.out.println(i + " = [late]const " + elt.value);
		} else {
		    elt.mapCol = e2.mapCol;
		    //System.out.println(i + " maps to: " + e2.mapCol);
		}
	    }
	}
	//System.out.println("ac8");

	/*
	System.out.println("matrixSize = " + matrixSize);
	for (j = 0; j != circuitMatrixSize; j++) {
	    for (i = 0; i != circuitMatrixSize; i++)
		System.out.print(circuitMatrix[j][i] + " ");
	    System.out.print("  " + circuitRightSide[j] + "\n");
	}
	System.out.print("\n");
	*/

	// make the new, simplified matrix
	int newsize = nn;
	double newmatx[][] = new double[newsize][newsize];
	double newrs  []   = new double[newsize];
	int ii = 0;
	for (i = 0; i != matrixSize; i++) {
	    RowInfo rri = circuitRowInfo[i];
	    if (rri.dropRow) {
		rri.mapRow = -1;
		continue;
	    }
	    newrs[ii] = circuitRightSide[i];
	    rri.mapRow = ii;
	    for (j = 0; j != matrixSize; j++) {
		RowInfo ri = circuitRowInfo[j];
		if (ri.type == RowInfo.ROW_CONST)
		    newrs[ii] -= ri.value*circuitMatrix[i][j];
		else
		    newmatx[ii][ri.mapCol] += circuitMatrix[i][j];
	    }
	    ii++;
	}

	circuitMatrix = newmatx;
	circuitRightSide = newrs;
	matrixSize = circuitMatrixSize = newsize;
	for (i = 0; i != matrixSize; i++)
	    origRightSide[i] = circuitRightSide[i];
	for (i = 0; i != matrixSize; i++)
	    for (j = 0; j != matrixSize; j++)
		origMatrix[i][j] = circuitMatrix[i][j];
	circuitNeedsMap = true;

	/*
	System.out.println("matrixSize = " + matrixSize + " " + circuitNonLinear);
	for (j = 0; j != circuitMatrixSize; j++) {
	    for (i = 0; i != circuitMatrixSize; i++)
		System.out.print(circuitMatrix[j][i] + " ");
	    System.out.print("  " + circuitRightSide[j] + "\n");
	}
	System.out.print("\n");
	*/

	// if a matrix is linear, we can do the lu_factor here instead of
	// needing to do it every frame
	if (!circuitNonLinear) {
	    if (!lu_factor(circuitMatrix, circuitMatrixSize, circuitPermute)) {
		stop("Singular matrix!", null);
		return;
	    }
	}
    }

    void stop(String s, CircuitElm ce) {
	stopMessage = s;
	circuitMatrix = null;
	stopElm = ce;
	stoppedCheck.setState(true);
	analyzeFlag = false;
	cv.repaint();
    }
    
    static final int FPI_INDUCT  = 1;
    static final int FPI_VOLTAGE = 2;
    static final int FPI_SHORT   = 3;
    static final int FPI_CAP_V   = 4;
    class FindPathInfo {
	boolean used[];
	int dest;
	CircuitElm firstElm;
	int type;
	FindPathInfo(int t, CircuitElm e, int d) {
	    dest = d;
	    type = t;
	    firstElm = e;
	    used = new boolean[nodeList.size()];
	}
	boolean findPath(int n1) {
	    if (n1 == dest)
		return true;
	    if (used[n1]) {
		//System.out.println("used " + n1);
		return false;
	    }
	    used[n1] = true;
	    int i;
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		if (ce == firstElm)
		    continue;
		if (type == FPI_INDUCT) {
		    if (ce instanceof CurrentElm)
			continue;
		}
		if (type == FPI_VOLTAGE) {
		    if (!(ce.isWire() || ce instanceof VoltageElm))
			continue;
		}
		if (type == FPI_SHORT && !ce.isWire())
		    continue;
		if (type == FPI_CAP_V) {
		    if (!(ce.isWire() || ce instanceof CapacitorElm ||
			  ce instanceof VoltageElm))
			continue;
		}
		if (n1 == 0) {
		    // look for posts which have a ground connection;
		    // our path can go through ground
		    int j;
		    for (j = 0; j != ce.getPostCount(); j++)
			if (ce.hasGroundConnection(j) &&
			      findPath(ce.getNode(j))) {
			    used[n1] = false;
			    return true;
			}
		}
		int j;
		for (j = 0; j != ce.getPostCount(); j++) {
		    //System.out.println(ce + " " + ce.getNode(j));
		    if (ce.getNode(j) == n1)
			break;
		}
		if (j == ce.getPostCount())
		    continue;
		if (ce.hasGroundConnection(j) && findPath(0)) {
		    //System.out.println(ce + " has ground");
		    used[n1] = false;
		    return true;
		}
		if (type == FPI_INDUCT && ce instanceof InductorElm) {
		    double c = ce.getCurrent();
		    if (j == 0)
			c = -c;
		    //System.out.println("matching " + c + " to " + firstElm.getCurrent());
		    //System.out.println(ce + " " + firstElm);
		    if (Math.abs(c-firstElm.getCurrent()) > 1e-10)
			continue;
		}
		int k;
		for (k = 0; k != ce.getPostCount(); k++) {
		    if (j == k)
			continue;
		    //System.out.println(ce + " " + ce.getNode(j) + "-" + ce.getNode(k));
		    if (ce.getConnection(j, k) && findPath(ce.getNode(k))) {
			//System.out.println("got findpath " + n1);
			used[n1] = false;
			return true;
		    }
		    //System.out.println("back on findpath " + n1);
		}
	    }
	    used[n1] = false;
	    //System.out.println(n1 + " failed");
	    return false;
	}
    }

    // stamp independent voltage source #vs, from n1 to n2, amount v
    void stampVoltageSource(int n1, int n2, int vs, double v) {
	int vn = nodeList.size()+vs;
	stampMatrix(vn, n1, -1);
	stampMatrix(vn, n2, 1);
	stampRightSide(vn, v);
	stampMatrix(n1, vn, 1);
	stampMatrix(n2, vn, -1);
    }

    // use this if the amount of voltage is going to be updated in doStep()
    void stampVoltageSource(int n1, int n2, int vs) {
	int vn = nodeList.size()+vs;
	stampMatrix(vn, n1, -1);
	stampMatrix(vn, n2, 1);
	stampRightSide(vn);
	stampMatrix(n1, vn, 1);
	stampMatrix(n2, vn, -1);
    }
    
    void updateVoltageSource(int n1, int n2, int vs, double v) {
	int vn = nodeList.size()+vs;
	stampRightSide(vn, v);
    }
    
    void stampResistor(int n1, int n2, double r) {
	double r0 = 1/r;
	if (Double.isNaN(r0) || Double.isInfinite(r0)) {
	    System.out.print("bad resistance " + r + " " + r0 + "\n");
	    int a = 0;
	    a /= a;
	}
	stampMatrix(n1, n1, r0);
	stampMatrix(n2, n2, r0);
	stampMatrix(n1, n2, -r0);
	stampMatrix(n2, n1, -r0);
    }

    void stampConductance(int n1, int n2, double r0) {
	stampMatrix(n1, n1, r0);
	stampMatrix(n2, n2, r0);
	stampMatrix(n1, n2, -r0);
	stampMatrix(n2, n1, -r0);
    }

    // current from cn1 to cn2 is equal to voltage from vn1 to 2, divided by g
    void stampVCCurrentSource(int cn1, int cn2, int vn1, int vn2, double g) {
	stampMatrix(cn1, vn1, g);
	stampMatrix(cn2, vn2, g);
	stampMatrix(cn1, vn2, -g);
	stampMatrix(cn2, vn1, -g);
    }

    void stampCurrentSource(int n1, int n2, double i) {
	stampRightSide(n1, -i);
	stampRightSide(n2, i);
    }

    // stamp value x in row i, column j, meaning that a voltage change
    // of dv in node j will increase the current into node i by x dv.
    // (Unless i or j is a voltage source node.)
    void stampMatrix(int i, int j, double x) {
	if (i > 0 && j > 0) {
	    if (circuitNeedsMap) {
		i = circuitRowInfo[i-1].mapRow;
		RowInfo ri = circuitRowInfo[j-1];
		if (ri.type == RowInfo.ROW_CONST) {
		    //System.out.println("Stamping constant " + i + " " + j + " " + x);
		    circuitRightSide[i] -= x*ri.value;
		    return;
		}
		j = ri.mapCol;
		//System.out.println("stamping " + i + " " + j + " " + x);
	    } else {
		i--;
		j--;
	    }
	    circuitMatrix[i][j] += x;
	}
    }

    // stamp value x on the right side of row i, representing an
    // independent current source flowing into node i
    void stampRightSide(int i, double x) {
	if (i > 0) {
	    if (circuitNeedsMap) {
		i = circuitRowInfo[i-1].mapRow;
		//System.out.println("stamping " + i + " " + x);
	    } else
		i--;
	    circuitRightSide[i] += x;
	}
    }

    // indicate that the value on the right side of row i changes in doStep()
    void stampRightSide(int i) {
	//System.out.println("rschanges true " + (i-1));
	if (i > 0)
	    circuitRowInfo[i-1].rsChanges = true;
    }
    
    // indicate that the values on the left side of row i change in doStep()
    void stampNonLinear(int i) {
	if (i > 0)
	    circuitRowInfo[i-1].lsChanges = true;
    }

    double getIterCount() {
	if (speedBar.getValue() == 0)
	    return 0;
	//return (Math.exp((speedBar.getValue()-1)/24.) + .5);
	return .1*Math.exp((speedBar.getValue()-61)/24.);
    }
    
    boolean converged;
    int subIterations;
    void runCircuit() {
	if (circuitMatrix == null || elmList.size() == 0) {
	    circuitMatrix = null;
	    return;
	}
	int iter;
	//int maxIter = getIterCount();
	boolean debugprint = dumpMatrix;
	dumpMatrix = false;
	long steprate = (long) (160*getIterCount());
	long tm = System.currentTimeMillis();
	long lit = lastIterTime;
	if (1000 >= steprate*(tm-lastIterTime))
	    return;
	for (iter = 1; ; iter++) {
	    int i, j, k, subiter;
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		ce.startIteration();
	    }
	    steps++;
	    final int subiterCount = 5000;
	    for (subiter = 0; subiter != subiterCount; subiter++) {
		converged = true;
		subIterations = subiter;
		for (i = 0; i != circuitMatrixSize; i++)
		    circuitRightSide[i] = origRightSide[i];
		if (circuitNonLinear) {
		    for (i = 0; i != circuitMatrixSize; i++)
			for (j = 0; j != circuitMatrixSize; j++)
			    circuitMatrix[i][j] = origMatrix[i][j];
		}
		for (i = 0; i != elmList.size(); i++) {
		    CircuitElm ce = getElm(i);
		    ce.doStep();
		}
		if (stopMessage != null)
		    return;
		boolean printit = debugprint;
		debugprint = false;
		for (j = 0; j != circuitMatrixSize; j++) {
		    for (i = 0; i != circuitMatrixSize; i++) {
			double x = circuitMatrix[i][j];
			if (Double.isNaN(x) || Double.isInfinite(x)) {
			    stop("nan/infinite matrix!", null);
			    return;
			}
		    }
		}
		if (printit) {
		    for (j = 0; j != circuitMatrixSize; j++) {
			for (i = 0; i != circuitMatrixSize; i++)
			    System.out.print(circuitMatrix[j][i] + ",");
			System.out.print("  " + circuitRightSide[j] + "\n");
		    }
		    System.out.print("\n");
		}
		if (circuitNonLinear) {
		    if (converged && subiter > 0)
			break;
		    if (!lu_factor(circuitMatrix, circuitMatrixSize,
				  circuitPermute)) {
			stop("Singular matrix!", null);
			return;
		    }
		}
		lu_solve(circuitMatrix, circuitMatrixSize, circuitPermute,
			 circuitRightSide);
		
		for (j = 0; j != circuitMatrixFullSize; j++) {
		    RowInfo ri = circuitRowInfo[j];
		    double res = 0;
		    if (ri.type == RowInfo.ROW_CONST)
			res = ri.value;
		    else
			res = circuitRightSide[ri.mapCol];
		    /*System.out.println(j + " " + res + " " +
		      ri.type + " " + ri.mapCol);*/
		    if (Double.isNaN(res)) {
			converged = false;
			//debugprint = true;
			break;
		    }
		    if (j < nodeList.size()-1) {
			CircuitNode cn = getCircuitNode(j+1);
			for (k = 0; k != cn.links.size(); k++) {
			    CircuitNodeLink cnl = (CircuitNodeLink)
				cn.links.elementAt(k);
			    cnl.elm.setNodeVoltage(cnl.num, res);
			}
		    } else {
			int ji = j-(nodeList.size()-1);
			//System.out.println("setting vsrc " + ji + " to " + res);
			voltageSources[ji].setCurrent(ji, res);
		    }
		}
		if (!circuitNonLinear)
		    break;
	    }
	    if (subiter > 5)
		System.out.print("converged after " + subiter + " iterations\n");
	    if (subiter == subiterCount) {
		stop("Convergence failed!", null);
		break;
	    }
	    t += timeStep;
	    for (i = 0; i != scopeCount; i++)
		scopes[i].timeStep();
	    tm = System.currentTimeMillis();
	    lit = tm;
	    if (iter*1000 >= steprate*(tm-lastIterTime) ||
		(tm-lastFrameTime > 500))
		break;
	}
	lastIterTime = lit;
	//System.out.println((System.currentTimeMillis()-lastFrameTime)/(double) iter);
    }

    int abs(int x) {
	return x < 0 ? -x : x;
    }

    int sign(int x) {
	return (x < 0) ? -1 : (x == 0) ? 0 : 1;
    }

    int min(int a, int b) { return (a < b) ? a : b; }
    int max(int a, int b) { return (a > b) ? a : b; }

    void editFuncPoint(int x, int y) {
	// XXX
	cv.repaint(pause);
    }

    public void componentHidden(ComponentEvent e){}
    public void componentMoved(ComponentEvent e){}
    public void componentShown(ComponentEvent e) {
	cv.repaint();
    }

    public void componentResized(ComponentEvent e) {
	handleResize();
	cv.repaint(100);
    }
    public void actionPerformed(ActionEvent e) {
	String ac = e.getActionCommand();
	if (e.getSource() == resetButton) {
	    int i;
	    
	    // on IE, drawImage() stops working inexplicably every once in
	    // a while.  Recreating it fixes the problem, so we do that here.
	    dbimage = main.createImage(winSize.width, winSize.height);
	    
	    for (i = 0; i != elmList.size(); i++)
		getElm(i).reset();
	    for (i = 0; i != scopeCount; i++)
		scopes[i].resetGraph();
	    analyzeFlag = true;
	    t = 0;
	    stoppedCheck.setState(false);
	    cv.repaint();
	}
	if (e.getSource() == dumpMatrixButton)
	    dumpMatrix = true;
	if (e.getSource() == exportItem)
	    doImport(false);
	if (e.getSource() == importItem)
	    doImport(true);
	if (e.getSource() == exitItem) {
	    applet.destroyFrame();
	    return;
	}
	if (ac.compareTo("stackAll") == 0)
	    stackAll();
	if (ac.compareTo("unstackAll") == 0)
	    unstackAll();
	if (e.getSource() == elmEditMenuItem)
	    doEdit();
	if (e.getSource() == elmDeleteMenuItem && menuElm != null)
	    deleteElm(menuElm);
	if (e.getSource() == elmScopeMenuItem && menuElm != null) {
	    int i;
	    for (i = 0; i != scopeCount; i++)
		if (scopes[i].elm == null)
		    break;
	    if (i == scopeCount) {
		if (scopeCount == scopes.length)
		    return;
		scopeCount++;
		scopes[i] = new Scope();
		scopes[i].position = i;
		handleResize();
	    }
	    scopes[i].setElm(menuElm);
	}
	if (menuScope != -1) {
	    if (ac.compareTo("remove") == 0)
		scopes[menuScope].setElm(null);
	    if (ac.compareTo("speed2") == 0)
		scopes[menuScope].speedUp();
	    if (ac.compareTo("speed1/2") == 0)
		scopes[menuScope].slowDown();
	    if (ac.compareTo("scale") == 0)
		scopes[menuScope].adjustScale(.5);
	    if (ac.compareTo("stack") == 0)
		stackScope(menuScope);
	    if (ac.compareTo("unstack") == 0)
		unstackScope(menuScope);
	    if (ac.compareTo("selecty") == 0)
		scopes[menuScope].selectY();
	    cv.repaint();
	}
	if (ac.indexOf("setup ") == 0)
	    readSetupFile(ac.substring(6),
			  ((MenuItem) e.getSource()).getLabel());
    }

    void stackScope(int s) {
	if (s == 0) {
	    if (scopeCount < 2)
		return;
	    s = 1;
	}
	if (scopes[s].position == scopes[s-1].position)
	    return;
	scopes[s].position = scopes[s-1].position;
	for (s++; s < scopeCount; s++)
	    scopes[s].position--;
    }
    
    void unstackScope(int s) {
	if (s == 0) {
	    if (scopeCount < 2)
		return;
	    s = 1;
	}
	if (scopes[s].position != scopes[s-1].position)
	    return;
	for (; s < scopeCount; s++)
	    scopes[s].position++;
    }

    void stackAll() {
	int i;
	for (i = 0; i != scopeCount; i++) {
	    scopes[i].position = 0;
	    scopes[i].showMax = false;
	}
    }

    void unstackAll() {
	int i;
	for (i = 0; i != scopeCount; i++) {
	    scopes[i].position = i;
	    scopes[i].showMax = true;
	}
    }
    
    void doEdit() {
	if (editDialog != null) {
	    requestFocus();
	    editDialog.setVisible(false);
	    editDialog = null;
	}
	editDialog = new EditDialog(menuElm, this);
	editDialog.show();
    }

    void doImport(boolean imp) {
	if (impDialog != null) {
	    requestFocus();
	    impDialog.setVisible(false);
	    impDialog = null;
	}
	String dump = "";
	if (!imp) {
	    int i;
	    int f = (dotsCheckItem.getState()) ? 1 : 0;
	    f |= (smallGridCheckItem.getState()) ? 2 : 0;
	    f |= (voltsCheckItem.getState()) ? 0 : 4;
	    f |= (powerCheckItem.getState()) ? 8 : 0;
	    f |= (showValuesCheckItem.getState()) ? 0 : 16;
	    dump = "$ " + f + " " +
		timeStep + " " + getIterCount() + " " +
		currentBar.getValue() + " " + voltageRange + " " +
		powerBar.getValue() + "\n";
	    for (i = 0; i != elmList.size(); i++)
		dump += getElm(i).dump() + "\n";
	    for (i = 0; i != scopeCount; i++) {
		String d = scopes[i].dump();
		if (d != null)
		    dump += d + "\n";
	    }
	    if (hintType != -1)
		dump += "h " + hintType + " " + hintItem1 + " " +
		    hintItem2 + "\n";
	}
	impDialog = new ImportDialog(this, dump);
	impDialog.show();
    }
    
    public void adjustmentValueChanged(AdjustmentEvent e) {
	System.out.print(((Scrollbar) e.getSource()).getValue() + "\n");
    }

    ByteArrayOutputStream readUrlData(URL url) throws java.io.IOException {
	Object o = url.getContent();
	FilterInputStream fis = (FilterInputStream) o;
	ByteArrayOutputStream ba = new ByteArrayOutputStream(fis.available());
	int blen = 1024;
	byte b[] = new byte[blen];
	while (true) {
	    int len = fis.read(b);
	    if (len <= 0)
		break;
	    ba.write(b, 0, len);
	}
	return ba;
    }
    
    void getSetupList(Menu menu, boolean retry) {
	System.out.println("getsetuplist " + retry);
	Menu stack[] = new Menu[6];
	int stackptr = 0;
	stack[stackptr++] = menu;
	try {
	    URL url = new URL(applet.getCodeBase() + "setuplist.txt");
	    ByteArrayOutputStream ba = readUrlData(url);
	    byte b[] = ba.toByteArray();
	    int len = ba.size();
	    int p;
	    if (len == 0 || b[0] != '#') {
		// got a redirect, try again
		getSetupList(menu, true);
		return;
	    }
	    for (p = 0; p < len; ) {
		int l;
		for (l = 0; l != len-p; l++)
		    if (b[l+p] == '\n') {
			l++;
			break;
		    }
		String line = new String(b, p, l-1);
		if (line.charAt(0) == '#')
		    ;
		else if (line.charAt(0) == '+') {
		    Menu n = new Menu(line.substring(1));
		    menu.add(n);
		    menu = stack[stackptr++] = n;
		} else if (line.charAt(0) == '-') {
		    menu = stack[--stackptr-1];
		} else {
		    int i = line.indexOf(' ');
		    if (i > 0)
			menu.add(getMenuItem(line.substring(i+1),
					     "setup " + line.substring(0, i)));
		}
		p += l;
	    }
	} catch (Exception e) {
	    e.printStackTrace();
	    stop("Can't read setuplist.txt!", null);
	}
    }

    void readSetup(String text) {
	readSetup(text.getBytes(), text.length());
	titleLabel.setText("untitled");
    }
    
    void readSetupFile(String str, String title) {
	t = 0;
	System.out.println(str);
	try {
	    URL url = new URL(applet.getCodeBase() + "circuits/" + str);
	    ByteArrayOutputStream ba = readUrlData(url);
	    readSetup(ba.toByteArray(), ba.size());
	} catch (Exception e) {
	    e.printStackTrace();
	    stop("Unable to read " + str + "!", null);
	}
	titleLabel.setText(title);
    }

    void readSetup(byte b[], int len) {
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    ce.delete();
	}
	elmList.removeAllElements();
	hintType = -1;
	timeStep = 5e-6;
	dotsCheckItem.setState(true);
	smallGridCheckItem.setState(false);
	powerCheckItem.setState(false);
	voltsCheckItem.setState(true);
	showValuesCheckItem.setState(true);
	setGrid();
	speedBar.setValue(117); // 57
	currentBar.setValue(50);
	powerBar.setValue(50);
	voltageRange = 5;
	cv.repaint();
	int p;
	scopeCount = 0;
	for (p = 0; p < len; ) {
	    int l;
	    int linelen = 0;
	    for (l = 0; l != len-p; l++)
		if (b[l+p] == '\n' || b[l+p] == '\r') {
		    linelen = l++;
		    if (l+p < b.length && b[l+p] == '\n')
			l++;
		    break;
		}
	    String line = new String(b, p, linelen);
	    StringTokenizer st = new StringTokenizer(line);
	    while (st.hasMoreTokens()) {
		String type = st.nextToken();
		int tint = type.charAt(0);
		try {
		    if (tint == 'o') {
			Scope sc = new Scope();
			sc.position = scopeCount;
			sc.undump(st);
			scopes[scopeCount++] = sc;
			break;
		    }
		    if (tint == 'h') {
			readHint(st);
			break;
		    }
		    if (tint == '$') {
			readOptions(st);
			break;
		    }
		    if (tint >= '0' && tint <= '9')
			tint = new Integer(type).intValue();
		    int x1 = new Integer(st.nextToken()).intValue();
		    int y1 = new Integer(st.nextToken()).intValue();
		    int x2 = new Integer(st.nextToken()).intValue();
		    int y2 = new Integer(st.nextToken()).intValue();
		    int f  = new Integer(st.nextToken()).intValue();
		    CircuitElm ce = null;
		    Class cls = dumpTypes[tint];
		    if (cls == null) {
			System.out.println("unrecognized dump type: " + type);
			break;
		    }
		    // find element class
		    Class carr[] = new Class[7];
		    carr[0] = getClass();
		    carr[1] = carr[2] = carr[3] = carr[4] = carr[5] =
			int.class;
		    carr[6] = StringTokenizer.class;
		    Constructor cstr = null;
		    cstr = cls.getConstructor(carr);
		
		    // invoke constructor with starting coordinates
		    Object oarr[] = new Object[7];
		    oarr[0] = this;
		    oarr[1] = new Integer(x1);
		    oarr[2] = new Integer(y1);
		    oarr[3] = new Integer(x2);
		    oarr[4] = new Integer(y2);
		    oarr[5] = new Integer(f );
		    oarr[6] = st;
		    ce = (CircuitElm) cstr.newInstance(oarr);
		    ce.setPoints();
		    elmList.addElement(ce);
		} catch (java.lang.reflect.InvocationTargetException ee) {
		    ee.getTargetException().printStackTrace();
		    break;
		} catch (Exception ee) {
		    ee.printStackTrace();
		    break;
		}
		break;
	    }
	    p += l;
	    
	}
	enableItems();
	handleResize(); // for scopes
	analyzeCircuit();
    }

    void readHint(StringTokenizer st) {
	hintType  = new Integer(st.nextToken()).intValue();
	hintItem1 = new Integer(st.nextToken()).intValue();
	hintItem2 = new Integer(st.nextToken()).intValue();
    }

    void readOptions(StringTokenizer st) {
	int flags = new Integer(st.nextToken()).intValue();
	dotsCheckItem.setState((flags & 1) != 0);
	smallGridCheckItem.setState((flags & 2) != 0);
	voltsCheckItem.setState((flags & 4) == 0);
	powerCheckItem.setState((flags & 8) == 8);
	showValuesCheckItem.setState((flags & 16) == 0);
	timeStep = new Double (st.nextToken()).doubleValue();
	double sp = new Double(st.nextToken()).doubleValue();
	int sp2 = (int) (Math.log(10*sp)*24+61.5);
	//int sp2 = (int) (Math.log(sp)*24+1.5);
	speedBar  .setValue(sp2);
	currentBar.setValue(new Integer(st.nextToken()).intValue());
	voltageRange = new Double (st.nextToken()).doubleValue();
	try {
	    powerBar.setValue(new Integer(st.nextToken()).intValue());
	} catch (Exception e) {
	}
	setGrid();
    }
    
    int snapGrid(int x) {
	return (x+gridRound) & gridMask;
    }

    boolean doSwitch(int x, int y) {
	if (mouseElm == null || !(mouseElm instanceof SwitchElm))
	    return false;
	SwitchElm se = (SwitchElm) mouseElm;
	se.toggle();
	if (se.momentary)
	    heldSwitchElm = se;
	analyzeCircuit();
	return true;
    }

    void deleteElm(CircuitElm elm) {
	int e = locateElm(elm);
	if (e >= 0) {
	    elm.delete();
	    elmList.removeElementAt(e);
	    analyzeCircuit();
	}
    }

    int locateElm(CircuitElm elm) {
	int i;
	for (i = 0; i != elmList.size(); i++)
	    if (elm == elmList.elementAt(i))
		return i;
	return -1;
    }
    
    public void mouseDragged(MouseEvent e) {
	if (!circuitArea.contains(e.getX(), e.getY()))
	    return;
	if (dragElm != null)
	    dragElm.drag(e.getX(), e.getY());
	switch (mouseMode) {
	case MODE_DRAG_ALL:
	    dragAll(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_ROW:
	    dragRow(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_COLUMN:
	    dragColumn(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_POST:
	    if (mouseElm != null)
		dragPost(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_SELECTED:
	    if (mouseElm != null)
		dragItem(e.getX(), e.getY());
	    break;
	}
	dragging = true;
	if (mouseMode == MODE_DRAG_SELECTED && mouseElm instanceof TextElm) {
	    dragX = e.getX(); dragY = e.getY();
	} else {
	    dragX = snapGrid(e.getX()); dragY = snapGrid(e.getY());
	}
	cv.repaint(pause);
    }

    void dragAll(int x, int y) {
	int dx = x-dragX;
	int dy = y-dragY;
	if (dx == 0 && dy == 0)
	    return;
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    ce.move(dx, dy);
	}
	removeZeroLengthElements();
    }

    void dragRow(int x, int y) {
	int dy = y-dragY;
	if (dy == 0)
	    return;
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.y  == dragY)
		ce.movePoint(0, 0, dy);
	    if (ce.y2 == dragY)
		ce.movePoint(1, 0, dy);
	}
	removeZeroLengthElements();
    }
    
    void dragColumn(int x, int y) {
	int dx = x-dragX;
	if (dx == 0)
	    return;
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.x  == dragX)
		ce.movePoint(0, dx, 0);
	    if (ce.x2 == dragX)
		ce.movePoint(1, dx, 0);
	}
	removeZeroLengthElements();
    }

    void dragItem(int x, int y) {
	if (!(mouseElm instanceof TextElm)) {
	    x = snapGrid(x);
	    y = snapGrid(y);
	}
	int dx = x-dragX;
	int dy = y-dragY;
	if (dx == 0 && dy == 0)
	    return;
	mouseElm.movePoint(0, dx, dy);
	mouseElm.movePoint(1, dx, dy);
	analyzeCircuit();
    }

    void dragPost(int x, int y) {
	if (draggingPost == -1) {
	    draggingPost =
		(distanceSq(mouseElm.x , mouseElm.y , x, y) >
		 distanceSq(mouseElm.x2, mouseElm.y2, x, y)) ? 1 : 0;
	}
	int dx = x-dragX;
	int dy = y-dragY;
	if (dx == 0 && dy == 0)
	    return;
	mouseElm.movePoint(draggingPost, dx, dy);
	analyzeCircuit();
    }

    void removeZeroLengthElements() {
	int i;
	boolean changed = false;
	for (i = elmList.size()-1; i >= 0; i--) {
	    CircuitElm ce = getElm(i);
	    if (ce.x == ce.x2 && ce.y == ce.y2) {
		elmList.removeElementAt(i);
		changed = true;
	    }
	}
	analyzeCircuit();
    }

    public void mouseMoved(MouseEvent e) {
	if ((e.getModifiers() & MouseEvent.BUTTON1_MASK) != 0)
	    return;
	int x = e.getX();
	int y = e.getY();
	dragX = snapGrid(x); dragY = snapGrid(y);
	draggingPost = -1;
	int i;
	CircuitElm origMouse = mouseElm;
	mouseElm = null;
	mousePost = -1;
	plotXElm = plotYElm = null;
	int bestDist = 100000;
	int bestArea = 100000;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.boundingBox.contains(x, y)) {
		int j;
		int area = ce.boundingBox.width * ce.boundingBox.height;
		int jn = ce.getPostCount();
		if (jn > 2)
		    jn = 2;
		for (j = 0; j != jn; j++) {
		    Point pt = ce.getPost(j);
		    int dist = distanceSq(x, y, pt.x, pt.y);

		    // if multiple elements have overlapping bounding boxes,
		    // we prefer selecting elements that have posts close
		    // to the mouse pointer and that have a small bounding
		    // box area.
		    if (dist <= bestDist && area <= bestArea) {
			bestDist = dist;
			bestArea = area;
			mouseElm = ce;
		    }
		}
		if (ce.getPostCount() == 0)
		    mouseElm = ce;
	    }
	}
	scopeSelected = -1;
	if (mouseElm == null) {
	    for (i = 0; i != scopeCount; i++) {
		Scope s = scopes[i];
		if (s.rect.contains(x, y)) {
		    s.select();
		    scopeSelected = i;
		}
	    }
	    // the mouse pointer was not in any of the bounding boxes, but we
	    // might still be close to a post
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		int j;
		int jn = ce.getPostCount();
		for (j = 0; j != jn; j++) {
		    Point pt = ce.getPost(j);
		    int dist = distanceSq(x, y, pt.x, pt.y);
		    if (distanceSq(pt.x, pt.y, x, y) < 26) {
			mouseElm = ce;
			mousePost = j;
			break;
		    }
		}
	    }
	} else {
	    mousePost = -1;
	    // look for post close to the mouse pointer
	    for (i = 0; i != mouseElm.getPostCount(); i++) {
		Point pt = mouseElm.getPost(i);
		if (distanceSq(pt.x, pt.y, x, y) < 26)
		    mousePost = i;
	    }
	}
	if (mouseElm != origMouse)
	    cv.repaint();
    }

    int distanceSq(int x1, int y1, int x2, int y2) {
	x2 -= x1;
	y2 -= y1;
	return x2*x2+y2*y2;
    }
    
    public void mouseClicked(MouseEvent e) {
    }
    public void mouseEntered(MouseEvent e) {
    }
    public void mouseExited(MouseEvent e) {
	scopeSelected = -1;
	mouseElm = plotXElm = plotYElm = null;
	cv.repaint();
    }
    public void mousePressed(MouseEvent e) {
	if (e.isPopupTrigger()) {
	    doPopupMenu(e);
	    return;
	}
	if ((e.getModifiers() & MouseEvent.BUTTON1_MASK) == 0)
	    return;
	if (doSwitch(e.getX(), e.getY()))
	    return;
	dragging = true;
	if (mouseMode != MODE_ADD_ELM || addingClass == null)
	    return;
	
	int x0 = snapGrid(e.getX());
	int y0 = snapGrid(e.getY());
	if (!circuitArea.contains(x0, y0))
	    return;

	dragElm = constructElement(addingClass, x0, y0);
    }

    CircuitElm constructElement(Class c, int x0, int y0) {
	// find element class
	Class carr[] = new Class[3];
	carr[0] = getClass();
	carr[1] = carr[2] = int.class;
	Constructor cstr = null;
	try {
	    cstr = c.getConstructor(carr);
	} catch (Exception ee) {
	    ee.printStackTrace();
	    return null;
	}

	// invoke constructor with starting coordinates
	Object oarr[] = new Object[3];
	oarr[0] = this;
	oarr[1] = new Integer(x0);
	oarr[2] = new Integer(y0);
	try {
	    return (CircuitElm) cstr.newInstance(oarr);
	} catch (Exception ee) { ee.printStackTrace(); }
	return null;
    }
    
    void doPopupMenu(MouseEvent e) {
	menuElm = mouseElm;
	menuScope = -1;
	if (scopeSelected != -1) {
	    PopupMenu m = scopes[scopeSelected].getMenu();
	    menuScope = scopeSelected;
	    if (m != null)
		m.show(e.getComponent(), e.getX(), e.getY());
	} else if (mouseElm != null) {
	    elmEditMenuItem .setEnabled(mouseElm.getEditInfo(0) != null);
	    elmScopeMenuItem.setEnabled(mouseElm.canViewInScope());
	    elmMenu.show(e.getComponent(), e.getX(), e.getY());
	} else {
	    doMainMenuChecks(mainMenu);
	    mainMenu.show(e.getComponent(), e.getX(), e.getY());
	}
    }

    void doMainMenuChecks(Menu m) {
	int i;
	if (m == optionsMenu)
	    return;
	for (i = 0; i != m.getItemCount(); i++) {
	    MenuItem mc = m.getItem(i);
	    if (mc instanceof Menu)
		doMainMenuChecks((Menu) mc);
	    if (mc instanceof CheckboxMenuItem) {
		CheckboxMenuItem cmi = (CheckboxMenuItem) mc;
		cmi.setState(
		      mouseModeStr.compareTo(cmi.getActionCommand()) == 0);
	    }
	}
    }
    
    public void mouseReleased(MouseEvent e) {
	if (e.isPopupTrigger()) {
	    doPopupMenu(e);
	    return;
	}
	if ((e.getModifiers() & MouseEvent.BUTTON1_MASK) == 0)
	    return;
	dragging = false;
	boolean circuitChanged = false;
	if (heldSwitchElm != null) {
	    heldSwitchElm.mouseUp();
	    heldSwitchElm = null;
	    circuitChanged = true;
	}
	if (dragElm != null &&
	    !(dragElm.x == dragElm.x2 && dragElm.y == dragElm.y2)) {
	    elmList.addElement(dragElm);
	    dragElm = null;
	    circuitChanged = true;
	}
	if (circuitChanged)
	    analyzeCircuit();
	if (dragElm != null)
	    dragElm.delete();
	dragElm = null;
	cv.repaint();
    }

    void enableItems() {
	if (powerCheckItem.getState()) {
	    powerBar.enable();
	    powerLabel.enable();
	} else {
	    powerBar.disable();
	    powerLabel.disable();
	}
    }
    
    public void itemStateChanged(ItemEvent e) {
	cv.repaint(pause);
	Object mi = e.getItemSelectable();
	if (mi == stoppedCheck)
	    return;
	if (mi == smallGridCheckItem)
	    setGrid();
	if (mi == powerCheckItem) {
	    if (powerCheckItem.getState())
		voltsCheckItem.setState(false);
	    else
		voltsCheckItem.setState(true);
	}
	if (mi == voltsCheckItem && voltsCheckItem.getState())
	    powerCheckItem.setState(false);
	enableItems();
	if (menuScope != -1) {
	    Scope sc = scopes[menuScope];
	    sc.handleMenu(e, mi);
	}
	if (mi instanceof CheckboxMenuItem) {
	    MenuItem mmi = (MenuItem) mi;
	    mouseMode = MODE_ADD_ELM;
	    String s = mmi.getActionCommand();
	    if (s.length() > 0)
		mouseModeStr = s;
	    if (s.compareTo("DragAll") == 0)
		mouseMode = MODE_DRAG_ALL;
	    else if (s.compareTo("DragRow") == 0)
		mouseMode = MODE_DRAG_ROW;
	    else if (s.compareTo("DragColumn") == 0)
		mouseMode = MODE_DRAG_COLUMN;
	    else if (s.compareTo("DragSelected") == 0)
		mouseMode = MODE_DRAG_SELECTED;
	    else if (s.compareTo("DragPost") == 0)
		mouseMode = MODE_DRAG_POST;
	    else if (s.length() > 0) {
		try {
		    addingClass = Class.forName("CircuitFrame$" + s);
		} catch (Exception ee) {
		    ee.printStackTrace();
		}
	    }
	}
    }

    void setGrid() {
	gridSize = (smallGridCheckItem.getState()) ? 8 : 16;
	gridMask = ~(gridSize-1);
	gridRound = gridSize/2-1;
    }
    
    String getVoltageDText(double v) {
	return getUnitText(Math.abs(v), "V");
    }
    String getVoltageText(double v) {
	return getUnitText(v, "V");
    }
    String getUnitText(double v, String u) {
	double va = Math.abs(v);
	if (va < 1e-12)
	    return "0 " + u;
	if (va < 1e-9)
	    return showFormat.format(v*1e12) + " p" + u;
	if (va < 1e-6)
	    return showFormat.format(v*1e9) + " n" + u;
	if (va < 1e-3)
	    return showFormat.format(v*1e6) + " " + muString + u;
	if (va < 1)
	    return showFormat.format(v*1e3) + " m" + u;
	if (va < 1e3)
	    return showFormat.format(v) + " " + u;
	if (va < 1e6)
	    return showFormat.format(v*1e-3) + " k" + u;
	if (va < 1e9)
	    return showFormat.format(v*1e-6) + " M" + u;
	return showFormat.format(v*1e-9) + " G" + u;
    }
    String getShortUnitText(double v, String u) {
	double va = Math.abs(v);
	if (va < 1e-12)
	    return null;
	if (va < 1e-9)
	    return shortFormat.format(v*1e12) + "p" + u;
	if (va < 1e-6)
	    return shortFormat.format(v*1e9) + "n" + u;
	if (va < 1e-3)
	    return shortFormat.format(v*1e6) + muString + u;
	if (va < 1)
	    return shortFormat.format(v*1e3) + "m" + u;
	if (va < 1e3)
	    return shortFormat.format(v) + u;
	if (va < 1e6)
	    return shortFormat.format(v*1e-3) + "k" + u;
	if (va < 1e9)
	    return shortFormat.format(v*1e-6) + "M" + u;
	return shortFormat.format(v*1e-9) + "G" + u;
    }
    String getCurrentText(double i) {
	return getUnitText(i, "A");
    }
    String getCurrentDText(double i) {
	return getUnitText(Math.abs(i), "A");
    }

    abstract class CircuitElm {
	int x, y, x2, y2, flags, nodes[], voltSource;
	int dx, dy, dsign;
	double dn, dpx1, dpy1;
	Point point1, point2, lead1, lead2;
	double volts[];
	double current, curcount;
	Rectangle boundingBox;
	boolean noDiagonal;
	int getDumpType() { return 0; }
	Class getDumpClass() { return getClass(); }
	int getDefaultFlags() { return 0; }
	CircuitElm(int xx, int yy) {
	    x = x2 = xx;
	    y = y2 = yy;
	    flags = getDefaultFlags();
	    allocNodes();
	    boundingBox = new Rectangle();
	}
	CircuitElm(int xa, int ya, int xb, int yb, int f) {
	    x = xa; y = ya; x2 = xb; y2 = yb; flags = f;
	    allocNodes();
	    boundingBox = new Rectangle();
	}
	void allocNodes() {
	    nodes = new int[getPostCount()+getInternalNodeCount()];
	    volts = new double[getPostCount()+getInternalNodeCount()];
	}
	String dump() {
	    int t = getDumpType();
	    return (t < 127 ? ((char)t)+" " : t+" ") + x + " " + y + " " +
		x2 + " " + y2 + " " + flags;
	}
	void reset() {
	    int i;
	    for (i = 0; i != getPostCount()+getInternalNodeCount(); i++)
		volts[i] = 0;
	    curcount = 0;
	}
	void draw(Graphics g) {}
	void setCurrent(int x, double c) { current = c; }
	double getCurrent() { return current; }
	void doStep() {}
	void delete() {}
	void startIteration() {}
	double getPostVoltage(int x) { return volts[x]; }
	void setNodeVoltage(int n, double c) {
	    volts[n] = c;
	    calculateCurrent();
	}
	void calculateCurrent() {}
	void setPoints() {
	    dx = x2-x; dy = y2-y;
	    dn = Math.sqrt(dx*dx+dy*dy);
	    dpx1 = dy/dn;
	    dpy1 = -dx/dn;
	    dsign = (dy == 0) ? sign(dx) : sign(dy);
	    point1 = new Point(x , y );
	    point2 = new Point(x2, y2);
	}
	void calcLeads(int len) {
	    if (dn < len || len == 0) {
		lead1 = point1;
		lead2 = point2;
		return;
	    }
	    lead1 = interpPoint(point1, point2, (dn-len)/(2*dn));
	    lead2 = interpPoint(point1, point2, (dn+len)/(2*dn));
	}
	Point interpPoint(Point a, Point b, double f) {
	    Point p = new Point();
	    interpPoint(a, b, p, f);
	    return p;
	}
	void interpPoint(Point a, Point b, Point c, double f) {
	    int xpd = b.x-a.x;
	    int ypd = b.y-a.y;
	    c.x = (int) (a.x*(1-f)+b.x*f+.48);
	    c.y = (int) (a.y*(1-f)+b.y*f+.48);
	}
	void interpPoint(Point a, Point b, Point c, double f, double g) {
	    int xpd = b.x-a.x;
	    int ypd = b.y-a.y;
	    int gx = b.y-a.y;
	    int gy = a.x-b.x;
	    g /= Math.sqrt(gx*gx+gy*gy);
	    c.x = (int) (a.x*(1-f)+b.x*f+g*gx+.48);
	    c.y = (int) (a.y*(1-f)+b.y*f+g*gy+.48);
	}
	Point interpPoint(Point a, Point b, double f, double g) {
	    Point p = new Point();
	    interpPoint(a, b, p, f, g);
	    return p;
	}
	void interpPoint2(Point a, Point b, Point c, Point d, double f, double g) {
	    int xpd = b.x-a.x;
	    int ypd = b.y-a.y;
	    int gx = b.y-a.y;
	    int gy = a.x-b.x;
	    g /= Math.sqrt(gx*gx+gy*gy);
	    c.x = (int) (a.x*(1-f)+b.x*f+g*gx+.48);
	    c.y = (int) (a.y*(1-f)+b.y*f+g*gy+.48);
	    d.x = (int) (a.x*(1-f)+b.x*f-g*gx+.48);
	    d.y = (int) (a.y*(1-f)+b.y*f-g*gy+.48);
	}
	void draw2Leads(Graphics g) {
	    // draw first lead
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);

	    // draw second lead
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, lead2, point2);
	}
	Point [] newPointArray(int n) {
	    Point a[] = new Point[n];
	    while (n > 0)
		a[--n] = new Point();
	    return a;
	}
	
	void drawDots(Graphics g, Point pa, Point pb, double pos) {
	    if (stoppedCheck.getState() || pos == 0 || !dotsCheckItem.getState())
		return;
	    int dx = pb.x-pa.x;
	    int dy = pb.y-pa.y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    g.setColor(Color.yellow);
	    int ds = 16;
	    pos %= ds;
	    if (pos < 0)
		pos += ds;
	    double di = 0;
	    for (di = pos; di < dn; di += ds) {
		int x0 = (int) (pa.x+di*dx/dn);
		int y0 = (int) (pa.y+di*dy/dn);
		g.fillRect(x0-1, y0-1, 4, 4);
	    }
	}

	Polygon calcArrow(Point a, Point b, double al, double aw) {
	    Polygon poly = new Polygon();
	    Point p1 = new Point();
	    Point p2 = new Point();
	    int adx = b.x-a.x;
	    int ady = b.y-a.y;
	    double l = Math.sqrt(adx*adx+ady*ady);
	    poly.addPoint(b.x, b.y);
	    interpPoint2(a, b, p1, p2, 1-al/l, aw);
	    poly.addPoint(p1.x, p1.y);
	    poly.addPoint(p2.x, p2.y);
	    return poly;
	}
	Polygon createPolygon(Point a, Point b, Point c) {
	    Polygon p = new Polygon();
	    p.addPoint(a.x, a.y);
	    p.addPoint(b.x, b.y);
	    p.addPoint(c.x, c.y);
	    return p;
	}
	Polygon createPolygon(Point a, Point b, Point c, Point d) {
	    Polygon p = new Polygon();
	    p.addPoint(a.x, a.y);
	    p.addPoint(b.x, b.y);
	    p.addPoint(c.x, c.y);
	    p.addPoint(d.x, d.y);
	    return p;
	}
	Polygon createPolygon(Point a[]) {
	    Polygon p = new Polygon();
	    int i;
	    for (i = 0; i != a.length; i++)
		p.addPoint(a[i].x, a[i].y);
	    return p;
	}
	void drag(int xx, int yy) {
	    xx = snapGrid(xx);
	    yy = snapGrid(yy);
	    if (noDiagonal) {
		if (abs(x-xx) < abs(y-yy))
		    xx = x;
		else
		    yy = y;
	    }
	    x2 = xx; y2 = yy;
	    setPoints();
	}
	void move(int dx, int dy) {
	    x += dx; y += dy; x2 += dx; y2 += dy;
	    setPoints();
	}
	void movePoint(int n, int dx, int dy) {
	    if (n == 0) {
		x += dx; y += dy;
	    } else {
		x2 += dx; y2 += dy;
	    }
	    setPoints();
	}
	void drawPosts(Graphics g) {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Point p = getPost(i);
		drawPost(g, p.x, p.y, nodes[i]);
	    }
	}
	void stamp() {}
	int getVoltageSourceCount() { return 0; }
	int getInternalNodeCount() { return 0; }
	void setNode(int p, int n) { nodes[p] = n; }
	void setVoltageSource(int n, int v) { voltSource = v; }
	int getVoltageSource() { return voltSource; }
	double getVoltageDiff() {
	    return volts[0] - volts[1];
	}
	boolean nonLinear() { return false; }
	int getPostCount() { return 2; }
	int getNode(int n) { return nodes[n]; }
	Point getPost(int n) {
	    return (n == 0) ? point1 : (n == 1) ? point2 : null;
	}
	void drawPost(Graphics g, int x0, int y0, int n) {
	    if (dragElm == null && mouseElm != this &&
		getCircuitNode(n).links.size() == 2)
		return;
	    if (mouseMode == MODE_DRAG_ROW || mouseMode == MODE_DRAG_COLUMN)
		return;
	    drawPost(g, x0, y0);
	}
	void drawPost(Graphics g, int x0, int y0) {
	    g.setColor(whiteColor);
	    g.fillOval(x0-3, y0-3, 7, 7);
	}
	void setBbox(int x1, int y1, int x2, int y2) {
	    if (x1 > x2) { int q = x1; x1 = x2; x2 = q; }
	    if (y1 > y2) { int q = y1; y1 = y2; y2 = q; }
	    boundingBox.setBounds(x1, y1, x2-x1+1, y2-y1+1);
	}
	void setBbox(Point p1, Point p2, double w) {
	    setBbox(p1.x, p1.y, p2.x, p2.y);
	    int gx = p2.y-p1.y;
	    int gy = p1.x-p2.x;
	    int dpx = (int) (dpx1*w);
	    int dpy = (int) (dpy1*w);
	    adjustBbox(p1.x+dpx, p1.y+dpy, p1.x-dpx, p1.y-dpy);
	}
	void adjustBbox(int x1, int y1, int x2, int y2) {
	    if (x1 > x2) { int q = x1; x1 = x2; x2 = q; }
	    if (y1 > y2) { int q = y1; y1 = y2; y2 = q; }
	    x1 = min(boundingBox.x, x1);
	    y1 = min(boundingBox.y, y1);
	    x2 = max(boundingBox.x+boundingBox.width-1,  x2);
	    y2 = max(boundingBox.y+boundingBox.height-1, y2);
	    boundingBox.setBounds(x1, y1, x2-x1, y2-y1);
	}
	boolean isCenteredText() { return false; }
	
	void drawCenteredText(Graphics g, String s, int x, int y, boolean cx) {
	    FontMetrics fm = g.getFontMetrics();
	    int w = fm.stringWidth(s);
	    if (cx)
		x -= w/2;
	    g.drawString(s, x, y+fm.getAscent()/2);
	    adjustBbox(x, y-fm.getAscent()/2,
		       x+w, y+fm.getAscent()/2+fm.getDescent());
	}
    
	void drawValues(Graphics g, String s, double hs) {
	    if (s == null)
		return;
	    g.setFont(unitsFont);
	    FontMetrics fm = g.getFontMetrics();
	    int w = fm.stringWidth(s);
	    g.setColor(whiteColor);
	    int ya = fm.getAscent()/2;
	    int xc, yc;
	    if (this instanceof RailElm || this instanceof SweepElm) {
		xc = x2;
		yc = y2;
	    } else {
		xc = (x2+x)/2;
		yc = (y2+y)/2;
	    }
	    int dpx = (int) (dpx1*hs);
	    int dpy = (int) (dpy1*hs);
	    if (dpx == 0) {
		g.drawString(s, xc-w/2, yc-abs(dpy)-2);
	    } else {
		int xx = xc+abs(dpx)+2;
		if (this instanceof VoltageElm || (x < x2 && y > y2))
		    xx = xc-(w+abs(dpx)+2);
		g.drawString(s, xx, yc+dpy+ya);
	    }
	}
	void drawCoil(Graphics g, int hs, Point p1, Point p2,
		      double v1, double v2) {
	    double len = Math.sqrt(distanceSq(p1.x, p1.y, p2.x, p2.y));
	    int segments = 30; // 10*(int) (len/10);
	    int i;
	    double segf = 1./segments;
	    
	    ps1.setLocation(p1);
	    for (i = 0; i != segments; i++) {
		double cx = (((i+1)*6.*segf) % 2)-1;
		double hsx = Math.sqrt(1-cx*cx);
		if (hsx < 0)
		    hsx = -hsx;
		interpPoint(p1, p2, ps2, i*segf, hsx*hs);
		double v = v1+(v2-v1)*i/segments;
		setVoltageColor(g, v);
		drawThickLine(g, ps1, ps2);
		ps1.setLocation(ps2);
	    }
	}
	void updateDotCount() {
	    curcount = updateDotCount(current, curcount);
	}
	double updateDotCount(double cur, double cc) {
	    if (stoppedCheck.getState())
		return cc;
	    double cadd = cur*currentMult;
	    /*if (cur != 0 && cadd <= .05 && cadd >= -.05)
	      cadd = (cadd < 0) ? -.05 : .05;*/
	    cadd %= 8;
	    /*if (cadd > 8)
		cadd = 8;
	    if (cadd < -8)
	    cadd = -8;*/
	    return cc + cadd;
	}
	void doDots(Graphics g) {
	    updateDotCount();
	    if (dragElm != this)
		drawDots(g, point1, point2, curcount);
	}
	void doAdjust() {}
	void setupAdjust() {}
	void getInfo(String arr[]) {
	}
	void getBasicInfo(String arr[]) {
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	    arr[2] = "Vd = " + getVoltageDText(getVoltageDiff());
	}
	void setVoltageColor(Graphics g, double volts) {
	    if (this == mouseElm) {
		g.setColor(selectColor);
		return;
	    }
	    if (!voltsCheckItem.getState()) {
		if (!powerCheckItem.getState()) // && !conductanceCheckItem.getState())
		    g.setColor(whiteColor);
		return;
	    }
	    int c = (int) ((volts+voltageRange)*(colorScaleCount-1)/
			   (voltageRange*2));
	    if (c < 0)
		c = 0;
	    if (c >= colorScaleCount)
		c = colorScaleCount-1;
	    g.setColor(colorScale[c]);
	}
	void setPowerColor(Graphics g, boolean yellow) {
	    /*if (conductanceCheckItem.getState()) {
		setConductanceColor(g, current/getVoltageDiff());
		return;
		}*/
	    if (!powerCheckItem.getState())
		return;
	    setPowerColor(g, getPower());
	}
	void setPowerColor(Graphics g, double w0) {
	    w0 *= powerMult;
	    //System.out.println(w);
	    double w = (w0 < 0) ? -w0 : w0;
	    if (w > 1)
		w = 1;
	    int rg = 128+(int) (w*127);
	    int b  = (int) (128*(1-w));
	    /*if (yellow)
		g.setColor(new Color(rg, rg, b));
		else */
	    if (w0 > 0)
		g.setColor(new Color(rg, b, b));
	    else
		g.setColor(new Color(b, rg, b));
	}
	void setConductanceColor(Graphics g, double w0) {
	    w0 *= powerMult;
	    //System.out.println(w);
	    double w = (w0 < 0) ? -w0 : w0;
	    if (w > 1)
		w = 1;
	    int rg = (int) (w*255);
	    g.setColor(new Color(rg, rg, rg));
	}
	double getPower() { return getVoltageDiff()*current; }
	double getScopeValue(int x) {
	    return (x == 1) ? getPower() : getVoltageDiff();
	}
	String getScopeUnits(int x) {
	    return (x == 1) ? "W" : "V";
	}
	EditInfo getEditInfo(int n) { return null; }
	void setEditValue(int n, EditInfo ei) {}
	boolean getConnection(int n1, int n2) { return true; }
	boolean hasGroundConnection(int n1) { return false; }
	boolean isWire() { return false; }
	boolean canViewInScope() { return getPostCount() <= 2; }
	boolean comparePair(int x1, int x2, int y1, int y2) {
	    return ((x1 == y1 && x2 == y2) || (x1 == y2 && x2 == y1));
	}
    }
    
    class WireElm extends CircuitElm {
	public WireElm(int xx, int yy) { super(xx, yy); }
	public WireElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	static final int FLAG_SHOWCURRENT = 1;
	void draw(Graphics g) {
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, point2);
	    doDots(g);
	    setBbox(point1, point2, 3);
	    if (mustShowCurrent()) {
	        String s = getShortUnitText(Math.abs(getCurrent()), "A");
	        drawValues(g, s, 4);
	    }
	    drawPosts(g);
	}
	void stamp() {
	    stampVoltageSource(nodes[0], nodes[1], voltSource, 0);
	}
	boolean mustShowCurrent() {
	    return (flags & FLAG_SHOWCURRENT) != 0;
	}
	int getVoltageSourceCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "wire";
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	    arr[2] = "V = " + getVoltageText(volts[0]);
	}
	int getDumpType() { return 'w'; }
	double getPower() { return 0; }
	double getVoltageDiff() { return volts[0]; }
	boolean isWire() { return true; }
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Show Current", mustShowCurrent());
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		flags = (ei.checkbox.getState()) ? FLAG_SHOWCURRENT : 0;
	}
    }

    void drawThickLine(Graphics g, int x, int y, int x2, int y2) {
	g.drawLine(x, y, x2, y2);
	g.drawLine(x+1, y, x2+1, y2);
	g.drawLine(x, y+1, x2, y2+1);
	g.drawLine(x+1, y+1, x2+1, y2+1);
    }

    void drawThickLine(Graphics g, Point pa, Point pb) {
	g.drawLine(pa.x, pa.y, pb.x, pb.y);
	g.drawLine(pa.x+1, pa.y, pb.x+1, pb.y);
	g.drawLine(pa.x, pa.y+1, pb.x, pb.y+1);
	g.drawLine(pa.x+1, pa.y+1, pb.x+1, pb.y+1);
    }

    void drawThickPolygon(Graphics g, int xs[], int ys[], int c) {
	int i;
	for (i = 0; i != c-1; i++)
	    drawThickLine(g, xs[i], ys[i], xs[i+1], ys[i+1]);
	drawThickLine(g, xs[i], ys[i], xs[0], ys[0]);
    }
    
    void drawThickPolygon(Graphics g, Polygon p) {
	drawThickPolygon(g, p.xpoints, p.ypoints, p.npoints);
    }
    
    void drawThickCircle(Graphics g, int cx, int cy, int ri) {
	int a;
	double m = pi/180;
	double r = ri*.98;
	for (a = 0; a != 360; a += 20) {
	    double ax = Math.cos(a*m)*r + cx;
	    double ay = Math.sin(a*m)*r + cy;
	    double bx = Math.cos((a+20)*m)*r + cx;
	    double by = Math.sin((a+20)*m)*r + cy;
	    drawThickLine(g, (int) ax, (int) ay, (int) bx, (int) by);
	}
    }
    
    class ResistorElm extends CircuitElm {
	double resistance;
	public ResistorElm(int xx, int yy) { super(xx, yy); resistance = 100; }
	public ResistorElm(int xa, int ya, int xb, int yb, int f,
		    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    resistance = new Double(st.nextToken()).doubleValue();
	}
	int getDumpType() { return 'r'; }
	String dump() {
	    return super.dump() + " " + resistance;
	}

	Point ps3, ps4;
	void setPoints() {
	    super.setPoints();
	    calcLeads(32);
	    ps3 = new Point();
	    ps4 = new Point();
	}
	
	void draw(Graphics g) {
	    int segments = 16;
	    int i;
	    int ox = 0;
	    int hs = euroResistorCheckItem.getState() ? 6 : 8;
	    double v1 = volts[0];
	    double v2 = volts[1];
	    setBbox(point1, point2, hs);
	    draw2Leads(g);
	    setPowerColor(g, true);
	    double segf = 1./segments;
	    if (!euroResistorCheckItem.getState()) {
		// draw zigzag
		for (i = 0; i != segments; i++) {
		    int nx = 0;
		    switch (i & 3) {
		    case 0: nx = 1; break;
		    case 2: nx = -1; break;
		    default: nx = 0; break;
		    }
		    double v = v1+(v2-v1)*i/segments;
		    setVoltageColor(g, v);
		    interpPoint(lead1, lead2, ps1, i*segf, hs*ox);
		    interpPoint(lead1, lead2, ps2, (i+1)*segf, hs*nx);
		    drawThickLine(g, ps1, ps2);
		    ox = nx;
		}
	    } else {
		// draw rectangle
		setVoltageColor(g, v1);
		interpPoint2(lead1, lead2, ps1, ps2, 0, hs);
		drawThickLine(g, ps1, ps2);
		for (i = 0; i != segments; i++) {
		    double v = v1+(v2-v1)*i/segments;
		    setVoltageColor(g, v);
		    interpPoint2(lead1, lead2, ps1, ps2, i*segf, hs);
		    interpPoint2(lead1, lead2, ps3, ps4, (i+1)*segf, hs);
		    drawThickLine(g, ps1, ps3);
		    drawThickLine(g, ps2, ps4);
		}
		interpPoint2(lead1, lead2, ps1, ps2, 1, hs);
		drawThickLine(g, ps1, ps2);
	    }
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(resistance, "");
		drawValues(g, s, hs);
	    }
	    doDots(g);
	    drawPosts(g);
	}
    
	void calculateCurrent() {
	    current = (volts[0]-volts[1])/resistance;
	    //System.out.print(this + " res current set to " + current + "\n");
	}
	void stamp() {
	    stampResistor(nodes[0], nodes[1], resistance);
	}
	void getInfo(String arr[]) {
	    arr[0] = "resistor";
	    getBasicInfo(arr);
	    arr[3] = "R = " + getUnitText(resistance, ohmString);
	    arr[4] = "P = " + getUnitText(getPower(), "W");
	}
	EditInfo getEditInfo(int n) {
	    // ohmString doesn't work here on linux
	    if (n == 0)
		return new EditInfo("Resistance (ohms)", resistance, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    resistance = ei.value;
	}
    }

    class CapacitorElm extends CircuitElm {
	double capacitance;
	double compResistance, voltdiff;
	Point plate1[], plate2[];
	public CapacitorElm(int xx, int yy) {
	    super(xx, yy);
	    capacitance = 1e-5;
	}
	public CapacitorElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    capacitance = new Double(st.nextToken()).doubleValue();
	    voltdiff = new Double(st.nextToken()).doubleValue();
	}
	void setNodeVoltage(int n, double c) {
	    super.setNodeVoltage(n, c);
	    voltdiff = volts[0]-volts[1];
	}
	void reset() {
	    current = curcount = 0;
	    // put small charge on caps when reset to start oscillators
	    voltdiff = 1e-3;
	}
	int getDumpType() { return 'c'; }
	String dump() {
	    return super.dump() + " " + capacitance + " " + voltdiff;
	}
	void setPoints() {
	    super.setPoints();
	    double f = (dn/2-4)/dn;
	    // calc leads
	    lead1 = interpPoint(point1, point2, f);
	    lead2 = interpPoint(point1, point2, 1-f);
	    // calc plates
	    plate1 = newPointArray(2);
	    plate2 = newPointArray(2);
	    interpPoint2(point1, point2, plate1[0], plate1[1], f, 12);
	    interpPoint2(point1, point2, plate2[0], plate2[1], 1-f, 12);
	}
	
	void draw(Graphics g) {
	    int hs = 12;
	    setBbox(point1, point2, hs);
	    
	    // draw first lead and plate
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);
	    setPowerColor(g, false);
	    drawThickLine(g, plate1[0], plate1[1]);
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);

	    // draw second lead and plate
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, point2, lead2);
	    setPowerColor(g, false);
	    drawThickLine(g, plate2[0], plate2[1]);
	    
	    updateDotCount();
	    if (dragElm != this) {
		drawDots(g, point1, lead1, curcount);
		drawDots(g, point2, lead2, -curcount);
	    }
	    drawPosts(g);
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(capacitance, "F");
		drawValues(g, s, hs);
	    }
	}
	void stamp() {
	    // capacitor companion model using trapezoidal approximation
	    // (Thevenin equivalent) consists of a voltage source in
	    // series with a resistor
	    stampVoltageSource(nodes[0], nodes[2], voltSource);
	    compResistance = timeStep/(2*capacitance);
	    stampResistor(nodes[2], nodes[1], compResistance);
	}
	void startIteration() {
	    voltSourceValue = -voltdiff-current*compResistance;
	}
	double voltSourceValue;
	void doStep() {
	    updateVoltageSource(nodes[0], nodes[2], voltSource,
				voltSourceValue);
 	}
	int getVoltageSourceCount() { return 1; }
	int getInternalNodeCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "capacitor";
	    getBasicInfo(arr);
	    arr[3] = "C = " + getUnitText(capacitance, "F");
	    arr[4] = "P = " + getUnitText(getPower(), "W");
	    //double v = getVoltageDiff();
	    //arr[4] = "U = " + getUnitText(.5*capacitance*v*v, "J");
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Capacitance (uF)", capacitance*1e6, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    capacitance = ei.value*1e-6;
	}
    }

    class InductorElm extends CircuitElm {
	double inductance;
	double compResistance;
	public InductorElm(int xx, int yy) { super(xx, yy); inductance = 1; }
	public InductorElm(int xa, int ya, int xb, int yb, int f,
		    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    inductance = new Double(st.nextToken()).doubleValue();
	    current = new Double(st.nextToken()).doubleValue();
	}
	int getDumpType() { return 'l'; }
	String dump() {
	    return super.dump() + " " + inductance + " " + current;
	}
	void setPoints() {
	    super.setPoints();
	    calcLeads(32);
	}
	void draw(Graphics g) {
	    double v1 = volts[0];
	    double v2 = volts[1];
	    int i;
	    int hs = 8;
	    setBbox(point1, point2, hs);
	    draw2Leads(g);
	    setPowerColor(g, false);
	    drawCoil(g, 8, lead1, lead2, v1, v2);
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(inductance, "H");
		drawValues(g, s, hs);
	    }
	    doDots(g);
	    drawPosts(g);
	}
	void reset() { current = volts[0] = volts[1] = curcount = 0; }
	void stamp() {
	    // inductor companion model using trapezoidal approximation
	    // (Norton equivalent) consists of a current source in
	    // parallel with a resistor.
	    compResistance = 2*inductance/timeStep;
	    stampResistor(nodes[0], nodes[1], compResistance);
	    stampRightSide(nodes[0]);
	    stampRightSide(nodes[1]);
	}
	void startIteration() {
	    double voltdiff = volts[0]-volts[1];
	    curSourceValue = voltdiff/compResistance+current;
	    //System.out.println("ind " + this + " " + current + " " + voltdiff);
	}
	void calculateCurrent() {
	    double voltdiff = volts[0]-volts[1];
	    // we check compResistance because this might get called
	    // before stamp(), which sets compResistance, causing
	    // infinite current
	    if (compResistance > 0)
		current = voltdiff/compResistance + curSourceValue;
	}
	double curSourceValue;
	void doStep() {
	    stampCurrentSource(nodes[0], nodes[1], curSourceValue);
 	}
	void getInfo(String arr[]) {
	    arr[0] = "inductor";
	    getBasicInfo(arr);
	    arr[3] = "L = " + getUnitText(inductance, "H");
	    arr[4] = "P = " + getUnitText(getPower(), "W");
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Inductance (H)", inductance, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    inductance = ei.value;
	}
    }

    class DCVoltageElm extends VoltageElm {
	public DCVoltageElm(int xx, int yy) { super(xx, yy, WF_DC); }
	Class getDumpClass() { return VoltageElm.class; }
    }
    class ACVoltageElm extends VoltageElm {
	public ACVoltageElm(int xx, int yy) { super(xx, yy, WF_AC); }
	Class getDumpClass() { return VoltageElm.class; }
    }
    
    class VoltageElm extends CircuitElm {
	static final int FLAG_COS = 2;
	int waveform;
	static final int WF_DC = 0;
	static final int WF_AC = 1;
	static final int WF_SQUARE = 2;
	static final int WF_TRIANGLE = 3;
	static final int WF_SAWTOOTH = 4;
	static final int WF_PULSE = 5;
	static final int WF_VAR = 6;
	double frequency, maxVoltage, freqTimeZero, bias,
	    phaseShift, dutyCycle;
	VoltageElm(int xx, int yy, int wf) {
	    super(xx, yy);
	    waveform = wf;
	    maxVoltage = 5;
	    frequency = 40;
	    dutyCycle = .5;
	    reset();
	}
	public VoltageElm(int xa, int ya, int xb, int yb, int f,
		   StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    maxVoltage = 5;
	    frequency = 40;
	    waveform = WF_DC;
	    dutyCycle = .5;
	    try {
		waveform = new Integer(st.nextToken()).intValue();
		frequency = new Double(st.nextToken()).doubleValue();
		maxVoltage = new Double(st.nextToken()).doubleValue();
		bias = new Double(st.nextToken()).doubleValue();
		phaseShift = new Double(st.nextToken()).doubleValue();
		dutyCycle = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
	    }
	    if ((flags & FLAG_COS) != 0) {
		flags &= ~FLAG_COS;
		phaseShift = pi/2;
	    }
	    reset();
	}
	int getDumpType() { return 'v'; }
	String dump() {
	    return super.dump() + " " + waveform + " " + frequency + " " +
		maxVoltage + " " + bias + " " + phaseShift + " " +
		dutyCycle;
	}
	/*void setCurrent(double c) {
	    current = c;
	    System.out.print("v current set to " + c + "\n");
	    }*/

	void reset() {
	    freqTimeZero = 0;
	    curcount = 0;
	}
	double triangleFunc(double x) {
	    if (x < pi)
		return x*(2/pi)-1;
	    return 1-(x-pi)*(2/pi);
	}
	void stamp() {
	    if (waveform == WF_DC)
		stampVoltageSource(nodes[0], nodes[1], voltSource,
				   getVoltage());
	    else
		stampVoltageSource(nodes[0], nodes[1], voltSource);
	}
	void doStep() {
	    if (waveform != WF_DC)
		updateVoltageSource(nodes[0], nodes[1], voltSource,
				    getVoltage());
	}
	double getVoltage() {
	    double w = 2*pi*(t-freqTimeZero)*frequency + phaseShift;
	    switch (waveform) {
	    case WF_DC: return maxVoltage+bias;
	    case WF_AC: return Math.sin(w)*maxVoltage+bias;
	    case WF_SQUARE:
		return bias+((w % (2*pi) > (2*pi*dutyCycle)) ?
			      -maxVoltage : maxVoltage);
	    case WF_TRIANGLE:
		return bias+triangleFunc(w % (2*pi))*maxVoltage;
	    case WF_SAWTOOTH:
		return bias+(w % (2*pi))*(maxVoltage/pi)-maxVoltage;
	    case WF_PULSE:
		return ((w % (2*pi)) < 1) ? maxVoltage+bias : bias;
	    default: return 0;
	    }
	}
	final int circleSize = 17;
	void setPoints() {
	    super.setPoints();
	    calcLeads((waveform == WF_DC || waveform == WF_VAR) ? 8 : circleSize*2);
	}
	void draw(Graphics g) {
	    setBbox(x, y, x2, y2);
	    draw2Leads(g);
	    if (waveform == WF_DC) {
		setPowerColor(g, false);
		setVoltageColor(g, volts[0]);
		interpPoint2(lead1, lead2, ps1, ps2, 0, 10);
		drawThickLine(g, ps1, ps2);
		setVoltageColor(g, volts[1]);
		int hs = 16;
		setBbox(point1, point2, hs);
		interpPoint2(lead1, lead2, ps1, ps2, 1, hs);
		drawThickLine(g, ps1, ps2);
	    } else {
		setBbox(point1, point2, circleSize);
		interpPoint(lead1, lead2, ps1, .5);
		drawWaveform(g, ps1);
	    }
	    updateDotCount();
	    if (dragElm != this) {
		if (waveform == WF_DC)
		    drawDots(g, point1, point2, curcount);
		else {
		    drawDots(g, point1, lead1, curcount);
		    drawDots(g, point2, lead2, -curcount);
		}
	    }
	    drawPosts(g);
	}
	
	void drawWaveform(Graphics g, Point center) {
	    g.setColor(this == mouseElm ? selectColor : Color.gray);
	    setPowerColor(g, false);
	    int xc = center.x; int yc = center.y;
	    drawThickCircle(g, xc, yc, circleSize);
	    int wl = 8;
	    adjustBbox(xc-circleSize, yc-circleSize,
		       xc+circleSize, yc+circleSize);
	    int xc2;
	    switch (waveform) {
	    case WF_DC:
		{
		    break;
		}
	    case WF_SQUARE:
		xc2 = (int) (wl*2*dutyCycle-wl+xc);
		xc2 = max(xc-wl+3, min(xc+wl-3, xc2));
		drawThickLine(g, xc-wl, yc-wl, xc-wl, yc   );
		drawThickLine(g, xc-wl, yc-wl, xc2  , yc-wl);
		drawThickLine(g, xc2  , yc-wl, xc2  , yc+wl);
		drawThickLine(g, xc+wl, yc+wl, xc2  , yc+wl);
		drawThickLine(g, xc+wl, yc   , xc+wl, yc+wl);
		break;
	    case WF_PULSE:
		yc += wl/2;
		drawThickLine(g, xc-wl, yc-wl, xc-wl, yc   );
		drawThickLine(g, xc-wl, yc-wl, xc-wl/2, yc-wl);
		drawThickLine(g, xc-wl/2, yc-wl, xc-wl/2, yc);
		drawThickLine(g, xc-wl/2, yc, xc+wl, yc);
		break;
	    case WF_SAWTOOTH:
		drawThickLine(g, xc   , yc-wl, xc-wl, yc   );
		drawThickLine(g, xc   , yc-wl, xc   , yc+wl);
		drawThickLine(g, xc   , yc+wl, xc+wl, yc   );
		break;
	    case WF_TRIANGLE:
		{
		    int xl = 5;
		    drawThickLine(g, xc-xl*2, yc   , xc-xl, yc-wl);
		    drawThickLine(g, xc-xl, yc-wl, xc, yc);
		    drawThickLine(g, xc   , yc, xc+xl, yc+wl);
		    drawThickLine(g, xc+xl, yc+wl, xc+xl*2, yc);
		    break;
		}
	    case WF_AC:
		{
		    int i;
		    int xl = 10;
		    int ox = -1, oy = -1;
		    for (i = -xl; i <= xl; i++) {
			int yy = yc+(int) (.95*Math.sin(i*pi/xl)*wl);
			if (ox != -1)
			    drawThickLine(g, ox, oy, xc+i, yy);
			ox = xc+i; oy = yy;
		    }
		    break;
		}
	    }
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(frequency, "Hz");
		if (dx == 0 || dy == 0)
		    drawValues(g, s, circleSize);
	    }
	}
	
	int getVoltageSourceCount() {
	    return 1;
	}
	double getPower() { return -getVoltageDiff()*current; }
	double getVoltageDiff() { return volts[1] - volts[0]; }
	void getInfo(String arr[]) {
	    switch (waveform) {
	    case WF_DC: case WF_VAR:
		arr[0] = "voltage source"; break;
	    case WF_AC:       arr[0] = "A/C source"; break;
	    case WF_SQUARE:   arr[0] = "square wave gen"; break;
	    case WF_PULSE:    arr[0] = "pulse gen"; break;
	    case WF_SAWTOOTH: arr[0] = "sawtooth gen"; break;
	    case WF_TRIANGLE: arr[0] = "triangle gen"; break;
	    }
	    arr[1] = "I = " + getCurrentText(getCurrent());
	    arr[2] = ((this instanceof RailElm) ? "V = " : "Vd = ") +
		getVoltageText(getVoltageDiff());
	    if (waveform != WF_DC && waveform != WF_VAR) {
		arr[3] = "f = " + getUnitText(frequency, "Hz");
		arr[4] = "Vmax = " + getVoltageText(maxVoltage);
		int i = 5;
		if (bias != 0)
		    arr[i++] = "Voff = " + getVoltageText(bias);
		else if (frequency > 500)
		    arr[i++] = "wavelength = " +
			getUnitText(2.9979e8/frequency, "m");
		arr[i++] = "P = " + getUnitText(getPower(), "W");
	    }
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo(waveform == WF_DC ? "Voltage" :
				    "Max Voltage", maxVoltage, -20, 20);
	    if (n == 1) {
		EditInfo ei =  new EditInfo("Waveform", waveform, -1, -1);
		ei.choice = new Choice();
		ei.choice.add("D/C");
		ei.choice.add("A/C");
		ei.choice.add("Square Wave");
		ei.choice.add("Triangle");
		ei.choice.add("Sawtooth");
		ei.choice.add("Pulse");
		ei.choice.select(waveform);
		return ei;
	    }
	    if (waveform == WF_DC)
		return null;
	    if (n == 2)
		return new EditInfo("Frequency (Hz)", frequency, 4, 500);
	    if (n == 3)
		return new EditInfo("DC Offset (V)", bias, -20, 20);
	    if (n == 4)
		return new EditInfo("Phase Offset (degrees)", phaseShift*180/pi,
				    -180, 180);
	    if (n == 5 && waveform == WF_SQUARE)
		return new EditInfo("Duty Cycle", dutyCycle*100, 0, 100);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		maxVoltage = ei.value;
	    if (n == 3)
		bias = ei.value;
	    if (n == 2) {
		// adjust time zero to maintain continuity ind the waveform
		// even though the frequency has changed.
		double oldfreq = frequency;
		frequency = ei.value;
		double maxfreq = 1/(8*timeStep);
		if (frequency > maxfreq)
		    frequency = maxfreq;
		double adj = frequency-oldfreq;
		freqTimeZero = t-oldfreq*(t-freqTimeZero)/frequency;
	    }
	    if (n == 1) {
		int ow = waveform;
		waveform = ei.choice.getSelectedIndex();
		if (waveform == WF_DC && ow != WF_DC) {
		    ei.newDialog = true;
		    bias = 0;
		} else if (waveform != WF_DC && ow == WF_DC) {
		    ei.newDialog = true;
		}
		if ((waveform == WF_SQUARE || ow == WF_SQUARE) &&
		    waveform != ow)
		    ei.newDialog = true;
		setPoints();
	    }
	    if (n == 4)
		phaseShift = ei.value*pi/180;
	    if (n == 5)
		dutyCycle = ei.value*.01;
	}
    }

    class CurrentElm extends CircuitElm {
	double currentValue;
	public CurrentElm(int xx, int yy) {
	    super(xx, yy);
	    currentValue = .01;
	}
	public CurrentElm(int xa, int ya, int xb, int yb, int f,
		   StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    try {
		currentValue = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		currentValue = .01;
	    }
	}
	String dump() {
	    return super.dump() + " " + currentValue;
	}
	int getDumpType() { return 'i'; }
	
	Polygon arrow;
	Point ashaft1, ashaft2, center;
	void setPoints() {
	    super.setPoints();
	    calcLeads(26);
	    ashaft1 = interpPoint(lead1, lead2, .25);
	    ashaft2 = interpPoint(lead1, lead2, .6);
	    center = interpPoint(lead1, lead2, .5);
	    Point p2 = interpPoint(lead1, lead2, .75);
	    arrow = calcArrow(center, p2, 4, 4);
	}
	void draw(Graphics g) {
	    int cr = 12;
	    draw2Leads(g);
	    setVoltageColor(g, (volts[0]+volts[1])/2);
	    setPowerColor(g, false);
	    
	    drawThickCircle(g, center.x, center.y, cr);
	    drawThickLine(g, ashaft1, ashaft2);

	    g.fillPolygon(arrow);
	    setBbox(point1, point2, cr);
	    doDots(g);
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(currentValue, "A");
		if (dx == 0 || dy == 0)
		    drawValues(g, s, cr);
	    }
	    drawPosts(g);
	}
	void stamp() {
	    current = currentValue;
	    stampCurrentSource(nodes[0], nodes[1], current);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Current (A)", currentValue, 0, .1);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    currentValue = ei.value;
	}
	void getInfo(String arr[]) {
	    arr[0] = "current source";
	    getBasicInfo(arr);
	}
	double getVoltageDiff() {
	    return volts[1] - volts[0];
	}
    }

    class PushSwitchElm extends SwitchElm {
	public PushSwitchElm(int xx, int yy) { super(xx, yy, true); }
	Class getDumpClass() { return SwitchElm.class; }
    }
    
    class SwitchElm extends CircuitElm {
	boolean momentary;
	// position 0 == closed, position 1 == open
	int position, posCount;
	public SwitchElm(int xx, int yy) {
	    super(xx, yy);
	    momentary = false;
	    position = 0;
	    posCount = 2;
	}
	SwitchElm(int xx, int yy, boolean mm) {
	    super(xx, yy);
	    position = (mm) ? 1 : 0;
	    momentary = mm;
	    posCount = 2;
	}
	public SwitchElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    String str = st.nextToken();
	    if (str.compareTo("true") == 0)
		position = (this instanceof LogicInputElm) ? 0 : 1;
	    else if (str.compareTo("false") == 0)
		position = (this instanceof LogicInputElm) ? 1 : 0;
	    else
		position = new Integer(str).intValue();
	    momentary = new Boolean(st.nextToken()).booleanValue();
	    posCount = 2;
	}
	int getDumpType() { return 's'; }
	String dump() {
	    return super.dump() + " " + position + " " + momentary;
	}

	Point ps;
	void setPoints() {
	    super.setPoints();
	    calcLeads(32);
	    ps = new Point();
	}
	
	void draw(Graphics g) {
	    int openhs = 16;
	    int hs = (position == 1) ? openhs : 0;
	    setBbox(point1, point2, openhs);

	    draw2Leads(g);
	    
	    if (position == 0)
		doDots(g);
	    
	    if (mouseElm != this)
		g.setColor(whiteColor);
	    interpPoint(lead1, lead2, ps, 1, hs);
	    
	    drawThickLine(g, lead1, ps);
	    drawPosts(g);
	}
	void calculateCurrent() {
	    if (position == 1)
		current = 0;
	}
	void stamp() {
	    if (position == 0)
		stampVoltageSource(nodes[0], nodes[1], voltSource, 0);
	}
	int getVoltageSourceCount() {
	    return (position == 1) ? 0 : 1;
	}
	void mouseUp() {
	    if (momentary)
		toggle();
	}
	void toggle() {
	    position++;
	    if (position >= posCount)
		position = 0;
	}
	void getInfo(String arr[]) {
	    arr[0] = (momentary) ? "push switch (SPST)" : "switch (SPST)";
	    if (position == 1) {
		arr[1] = "open";
		arr[2] = "Vd = " + getVoltageDText(getVoltageDiff());
	    } else {
		arr[1] = "closed";
		arr[2] = "V = " + getVoltageText(volts[0]);
		arr[3] = "I = " + getCurrentDText(getCurrent());
	    }
	}
	boolean getConnection(int n1, int n2) { return position == 0; }
	boolean isWire() { return true; }
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Momentary Switch", momentary);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		momentary = ei.checkbox.getState();
	}
    }

    // DPST switch
    class Switch2Elm extends SwitchElm {
	int link;
	
	public Switch2Elm(int xx, int yy) {
	    super(xx, yy, false);
	    noDiagonal = true;
	}
	Switch2Elm(int xx, int yy, boolean mm) {
	    super(xx, yy, mm);
	    noDiagonal = true;
	}
	public Switch2Elm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    link = new Integer(st.nextToken()).intValue();
	    noDiagonal = true;
	}
	int getDumpType() { return 'S'; }
	String dump() {
	    return super.dump() + " " + position + " " + momentary + " " + link;
	}

	final int openhs = 16;
	Point swposts[], swpoles[];
	void setPoints() {
	    super.setPoints();
	    calcLeads(32);
	    swposts = newPointArray(2);
	    swpoles = newPointArray(2);
	    interpPoint2(lead1,  lead2,  swpoles[0], swpoles[1], 1, openhs);
	    interpPoint2(point1, point2, swposts[0], swposts[1], 1, openhs);
	}
	
	void draw(Graphics g) {
	    setBbox(point1, point2, openhs);

	    // draw first lead
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);

	    // draw second lead
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, swpoles[0], swposts[0]);
	    
	    // draw third lead
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, swpoles[1], swposts[1]);

	    // draw switch
	    if (mouseElm != this)
		g.setColor(whiteColor);
	    drawThickLine(g, lead1, swpoles[position]);
	    
	    updateDotCount();
	    drawDots(g, point1, lead1, curcount);
	    drawDots(g, swpoles[position], swposts[position], curcount);
	    drawPosts(g);
	}
	Point getPost(int n) {
	    return (n == 0) ? point1 : swposts[n-1];
	}
	int getPostCount() { return 3; }
	void calculateCurrent() { }
	void stamp() {
	    stampVoltageSource(nodes[0], nodes[position+1], voltSource, 0);
	}
	int getVoltageSourceCount() {
	    return 1;
	}
	void toggle() {
	    super.toggle();
	    if (link != 0) {
		int i;
		for (i = 0; i != elmList.size(); i++) {
		    Object o = elmList.elementAt(i);
		    if (o instanceof Switch2Elm) {
			Switch2Elm s2 = (Switch2Elm) o;
			if (s2.link == link)
			    s2.position = position;
		    }
		}
	    }
	}
	boolean getConnection(int n1, int n2) {
	    return comparePair(n1, n2, 0, 1+position);
	}
	void getInfo(String arr[]) {
	    arr[0] = (link == 0) ? "switch (DPST)" : "switch (DPDT)";
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	}
    }

    class DiodeElm extends CircuitElm {
	static final int FLAG_FWDROP = 1;
	public DiodeElm(int xx, int yy) {
	    super(xx, yy);
	    fwdrop = defaultdrop;
	    setup();
	}
	public DiodeElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    fwdrop = defaultdrop;
	    if ((f & FLAG_FWDROP) > 0) {
		try {
		    fwdrop = new Double(st.nextToken()).doubleValue();
		} catch (Exception e) {
		}
	    }
	    setup();
	}
	boolean nonLinear() { return true; }
	void setup() {
	    vdcoef = Math.log(1/leakage + 1)/fwdrop;
	    vt = 1/vdcoef;
	    vcrit = vt * Math.log(vt/(Math.sqrt(2)*leakage));
	}
	int getDumpType() { return 'd'; }
	String dump() {
	    flags |= FLAG_FWDROP;
	    return super.dump() + " " + fwdrop;
	}
	
	final int hs = 8;
	Polygon poly;
	Point cathode[];
	
	void setPoints() {
	    super.setPoints();
	    calcLeads(16);
	    cathode = newPointArray(2);
	    Point pa[] = newPointArray(2);
	    interpPoint2(lead1, lead2, pa[0], pa[1], 0, hs);
	    interpPoint2(lead1, lead2, cathode[0], cathode[1], 1, hs);
	    poly = createPolygon(pa[0], pa[1], lead2);
	}
	
	void draw(Graphics g) {
	    drawDiode(g);
	    doDots(g);
	    drawPosts(g);
	}
	
	void reset() {
	    lastvoltdiff = volts[0] = volts[1] = curcount = 0;
	}
	
	void drawDiode(Graphics g) {
	    setBbox(point1, point2, hs);

	    double v1 = volts[0];
	    double v2 = volts[1];

	    draw2Leads(g);

	    // draw arrow thingy
	    setPowerColor(g, true);
	    setVoltageColor(g, v1);
	    g.fillPolygon(poly);

	    // draw thing arrow is pointing to
	    setVoltageColor(g, v2);
	    drawThickLine(g, cathode[0], cathode[1]);
	}
	
	double leakage = 1e-14; // was 1e-9;
	final double defaultdrop = .805904783;
	double vt, vdcoef, fwdrop;
	double lastvoltdiff;
	double vcrit;
	double limitStep(double vnew, double vold) {
	    double arg;
	    double oo = vnew;
	    
	    if (vnew > vcrit && Math.abs(vnew - vold) > (vt + vt)) {
		if(vold > 0) {
		    arg = 1 + (vnew - vold) / vt;
		    if(arg > 0) {
			vnew = vold + vt * Math.log(arg);
			double v0 = Math.log(1e-6/leakage)*vt;
			vnew = Math.max(v0, vnew);
			//System.out.println(oo + " " + vnew);
		    } else {
			vnew = vcrit;
		    }
		} else {
		    vnew = vt *Math.log(vnew/vt);
		}
		converged = false;
		//System.out.println(vnew + " " + oo + " " + vold);
	    }
	    return(vnew);
	}
	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	}
	void doStep() {
	    double voltdiff = volts[0] - volts[1];
	    // used to have .1 here, but needed .01 for peak detector
	    if (Math.abs(voltdiff-lastvoltdiff) > .01)
		converged = false;
	    voltdiff = limitStep(voltdiff, lastvoltdiff);
	    lastvoltdiff = voltdiff;
	    double eval = Math.exp(voltdiff*vdcoef);
	    // make diode linear with negative voltages; aids convergence
	    if (voltdiff < 0)
		eval = 1;
	    double geq = vdcoef*leakage*eval;
	    double nc = (eval-1)*leakage - geq*voltdiff;
	    stampConductance(nodes[0], nodes[1], geq);
	    stampCurrentSource(nodes[0], nodes[1], nc);
	}
	void calculateCurrent() {
	    double voltdiff = volts[0] - volts[1];
	    current = leakage*(Math.exp(voltdiff*vdcoef)-1);
	}
	void getInfo(String arr[]) {
	    arr[0] = "diode";
	    arr[1] = "I = " + getCurrentText(getCurrent());
	    arr[2] = "Vd = " + getVoltageText(getVoltageDiff());
	    arr[3] = "P = " + getUnitText(getPower(), "W");
	    arr[4] = "Vf = " + getVoltageText(fwdrop);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Fwd Voltage @ 1A", fwdrop, 10, 1000);
	    return null;
	} 
	void setEditValue(int n, EditInfo ei) {
	    fwdrop = ei.value;
	    setup();
	}
    }

    // Zener code contributed by J. Mike Rollins
    // http://www.camotruck.net/rollins/simulator.html
    class ZenerElm extends DiodeElm {
	public ZenerElm(int xx, int yy) {
	    super(xx, yy);
	    zvoltage = default_zvoltage;
	    setup();
	}
	public ZenerElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    zvoltage = new Double(st.nextToken()).doubleValue();
	    setup();
	}
	void setup() {
	    super.setup();
	    leakage = 5e-6; // 1N4004 is 5.0 uAmp
	    // calculate offset which will give us 5mA at zvoltage
	    double i = -.005;
	    zoffset = zvoltage-Math.log(-(1+i/leakage))/vdcoef;
	}
	int getDumpType() { return 'z'; }
	String dump() {
	    return super.dump() + " " + zvoltage;
	}
	
	final int hs = 8;
	Polygon poly;
	Point cathode[];
	Point wing[];
	
	void setPoints() {
	    super.setPoints();
	    calcLeads(16);
	    cathode = newPointArray(2);
	    wing = newPointArray(2);
	    Point pa[] = newPointArray(2);
	    interpPoint2(lead1, lead2, pa[0], pa[1], 0, hs);
	    interpPoint2(lead1, lead2, cathode[0], cathode[1], 1, hs);
	    interpPoint(cathode[0], cathode[1], wing[0], -0.2, -hs);
	    interpPoint(cathode[1], cathode[0], wing[1], -0.2, -hs);
	    poly = createPolygon(pa[0], pa[1], lead2);
	}
	
	void draw(Graphics g) {
	    setBbox(point1, point2, hs);

	    double v1 = volts[0];
	    double v2 = volts[1];

	    draw2Leads(g);

	    // draw arrow thingy
	    setPowerColor(g, true);
	    setVoltageColor(g, v1);
	    g.fillPolygon(poly);

	    // draw thing arrow is pointing to
	    setVoltageColor(g, v2);
	    drawThickLine(g, cathode[0], cathode[1]);

            // draw wings on cathode
	    drawThickLine(g, wing[0], cathode[0]);
	    drawThickLine(g, wing[1], cathode[1]);
	    
	    doDots(g);
	    drawPosts(g);
	}
	
        double zvoltage, zoffset;
	final double default_zvoltage = 5.6;

        //boolean my_debug = true;
        boolean my_debug = false;

	double limitStep(double vnew, double vold) {
	    double arg;
	    double oo = vnew;
	    
            if (vnew >= 0)
              {
	        if (vnew > vcrit && Math.abs(vnew - vold) > (vt + vt)) 
                  {
		    if(vold > 0) {
		        arg = 1 + (vnew - vold) / vt;
		        if(arg > 0) {
			    vnew = vold + vt * Math.log(arg);
			    double v0 = Math.log(1e-6/leakage)*vt;
			    vnew = Math.max(v0, vnew);
			    //System.out.println(oo + " " + vnew);
		        } else {
			    vnew = vcrit;
		        }
		    } else {
		        vnew = vt *Math.log(vnew/vt);
		    }
		    converged = false;
		    //System.out.println(vnew + " " + oo + " " + vold);
	          }
	        return(vnew);
	      }
            else
              {

                if (my_debug) System.out.println(vnew + " " + vold);

                // use the same logic with the regular diode but translate the values
                vnew = -vnew - zoffset;
                vold = -vold - zoffset;

                if (vnew > vcrit && Math.abs(vnew - vold) > (vt + vt))
                  {
                    if(vold > 0) {
                        arg = 1 + (vnew - vold) / vt;
                        if(arg > 0) {
                            vnew = vold + vt * Math.log(arg);
                            double v0 = Math.log(1e-6/leakage)*vt;
                            vnew = Math.max(v0, vnew);
                            //System.out.println(oo + " " + vnew);
                        } else {
                            vnew = vcrit;
                        }
                    } else {
                        vnew = vt *Math.log(vnew/vt);
                    }
                    converged = false;
                  }
                else
                  {
                    //vnew = (vnew+vold)/2;
                  }
                vnew = -(vnew+zoffset);
                return(vnew);
              }
	}

	void doStep() {
	    double voltdiff = volts[0] - volts[1];

            if (my_debug) 
              System.out.println("Voltdiff: " + volts[0] + " - " + volts[1] 
                                 + " = " + voltdiff);

	    if (Math.abs(voltdiff-lastvoltdiff) > .01)
		converged = false;
	    voltdiff = limitStep(voltdiff, lastvoltdiff);
	    lastvoltdiff = voltdiff;

	    if (voltdiff >= 0) {
		double eval = Math.exp(voltdiff*vdcoef);
		double geq = vdcoef*leakage*eval;   
		double nc = (eval-1)*leakage + geq*(-voltdiff); 
		stampConductance(nodes[0], nodes[1], geq);
		stampCurrentSource(nodes[0], nodes[1], nc);
	    } else {
                /* 
                 * I(Vd) = Is * (exp[Vd*C] - exp[(-Vd-Vz)*C] - 1 )
                 *
                 * geq is I'(Vd)
                 * nc is I(Vd) + I'(Vd)*(-Vd)
                 */

		double geq = leakage*vdcoef*
			       ( 
				 Math.exp(voltdiff*vdcoef) 
				 + Math.exp((-voltdiff-zoffset)*vdcoef)
			       );

		double nc = leakage*
			      (
				Math.exp(voltdiff*vdcoef) 
				- Math.exp((-voltdiff-zoffset)*vdcoef) 
				- 1
			      )
			    + geq*(-voltdiff);

		stampConductance(nodes[0], nodes[1], geq);
		stampCurrentSource(nodes[0], nodes[1],  nc);
	    }
	}

	void calculateCurrent() {
	    double voltdiff = (volts[0] - volts[1]);
	    
	    if (voltdiff >= 0)
		current = leakage*(Math.exp(voltdiff*vdcoef)-1);
	    else {
		current = leakage* (
			Math.exp(voltdiff*vdcoef)  
			- Math.exp((-voltdiff-zoffset)*vdcoef)  
			- 1
			);
	    }
	}

	void getInfo(String arr[]) {
	    super.getInfo(arr);
	    arr[0] = "Zener diode";
	    arr[5] = "Vz = " + getVoltageText(zvoltage);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Fwd Voltage @ 1A", fwdrop, 10, 1000);
	    if (n == 1)
		return new EditInfo("Zener Voltage @ 5mA", zvoltage, 1, 25);
	    return null;
	} 
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		fwdrop = ei.value;
	    if (n == 1)
		zvoltage = ei.value;
	    setup();
	}
    }
    
    class LEDElm extends DiodeElm {
	double colorR, colorG, colorB;
	public LEDElm(int xx, int yy) {
	    super(xx, yy);
	    fwdrop = 2.1024259;
	    setup();
	    colorR = 1; colorG = colorB = 0;
	}
	public LEDElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    if ((f & FLAG_FWDROP) == 0)
		fwdrop = 2.1024259;
	    setup();
	    colorR = new Double(st.nextToken()).doubleValue();
	    colorG = new Double(st.nextToken()).doubleValue();
	    colorB = new Double(st.nextToken()).doubleValue();
	}
	int getDumpType() { return 162; }
	String dump() {
	    return super.dump() + " " + colorR + " " + colorG + " " + colorB;
	}

	Point ledLead1, ledLead2, ledCenter;
	void setPoints() {
	    super.setPoints();
	    int cr = 12;
	    ledLead1  = interpPoint(point1, point2, .5-cr/dn);
	    ledLead2  = interpPoint(point1, point2, .5+cr/dn);
	    ledCenter = interpPoint(point1, point2, .5);
	}
	
	void draw(Graphics g) {
	    if (this == mouseElm || this == dragElm) {
		super.draw(g);
		return;
	    }
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, ledLead1);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, ledLead2, point2);
	    
	    g.setColor(Color.gray);
	    int cr = 12;
	    drawThickCircle(g, ledCenter.x, ledCenter.y, cr);
	    cr -= 4;
	    double w = 255*current/.01;
	    if (w > 255)
		w = 255;
	    Color cc = new Color((int) (colorR*w), (int) (colorG*w),
				 (int) (colorB*w));
	    g.setColor(cc);
	    g.fillOval(ledCenter.x-cr, ledCenter.y-cr, cr*2, cr*2);
	    setBbox(point1, point2, cr);
	    updateDotCount();
	    drawDots(g, point1, ledLead1, curcount);
	    drawDots(g, point2, ledLead2, -curcount);
	    drawPosts(g);
	}

	void getInfo(String arr[]) {
	    super.getInfo(arr);
	    arr[0] = "LED";
	}
    }

    class NTransistorElm extends TransistorElm {
	public NTransistorElm(int xx, int yy) { super(xx, yy, false); }
	Class getDumpClass() { return TransistorElm.class; }
    }
    class PTransistorElm extends TransistorElm {
	public PTransistorElm(int xx, int yy) { super(xx, yy, true); }
	Class getDumpClass() { return TransistorElm.class; }
    }
    
    class TransistorElm extends CircuitElm {
	int pnp;
	double beta;
	double fgain;
	final int FLAG_FLIP = 1;
	TransistorElm(int xx, int yy, boolean pnpflag) {
	    super(xx, yy);
	    pnp = (pnpflag) ? -1 : 1;
	    beta = 100;
	    setup();
	}
	public TransistorElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    pnp = new Integer(st.nextToken()).intValue();
	    beta = 100;
	    try {
		lastvbe = new Double(st.nextToken()).doubleValue();
		lastvbc = new Double(st.nextToken()).doubleValue();
		volts[0] = 0;
		volts[1] = -lastvbe;
		volts[2] = -lastvbc;
		beta = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
	    }
	    setup();
	}
	void setup() {
	    vcrit = vt * Math.log(vt/(Math.sqrt(2)*leakage));
	    fgain = beta/(beta+1);
	    noDiagonal = true;
	}
	boolean nonLinear() { return true; }
	void reset() {
	    volts[0] = volts[1] = volts[2] = 0;
	    lastvbc = lastvbe = curcount = 0;
	}
	int getDumpType() { return 't'; }
	String dump() {
	    return super.dump() + " " + pnp + " " + (volts[0]-volts[1]) + " " +
		(volts[0]-volts[2]) + " " + beta;
	}
	double ic, ie, ib, curcount_c, curcount_e, curcount_b;
	Polygon rectPoly, arrowPoly;
	
	void draw(Graphics g) {
	    setBbox(point1, point2, 16);
	    setPowerColor(g, true);
	    // draw collector
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, coll[0], coll[1]);
	    // draw emitter
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, emit[0], emit[1]);
	    // draw arrow
	    g.setColor(lightGrayColor);
	    g.fillPolygon(arrowPoly);
	    // draw base
	    setVoltageColor(g, volts[0]);
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);
	    drawThickLine(g, point1, base);
	    // draw dots
	    curcount_b = updateDotCount(-ib, curcount_b);
	    drawDots(g, base, point1, curcount_b);
	    curcount_c = updateDotCount(-ic, curcount_c);
	    drawDots(g, coll[1], coll[0], curcount_c);
	    curcount_e = updateDotCount(-ie, curcount_e);
	    drawDots(g, emit[1], emit[0], curcount_e);
	    // draw base rectangle
	    setVoltageColor(g, volts[0]);
	    setPowerColor(g, true);
	    g.fillPolygon(rectPoly);

	    if ((mouseElm == this || dragElm == this) && dy == 0) {
		g.setColor(Color.white);
		g.setFont(unitsFont);
		int ds = sign(dx);
		g.drawString("B", base.x-10*ds, base.y-5);
		g.drawString("C", coll[0].x-3+9*ds, coll[0].y+4); // x+6 if ds=1, -12 if -1
		g.drawString("E", emit[0].x-3+9*ds, emit[0].y+4);
	    }
	    drawPosts(g);
	}
	Point getPost(int n) {
	    return (n == 0) ? point1 : (n == 1) ? coll[0] : emit[0];
	}
	
	int getPostCount() { return 3; }
	double getPower() {
	    return (volts[0]-volts[2])*ib + (volts[1]-volts[2])*ic;
	}

	Point rect[], coll[], emit[], base;
	void setPoints() {
	    super.setPoints();
	    int hs = 16;
	    if ((flags & FLAG_FLIP) != 0)
		dsign = -dsign;
	    int hs2 = hs*dsign*pnp;
	    // calc collector, emitter posts
	    coll = newPointArray(2);
	    emit = newPointArray(2);
	    interpPoint2(point1, point2, coll[0], emit[0], 1, hs2);
	    // calc rectangle edges
	    rect = newPointArray(4);
	    interpPoint2(point1, point2, rect[0], rect[1], 1-16/dn, hs);
	    interpPoint2(point1, point2, rect[2], rect[3], 1-13/dn, hs);
	    // calc points where collector/emitter leads contact rectangle
	    interpPoint2(point1, point2, coll[1], emit[1], 1-13/dn, 6*dsign*pnp);
	    // calc point where base lead contacts rectangle
	    base = new Point();
	    interpPoint (point1, point2, base, 1-16/dn);

	    // rectangle
	    rectPoly = createPolygon(rect[0], rect[2], rect[3], rect[1]);

	    // arrow
	    if (pnp == 1)
		arrowPoly = calcArrow(emit[1], emit[0], 8, 4);
	    else {
		Point pt = interpPoint(point1, point2, 1-11/dn, -5*dsign*pnp);
		arrowPoly = calcArrow(emit[0], pt, 8, 4);
	    }
	}
	
	static final double leakage = 1e-13; // 1e-6;
	static final double vt = .025;
	static final double vdcoef = 1/vt;
	static final double rgain = .5;
	double vcrit;
	double lastvbc, lastvbe;
	double limitStep(double vnew, double vold) {
	    double arg;
	    double oo = vnew;
	    
	    if (vnew > vcrit && Math.abs(vnew - vold) > (vt + vt)) {
		if(vold > 0) {
		    arg = 1 + (vnew - vold) / vt;
		    if(arg > 0) {
			vnew = vold + vt * Math.log(arg);
		    } else {
			vnew = vcrit;
		    }
		} else {
		    vnew = vt *Math.log(vnew/vt);
		}
		converged = false;
		//System.out.println(vnew + " " + oo + " " + vold);
	    }
	    return(vnew);
	}
	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	    stampNonLinear(nodes[2]);
	}
	void doStep() {
	    double vbc = volts[0]-volts[1]; // typically negative
	    double vbe = volts[0]-volts[2]; // typically positive
	    if (Math.abs(vbc-lastvbc) > .01 || // .01
		Math.abs(vbe-lastvbe) > .01)
		converged = false;
	    //if (subIterations > 300 && subIterations < 330)
	    //System.out.print("T " + vbc + " " + vbe + "\n");
	    vbc = pnp*limitStep(pnp*vbc, pnp*lastvbc);
	    vbe = pnp*limitStep(pnp*vbe, pnp*lastvbe);
	    lastvbc = vbc;
	    lastvbe = vbe;
	    double pcoef = vdcoef*pnp;
	    double expbc = Math.exp(vbc*pcoef);
	    /*if (expbc > 1e13 || Double.isInfinite(expbc))
	      expbc = 1e13;*/
	    double expbe = Math.exp(vbe*pcoef);
	    if (expbe < 1)
		expbe = 1;
	    /*if (expbe > 1e13 || Double.isInfinite(expbe))
	      expbe = 1e13;*/
	    ie = pnp*leakage*(-(expbe-1)+rgain*(expbc-1));
	    ic = pnp*leakage*(fgain*(expbe-1)-(expbc-1));
	    ib = -(ie+ic);
	    //System.out.println("gain " + ic/ib);
	    //System.out.print("T " + vbc + " " + vbe + " " + ie + " " + ic + "\n");
	    double gee = -leakage*vdcoef*expbe;
	    double gec = rgain*leakage*vdcoef*expbc;
	    double gce = -gee*fgain;
	    double gcc = -gec*(1/rgain);
	    /*System.out.print("gee = " + gee + "\n");
	    System.out.print("gec = " + gec + "\n");
	    System.out.print("gce = " + gce + "\n");
	    System.out.print("gcc = " + gcc + "\n");
	    System.out.print("gce+gcc = " + (gce+gcc) + "\n");
	    System.out.print("gee+gec = " + (gee+gec) + "\n");*/
	    
	    // stamps from page 302 of Pillage.  Node 0 is the base,
	    // node 1 the collector, node 2 the emitter
	    stampMatrix(nodes[0], nodes[0], -gee-gec-gce-gcc);
	    stampMatrix(nodes[0], nodes[1], gec+gcc);
	    stampMatrix(nodes[0], nodes[2], gee+gce);
	    stampMatrix(nodes[1], nodes[0], gce+gcc);
	    stampMatrix(nodes[1], nodes[1], -gcc);
	    stampMatrix(nodes[1], nodes[2], -gce);
	    stampMatrix(nodes[2], nodes[0], gee+gec);
	    stampMatrix(nodes[2], nodes[1], -gec);
	    stampMatrix(nodes[2], nodes[2], -gee);

	    // we are solving for v(k+1), not delta v, so we use formula
	    // 10.5.13, multiplying J by v(k)
	    stampRightSide(nodes[0], -ib - (gec+gcc)*vbc - (gee+gce)*vbe);
	    stampRightSide(nodes[1], -ic + gce*vbe + gcc*vbc);
	    stampRightSide(nodes[2], -ie + gee*vbe + gec*vbc);
	}
	void getInfo(String arr[]) {
	    arr[0] = "transistor (" + ((pnp == -1) ? "PNP)" : "NPN)") + " beta=" +
		showFormat.format(beta);
	    double vbc = volts[0]-volts[1];
	    double vbe = volts[0]-volts[2];
	    double vce = volts[1]-volts[2];
	    if (vbc*pnp > .2)
		arr[1] = vbe*pnp > .2 ? "saturation" : "reverse active";
	    else
		arr[1] = vbe*pnp > .2 ? "fwd active" : "cutoff";
	    arr[2] = "Ic = " + getCurrentText(ic);
	    arr[3] = "Ib = " + getCurrentText(ib);
	    arr[4] = "Vbe = " + getVoltageText(vbe);
	    arr[5] = "Vbc = " + getVoltageText(vbc);
	    arr[6] = "Vce = " + getVoltageText(vce);
	}
	double getScopeValue(int x) {
	    switch (x) {
	    case SCOPEVAL_IB: return ib;
	    case SCOPEVAL_IC: return ic;
	    case SCOPEVAL_IE: return ie;
	    case SCOPEVAL_VBE: return volts[0]-volts[2];
	    case SCOPEVAL_VBC: return volts[0]-volts[1];
	    case SCOPEVAL_VCE: return volts[1]-volts[2];
	    }
	    return 0;
	}
	String getScopeUnits(int x) {
	    switch (x) {
	    case SCOPEVAL_IB: case SCOPEVAL_IC:
	    case SCOPEVAL_IE: return "A";
	    default: return "V";
	    }
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Beta/hFE", beta, 10, 1000);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    beta = ei.value;
	    setup();
	}
	boolean canViewInScope() { return true; }
    }

    class Scope {
	final int FLAG_YELM = 32;
	double minV[], maxV[], minMaxV;
	double minI[], maxI[], minMaxI;
	int scopePointCount = 128;
	int ptr, ctr, speed, position;
	int value;
	String text;
	Rectangle rect;
	boolean showI, showV, showMax, showFreq, lockScale, plot2d, plotXY;
	CircuitElm elm, xElm, yElm;
	MemoryImageSource imageSource;
	Image image;
	int pixels[];
	int draw_ox, draw_oy;
	float dpixels[];
	Scope() {
	    rect = new Rectangle();
	    reset();
	}
	void showCurrent(boolean b) { showI = b; value = 0; }
	void showVoltage(boolean b) { showV = b; value = 0; }
	void showMax    (boolean b) { showMax = b; }
	void showFreq   (boolean b) { showFreq = b; }
	void setLockScale  (boolean b) { lockScale = b; }
	void resetGraph() {
	    scopePointCount = 1;
	    while (scopePointCount < rect.width)
		scopePointCount *= 2;
	    minV = new double[scopePointCount];
	    maxV = new double[scopePointCount];
	    minI = new double[scopePointCount];
	    maxI = new double[scopePointCount];
	    ptr = ctr = 0;
	    allocImage();
	}
	boolean active() { return elm != null; }
	void reset() {
	    resetGraph();
	    minMaxV = 5;
	    minMaxI = .1;
	    speed = 64;
	    showI = showV = showMax = true;
	    showFreq = lockScale = false;
	    // no showI for Output
	    if (elm != null && (elm instanceof OutputElm ||
				elm instanceof LogicOutputElm ||
				elm instanceof ProbeElm))
		showI = false;
	    value = 0;
	    if (elm instanceof TransistorElm)
		value = SCOPEVAL_VCE;
	}
	void setRect(Rectangle r) {
	    rect = r;
	    resetGraph();
	}
	int getWidth() { return rect.width; }
	int rightEdge() { return rect.x+rect.width; }
	
	void setElm(CircuitElm ce) {
	    elm = ce;
	    reset();
	}
	
	void timeStep() {
	    if (elm == null)
		return;
	    double v = elm.getScopeValue(value);
	    if (v < minV[ptr])
		minV[ptr] = v;
	    if (v > maxV[ptr])
		maxV[ptr] = v;
	    double i = 0;
	    if (value == 0) {
		i = elm.getCurrent();
		if (i < minI[ptr])
		    minI[ptr] = i;
		if (i > maxI[ptr])
		    maxI[ptr] = i;
	    }

	    if (plot2d && dpixels != null) {
		boolean newscale = false;
		while (v > minMaxV || v < -minMaxV) {
		    minMaxV *= 2;
		    newscale = true;
		}
		double yval = i;
		if (plotXY)
		    yval = (yElm == null) ? 0 : yElm.getVoltageDiff();
		while (yval > minMaxI || yval < -minMaxI) {
		    minMaxI *= 2;
		    newscale = true;
		}
		if (newscale)
		    clear2dView();
		double xa = v/minMaxV;
		double ya = yval/minMaxI;
		int x = (int) (rect.width *(1+xa)*.499);
		int y = (int) (rect.height*(1-ya)*.499);
		drawTo(x, y);
	    } else {
		ctr++;
		if (ctr >= speed) {
		    ptr = (ptr+1) & (scopePointCount-1);
		    minV[ptr] = maxV[ptr] = v;
		    minI[ptr] = maxI[ptr] = i;
		    ctr = 0;
		}
	    }
	}

	void drawTo(int x2, int y2) {
	    if (draw_ox == -1) {
		draw_ox = x2;
		draw_oy = y2;
	    }
	    // need to draw a line from x1,y1 to x2,y2
	    if (draw_ox == x2 && draw_oy == y2) {
		dpixels[x2+rect.width*y2] = 1;
	    } else if (abs(y2-draw_oy) > abs(x2-draw_ox)) {
		// y difference is greater, so we step along y's
		// from min to max y and calculate x for each step
		double sgn = sign(y2-draw_oy);
		int x, y;
		for (y = draw_oy; y != y2+sgn; y += sgn) {
		    x = draw_ox+(x2-draw_ox)*(y-draw_oy)/(y2-draw_oy);
		    dpixels[x+rect.width*y] = 1;
		}
	    } else {
		// x difference is greater, so we step along x's
		// from min to max x and calculate y for each step
		double sgn = sign(x2-draw_ox);
		int x, y;
		for (x = draw_ox; x != x2+sgn; x += sgn) {
		    y = draw_oy+(y2-draw_oy)*(x-draw_ox)/(x2-draw_ox);
		    dpixels[x+rect.width*y] = 1;
		}
	    }
	    draw_ox = x2;
	    draw_oy = y2;
	}
	
	void clear2dView() {
	    int i;
	    for (i = 0; i != dpixels.length; i++)
		dpixels[i] = 0;
	    draw_ox = draw_oy = -1;
	}
	
	void adjustScale(double x) {
	    minMaxV *= x;
	    minMaxI *= x;
	}

	void draw2d(Graphics g) {
	    int i;
	    if (pixels == null || dpixels == null)
		return;
	    int col = (printableCheckItem.getState()) ? 0xFFFFFFFF : 0;
	    for (i = 0; i != pixels.length; i++)
		pixels[i] = col;
	    for (i = 0; i != rect.width; i++)
		pixels[i+rect.width*(rect.height/2)] = 0xFF00FF00;
	    int ycol = (plotXY) ? 0xFF00FF00 : 0xFFFFFF00;
	    for (i = 0; i != rect.height; i++)
		pixels[rect.width/2+rect.width*i] = ycol;
	    for (i = 0; i != pixels.length; i++) {
		int q = (int) (255*dpixels[i]);
		if (q > 0)
		    pixels[i] = 0xFF000000 | (0x10101*q);
		dpixels[i] *= .99;
	    }
	    g.drawImage(image, rect.x, rect.y, null);
	    g.setColor(whiteColor);
	    g.fillOval(rect.x+draw_ox-2, rect.y+draw_oy-2, 5, 5);
	    int yt = rect.y+10;
	    int x = rect.x;
	    if (text != null && rect.y + rect.height > yt+5) {
		g.drawString(text, x, yt);
		yt += 15;
	    }
	}
	
	void draw(Graphics g) {
	    if (elm == null)
		return;
	    if (plot2d) {
		draw2d(g);
		return;
	    }
	    if (pixels == null)
		return;
	    int i;
	    int col = (printableCheckItem.getState()) ? 0xFFFFFFFF : 0;
	    for (i = 0; i != pixels.length; i++)
		pixels[i] = col;
	    int x = 0;
	    int maxy = (rect.height-1)/2;
	    int y = maxy;

	    boolean gotI = false;
	    boolean gotV = false;
	    int minRange = 4;
	    double realMaxV = -1e8;
	    double realMaxI = -1e8;
	    int curColor = 0xFFFFFF00;
	    int voltColor = (value > 0) ? 0xFFFFFFFF : 0xFF00FF00;
	    if (scopeSelected == -1 && elm == mouseElm)
		curColor = voltColor = 0xFF00FFFF;
	    int ipa = ptr+scopePointCount-rect.width;
	    for (i = 0; i != rect.width; i++) {
		int ip = (i+ipa) & (scopePointCount-1);
		while (maxV[ip] > minMaxV)
		    minMaxV *= 2;
		while (minV[ip] < -minMaxV)
		    minMaxV *= 2;
		while (maxI[ip] > minMaxI)
		    minMaxI *= 2;
		while (minI[ip] < -minMaxI)
		    minMaxI *= 2;
	    }

	    double gridStep = 1e-8;
	    double gridMax = (showI ? minMaxI : minMaxV);
	    while (gridStep*100 < gridMax)
		gridStep *= 10;
	    if (maxy*gridStep/gridMax < .3)
		gridStep = 0;
	    
	    int ll;
	    boolean sublines = (maxy*gridStep/gridMax > 3);
	    for (ll = -100; ll <= 100; ll++) {
		// don't show gridlines if plotting multiple values,
		// or if lines are too close together (except for center line)
		if (ll != 0 && ((showI && showV) || gridStep == 0))
		    continue;
		int yl = maxy-(int) (maxy*ll*gridStep/gridMax);
		if (yl < 0 || yl >= rect.height-1)
		    continue;
		col = ll == 0 ? 0xFF909090 : 0xFF404040;
		if (ll % 10 != 0) {
		    col = 0xFF101010;
		    if (!sublines)
			continue;
		}
		for (i = 0; i != rect.width; i++)
		    pixels[i+yl*rect.width] = col;
	    }

	    gridStep = 1e-15;
	    double ts = timeStep*speed;
	    while (gridStep < ts*5)
		gridStep *= 10;
	    double tstart = t-timeStep*speed*rect.width;
	    double tx = t-(t % gridStep);
	    int first = 1;
	    for (ll = 0; ; ll++) {
		double tl = tx-gridStep*ll;
		int gx = (int) ((tl-tstart)/ts);
		if (gx < 0)
		    break;
		if (gx >= rect.width)
		    continue;
		if (tl < 0)
		    continue;
		col = 0xFF202020;
		first = 0;
		if (((tl+gridStep/4) % (gridStep*10)) < gridStep) {
		    col = 0xFF909090;
		    if (((tl+gridStep/4) % (gridStep*100)) < gridStep)
			col = 0xFF4040D0;
		}
		for (i = 0; i < pixels.length; i += rect.width)
		    pixels[i+gx] = col;
	    }
	    
	    // these two loops are pretty much the same, and should be
	    // combined!
	    if (value == 0 && showI) {
		int ox = -1, oy = -1;
		int j;
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    int miniy = (int) ((maxy/minMaxI)*minI[ip]);
		    int maxiy = (int) ((maxy/minMaxI)*maxI[ip]);
		    if (maxI[ip] > realMaxI)
			realMaxI = maxI[ip];
		    if (miniy <= maxy) {
			if (miniy < -minRange || maxiy > minRange)
			    gotI = true;
			if (ox != -1) {
			    if (miniy == oy && maxiy == oy)
				continue;
			    for (j = ox; j != x+i; j++)
				pixels[j+rect.width*(y-oy)] = curColor;
			    ox = oy = -1;
			}
			if (miniy == maxiy) {
			    ox = x+i;
			    oy = miniy;
			    continue;
			}
			for (j = miniy; j <= maxiy; j++)
			    pixels[x+i+rect.width*(y-j)] = curColor;
		    }
		}
		if (ox != -1)
		    for (j = ox; j != x+i; j++)
			pixels[j+rect.width*(y-oy)] = curColor;
	    }
	    if (value != 0 || showV) {
		int ox = -1, oy = -1, j;
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    int minvy = (int) ((maxy/minMaxV)*minV[ip]);
		    int maxvy = (int) ((maxy/minMaxV)*maxV[ip]);
		    if (maxV[ip] > realMaxV)
			realMaxV = maxV[ip];
		    if ((value != 0 || showV) && minvy <= maxy) {
			if (minvy < -minRange || maxvy > minRange)
			    gotV = true;
			if (ox != -1) {
			    if (minvy == oy && maxvy == oy)
				continue;
			    for (j = ox; j != x+i; j++)
				pixels[j+rect.width*(y-oy)] = voltColor;
			    ox = oy = -1;
			}
			if (minvy == maxvy) {
			    ox = x+i;
			    oy = minvy;
			    continue;
			}
			for (j = minvy; j <= maxvy; j++)
			    pixels[x+i+rect.width*(y-j)] = voltColor;
		    }
		}
		if (ox != -1)
		    for (j = ox; j != x+i; j++)
			pixels[j+rect.width*(y-oy)] = voltColor;
	    }
	    double freq = 0;
	    if (showFreq) {
		// try to get frequency
		// get average
		double avg = 0;
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    avg += minV[ip]+maxV[ip];
		}
		avg /= i*2;
		int state = 0;
		double thresh = avg*.05;
		int oi = 0;
		double avperiod = 0;
		int periodct = -1;
		double avperiod2 = 0;
		// count period lengths
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    double q = maxV[ip]-avg;
		    int os = state;
		    if (q < thresh)
			state = 1;
		    else if (q > -thresh)
			state = 2;
		    if (state == 2 && os == 1) {
			int pd = i-oi;
			oi = i;
			// short periods can't be counted properly
			if (pd < 12)
			    continue;
			// skip first period, it might be too short
			if (periodct >= 0) {
			    avperiod += pd;
			    avperiod2 += pd*pd;
			}
			periodct++;
		    }
		}
		avperiod /= periodct;
		avperiod2 /= periodct;
		double periodstd = Math.sqrt(avperiod2-avperiod*avperiod);
		freq = 1/(avperiod*timeStep*speed);
		// don't show freq if standard deviation is too great
		if (periodct < 1 || periodstd > 2)
		    freq = 0;
		// System.out.println(freq + " " + periodstd + " " + periodct);
	    }
	    g.drawImage(image, rect.x, rect.y, null);
	    g.setColor(whiteColor);
	    int yt = rect.y+10;
	    x += rect.x;
	    if (showMax) {
		if (value != 0)
		    g.drawString(getUnitText(realMaxV,
					     elm.getScopeUnits(value)),
				 x, yt);
		else if (showV)
		    g.drawString(getVoltageText(realMaxV), x, yt);
		else if (showI)
		    g.drawString(getCurrentText(realMaxI), x, yt);
		yt += 15;
	    }
	    if (text != null && rect.y + rect.height > yt+5) {
		g.drawString(text, x, yt);
		yt += 15;
	    }
	    if (showFreq && freq != 0 && rect.y + rect.height > yt+5)
		g.drawString(getUnitText(freq, "Hz"), x, yt);
	    if (ptr > 5 && !lockScale) {
		if (!gotI && minMaxI > 1e-4)
		    minMaxI /= 2;
		if (!gotV && minMaxV > 1e-4)
		    minMaxV /= 2;
	    }
	}
	
	void speedUp() {
	    if (speed > 1) {
		speed /= 2;
		resetGraph();
	    }
	}
	void slowDown() {
	    speed *= 2;
	    resetGraph();
	}
	
	PopupMenu getMenu() {
	    if (elm == null)
		return null;
	    if (elm instanceof TransistorElm) {
		scopeIbMenuItem.setState(value == SCOPEVAL_IB);
		scopeIcMenuItem.setState(value == SCOPEVAL_IC);
		scopeIeMenuItem.setState(value == SCOPEVAL_IE);
		scopeVbeMenuItem.setState(value == SCOPEVAL_VBE);
		scopeVbcMenuItem.setState(value == SCOPEVAL_VBC);
		scopeVceMenuItem.setState(value == SCOPEVAL_VCE);
		return transScopeMenu;
	    } else {
		scopeVMenuItem    .setState(showV && value == 0);
		scopeIMenuItem    .setState(showI && value == 0);
		scopeMaxMenuItem  .setState(showMax);
		scopeFreqMenuItem .setState(showFreq);
		scopePowerMenuItem.setState(value == SCOPEVAL_POWER);
		scopeVIMenuItem   .setState(plot2d && !plotXY);
		scopeXYMenuItem   .setState(plotXY);
		scopeSelectYMenuItem.setEnabled(plotXY);
		scopeResistMenuItem.setState(value == SCOPEVAL_R);
		scopeResistMenuItem.setEnabled(elm instanceof MemristorElm);
		return scopeMenu;
	    }
	}
	void setValue(int x) { reset(); value = x; }
	String dump() {
	    if (elm == null)
		return null;
	    int flags = (showI ? 1 : 0) | (showV ? 2 : 0) |
		(showMax ? 0 : 4) | (showFreq ? 8 : 0) |
		(lockScale ? 16 : 0) | (plot2d ? 64 : 0) |
		(plotXY ? 128 : 0);
	    flags |= FLAG_YELM; // yelm present
	    int eno = locateElm(elm);
	    if (eno < 0)
		return null;
	    int yno = yElm == null ? -1 : locateElm(yElm);
	    String x = "o " + eno + " " +
		speed + " " + value + " " + flags + " " +
		minMaxV + " " + minMaxI + " " + position + " " + yno;
	    if (text != null)
		x += " " + text;
	    return x;
	}
	void undump(StringTokenizer st) {
	    reset();
	    int e = new Integer(st.nextToken()).intValue();
	    if (e == -1)
		return;
	    elm = getElm(e);
	    speed = new Integer(st.nextToken()).intValue();
	    value = new Integer(st.nextToken()).intValue();
	    int flags = new Integer(st.nextToken()).intValue();
	    minMaxV = new Double(st.nextToken()).doubleValue();
	    minMaxI = new Double(st.nextToken()).doubleValue();
	    if (minMaxV == 0)
		minMaxV = .5;
	    if (minMaxI == 0)
		minMaxI = 1;
	    text = null;
	    yElm = null;
	    try {
		position = new Integer(st.nextToken()).intValue();
		int ye = -1;
		if ((flags & FLAG_YELM) != 0) {
		    ye = new Integer(st.nextToken()).intValue();
		    if (ye != -1)
			yElm = getElm(ye);
		}
		while (st.hasMoreTokens()) {
		    if (text == null)
			text = st.nextToken();
		    else
			text += " " + st.nextToken();
		}
	    } catch (Exception ee) {
	    }
	    showI = (flags & 1) != 0;
	    showV = (flags & 2) != 0;
	    showMax = (flags & 4) == 0;
	    showFreq = (flags & 8) != 0;
	    lockScale = (flags & 16) != 0;
	    plot2d = (flags & 64) != 0;
	    plotXY = (flags & 128) != 0;
	}
	void allocImage() {
	    pixels = null;
	    int w = rect.width;
	    int h = rect.height;
	    if (w == 0 || h == 0)
		return;
	    if (useBufferedImage) {
		try {
		    /* simulate the following code using reflection:
		       dbimage = new BufferedImage(d.width, d.height,
		       BufferedImage.TYPE_INT_RGB);
		       DataBuffer db = (DataBuffer)(((BufferedImage)dbimage).
		       getRaster().getDataBuffer());
		       DataBufferInt dbi = (DataBufferInt) db;
		       pixels = dbi.getData();
		    */
		    Class biclass = Class.forName("java.awt.image.BufferedImage");
		    Class dbiclass = Class.forName("java.awt.image.DataBufferInt");
		    Class rasclass = Class.forName("java.awt.image.Raster");
		    Constructor cstr = biclass.getConstructor(
			new Class[] { int.class, int.class, int.class });
		    image = (Image) cstr.newInstance(new Object[] {
				 new Integer(w), new Integer(h),
				 new Integer(BufferedImage.TYPE_INT_RGB)});
		    Method m = biclass.getMethod("getRaster", null);
		    Object ras = m.invoke(image, null);
		    Object db = rasclass.getMethod("getDataBuffer", null).
			invoke(ras, null);
		    pixels = (int[])
			dbiclass.getMethod("getData", null).invoke(db, null);
		} catch (Exception ee) {
		    // ee.printStackTrace();
		    System.out.println("BufferedImage failed");
		}
	    }
	    if (pixels == null) {
		pixels = new int[w*h];
		int i;
		for (i = 0; i != w*h; i++)
		    pixels[i] = 0xFF000000;
		imageSource = new MemoryImageSource(w, h, pixels, 0, w);
		imageSource.setAnimated(true);
		imageSource.setFullBufferUpdates(true);
		image = cv.createImage(imageSource);
	    }
	    dpixels = new float[w*h];
	    draw_ox = draw_oy = -1;
	}

	void handleMenu(ItemEvent e, Object mi) {
	    if (mi == scopeVMenuItem)
		showVoltage(scopeVMenuItem.getState());
	    if (mi == scopeIMenuItem)
		showCurrent(scopeIMenuItem.getState());
	    if (mi == scopeMaxMenuItem)
		showMax(scopeMaxMenuItem.getState());
	    if (mi == scopeFreqMenuItem)
		showFreq(scopeFreqMenuItem.getState());
	    if (mi == scopePowerMenuItem)
		setValue(SCOPEVAL_POWER);
	    if (mi == scopeIbMenuItem)
		setValue(SCOPEVAL_IB);
	    if (mi == scopeIcMenuItem)
		setValue(SCOPEVAL_IC);
	    if (mi == scopeIeMenuItem)
		setValue(SCOPEVAL_IE);
	    if (mi == scopeVbeMenuItem)
		setValue(SCOPEVAL_VBE);
	    if (mi == scopeVbcMenuItem)
		setValue(SCOPEVAL_VBC);
	    if (mi == scopeVceMenuItem)
		setValue(SCOPEVAL_VCE);
	    if (mi == scopeVIMenuItem) {
		plot2d = scopeVIMenuItem.getState();
		plotXY = false;
		resetGraph();
	    }
	    if (mi == scopeXYMenuItem) {
		plotXY = plot2d = scopeXYMenuItem.getState();
		if (yElm == null)
		    selectY();
		resetGraph();
	    }
	    if (mi == scopeResistMenuItem)
		setValue(SCOPEVAL_R);
	}

	void select() {
	    mouseElm = elm;
	    if (plotXY) {
		plotXElm = elm;
		plotYElm = yElm;
	    }
	}

	void selectY() {
	    int e = yElm == null ? -1 : locateElm(yElm);
	    int firstE = e;
	    while (true) {
		for (e++; e < elmList.size(); e++) {
		    CircuitElm ce = getElm(e);
		    if ((ce instanceof OutputElm || ce instanceof ProbeElm) &&
			ce != elm) {
			yElm = ce;
			return;
		    }
		}
		if (firstE == -1)
		    return;
		e = firstE = -1;
	    }
	}
    }

    class NMosfetElm extends MosfetElm {
	public NMosfetElm(int xx, int yy) { super(xx, yy, false); }
	Class getDumpClass() { return MosfetElm.class; }
    }
    class PMosfetElm extends MosfetElm {
	public PMosfetElm(int xx, int yy) { super(xx, yy, true); }
	Class getDumpClass() { return MosfetElm.class; }
    }
    class MosfetElm extends CircuitElm {
	int pnp;
	int FLAG_PNP = 1;
	int FLAG_SHOWVT = 2;
	int FLAG_DIGITAL = 4;
	double vt;
	MosfetElm(int xx, int yy, boolean pnpflag) {
	    super(xx, yy);
	    pnp = (pnpflag) ? -1 : 1;
	    flags = (pnpflag) ? FLAG_PNP : 0;
	    noDiagonal = true;
	    vt = getDefaultThreshold();
	}
	public MosfetElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    pnp = ((f & FLAG_PNP) != 0) ? -1 : 1;
	    noDiagonal = true;
	    vt = getDefaultThreshold();
	    try {
		vt = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {}
	}
	double getDefaultThreshold() { return 1.5; }
	double getBeta() { return .02; }
	boolean nonLinear() { return true; }
	boolean drawDigital() { return (flags & FLAG_DIGITAL) != 0; }
	void reset() {
	    lastv1 = lastv2 = volts[0] = volts[1] = volts[2] = curcount = 0;
	}
	String dump() {
	    return super.dump() + " " + vt;
	}
	int getDumpType() { return 'f'; }
	final int hs = 16;
	
	void draw(Graphics g) {
	    setBbox(point1, point2, hs);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, src[0], src[1]);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, drn[0], drn[1]);
	    int segments = 6;
	    int i;
	    setPowerColor(g, true);
	    double segf = 1./segments;
	    for (i = 0; i != segments; i++) {
		double v = volts[1]+(volts[2]-volts[1])*i/segments;
		setVoltageColor(g, v);
		interpPoint(src[1], drn[1], ps1, i*segf);
		interpPoint(src[1], drn[1], ps2, (i+1)*segf);
		drawThickLine(g, ps1, ps2);
	    }
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, src[1], src[2]);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, drn[1], drn[2]);
	    if (!drawDigital()) {
		setVoltageColor(g, pnp == 1 ? volts[1] : volts[2]);
		g.fillPolygon(arrowPoly);
	    }
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, gate[1]);
	    drawThickLine(g, gate[0], gate[2]);
	    if (drawDigital() && pnp == -1)
		drawThickCircle(g, pcircle.x, pcircle.y, pcircler);
	    if ((flags & FLAG_SHOWVT) != 0) {
		String s = "" + (vt*pnp);
		g.setColor(whiteColor);
		g.setFont(unitsFont);
		drawCenteredText(g, s, x2+2, y2, false);
	    }
	    if ((mouseElm == this || dragElm == this) && dy == 0) {
		g.setColor(Color.white);
		g.setFont(unitsFont);
		int ds = sign(dx);
		g.drawString("G", gate[1].x-10*ds, gate[1].y-5);
		g.drawString(pnp == -1 ? "D" : "S", src[0].x-3+9*ds, src[0].y+4); // x+6 if ds=1, -12 if -1
		g.drawString(pnp == -1 ? "S" : "D", drn[0].x-3+9*ds, drn[0].y+4);
	    }	    
	    curcount = updateDotCount(-ids, curcount);
	    drawDots(g, src[0], src[1], curcount);
	    drawDots(g, src[1], drn[1], curcount);
	    drawDots(g, drn[1], drn[0], curcount);
	    drawPosts(g);
	}
	Point getPost(int n) {
	    return (n == 0) ? point1 : (n == 1) ? src[0] : drn[0];
	}
	double getCurrent() { return ids; }
	double getPower() { return ids*(volts[2]-volts[1]); }
	int getPostCount() { return 3; }

	int pcircler;
	Point src[], drn[], gate[], pcircle;
	Polygon arrowPoly;
	
	void setPoints() {
	    super.setPoints();

	    // find the coordinates of the various points we need to draw
	    // the MOSFET.
	    int hs2 = hs*dsign;
	    src = newPointArray(3);
	    drn = newPointArray(3);
	    interpPoint2(point1, point2, src[0], drn[0], 1, -hs2);
	    interpPoint2(point1, point2, src[1], drn[1], 1-22/dn, -hs2);
	    interpPoint2(point1, point2, src[2], drn[2], 1-22/dn, -hs2*4/3);

	    gate = newPointArray(3);
	    interpPoint2(point1, point2, gate[0], gate[2], 1-28/dn, hs2/2); // was 1-20/dn
	    interpPoint(gate[0], gate[2], gate[1], .5);

	    if (!drawDigital()) {
		if (pnp == 1)
		    arrowPoly = calcArrow(src[1], src[0], 10, 4);
		else
		    arrowPoly = calcArrow(drn[0], drn[1], 12, 5);
	    } else if (pnp == -1) {
		interpPoint(point1, point2, gate[1], 1-36/dn);
		int dist = (dsign < 0) ? 32 : 31;
		pcircle = interpPoint(point1, point2, 1-dist/dn);
		pcircler = 3;
	    }
	}

	double lastv1, lastv2;
	double ids;
	int mode = 0;
	double gm = 0;
	
	void stamp() {
	    stampNonLinear(nodes[1]);
	    stampNonLinear(nodes[2]);
	}
	void doStep() {
	    double vs[] = new double[3];
	    vs[0] = volts[0];
	    vs[1] = volts[1];
	    vs[2] = volts[2];
	    if (vs[1] > lastv1 + .5)
		vs[1] = lastv1 + .5;
	    if (vs[1] < lastv1 - .5)
		vs[1] = lastv1 - .5;
	    if (vs[2] > lastv2 + .5)
		vs[2] = lastv2 + .5;
	    if (vs[2] < lastv2 - .5)
		vs[2] = lastv2 - .5;
	    int source = 1;
	    int drain = 2;
	    if (pnp*vs[1] > pnp*vs[2]) {
		source = 2;
		drain = 1;
	    }
	    int gate = 0;
	    double vgs = vs[gate ]-vs[source];
	    double vds = vs[drain]-vs[source];
	    if (Math.abs(lastv1-vs[1]) > .01 ||
		Math.abs(lastv2-vs[2]) > .01)
		converged = false;
	    lastv1 = vs[1];
	    lastv2 = vs[2];
	    double realvgs = vgs;
	    double realvds = vds;
	    vgs *= pnp;
	    vds *= pnp;
	    ids = 0;
	    gm = 0;
	    double Gds = 0;
	    double beta = getBeta();
	    if (vgs > .5 && this instanceof JfetElm) {
		stop("JFET is reverse biased!", this);
		return;
	    }
	    if (vgs < vt) {
		// should be all zero, but that causes a singular matrix,
		// so instead we treat it as a large resistor
		Gds = 1e-8;
		ids = vds*Gds;
		mode = 0;
	    } else if (vds < vgs-vt) {
		// linear
		ids = beta*((vgs-vt)*vds - vds*vds*.5);
		gm  = beta*vds;
		Gds = beta*(vgs-vds-vt);
		mode = 1;
	    } else {
		// saturation; Gds = 0
		gm  = beta*(vgs-vt);
		// use very small Gds to avoid nonconvergence
		Gds = 1e-8;
		ids = .5*beta*(vgs-vt)*(vgs-vt) + (vds-(vgs-vt))*Gds;
		mode = 2;
	    }
	    double rs = -pnp*ids + Gds*realvds + gm*realvgs;
	    //System.out.println("M " + vds + " " + vgs + " " + ids + " " + gm + " "+ Gds + " " + volts[0] + " " + volts[1] + " " + volts[2] + " " + source + " " + rs + " " + this);
	    stampMatrix(nodes[drain],  nodes[drain],  Gds);
	    stampMatrix(nodes[drain],  nodes[source], -Gds-gm); 
	    stampMatrix(nodes[drain],  nodes[gate],   gm);
	    
	    stampMatrix(nodes[source], nodes[drain],  -Gds);
	    stampMatrix(nodes[source], nodes[source], Gds+gm); 
	    stampMatrix(nodes[source], nodes[gate],  -gm);
	    
	    stampRightSide(nodes[drain],  rs);
	    stampRightSide(nodes[source], -rs);
	    if (source == 2 && pnp == 1 ||
		source == 1 && pnp == -1)
		ids = -ids;
	}
	void getFetInfo(String arr[], String n) {
	    arr[0] = ((pnp == -1) ? "p-" : "n-") + n;
	    arr[0] += " (Vt = " + getVoltageText(pnp*vt) + ")";
	    arr[1] = ((pnp == 1) ? "Ids = " : "Isd = ") + getCurrentText(ids);
	    arr[2] = "Vgs = " + getVoltageText(volts[0]-volts[pnp == -1 ? 2 : 1]);
	    arr[3] = ((pnp == 1) ? "Vds = " : "Vsd = ") + getVoltageText(volts[2]-volts[1]);
	    arr[4] = (mode == 0) ? "off" :
		(mode == 1) ? "linear" : "saturation";
	    arr[5] = "gm = " + getUnitText(gm, "A/V");
	}
	void getInfo(String arr[]) {
	    getFetInfo(arr, "MOSFET");
	}
	boolean canViewInScope() { return true; }
	double getVoltageDiff() { return volts[2] - volts[1]; }
	boolean getConnection(int n1, int n2) {
	    return !(n1 == 0 || n2 == 0);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Threshold Voltage", pnp*vt, .01, 5);
	    if (n == 1) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Digital Symbol", drawDigital());
		return ei;
	    }
		
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		vt = pnp*ei.value;
	    if (n == 1) {
		flags = (ei.checkbox.getState()) ? (flags | FLAG_DIGITAL) :
		    (flags & ~FLAG_DIGITAL);
		setPoints();
	    }
	}
    }
    class NJfetElm extends JfetElm {
	public NJfetElm(int xx, int yy) { super(xx, yy, false); }
	Class getDumpClass() { return JfetElm.class; }
    }
    class JfetElm extends MosfetElm {
	JfetElm(int xx, int yy, boolean pnpflag) {
	    super(xx, yy, pnpflag);
	    noDiagonal = true;
	}
	public JfetElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    noDiagonal = true;
	}
	
	Polygon gatePoly;
	Polygon arrowPoly;
	Point gatePt;

	void draw(Graphics g) {
	    setBbox(point1, point2, hs);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, src[0], src[1]);
	    drawThickLine(g, src[1], src[2]);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, drn[0], drn[1]);
	    drawThickLine(g, drn[1], drn[2]);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, gatePt);
	    g.fillPolygon(arrowPoly);
	    setPowerColor(g, true);
	    g.fillPolygon(gatePoly);
	    curcount = updateDotCount(-ids, curcount);
	    if (curcount != 0) {
		drawDots(g, src[0], src[1], curcount);
		drawDots(g, src[1], src[2], curcount+8);
		drawDots(g, drn[0], drn[1], -curcount);
		drawDots(g, drn[1], drn[2], -(curcount+8));
	    }
	    drawPosts(g);
	}
	void setPoints() {
	    super.setPoints();

	    // find the coordinates of the various points we need to draw
	    // the JFET.
	    int hs2 = hs*dsign;
	    src = newPointArray(3);
	    drn = newPointArray(3);
	    interpPoint2(point1, point2, src[0], drn[0], 1, hs2);
	    interpPoint2(point1, point2, src[1], drn[1], 1, hs2/2);
	    interpPoint2(point1, point2, src[2], drn[2], 1-10/dn, hs2/2);

	    gatePt = interpPoint(point1, point2, 1-14/dn);

	    Point ra[] = newPointArray(4);
	    interpPoint2(point1, point2, ra[0], ra[1], 1-13/dn, hs);
	    interpPoint2(point1, point2, ra[2], ra[3], 1-10/dn, hs);
	    gatePoly = createPolygon(ra[0], ra[1], ra[3], ra[2]);
	    arrowPoly = calcArrow(point1, gatePt, 8, 3);
	}
	int getDumpType() { return 'j'; }
	// these values are taken from Hayes+Horowitz p155
	double getDefaultThreshold() { return -4; }
	double getBeta() { return .00125; }
	void getInfo(String arr[]) {
	    getFetInfo(arr, "JFET");
	}
    }

    class OpAmpSwapElm extends OpAmpElm {
	public OpAmpSwapElm(int xx, int yy) {
	    super(xx, yy);
	    flags |= FLAG_SWAP;
	}
	Class getDumpClass() { return OpAmpElm.class; }
    }
    
    class OpAmpElm extends CircuitElm {
	int opsize, opheight, opwidth, opaddtext;
	double maxOut, minOut, gain;
	boolean reset;
	final int FLAG_SWAP = 1;
	final int FLAG_SMALL = 2;
	final int FLAG_LOWGAIN = 4;
	public OpAmpElm(int xx, int yy) {
	    super(xx, yy);
	    noDiagonal = true;
	    maxOut = 15;
	    minOut = -15;
	    setSize(smallGridCheckItem.getState() ? 1 : 2);
	    setGain();
	}
	public OpAmpElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    try {
		maxOut = new Double(st.nextToken()).doubleValue();
		minOut = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		maxOut = 15; minOut = -15;
	    }
	    noDiagonal = true;
	    setSize((f & FLAG_SMALL) != 0 ? 1 : 2);
	    setGain();
	}
	void setGain() {
	    // gain of 100000 breaks e-amp-dfdx.txt
	    // gain was 1000, but it broke amp-schmitt.txt
	    gain = ((flags & FLAG_LOWGAIN) != 0) ? 1000 : 100000;
	    
	}
	String dump() {
	    return super.dump() + " " + maxOut + " " + minOut;
	}
	boolean nonLinear() { return true; }
	void draw(Graphics g) {
	    setBbox(point1, point2, opheight*2);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, in1p[0], in1p[1]);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, in2p[0], in2p[1]);
	    g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    setPowerColor(g, true);
	    drawThickPolygon(g, triangle);
	    g.setFont(plusFont);
	    drawCenteredText(g, "-", textp[0].x, textp[0].y-2, true);
	    drawCenteredText(g, "+", textp[1].x, textp[1].y  , true);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, lead2, point2);
	    curcount = updateDotCount(current, curcount);
	    drawDots(g, point2, lead2, curcount);
	    drawPosts(g);
	}
	double getPower() { return volts[2]*current; }
	Point in1p[], in2p[], textp[];
	Polygon triangle;
	Font plusFont;
	void setSize(int s) {
	    opsize = s;
	    opheight = 8*s;
	    opwidth = 13*s;
	    flags = (flags & ~FLAG_SMALL) | ((s == 1) ? FLAG_SMALL : 0);
	}
	void setPoints() {
	    super.setPoints();
	    if (dn > 150 && this == dragElm)
		setSize(2);
	    int ww = opwidth;
	    if (ww > dn/2)
		ww = (int) (dn/2);
	    calcLeads(ww*2);
	    int hs = opheight*dsign;
	    if ((flags & FLAG_SWAP) != 0)
		hs = -hs;
	    in1p = newPointArray(2);
	    in2p = newPointArray(2);
	    textp = newPointArray(2);
	    interpPoint2(point1, point2, in1p[0],  in2p[0], 0, hs);
	    interpPoint2(lead1 , lead2,  in1p[1],  in2p[1], 0, hs);
	    interpPoint2(lead1 , lead2,  textp[0], textp[1], .2, hs);
	    Point tris[] = newPointArray(2);
	    interpPoint2(lead1,  lead2,  tris[0], tris[1],  0, hs*2);
	    triangle = createPolygon(tris[0], tris[1], lead2);
	    plusFont = new Font("SansSerif", 0, opsize == 2 ? 14 : 10);
	}
	int getPostCount() { return 3; }
	Point getPost(int n) {
	    return (n == 0) ? in1p[0] : (n == 1) ? in2p[0] : point2;
	}
	int getVoltageSourceCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "op-amp";
	    arr[1] = "V+ = " + getVoltageText(volts[1]);
	    arr[2] = "V- = " + getVoltageText(volts[0]);
	    // sometimes the voltage goes slightly outside range, to make
	    // convergence easier.  so we hide that here.
	    double vo = Math.max(Math.min(volts[2], maxOut), minOut);
	    arr[3] = "Vout = " + getVoltageText(vo);
	    arr[4] = "Iout = " + getCurrentText(getCurrent());
	    arr[5] = "range = " + getVoltageText(minOut) + " to " +
		getVoltageText(maxOut);
	}

	double lastvd;

	void stamp() {
	    int vn = nodeList.size()+voltSource;
	    stampNonLinear(vn);
	    stampMatrix(nodes[2], vn, 1);
	}
	void doStep() {
	    double vd = volts[1] - volts[0];
	    if (Math.abs(lastvd-vd) > .1)
		converged = false;
	    else if (volts[2] > maxOut+.1 || volts[2] < minOut-.1)
		converged = false;
	    double x = 0;
	    int vn = nodeList.size()+voltSource;
	    double dx = 0;
	    if (vd >= maxOut/gain && (lastvd >= 0 || getrand(4) == 1)) {
		dx = 1e-4;
		x = maxOut - dx*maxOut/gain;
	    } else if (vd <= minOut/gain && (lastvd <= 0 || getrand(4) == 1)) {
		dx = 1e-4;
		x = minOut - dx*minOut/gain;
	    } else
		dx = gain;
	    //System.out.println("opamp " + vd + " " + volts[2] + " " + dx + " "  + x + " " + lastvd + " " + converged);
	    
	    // newton-raphson
	    stampMatrix(vn, nodes[0], dx);
	    stampMatrix(vn, nodes[1], -dx);
	    stampMatrix(vn, nodes[2], 1);
	    stampRightSide(vn, x);
	    
	    lastvd = vd;
	    /*if (converged)
	      System.out.println((volts[1]-volts[0]) + " " + volts[2] + " " + initvd);*/
	}
	// there is no current path through the op-amp inputs, but there
	// is an indirect path through the output to ground.
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return (n1 == 2);
	}
	double getVoltageDiff() { return volts[2] - volts[1]; }
	int getDumpType() { return 'a'; }
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Max Output (V)", maxOut, 1, 20);
	    if (n == 1)
		return new EditInfo("Min Output (V)", minOut, -20, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		maxOut = ei.value;
	    if (n == 1)
		minOut = ei.value;
	}
    }
    
    abstract class GateElm extends CircuitElm {
	final int FLAG_SMALL = 1;
	int inputCount = 2;
	boolean lastOutput;
	
	public GateElm(int xx, int yy) {
	    super(xx, yy);
	    noDiagonal = true;
	    inputCount = 2;
	    setSize(smallGridCheckItem.getState() ? 1 : 2);
	}
	public GateElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    inputCount = new Integer(st.nextToken()).intValue();
	    lastOutput = new Double (st.nextToken()).doubleValue() > 2.5;
	    noDiagonal = true;
	    setSize((f & FLAG_SMALL) != 0 ? 1 : 2);
	}
	boolean isInverting() { return false; }
	int gsize, gwidth, gwidth2, gheight, hs2;
	void setSize(int s) {
	    gsize = s;
	    gwidth = 7*s;
	    gwidth2 = 14*s;
	    gheight = 8*s;
	    flags = (s == 1) ? FLAG_SMALL : 0;
	}
	String dump() {
	    return super.dump() + " " + inputCount + " " + volts[inputCount];
	}
	Point inPosts[], inGates[];
	int ww;
	void setPoints() {
	    super.setPoints();
	    if (dn > 150 && this == dragElm)
		setSize(2);
	    int hs = gheight;
	    int i;
	    ww = gwidth2; // was 24
	    if (ww > dn/2)
		ww = (int) (dn/2);
	    if (isInverting() && ww+8 > dn/2)
		ww = (int) (dn/2-8);
	    calcLeads(ww*2);
	    inPosts = new Point[inputCount];
	    inGates = new Point[inputCount];
	    allocNodes();
	    int i0 = -inputCount/2;
	    for (i = 0; i != inputCount; i++, i0++) {
		if (i0 == 0 && (inputCount & 1) == 0)
		    i0++;
		inPosts[i] = interpPoint(point1, point2, 0, hs*i0);
		inGates[i] = interpPoint(lead1,  lead2,  0, hs*i0);
		volts[i] = (lastOutput ^ isInverting()) ? 5 : 0;
	    }
	    hs2 = gwidth*(inputCount/2+1);
	    setBbox(point1, point2, hs2);
	}
	void draw(Graphics g) {
	    int i;
	    for (i = 0; i != inputCount; i++) {
		setVoltageColor(g, volts[i]);
		drawThickLine(g, inPosts[i], inGates[i]);
	    }
	    setVoltageColor(g, volts[inputCount]);
	    drawThickLine(g, lead2, point2);
	    g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    drawThickPolygon(g, gatePoly);
	    if (linePoints != null)
		for (i = 0; i != linePoints.length-1; i++)
		    drawThickLine(g, linePoints[i], linePoints[i+1]);
	    if (isInverting())
		drawThickCircle(g, pcircle.x, pcircle.y, 3);
	    curcount = updateDotCount(current, curcount);
	    drawDots(g, lead2, point2, curcount);
	    drawPosts(g);
	}
	Polygon gatePoly;
	Point pcircle, linePoints[];
	int getPostCount() { return inputCount+1; }
	Point getPost(int n) {
	    if (n == inputCount)
		return point2;
	    return inPosts[n];
	}
	int getVoltageSourceCount() { return 1; }
	abstract String getGateName();
	void getInfo(String arr[]) {
	    arr[0] = getGateName();
	    arr[1] = "Vout = " + getVoltageText(volts[inputCount]);
	    arr[2] = "Iout = " + getCurrentText(getCurrent());
	}
	void stamp() {
	    stampVoltageSource(0, nodes[inputCount], voltSource);
	}
	boolean getInput(int x) {
	    return volts[x] > 2.5;
	}
	abstract boolean calcFunction();
	void doStep() {
	    int i;
	    boolean f = calcFunction();
	    if (isInverting())
		f = !f;
	    lastOutput = f;
	    double res = f ? 5 : 0;
	    updateVoltageSource(0, nodes[inputCount], voltSource, res);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Inputs", inputCount, 1, 8);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    inputCount = (int) ei.value;
	    setPoints();
	}
	// there is no current path through the gate inputs, but there
	// is an indirect path through the output to ground.
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return (n1 == inputCount);
	}
    }

    class AndGateElm extends GateElm {
	public AndGateElm(int xx, int yy) { super(xx, yy); }
	public AndGateElm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	void setPoints() {
	    super.setPoints();
	    
	    // 0=topleft, 1-10 = top curve, 11 = right, 12-21=bottom curve,
	    // 22 = bottom left
	    Point triPoints[] = newPointArray(23);
	    interpPoint2(lead1, lead2, triPoints[0], triPoints[22], 0, hs2);
	    int i;
	    for (i = 0; i != 10; i++) {
		double a = i*.1;
		double b = Math.sqrt(1-a*a);
		interpPoint2(lead1, lead2,
			     triPoints[i+1], triPoints[21-i],
			     .5+a/2, b*hs2);
	    }
	    triPoints[11] = new Point(lead2);
	    if (isInverting()) {
		pcircle = interpPoint(point1, point2, .5+(ww+4)/dn);
		lead2 = interpPoint(point1, point2, .5+(ww+8)/dn);
	    }
	    gatePoly = createPolygon(triPoints);
	}
	String getGateName() { return "AND gate"; }
	boolean calcFunction() {
	    int i;
	    boolean f = true;
	    for (i = 0; i != inputCount; i++)
		f &= getInput(i);
	    return f;
	}
	int getDumpType() { return 150; }
    }
    
    class NandGateElm extends AndGateElm {
	public NandGateElm(int xx, int yy) { super(xx, yy); }
	public NandGateElm(int xa, int ya, int xb, int yb, int f,
			   StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	boolean isInverting() { return true; }
	String getGateName() { return "NAND gate"; }
	int getDumpType() { return 151; }
    }
    
    class OrGateElm extends GateElm {
	public OrGateElm(int xx, int yy) { super(xx, yy); }
	public OrGateElm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getGateName() { return "OR gate"; }
	void setPoints() {
	    super.setPoints();

	    // 0-15 = top curve, 16 = right, 17-32=bottom curve,
	    // 33-37 = left curve
	    Point triPoints[] = newPointArray(38);
	    if (this instanceof XorGateElm)
		linePoints = new Point[5];
	    int i;
	    for (i = 0; i != 16; i++) {
		double a = i/16.;
		double b = 1-a*a;
		interpPoint2(lead1, lead2,
			     triPoints[i], triPoints[32-i],
			     .5+a/2, b*hs2);
	    }
	    double ww2 = (ww == 0) ? dn*2 : ww*2;
	    for (i = 0; i != 5; i++) {
		double a = (i-2)/2.;
		double b = 4*(1-a*a)-2;
		interpPoint(lead1, lead2,
			    triPoints[33+i], b/(ww2), a*hs2);
		if (this instanceof XorGateElm)
		    linePoints[i] = interpPoint(lead1, lead2,
						(b-5)/(ww2), a*hs2);
	    }
	    triPoints[16] = new Point(lead2);
	    if (isInverting()) {
		pcircle = interpPoint(point1, point2, .5+(ww+4)/dn);
		lead2 = interpPoint(point1, point2, .5+(ww+8)/dn);
	    }
	    gatePoly = createPolygon(triPoints);
	}
	boolean calcFunction() {
	    int i;
	    boolean f = false;
	    for (i = 0; i != inputCount; i++)
		f |= getInput(i);
	    return f;
	}
	int getDumpType() { return 152; }
    }
    
    class NorGateElm extends OrGateElm {
	public NorGateElm(int xx, int yy) { super(xx, yy); }
	public NorGateElm(int xa, int ya, int xb, int yb, int f,
			   StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getGateName() { return "NOR gate"; }
	boolean isInverting() { return true; }
	int getDumpType() { return 153; }
    }
    
    class XorGateElm extends OrGateElm {
	public XorGateElm(int xx, int yy) { super(xx, yy); }
	public XorGateElm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getGateName() { return "XOR gate"; }
	boolean calcFunction() {
	    int i;
	    boolean f = false;
	    for (i = 0; i != inputCount; i++)
		f ^= getInput(i);
	    return f;
	}
	int getDumpType() { return 154; }
    }
    
    class TransformerElm extends CircuitElm {
	double inductance, ratio;
	Point ptEnds[], ptCoil[], ptCore[];
	double current[], curcount[];
	int width;
	public TransformerElm(int xx, int yy) {
	    super(xx, yy);
	    inductance = 4;
	    ratio = 1;
	    width = 32;
	    noDiagonal = true;
	    current  = new double[2];
	    curcount = new double[2];
	}
	public TransformerElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    width = 32;
	    inductance = new Double(st.nextToken()).doubleValue();
	    ratio = new Double(st.nextToken()).doubleValue();
	    current  = new double[2];
	    curcount = new double[2];
	    current[0] = new Double(st.nextToken()).doubleValue();
	    current[1] = new Double(st.nextToken()).doubleValue();
	    noDiagonal = true;
	}
	void drag(int xx, int yy) {
	    xx = snapGrid(xx);
	    yy = snapGrid(yy);
	    width = max(32, abs(yy-y));
	    yy = y;
	    x2 = xx; y2 = yy;
	    setPoints();
	}
	int getDumpType() { return 'T'; }
	String dump() {
	    return super.dump() + " " + inductance + " " + ratio + " " +
		current[0] + " " + current[1];
	}
	void draw(Graphics g) {
	    int i;
	    for (i = 0; i != 4; i++) {
		setVoltageColor(g, volts[i]);
		drawThickLine(g, ptEnds[i], ptCoil[i]);
	    }
	    for (i = 0; i != 2; i++) {
		setPowerColor(g, current[i]*(volts[i]-volts[i+2]));
		drawCoil(g, dsign*(i == 1 ? -6 : 6),
			 ptCoil[i], ptCoil[i+2], volts[i], volts[i+2]);
	    }
	    g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    for (i = 0; i != 2; i++) {
		drawThickLine(g, ptCore[i], ptCore[i+2]);
		curcount[i] = updateDotCount(current[i], curcount[i]);
	    }
	    for (i = 0; i != 2; i++) {
		drawDots(g, ptEnds[i],   ptCoil[i],    curcount[i]);
		drawDots(g, ptCoil[i],   ptCoil[i+2],  curcount[i]);
		drawDots(g, ptEnds[i+2], ptCoil[i+2],  -curcount[i]);
	    }
	    
	    drawPosts(g);
	    setBbox(ptEnds[0], ptEnds[3], 0);
	}
	
	void setPoints() {
	    super.setPoints();
	    ptEnds = newPointArray(4);
	    ptCoil = newPointArray(4);
	    ptCore = newPointArray(4);
	    ptEnds[0] = point1;
	    ptEnds[1] = point2;
	    interpPoint(point1, point2, ptEnds[2], 0, -dsign*width);
	    interpPoint(point1, point2, ptEnds[3], 1, -dsign*width);
	    double ce = .5-12/dn;
	    double cd = .5-2/dn;
	    int i;
	    for (i = 0; i != 4; i += 2) {
		interpPoint(ptEnds[i], ptEnds[i+1], ptCoil[i],   ce);
		interpPoint(ptEnds[i], ptEnds[i+1], ptCoil[i+1], 1-ce);
		interpPoint(ptEnds[i], ptEnds[i+1], ptCore[i],   cd);
		interpPoint(ptEnds[i], ptEnds[i+1], ptCore[i+1], 1-cd);
	    }
	}
	Point getPost(int n) {
	    return ptEnds[n];
	}
	int getPostCount() { return 4; }
	void reset() {
	    current[0] = current[1] = volts[0] = volts[1] = volts[2] =
		volts[3] = curcount[0] = curcount[1] = 0;
	}
	double a1, a2, a3, a4;
	void stamp() {
	    // equations for transformer:
	    //   v1 = L1 di1/dt + M  di2/dt
	    //   v2 = M  di1/dt + L2 di2/dt
	    // we invert that to get:
	    //   di1/dt = a1 v1 + a2 v2
	    //   di2/dt = a3 v1 + a4 v2
	    // integrate di1/dt using trapezoidal approx and we get:
	    //   i1(t2) = i1(t1) + dt/2 (i1(t1) + i1(t2))
	    //          = i1(t1) + a1 dt/2 v1(t1) + a2 dt/2 v2(t1) +
	    //                     a1 dt/2 v1(t2) + a2 dt/2 v2(t2)
	    // the norton equivalent of this for i1 is:
	    //  a. current source, I = i1(t1) + a1 dt/2 v1(t1) + a2 dt/2 v2(t1)
	    //  b. resistor, G = a1 dt/2
	    //  c. current source controlled by voltage v2, G = a2 dt/2
	    // and for i2:
	    //  a. current source, I = i2(t1) + a3 dt/2 v1(t1) + a4 dt/2 v2(t1)
	    //  b. resistor, G = a3 dt/2
	    //  c. current source controlled by voltage v2, G = a4 dt/2
	    // 
	    // first winding goes from node 0 to 2, second is from 1 to 3
	    double l1 = inductance;
	    double l2 = inductance*ratio*ratio;
	    double m = .999*Math.sqrt(l1*l2);
	    double deti = 1/(l1*l2-m*m);
	    a1 = l2*deti*timeStep/2; // we multiply dt/2 into a1..a4 here
	    a2 = -m*deti*timeStep/2;
	    a3 = -m*deti*timeStep/2;
	    a4 = l1*deti*timeStep/2;
	    stampConductance(nodes[0], nodes[2], a1);
	    stampVCCurrentSource(nodes[0], nodes[2], nodes[1], nodes[3], a2);
	    stampVCCurrentSource(nodes[1], nodes[3], nodes[0], nodes[2], a3);
	    stampConductance(nodes[1], nodes[3], a4);
	    stampRightSide(nodes[0]);
	    stampRightSide(nodes[1]);
	    stampRightSide(nodes[2]);
	    stampRightSide(nodes[3]);
	}
	void startIteration() {
	    double voltdiff1 = volts[0]-volts[2];
	    double voltdiff2 = volts[1]-volts[3];
	    curSourceValue1 = voltdiff1*a1+voltdiff2*a2+current[0];
	    curSourceValue2 = voltdiff1*a3+voltdiff2*a4+current[1];
	}
	double curSourceValue1, curSourceValue2;
	void doStep() {
	    stampCurrentSource(nodes[0], nodes[2], curSourceValue1);
	    stampCurrentSource(nodes[1], nodes[3], curSourceValue2);
 	}
	void calculateCurrent() {
	    double voltdiff1 = volts[0]-volts[2];
	    double voltdiff2 = volts[1]-volts[3];
	    current[0] = voltdiff1*a1 + voltdiff2*a2 + curSourceValue1;
	    current[1] = voltdiff1*a3 + voltdiff2*a4 + curSourceValue2;
	}
	void getInfo(String arr[]) {
	    arr[0] = "transformer";
	    arr[1] = "L = " + getUnitText(inductance, "H");
	    arr[2] = "Ratio = 1:" + ratio;
	    //arr[3] = "I1 = " + getCurrentText(current1);
	    arr[3] = "Vd1 = " + getVoltageText(volts[0]-volts[2]);
	    //arr[5] = "I2 = " + getCurrentText(current2);
	    arr[4] = "Vd2 = " + getVoltageText(volts[1]-volts[3]);
	}
	boolean getConnection(int n1, int n2) {
	    if (comparePair(n1, n2, 0, 2))
		return true;
	    if (comparePair(n1, n2, 1, 3))
		return true;
	    return false;
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Inductance (H)", inductance, .01, 5);
	    if (n == 1)
		return new EditInfo("Ratio", ratio, 1, 10);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		inductance = ei.value;
	    if (n == 1)
		ratio = ei.value;
	}
    }

    class TappedTransformerElm extends CircuitElm {
	double inductance, ratio;
	Point ptEnds[], ptCoil[], ptCore[];
	double current[], curcount[];
	public TappedTransformerElm(int xx, int yy) {
	    super(xx, yy);
	    inductance = 4;
	    ratio = 1;
	    noDiagonal = true;
	    current  = new double[4];
	    curcount = new double[4];
	}
	public TappedTransformerElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    inductance = new Double(st.nextToken()).doubleValue();
	    ratio = new Double(st.nextToken()).doubleValue();
	    current  = new double[2];
	    curcount = new double[2];
	    current[0] = new Double(st.nextToken()).doubleValue();
	    current[1] = new Double(st.nextToken()).doubleValue();
	    noDiagonal = true;
	}
	int getDumpType() { return 169; }
	String dump() {
	    return super.dump() + " " + inductance + " " + ratio + " " +
		current[0] + " " + current[1];
	}
	void draw(Graphics g) {
	    int i;
	    for (i = 0; i != 5; i++) {
		setVoltageColor(g, volts[i]);
		drawThickLine(g, ptEnds[i], ptCoil[i]);
	    }
	    for (i = 0; i != 4; i++) {
		if (i == 1)
		    continue;
		setPowerColor(g, current[i]*(volts[i]-volts[i+1]));
		drawCoil(g, i > 1 ? -6 : 6,
			 ptCoil[i], ptCoil[i+1], volts[i], volts[i+1]);
	    }
	    g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    for (i = 0; i != 4; i += 2) {
		drawThickLine(g, ptCore[i], ptCore[i+1]);
		//curcount[i] = updateDotCount(current[i], curcount[i]);
	    }
	    /*for (i = 0; i != 2; i++) {
		drawDots(g, ptEnds[i],   ptCoil[i],    curcount[i]);
		drawDots(g, ptCoil[i],   ptCoil[i+2],  curcount[i]);
		drawDots(g, ptEnds[i+2], ptCoil[i+2],  -curcount[i]);
		}*/
	    
	    drawPosts(g);
	    setBbox(ptEnds[0], ptEnds[4], 0);
	}
	
	void setPoints() {
	    super.setPoints();
	    int hs = 32;
	    ptEnds = newPointArray(5);
	    ptCoil = newPointArray(5);
	    ptCore = newPointArray(4);
	    ptEnds[0] = point1;
	    ptEnds[2] = point2;
	    interpPoint(point1, point2, ptEnds[1], 0, -hs*2);
	    interpPoint(point1, point2, ptEnds[3], 1, -hs);
	    interpPoint(point1, point2, ptEnds[4], 1, -hs*2);
	    double ce = .5-12/dn;
	    double cd = .5-2/dn;
	    int i;
	    interpPoint(ptEnds[0], ptEnds[2], ptCoil[0], ce);
	    interpPoint(ptEnds[0], ptEnds[2], ptCoil[1], ce, -hs*2);
	    interpPoint(ptEnds[0], ptEnds[2], ptCoil[2], 1-ce);
	    interpPoint(ptEnds[0], ptEnds[2], ptCoil[3], 1-ce, -hs);
	    interpPoint(ptEnds[0], ptEnds[2], ptCoil[4], 1-ce, -hs*2);
	    for (i = 0; i != 2; i++) {
		int b = -hs*i*2;
		interpPoint(ptEnds[0], ptEnds[2], ptCore[i],   cd,   b);
		interpPoint(ptEnds[0], ptEnds[2], ptCore[i+2], 1-cd, b);
	    }
	}
	Point getPost(int n) {
	    return ptEnds[n];
	}
	int getPostCount() { return 5; }
	void reset() {
	    current[0] = current[1] = volts[0] = volts[1] = volts[2] =
		volts[3] = curcount[0] = curcount[1] = 0;
	}
	double a[];
	void stamp() {
	    // equations for transformer:
	    //   v1 = L1 di1/dt + M1 di2/dt + M1 di3/dt
	    //   v2 = M1 di1/dt + L2 di2/dt + M2 di3/dt
	    //   v3 = M1 di1/dt + M2 di2/dt + L2 di3/dt
	    // we invert that to get:
	    //   di1/dt = a1 v1 + a2 v2 + a3 v3
	    //   di2/dt = a4 v1 + a5 v2 + a6 v3
	    //   di3/dt = a7 v1 + a8 v2 + a9 v3
	    // integrate di1/dt using trapezoidal approx and we get:
	    //   i1(t2) = i1(t1) + dt/2 (i1(t1) + i1(t2))
	    //          = i1(t1) + a1 dt/2 v1(t1)+a2 dt/2 v2(t1)+a3 dt/2 v3(t3) +
	    //                     a1 dt/2 v1(t2)+a2 dt/2 v2(t2)+a3 dt/2 v3(t3)
	    // the norton equivalent of this for i1 is:
	    //  a. current source, I = i1(t1) + a1 dt/2 v1(t1) + a2 dt/2 v2(t1)
	    //                                + a3 dt/2 v3(t1)
	    //  b. resistor, G = a1 dt/2
	    //  c. current source controlled by voltage v2, G = a2 dt/2
	    //  d. current source controlled by voltage v3, G = a3 dt/2
	    // and similarly for i2
	    // 
	    // first winding goes from node 0 to 1, second is from 2 to 3 to 4
	    double l1 = inductance;
	    // second winding is split in half, so each part has half the turns;
	    // we square the 1/2 to divide by 4
	    double l2 = inductance*ratio*ratio/4;
	    double cc = .99;
	    //double m1 = .999*Math.sqrt(l1*l2);
	    // mutual inductance between two halves of the second winding
	    // is equal to self-inductance of either half (slightly less
	    // because the coupling is not perfect)
	    //double m2 = .999*l2;
	    a = new double[9];
	    a[0] = (1+cc)/(l1*(1+cc-2*cc*cc));
	    a[1] = a[2] = a[3] = a[6] = 2*cc/((2*cc*cc-cc-1)*inductance*ratio);
	    a[4] = a[8] = -4*(1+cc)/((2*cc*cc-cc-1)*l1*ratio*ratio);
	    a[5] = a[7] = 4*cc/((2*cc*cc-cc-1)*l1*ratio*ratio);
	    int i;
	    for (i = 0; i != 9; i++)
		a[i] *= timeStep/2;
	    stampConductance(nodes[0], nodes[1], a[0]);
	    stampVCCurrentSource(nodes[0], nodes[1], nodes[2], nodes[3], a[1]);
	    stampVCCurrentSource(nodes[0], nodes[1], nodes[3], nodes[4], a[2]);
	    
	    stampVCCurrentSource(nodes[2], nodes[3], nodes[0], nodes[1], a[3]);
	    stampConductance    (nodes[2], nodes[3], a[4]);
	    stampVCCurrentSource(nodes[2], nodes[3], nodes[3], nodes[4], a[5]);
	    
	    stampVCCurrentSource(nodes[3], nodes[4], nodes[0], nodes[1], a[6]);
	    stampVCCurrentSource(nodes[3], nodes[4], nodes[2], nodes[3], a[7]);
	    stampConductance    (nodes[3], nodes[4], a[8]);

	    for (i = 0; i != 5; i++)
		stampRightSide(nodes[i]);
	    voltdiff = new double[3];
	    curSourceValue = new double[3];
	}
	void startIteration() {
	    voltdiff[0] = volts[0]-volts[1];
	    voltdiff[1] = volts[2]-volts[3];
	    voltdiff[2] = volts[3]-volts[4];
	    int i, j;
	    for (i = 0; i != 3; i++) {
		curSourceValue[i] = current[i];
		for (j = 0; j != 3; j++)
		    curSourceValue[i] += a[i*3+j]*voltdiff[j];
	    }
	}
	double curSourceValue[], voltdiff[];
	void doStep() {
	    stampCurrentSource(nodes[0], nodes[1], curSourceValue[0]);
	    stampCurrentSource(nodes[2], nodes[3], curSourceValue[1]);
	    stampCurrentSource(nodes[3], nodes[4], curSourceValue[2]);
 	}
	void calculateCurrent() {
	    voltdiff[0] = volts[0]-volts[1];
	    voltdiff[1] = volts[2]-volts[3];
	    voltdiff[2] = volts[3]-volts[4];
	    int i, j;
	    for (i = 0; i != 3; i++) {
		current[i] = curSourceValue[i];
		for (j = 0; j != 3; j++)
		    current[i] += a[i*3+j]*voltdiff[j];
	    }
	}
	void getInfo(String arr[]) {
	    arr[0] = "transformer";
	    arr[1] = "L = " + getUnitText(inductance, "H");
	    arr[2] = "Ratio = " + ratio;
	    //arr[3] = "I1 = " + getCurrentText(current1);
	    arr[3] = "Vd1 = " + getVoltageText(volts[0]-volts[2]);
	    //arr[5] = "I2 = " + getCurrentText(current2);
	    arr[4] = "Vd2 = " + getVoltageText(volts[1]-volts[3]);
	}
	boolean getConnection(int n1, int n2) {
	    if (comparePair(n1, n2, 0, 1))
		return true;
	    if (comparePair(n1, n2, 2, 3))
		return true;
	    if (comparePair(n1, n2, 3, 4))
		return true;
	    if (comparePair(n1, n2, 2, 4))
		return true;
	    return false;
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Inductance (H)", inductance, .01, 5);
	    if (n == 1)
		return new EditInfo("Ratio", ratio, 1, 10);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		inductance = ei.value;
	    if (n == 1)
		ratio = ei.value;
	}
    }

    class ACRailElm extends RailElm {
	public ACRailElm(int xx, int yy) { super(xx, yy, WF_AC); }
	Class getDumpClass() { return RailElm.class; }
    }
    class SquareRailElm extends RailElm {
	public SquareRailElm(int xx, int yy) { super(xx, yy, WF_SQUARE); }
	Class getDumpClass() { return RailElm.class; }
    }
    class ClockElm extends RailElm {
	public ClockElm(int xx, int yy) {
	    super(xx, yy, WF_SQUARE);
	    maxVoltage = 2.5;
	    bias = 2.5;
	    frequency = 100;
	    flags |= FLAG_CLOCK;
	}
	Class getDumpClass() { return RailElm.class; }
    }
    
    class RailElm extends VoltageElm {
	public RailElm(int xx, int yy) { super(xx, yy, WF_DC); }
	RailElm(int xx, int yy, int wf) { super(xx, yy, wf); }
	public RailElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	final int FLAG_CLOCK = 1;
	int getDumpType() { return 'R'; }
	int getPostCount() { return 1; }
	
	void setPoints() {
	    super.setPoints();
	    lead1 = interpPoint(point1, point2, 1-circleSize/dn);
	}
	void draw(Graphics g) {
	    setBbox(point1, point2, circleSize);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);
	    boolean clock = waveform == WF_SQUARE && (flags & FLAG_CLOCK) != 0;
	    if (waveform == WF_DC || waveform == WF_VAR || clock) {
		Font f = new Font("SansSerif", 0, 12);
		g.setFont(f);
		g.setColor(this == mouseElm ? selectColor : whiteColor);
		setPowerColor(g, false);
		String s = showFormat.format(getVoltage()) + "V";
		if (getVoltage() > 0)
		    s = "+" + s;
		if (this instanceof AntennaElm)
		    s = "Ant";
		if (clock)
		    s = "CLK";
		drawCenteredText(g, s, x2, y2, true);
	    } else {
		drawWaveform(g, point2);
	    }
	    drawPosts(g);
	    curcount = updateDotCount(-current, curcount);
	    if (dragElm != this)
		drawDots(g, point1, lead1, curcount);
	}
	double getVoltageDiff() { return volts[0]; }
	void stamp() {
	    if (waveform == WF_DC)
		stampVoltageSource(0, nodes[0], voltSource, getVoltage());
	    else
		stampVoltageSource(0, nodes[0], voltSource);
	}
	void doStep() {
	    if (waveform != WF_DC)
		updateVoltageSource(0, nodes[0], voltSource, getVoltage());
	}
	boolean hasGroundConnection(int n1) { return true; }
    }

    class VarRailElm extends RailElm {
	Scrollbar slider;
	Label label;
	String sliderText;
	public VarRailElm(int xx, int yy) {
	    super(xx, yy, WF_VAR);
	    sliderText = "Voltage";
	    frequency = maxVoltage;
	    createSlider();
	}
	public VarRailElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    sliderText = st.nextToken();
	    while (st.hasMoreTokens())
		sliderText += ' ' + st.nextToken();
	    createSlider();
	}
	String dump() {
	    return super.dump() + " " + sliderText;
	}
	int getDumpType() { return 172; }
	void createSlider() {
	    waveform = WF_VAR;
	    main.add(label = new Label(sliderText, Label.CENTER));
	    int value = (int) ((frequency-bias)*100/(maxVoltage-bias));
	    main.add(slider = new Scrollbar(Scrollbar.HORIZONTAL, value, 1, 0, 101));
	    main.validate();
	}
	double getVoltage() {
	    frequency = slider.getValue() * (maxVoltage-bias) / 100. + bias;
	    return frequency;
	}
	void delete() {
	    main.remove(label);
	    main.remove(slider);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Min Voltage", bias, -20, 20);
	    if (n == 1)
		return new EditInfo("Max Voltage", maxVoltage, -20, 20);
	    if (n == 2) {
		EditInfo ei = new EditInfo("Slider Text", 0, -1, -1);
		ei.text = sliderText;
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		bias = ei.value;
	    if (n == 1)
		maxVoltage = ei.value;
	    if (n == 2) {
		sliderText = ei.textf.getText();
		label.setText(sliderText);
	    }
	}
    }
    
    class GroundElm extends CircuitElm {
	public GroundElm(int xx, int yy) { super(xx, yy); }
	public GroundElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 'g'; }
	int getPostCount() { return 1; }
	void draw(Graphics g) {
	    setVoltageColor(g, 0);
	    drawThickLine(g, point1, point2);
	    int i;
	    for (i = 0; i != 3; i++) {
		int a = 10-i*4;
		int b = i*5; // -10;
		interpPoint2(point1, point2, ps1, ps2, 1+b/dn, a);
		drawThickLine(g, ps1, ps2);
	    }
	    doDots(g);
	    interpPoint(point1, point2, ps2, 1+11./dn);
	    setBbox(point1, ps2, 11);
	    drawPost(g, x, y, nodes[0]);
	}
	void setCurrent(int x, double c) { current = -c; }
	void stamp() {
	    stampVoltageSource(0, nodes[0], voltSource, 0);
	}
	double getVoltageDiff() { return 0; }
	int getVoltageSourceCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "ground";
	    arr[1] = "I = " + getCurrentText(getCurrent());
	}
	boolean hasGroundConnection(int n1) { return true; }
    }

    class AntennaElm extends RailElm {
	public AntennaElm(int xx, int yy) { super(xx, yy, WF_DC); }
	public AntennaElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    waveform = WF_DC;
	}
	double fmphase;
	void stamp() {
	    stampVoltageSource(0, nodes[0], voltSource);
	}
	void doStep() {
	    updateVoltageSource(0, nodes[0], voltSource, getVoltage());
	}
	double getVoltage() {
	    fmphase += 2*pi*(2200+Math.sin(2*pi*t*13)*100)*timeStep;
	    double fm = 3*Math.sin(fmphase);
	    return Math.sin(2*pi*t*3000)*(1.3+Math.sin(2*pi*t*12))*3 +
	           Math.sin(2*pi*t*2710)*(1.3+Math.sin(2*pi*t*13))*3 +
		   Math.sin(2*pi*t*2433)*(1.3+Math.sin(2*pi*t*14))*3 + fm;
	}
	int getDumpType() { return 'A'; }
    }

    class LogicInputElm extends SwitchElm {
	final int FLAG_TERNARY = 1;
	final int FLAG_NUMERIC = 2;
	double hiV, loV;
	public LogicInputElm(int xx, int yy) {
	    super(xx, yy, false);
	    hiV = 5;
	    loV = 0;
	}
	public LogicInputElm(int xa, int ya, int xb, int yb, int f,
			     StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    try {
		hiV = new Double(st.nextToken()).doubleValue();
		loV = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		hiV = 5;
		loV = 0;
	    }
	    if (isTernary())
		posCount = 3;
	}
	boolean isTernary() { return (flags & FLAG_TERNARY) != 0; }
	boolean isNumeric() { return (flags & (FLAG_TERNARY|FLAG_NUMERIC)) != 0; }
	int getDumpType() { return 'L'; }
	String dump() {
	    return super.dump() + " " + hiV + " " + loV;
	}
	int getPostCount() { return 1; }
	void setPoints() {
	    super.setPoints();
	    lead1 = interpPoint(point1, point2, 1-12/dn);
	}
	void draw(Graphics g) {
	    Font f = new Font("SansSerif", Font.BOLD, 20);
	    g.setFont(f);
	    g.setColor(this == mouseElm ? selectColor : whiteColor);
	    String s = position == 0 ? "L" : "H";
	    if (isNumeric())
		s = "" + position;
	    setBbox(point1, lead1, 0);
	    drawCenteredText(g, s, x2, y2, true);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);
	    updateDotCount();
	    drawDots(g, point1, lead1, curcount);
	    drawPosts(g);
	}
	void setCurrent(int vs, double c) { current = -c; }
	void stamp() {
	    double v = (position == 0) ? loV : hiV;
	    if (isTernary())
		v = position * 2.5;
	    stampVoltageSource(0, nodes[0], voltSource, v);
	}
	int getVoltageSourceCount() { return 1; }
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "logic input";
	    arr[1] = (position == 0) ? "low" : "high";
	    if (isNumeric())
		arr[1] = "" + position;
	    arr[1] += " (" + getVoltageText(volts[0]) + ")";
	    arr[2] = "I = " + getCurrentText(getCurrent());
	}
	boolean hasGroundConnection(int n1) { return true; }
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, 0, 0);
		ei.checkbox = new Checkbox("Momentary Switch", momentary);
		return ei;
	    }
	    if (n == 1)
		return new EditInfo("High Voltage", hiV, 10, -10);
	    if (n == 2)
		return new EditInfo("Low Voltage", loV, 10, -10);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		momentary = ei.checkbox.getState();
	    if (n == 1)
		hiV = ei.value;
	    if (n == 2)
		loV = ei.value;
	}
    }

    class LogicOutputElm extends CircuitElm {
	final int FLAG_TERNARY = 1;
	final int FLAG_NUMERIC = 2;
	double threshold;
	String value;
	public LogicOutputElm(int xx, int yy) {
	    super(xx, yy);
	    threshold = 2.5;
	}
	public LogicOutputElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    try {
		threshold = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		threshold = 2.5;
	    }
	}
	String dump() {
	    return super.dump() + " " + threshold;
	}
	int getDumpType() { return 'M'; }
	int getPostCount() { return 1; }
	boolean isTernary() { return (flags & FLAG_TERNARY) != 0; }
	boolean isNumeric() { return (flags & (FLAG_TERNARY|FLAG_NUMERIC)) != 0; }
	void setPoints() {
	    super.setPoints();
	    lead1 = interpPoint(point1, point2, 1-12/dn);
	}
	void draw(Graphics g) {
	    Font f = new Font("SansSerif", Font.BOLD, 20);
	    g.setFont(f);
	    //g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    g.setColor(lightGrayColor);
	    String s = (volts[0] < threshold) ? "L" : "H";
	    if (isTernary()) {
		if (volts[0] > 3.75)
		    s = "2";
		else if (volts[0] > 1.25)
		    s = "1";
		else
		    s = "0";
	    } else if (isNumeric())
		s = (volts[0] < threshold) ? "0" : "1";
	    value = s;
	    setBbox(point1, lead1, 0);
	    drawCenteredText(g, s, x2, y2, true);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);
	    drawPosts(g);
	}
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "logic output";
	    arr[1] = (volts[0] < threshold) ? "low" : "high";
	    if (isNumeric())
		arr[1] = value;
	    arr[2] = "V = " + getVoltageText(volts[0]);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Threshold", threshold, 10, -10);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		threshold = ei.value;
	}
    }

    class OutputElm extends CircuitElm {
	final int FLAG_VALUE = 1;
	public OutputElm(int xx, int yy) { super(xx, yy); }
	public OutputElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 'O'; }
	int getPostCount() { return 1; }
	void setPoints() {
	    super.setPoints();
	    lead1 = new Point();
	}
	void draw(Graphics g) {
	    boolean selected = (mouseElm == this || plotYElm == this);
	    Font f = new Font("SansSerif", selected ? Font.BOLD : 0, 14);
	    g.setFont(f);
	    g.setColor(selected ? selectColor : whiteColor);
	    String s = (flags & FLAG_VALUE) != 0 ? getVoltageText(volts[0]) : "out";
	    FontMetrics fm = g.getFontMetrics();
	    if (this == plotXElm)
		s = "X";
	    if (this == plotYElm)
		s = "Y";
	    interpPoint(point1, point2, lead1, 1-(fm.stringWidth(s)/2+8)/dn);
	    setBbox(point1, lead1, 0);
	    drawCenteredText(g, s, x2, y2, true);
	    setVoltageColor(g, volts[0]);
	    if (selected)
		g.setColor(selectColor);
	    drawThickLine(g, point1, lead1);
	    drawPosts(g);
	}
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "output";
	    arr[1] = "V = " + getVoltageText(volts[0]);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Show Voltage",
					   (flags & FLAG_VALUE) != 0);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		flags = (ei.checkbox.getState()) ?
		    (flags | FLAG_VALUE) :
		    (flags & ~FLAG_VALUE);
	}
    }

    class InverterElm extends CircuitElm {
	double slewRate; // V/ns
	public InverterElm(int xx, int yy) {
	    super(xx, yy);
	    noDiagonal = true;
	    slewRate = .5;
	}
	public InverterElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    noDiagonal = true;
	    try {
		slewRate = new Double (st.nextToken()).doubleValue();
	    } catch (Exception e) {
		slewRate = .5;
	    }
	}
	String dump() {
	    return super.dump() + " " + slewRate;
	}
	
	int getDumpType() { return 'I'; }
	void draw(Graphics g) {
	    drawPosts(g);
	    draw2Leads(g);
	    g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    drawThickPolygon(g, gatePoly);
	    drawThickCircle(g, pcircle.x, pcircle.y, 3);
	    curcount = updateDotCount(current, curcount);
	    drawDots(g, lead2, point2, curcount);
	}
	Polygon gatePoly;
	Point pcircle;
	void setPoints() {
	    super.setPoints();
	    int hs = 16;
	    int ww = 16;
	    if (ww > dn/2)
		ww = (int) (dn/2);
	    lead1 = interpPoint(point1, point2, .5-ww/dn);
	    lead2 = interpPoint(point1, point2, .5+(ww+2)/dn);
	    pcircle = interpPoint(point1, point2, .5+(ww-2)/dn);
	    Point triPoints[] = newPointArray(3);
	    interpPoint2(lead1, lead2, triPoints[0], triPoints[1], 0, hs);
	    triPoints[2] = interpPoint(point1, point2, .5+(ww-5)/dn);
	    gatePoly = createPolygon(triPoints);
	    setBbox(point1, point2, hs);
	}
	int getVoltageSourceCount() { return 1; }
	void stamp() {
	    stampVoltageSource(0, nodes[1], voltSource);
	}
	void doStep() {
	    double v0 = volts[1];
	    double out = volts[0] > 2.5 ? 0 : 5;
	    double maxStep = slewRate * timeStep * 1e9;
	    out = Math.max(Math.min(v0+maxStep, out), v0-maxStep);
	    updateVoltageSource(0, nodes[1], voltSource, out);
	}
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "inverter";
	    arr[1] = "Vi = " + getVoltageText(volts[0]);
	    arr[2] = "Vo = " + getVoltageText(volts[1]);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Slew Rate (V/ns)", slewRate, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    slewRate = ei.value;
	}
	// there is no current path through the inverter input, but there
	// is an indirect path through the output to ground.
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return (n1 == 1);
	}
    }

    abstract class ChipElm extends CircuitElm {
	int csize, cspc, cspc2;
	int bits;
	final int FLAG_SMALL = 1;
	public ChipElm(int xx, int yy) {
	    super(xx, yy);
	    if (needsBits())
		bits = (this instanceof DecadeElm) ? 10 : 4;
	    noDiagonal = true;
	    setupPins();
	    setSize(smallGridCheckItem.getState() ? 1 : 2);
	}
	public ChipElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    if (needsBits())
		bits = new Integer(st.nextToken()).intValue();
	    noDiagonal = true;
	    setupPins();
	    setSize((f & FLAG_SMALL) != 0 ? 1 : 2);
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		if (pins[i].state) {
		    volts[i] = new Double(st.nextToken()).doubleValue();
		    pins[i].value = volts[i] > 2.5;
		}
	    }
	}
	boolean needsBits() { return false; }
	void setSize(int s) {
	    csize = s;
	    cspc = 8*s;
	    cspc2 = cspc*2;
	    flags &= ~FLAG_SMALL;
	    flags |= (s == 1) ? FLAG_SMALL : 0;
	}
	abstract void setupPins();
	void draw(Graphics g) {
	    drawChip(g);
	}
	void drawChip(Graphics g) {
	    int i;
	    Font f = new Font("SansSerif", 0, 10*csize);
	    g.setFont(f);
	    FontMetrics fm = g.getFontMetrics();
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		setVoltageColor(g, volts[i]);
		Point a = p.post;
		Point b = p.stub;
		drawThickLine(g, a, b);
		p.curcount = updateDotCount(p.current, p.curcount);
		drawDots(g, b, a, p.curcount);
		if (p.bubble) {
		    g.setColor(printableCheckItem.getState() ?
			       Color.white : Color.black);
		    drawThickCircle(g, p.bubbleX, p.bubbleY, 1);
		    g.setColor(lightGrayColor);
		    drawThickCircle(g, p.bubbleX, p.bubbleY, 3);
		}
		g.setColor(whiteColor);
		int sw = fm.stringWidth(p.text);
		g.drawString(p.text, p.textloc.x-sw/2,
			     p.textloc.y+fm.getAscent()/2);
		if (p.lineOver) {
		    int ya = p.textloc.y-fm.getAscent()/2;
		    g.drawLine(p.textloc.x-sw/2, ya, p.textloc.x+sw/2, ya);
		}
	    }
	    g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    drawThickPolygon(g, rectPointsX, rectPointsY, 4);
	    if (clockPointsX != null)
		g.drawPolyline(clockPointsX, clockPointsY, 3);
	    for (i = 0; i != getPostCount(); i++)
		drawPost(g, pins[i].post.x, pins[i].post.y, nodes[i]);
	}
	int rectPointsX[], rectPointsY[];
	int clockPointsX[], clockPointsY[];
	Pin pins[];
	int sizeX, sizeY;
	boolean lastClock;
	void drag(int xx, int yy) {
	    yy = snapGrid(yy);
	    if (xx < x) {
		xx = x; yy = y;
	    } else {
		y = y2 = yy;
		x2 = snapGrid(xx);
	    }
	    setPoints();
	}
	void setPoints() {
	    if (x2-x > sizeX*cspc2 && this == dragElm)
		setSize(2);
	    int hs = cspc;
	    int x0 = x+cspc2; int y0 = y;
	    int xr = x0-cspc;
	    int yr = y0-cspc;
	    int xs = sizeX*cspc2;
	    int ys = sizeY*cspc2;
	    rectPointsX = new int[] { xr, xr+xs, xr+xs, xr };
	    rectPointsY = new int[] { yr, yr, yr+ys, yr+ys };
	    setBbox(xr, yr, rectPointsX[2], rectPointsY[2]);
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		switch (p.side) {
		case SIDE_N: p.setPoint(x0, y0, 1, 0, 0, -1, 0, 0); break;
		case SIDE_S: p.setPoint(x0, y0, 1, 0, 0,  1, 0, ys-cspc2);break;
		case SIDE_W: p.setPoint(x0, y0, 0, 1, -1, 0, 0, 0); break;
		case SIDE_E: p.setPoint(x0, y0, 0, 1,  1, 0, xs-cspc2, 0);break;
		}
	    }
	}
	Point getPost(int n) {
	    return pins[n].post;
	}
	abstract int getVoltageSourceCount(); // output count
	void setVoltageSource(int j, int vs) {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (p.output && j-- == 0) {
		    p.voltSource = vs;
		    return;
		}
	    }
	    System.out.println("setVoltageSource failed for " + this);
	}
	void stamp() {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (p.output)
		    stampVoltageSource(0, nodes[i], p.voltSource);
	    }
	}
	void execute() {}
	void doStep() {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (!p.output)
		    p.value = volts[i] > 2.5;
	    }
	    execute();
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (p.output)
		    updateVoltageSource(0, nodes[i], p.voltSource,
					p.value ? 5 : 0);
	    }
	}
	void reset() {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		pins[i].value = false;
		pins[i].curcount = 0;
		volts[i] = 0;
	    }
	    lastClock = false;
	}
	
	String dump() {
	    int t = getDumpType();
	    String s = super.dump();
	    if (needsBits())
		s += " " + bits;
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		if (pins[i].state)
		    s += " " + volts[i];
	    }
	    return s;
	}
	
	void getInfo(String arr[]) {
	    arr[0] = getChipName();
	    int i, a = 1;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (arr[a] != null)
		    arr[a] += "; ";
		else
		    arr[a] = "";
		String t = p.text;
		if (p.lineOver)
		    t += '\'';
		if (p.clock)
		    t = "Clk";
		arr[a] += t + " = " + getVoltageText(volts[i]);
		if (i % 2 == 1)
		    a++;
	    }
	}
	void setCurrent(int x, double c) {
	    int i;
	    for (i = 0; i != getPostCount(); i++)
		if (pins[i].output && pins[i].voltSource == x)
		    pins[i].current = c;
	}
	String getChipName() { return "chip"; }
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return pins[n1].output;
	}

	final int SIDE_N = 0;
	final int SIDE_S = 1;
	final int SIDE_W = 2;
	final int SIDE_E = 3;
	class Pin {
	    Pin(int p, int s, String t) {
		pos = p; side = s; text = t;
	    }
	    Point post, stub;
	    Point textloc;
	    int pos, side, voltSource, bubbleX, bubbleY;
	    String text;
	    boolean lineOver, bubble, clock, output, value, state;
	    double curcount, current;
	    void setPoint(int x, int y, int dx, int dy, int dax, int day,
			  int sx, int sy) {
		int xa = x+cspc2*dx*pos+sx;
		int ya = y+cspc2*dy*pos+sy;
		post    = new Point(xa+dax*cspc2, ya+day*cspc2);
		stub    = new Point(xa+dax*cspc , ya+day*cspc );
		textloc = new Point(xa       , ya       );
		if (bubble) {
		    bubbleX = xa+dax*10*csize;
		    bubbleY = ya+day*10*csize;
		}
		if (clock) {
		    clockPointsX = new int[3];
		    clockPointsY = new int[3];
		    clockPointsX[0] = xa+dax*cspc-dx*cspc/2;
		    clockPointsY[0] = ya+day*cspc-dy*cspc/2;
		    clockPointsX[1] = xa;
		    clockPointsY[1] = ya;
		    clockPointsX[2] = xa+dax*cspc+dx*cspc/2;
		    clockPointsY[2] = ya+day*cspc+dy*cspc/2;
		}
	    }
	}
    }

    class DFlipFlopElm extends ChipElm {
	final int FLAG_RESET = 2;
	public DFlipFlopElm(int xx, int yy) { super(xx, yy); }
	public DFlipFlopElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    pins[2].value = !pins[1].value;
	}
	String getChipName() { return "D flip-flop"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 3;
	    pins = new Pin[getPostCount()];
	    pins[0] = new Pin(0, SIDE_W, "D");
	    pins[1] = new Pin(0, SIDE_E, "Q");
	    pins[1].output = pins[1].state = true;
	    pins[2] = new Pin(2, SIDE_E, "Q");
	    pins[2].output = true;
	    pins[2].lineOver = true;
	    pins[3] = new Pin(1, SIDE_W, "");
	    pins[3].clock = true;
	    if ((flags & FLAG_RESET) != 0)
		pins[4] = new Pin(2, SIDE_W, "R");
	}
	int getPostCount() {
	    return ((flags & FLAG_RESET) != 0) ? 5 : 4;
	}
	int getVoltageSourceCount() { return 2; }
	void execute() {
	    if (pins[3].value && !lastClock) {
		pins[1].value =  pins[0].value;
		pins[2].value = !pins[0].value;
	    }
	    if (pins.length > 4 && pins[4].value) {
		pins[1].value = false;
		pins[2].value = true;
	    }
	    lastClock = pins[3].value;
	}
	int getDumpType() { return 155; }
    }
    
    class JKFlipFlopElm extends ChipElm {
	public JKFlipFlopElm(int xx, int yy) { super(xx, yy); }
	public JKFlipFlopElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    pins[4].value = !pins[3].value;
	}
	String getChipName() { return "JK flip-flop"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 3;
	    pins = new Pin[5];
	    pins[0] = new Pin(0, SIDE_W, "J");
	    pins[1] = new Pin(1, SIDE_W, "");
	    pins[1].clock = true;
	    pins[1].bubble = true;
	    pins[2] = new Pin(2, SIDE_W, "K");
	    pins[3] = new Pin(0, SIDE_E, "Q");
	    pins[3].output = pins[3].state = true;
	    pins[4] = new Pin(2, SIDE_E, "Q");
	    pins[4].output = true;
	    pins[4].lineOver = true;
	}
	int getPostCount() { return 5; }
	int getVoltageSourceCount() { return 2; }
	void execute() {
	    if (!pins[1].value && lastClock) {
		boolean q = pins[3].value;
		if (pins[0].value) {
		    if (pins[2].value)
			q = !q;
		    else
			q = true;
		} else if (pins[2].value)
		    q = false;
		pins[3].value = q;
		pins[4].value = !q;
	    }
	    lastClock = pins[1].value;
	}
	int getDumpType() { return 156; }
    }
    
    class DecadeElm extends ChipElm {
	public DecadeElm(int xx, int yy) { super(xx, yy); }
	public DecadeElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "decade counter"; }
	boolean needsBits() { return true; }
	void setupPins() {
	    sizeX = bits > 2 ? bits : 2;
	    sizeY = 2;
	    pins = new Pin[getPostCount()];
	    pins[0] = new Pin(1, SIDE_W, "");
	    pins[0].clock = true;
	    pins[1] = new Pin(sizeX-1, SIDE_S, "R");
	    pins[1].bubble = true;
	    int i;
	    for (i = 0; i != bits; i++) {
		int ii = i+2;
		pins[ii] = new Pin(i, SIDE_N, "Q" + i);
		pins[ii].output = pins[ii].state = true;
	    }
	    allocNodes();
	}
	int getPostCount() { return bits+2; }
	int getVoltageSourceCount() { return bits; }
	void execute() {
	    int i;
	    if (pins[0].value && !lastClock) {
		for (i = 0; i != bits; i++)
		    if (pins[i+2].value)
			break;
		if (i < bits)
		    pins[i++ +2].value = false;
		i %= bits;
		pins[i+2].value = true;
	    }
	    if (!pins[1].value) {
		for (i = 1; i != bits; i++)
		    pins[i+2].value = false;
		pins[2].value = true;
	    }
	    lastClock = pins[0].value;
	}
	int getDumpType() { return 163; }
    }
    
    class CounterElm extends ChipElm {
	final int FLAG_ENABLE = 2;
	public CounterElm(int xx, int yy) { super(xx, yy); }
	public CounterElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	boolean needsBits() { return true; }
	String getChipName() { return "Counter"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = bits > 2 ? bits : 2;
	    pins = new Pin[getPostCount()];
	    pins[0] = new Pin(0, SIDE_W, "");
	    pins[0].clock = true;
	    pins[1] = new Pin(sizeY-1, SIDE_W, "R");
	    pins[1].bubble = true;
	    int i;
	    for (i = 0; i != bits; i++) {
		int ii = i+2;
		pins[ii] = new Pin(i, SIDE_E, "Q" + (bits-i-1));
		pins[ii].output = pins[ii].state = true;
	    }
	    if (hasEnable())
		pins[bits+2] = new Pin(sizeY-2, SIDE_W, "En");
	    allocNodes();
	}
	int getPostCount() {
	    if (hasEnable())
		return bits+3;
	    return bits+2;
	}
	boolean hasEnable() { return (flags & FLAG_ENABLE) != 0; }
	int getVoltageSourceCount() { return bits; }
	void execute() {
	    boolean en = true;
	    if (hasEnable())
		en = pins[bits+2].value;
	    if (pins[0].value && !lastClock && en) {
		int i;
		for (i = bits-1; i >= 0; i--) {
		    int ii = i+2;
		    if (!pins[ii].value) {
			pins[ii].value = true;
			break;
		    }
		    pins[ii].value = false;
		}
	    }
	    if (!pins[1].value) {
		int i;
		for (i = 0; i != bits; i++)
		    pins[i+2].value = false;
	    }
	    lastClock = pins[0].value;
	}
	int getDumpType() { return 164; }
    }
    
    class SevenSegElm extends ChipElm {
	public SevenSegElm(int xx, int yy) { super(xx, yy); }
	public SevenSegElm(int xa, int ya, int xb, int yb, int f,
			   StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "7-segment driver/display"; }
	Color darkred;
	void setupPins() {
	    darkred = new Color(30, 0, 0);
	    sizeX = 4;
	    sizeY = 4;
	    pins = new Pin[7];
	    pins[0] = new Pin(0, SIDE_W, "a");
	    pins[1] = new Pin(1, SIDE_W, "b");
	    pins[2] = new Pin(2, SIDE_W, "c");
	    pins[3] = new Pin(3, SIDE_W, "d");
	    pins[4] = new Pin(1, SIDE_S, "e");
	    pins[5] = new Pin(2, SIDE_S, "f");
	    pins[6] = new Pin(3, SIDE_S, "g");
	}
	void draw(Graphics g) {
	    drawChip(g);
	    g.setColor(Color.red);
	    int xl = x+cspc*5;
	    int yl = y+cspc;
	    setColor(g, 0);
	    drawThickLine(g, xl, yl, xl+cspc, yl);
	    setColor(g, 1);
	    drawThickLine(g, xl+cspc, yl, xl+cspc, yl+cspc);
	    setColor(g, 2);
	    drawThickLine(g, xl+cspc, yl+cspc, xl+cspc, yl+cspc2);
	    setColor(g, 3);
	    drawThickLine(g, xl, yl+cspc2, xl+cspc, yl+cspc2);
	    setColor(g, 4);
	    drawThickLine(g, xl, yl+cspc, xl, yl+cspc2);
	    setColor(g, 5);
	    drawThickLine(g, xl, yl, xl, yl+cspc);
	    setColor(g, 6);
	    drawThickLine(g, xl, yl+cspc, xl+cspc, yl+cspc);
	}
	void setColor(Graphics g, int p) {
	    g.setColor(pins[p].value ? Color.red :
		       printableCheckItem.getState() ? Color.white : darkred);
	}
	int getPostCount() { return 7; }
	int getVoltageSourceCount() { return 0; }
	int getDumpType() { return 157; }
    }
    
    class VCOElm extends ChipElm {
	public VCOElm(int xx, int yy) { super(xx, yy); }
	public VCOElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "VCO"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 4;
	    pins = new Pin[6];
	    pins[0] = new Pin(0, SIDE_W, "Vi");
	    pins[1] = new Pin(3, SIDE_W, "Vo");
	    pins[1].output = true;
	    pins[2] = new Pin(0, SIDE_E, "C");
	    pins[3] = new Pin(1, SIDE_E, "C");
	    pins[4] = new Pin(2, SIDE_E, "R1");
	    pins[4].output = true;
	    pins[5] = new Pin(3, SIDE_E, "R2");
	    pins[5].output = true;
	}
	boolean nonLinear() { return true; }
	void stamp() {
	    // output pin
	    stampVoltageSource(0, nodes[1], pins[1].voltSource);
	    // attach Vi to R1 pin so its current is proportional to Vi
	    stampVoltageSource(nodes[0], nodes[4], pins[4].voltSource, 0);
	    // attach 5V to R2 pin so we get a current going
	    stampVoltageSource(0, nodes[5], pins[5].voltSource, 5);
	    // put resistor across cap pins to give current somewhere to go
	    // in case cap is not connected
	    stampResistor(nodes[2], nodes[3], cResistance);
	    stampNonLinear(nodes[2]);
	    stampNonLinear(nodes[3]);
	}
	final double cResistance = 1e6;
	double cCurrent;
	int cDir;
	void doStep() {
	    double vc = volts[3]-volts[2];
	    double vo = volts[1];
	    int dir = (vo < 2.5) ? 1 : -1;
	    // switch direction of current through cap as we oscillate
	    if (vo < 2.5 && vc > 4.5) {
		vo = 5;
		dir = -1;
	    }
	    if (vo > 2.5 && vc < .5) {
		vo = 0;
		dir = 1;
	    }

	    // generate output voltage
	    updateVoltageSource(0, nodes[1], pins[1].voltSource, vo);
	    // now we set the current through the cap to be equal to the
	    // current through R1 and R2, so we can measure the voltage
	    // across the cap
	    int cur1 = nodeList.size() + pins[4].voltSource;
	    int cur2 = nodeList.size() + pins[5].voltSource;
	    stampMatrix(nodes[2], cur1, dir);
	    stampMatrix(nodes[2], cur2, dir);
	    stampMatrix(nodes[3], cur1, -dir);
	    stampMatrix(nodes[3], cur2, -dir);
	    cDir = dir;
	}
	// can't do this in calculateCurrent() because it's called before
        // we get pins[4].current and pins[5].current, which we need
	void computeCurrent() {
	    if (cResistance == 0)
		return;
	    double c = cDir*(pins[4].current + pins[5].current) +
		(volts[3]-volts[2])/cResistance;
	    pins[2].current = -c;
	    pins[3].current = c;
	    pins[0].current = -pins[4].current;
	}
	void draw(Graphics g) {
	    computeCurrent();
	    drawChip(g);
	}
	int getPostCount() { return 6; }
	int getVoltageSourceCount() { return 3; }
	int getDumpType() { return 158; }
    }
    
    class TimerElm extends ChipElm {
	final int FLAG_RESET = 2;
	int getDefaultFlags() { return FLAG_RESET; }
	public TimerElm(int xx, int yy) { super(xx, yy); }
	public TimerElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "555 Timer"; }
	void setupPins() {
	    sizeX = 3;
	    sizeY = 5;
	    pins = new Pin[7];
	    pins[0] = new Pin(1, SIDE_W, "dis");
	    pins[1] = new Pin(3, SIDE_W, "tr");
	    pins[1].lineOver = true;
	    pins[2] = new Pin(4, SIDE_W, "th");
	    pins[3] = new Pin(1, SIDE_N, "Vin");
	    pins[4] = new Pin(1, SIDE_S, "ctl");
	    pins[5] = new Pin(2, SIDE_E, "out");
	    pins[5].output = pins[5].state = true;
	    pins[6] = new Pin(1, SIDE_E, "rst");
	}
	boolean nonLinear() { return true; }
	boolean hasReset() { return (flags & FLAG_RESET) != 0; }
	void stamp() {
	    // stamp voltage divider to put ctl pin at 2/3 V
	    stampResistor(nodes[3], nodes[4],  5000);
	    stampResistor(nodes[4], 0,        10000);
	    // output pin
	    stampVoltageSource(0, nodes[5], pins[5].voltSource);
	    // discharge pin
	    stampNonLinear(0);
	    stampNonLinear(nodes[0]);
	}
	void calculateCurrent() {
	    // need current for V, discharge, control; output current is
	    // calculated for us, and other pins have no current
	    pins[3].current = (volts[4]-volts[3])/5000;
	    pins[4].current = -volts[4]/10000 - pins[3].current;
	    pins[0].current = out ? 0 : -volts[0]/10;
	}
	boolean out;
	void startIteration() {
	    out = volts[5] > volts[3]/2;
	    // check comparators
	    if (volts[4]/2 > volts[1])
		out = true;
	    if (volts[2] > volts[4] || (hasReset() && volts[6] < .7))
		out = false;
	}
	void doStep() {
	    // if output is low, discharge pin 0.  we use a small
	    // resistor because it's easier, and sometimes people tie
	    // the discharge pin to the trigger and threshold pins.
	    if (!out)
		stampResistor(nodes[0], 0, 10);
	    // output
	    updateVoltageSource(0, nodes[5], pins[5].voltSource,
				out ? volts[3] : 0);
	}
	int getPostCount() { return hasReset() ? 7 : 6; }
	int getVoltageSourceCount() { return 1; }
	int getDumpType() { return 165; }
    }
    
    class PhaseCompElm extends ChipElm {
	public PhaseCompElm(int xx, int yy) { super(xx, yy); }
	public PhaseCompElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "phase comparator"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 2;
	    pins = new Pin[3];
	    pins[0] = new Pin(0, SIDE_W, "I1");
	    pins[1] = new Pin(1, SIDE_W, "I2");
	    pins[2] = new Pin(0, SIDE_E, "O");
	    pins[2].output = true;
	}
	boolean nonLinear() { return true; }
	void stamp() {
	    int vn = nodeList.size()+pins[2].voltSource;
	    stampNonLinear(vn);
	    stampNonLinear(0);
	    stampNonLinear(nodes[2]);
	}
	boolean ff1, ff2;
	void doStep() {
	    boolean v1 = volts[0] > 2.5;
	    boolean v2 = volts[1] > 2.5;
	    if (v1 && !pins[0].value)
		ff1 = true;
	    if (v2 && !pins[1].value)
		ff2 = true;
	    if (ff1 && ff2)
		ff1 = ff2 = false;
	    double out = (ff1) ? 5 : (ff2) ? 0 : -1;
	    //System.out.println(out + " " + v1 + " " + v2);
	    if (out != -1)
		stampVoltageSource(0, nodes[2], pins[2].voltSource, out);
	    else {
		// tie current through output pin to 0
		int vn = nodeList.size()+pins[2].voltSource;
		stampMatrix(vn, vn, 1);
	    }
	    pins[0].value = v1;
	    pins[1].value = v2;
	}
	int getPostCount() { return 3; }
	int getVoltageSourceCount() { return 1; }
	int getDumpType() { return 161; }
    }
    
    class DACElm extends ChipElm {
	public DACElm(int xx, int yy) { super(xx, yy); }
	public DACElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "DAC"; }
	boolean needsBits() { return true; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = bits > 2 ? bits : 2;
	    pins = new Pin[getPostCount()];
	    int i;
	    for (i = 0; i != bits; i++)
		pins[i] = new Pin(bits-1-i, SIDE_W, "D" + i);
	    pins[bits]   = new Pin(0, SIDE_E, "O");
	    pins[bits].output = true;
	    pins[bits+1] = new Pin(sizeY-1, SIDE_E, "V+");
	    allocNodes();
	}
	void doStep() {
	    int ival = 0;
	    int i;
	    for (i = 0; i != bits; i++)
		if (volts[i] > 2.5)
		    ival |= 1<<i;
	    int ivalmax = (1<<bits)-1;
	    double v = ival*volts[bits+1]/ivalmax;
	    updateVoltageSource(0, nodes[bits], pins[bits].voltSource, v);
	}
	int getVoltageSourceCount() { return 1; }
	int getPostCount() { return bits+2; }
	int getDumpType() { return 166; }
    }
    
    class ADCElm extends ChipElm {
	public ADCElm(int xx, int yy) { super(xx, yy); }
	public ADCElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "ADC"; }
	boolean needsBits() { return true; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = bits > 2 ? bits : 2;
	    pins = new Pin[getPostCount()];
	    int i;
	    for (i = 0; i != bits; i++) {
		pins[i] = new Pin(bits-1-i, SIDE_E, "D" + i);
		pins[i].output = true;
	    }
	    pins[bits]   = new Pin(0, SIDE_W, "In");
	    pins[bits+1] = new Pin(sizeY-1, SIDE_W, "V+");
	    allocNodes();
	}
	void execute() {
	    int imax = (1<<bits)-1;
	    // if we round, the half-flash doesn't work
	    double val = imax*volts[bits]/volts[bits+1]; // + .5;
	    int ival = (int) val;
	    ival = min(imax, max(0, ival));
	    int i;
	    for (i = 0; i != bits; i++)
		pins[i].value = ((ival & (1<<i)) != 0);
	}
	int getVoltageSourceCount() { return bits; }
	int getPostCount() { return bits+2; }
	int getDumpType() { return 167; }
    }
    
    class LatchElm extends ChipElm {
	public LatchElm(int xx, int yy) { super(xx, yy); }
	public LatchElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "Latch"; }
	boolean needsBits() { return true; }
	int loadPin;
	void setupPins() {
	    sizeX = 2;
	    sizeY = bits+1;
	    pins = new Pin[getPostCount()];
	    int i;
	    for (i = 0; i != bits; i++)
		pins[i] = new Pin(bits-1-i, SIDE_W, "I" + i);
	    for (i = 0; i != bits; i++) {
		pins[i+bits] = new Pin(bits-1-i, SIDE_E, "O");
		pins[i+bits].output = true;
	    }
	    pins[loadPin = bits*2] = new Pin(bits, SIDE_W, "Ld");
	    allocNodes();
	}
	boolean lastLoad = false;
	void execute() {
	    int i;
	    if (pins[loadPin].value && !lastLoad)
		for (i = 0; i != bits; i++)
		    pins[i+bits].value = pins[i].value;
	    lastLoad = pins[loadPin].value;
	}
	int getVoltageSourceCount() { return bits; }
	int getPostCount() { return bits*2+1; }
	int getDumpType() { return 168; }
    }
    
    class TextElm extends CircuitElm {
	String text;
	Vector lines;
	int size;
	final int FLAG_CENTER = 1;
	final int FLAG_BAR = 2;
	public TextElm(int xx, int yy) {
	    super(xx, yy);
	    text = "hello";
	    lines = new Vector();
	    lines.add(text);
	    size = 24;
	}
	public TextElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    size = new Integer(st.nextToken()).intValue();
	    text = st.nextToken();
	    while (st.hasMoreTokens())
		text += ' ' + st.nextToken();
	    split();
	}
	void split() {
	    int i;
	    lines = new Vector();
	    StringBuffer sb = new StringBuffer(text);
	    for (i = 0; i < sb.length(); i++) {
		char c = sb.charAt(i);
		if (c == '\\') {
		    sb.deleteCharAt(i);
		    c = sb.charAt(i);
		    if (c == 'n') {
			lines.add(sb.substring(0, i));
			sb.delete(0, i+1);
			i = -1;
			continue;
		    }
		}
	    }
	    lines.add(sb.toString());
	}
	String dump() {
	    return super.dump() + " " + size + " " + text;
	}
	int getDumpType() { return 'x'; }
	void drag(int xx, int yy) {
	    x = xx;
	    y = yy;
	    x2 = xx+16;
	    y2 = yy;
	}
	void draw(Graphics g) {
	    g.setColor(this == mouseElm ? selectColor : lightGrayColor);
	    Font f = new Font("SansSerif", 0, size);
	    g.setFont(f);
	    FontMetrics fm = g.getFontMetrics();
	    int i;
	    int maxw = -1;
	    for (i = 0; i != lines.size(); i++) {
		int w = fm.stringWidth((String) (lines.elementAt(i)));
		if (w > maxw)
		    maxw = w;
	    }
	    int cury = y;
	    setBbox(x, y, x, y);
	    for (i = 0; i != lines.size(); i++) {
		String s = (String) (lines.elementAt(i));
		if ((flags & FLAG_CENTER) != 0)
		    x = (winSize.width-fm.stringWidth(s))/2;
		g.drawString(s, x, cury);
		if ((flags & FLAG_BAR) != 0) {
		    int by = cury-fm.getAscent();
		    g.drawLine(x, by, x+fm.stringWidth(s)-1, by);
		}
		adjustBbox(x, cury-fm.getAscent(),
			   x+fm.stringWidth(s), cury+fm.getDescent());
		cury += fm.getHeight();
	    }
	    x2 = boundingBox.x + boundingBox.width;
	    y2 = boundingBox.y + boundingBox.height;
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("Text", 0, -1, -1);
		ei.text = text;
		return ei;
	    }
	    if (n == 1)
		return new EditInfo("Size", size, 5, 100);
	    if (n == 2) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox =
		    new Checkbox("Center", (flags & FLAG_CENTER) != 0);
		return ei;
	    }
	    if (n == 3) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox =
		    new Checkbox("Draw Bar On Top", (flags & FLAG_BAR) != 0);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0) {
		text = ei.textf.getText();
		split();
	    }
	    if (n == 1)
		size = (int) ei.value;
	    if (n == 3) {
		if (ei.checkbox.getState())
		    flags |= FLAG_BAR;
		else
		    flags &= ~FLAG_BAR;
	    }
	    if (n == 2) {
		if (ei.checkbox.getState())
		    flags |= FLAG_CENTER;
		else
		    flags &= ~FLAG_CENTER;
	    }
	}
	boolean isCenteredText() { return (flags & FLAG_CENTER) != 0; }
	void getInfo(String arr[]) {
	    arr[0] = text;
	}
	int getPostCount() { return 0; }
    }

    class ProbeElm extends CircuitElm {
	public ProbeElm(int xx, int yy) { super(xx, yy); }
	public ProbeElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 'p'; }
	
	Point center;
	void setPoints() {
	    super.setPoints();
	    // swap points so that we subtract higher from lower
	    if (point2.y < point1.y) {
		Point x = point1;
		point1 = point2;
		point2 = x;
	    }
	    center = interpPoint(point1, point2, .5);
	}
	void draw(Graphics g) {
	    int hs = 8;
	    setBbox(point1, point2, hs);
	    boolean selected = (mouseElm == this || plotYElm == this);
	    double len = (selected || dragElm == this) ? 16 : dn-32;
	    calcLeads((int) len);
	    setVoltageColor(g, volts[0]);
	    if (selected)
		g.setColor(selectColor);
	    drawThickLine(g, point1, lead1);
	    setVoltageColor(g, volts[1]);
	    if (selected)
		g.setColor(selectColor);
	    drawThickLine(g, lead2, point2);
	    Font f = new Font("SansSerif", Font.BOLD, 14);
	    g.setFont(f);
	    if (this == plotXElm)
		drawCenteredText(g, "X", center.x, center.y, true);
	    if (this == plotYElm)
		drawCenteredText(g, "Y", center.x, center.y, true);
	    drawPosts(g);
	}
	
	void getInfo(String arr[]) {
	    arr[0] = "scope probe";
	    arr[1] = "Vd = " + getVoltageText(getVoltageDiff());
	}
	boolean getConnection(int n1, int n2) { return false; }
    }
    
    class AnalogSwitchElm extends CircuitElm {
	final int FLAG_INVERT = 1;
	public AnalogSwitchElm(int xx, int yy) {
	    super(xx, yy);
	}
	public AnalogSwitchElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 159; }
	boolean open;
	
	Point ps, point3, lead3;
	void setPoints() {
	    super.setPoints();
	    calcLeads(32);
	    ps = new Point();
	    int openhs = 16;
	    point3 = interpPoint(point1, point2, .5, -openhs);
	    lead3  = interpPoint(point1, point2, .5, -openhs/2);
	}
	
	void draw(Graphics g) {
	    int openhs = 16;
	    int hs = (open) ? openhs : 0;
	    setBbox(point1, point2, openhs);

	    draw2Leads(g);
	    
	    g.setColor(lightGrayColor);
	    interpPoint(lead1, lead2, ps, 1, hs);
	    drawThickLine(g, lead1, ps);

	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, point3, lead3);
	    
	    if (!open)
		doDots(g);
	    drawPosts(g);
	}
	final double resistance = 20;
	void calculateCurrent() {
	    if (open)
		current = 0;
	    else
		current = (volts[0]-volts[1])/resistance;
	}
	
	// we need this to be able to change the matrix for each step
	boolean nonLinear() { return true; }

	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	}
	void doStep() {
	    open = (volts[2] < 2.5);
	    if ((flags & FLAG_INVERT) != 0)
		open = !open;
	    if (!open)
		stampResistor(nodes[0], nodes[1], resistance);
	    else
		stampResistor(nodes[0], nodes[1], 1e10);
	}
	void drag(int xx, int yy) {
	    xx = snapGrid(xx);
	    yy = snapGrid(yy);
	    if (abs(x-xx) < abs(y-yy))
		xx = x;
	    else
		yy = y;
	    int q1 = abs(x-xx)+abs(y-yy);
	    int q2 = (q1/2) % gridSize;
	    if (q2 != 0)
		return;
	    x2 = xx; y2 = yy;
	    setPoints();
	}
	int getPostCount() { return 3; }
	Point getPost(int n) {
	    return (n == 0) ? point1 : (n == 1) ? point2 : point3;
	}
	void getInfo(String arr[]) {
	    arr[0] = "analog switch";
	    arr[1] = open ? "open" : "closed";
	    arr[2] = "Vd = " + getVoltageDText(getVoltageDiff());
	    arr[3] = "I = " + getCurrentDText(getCurrent());
	    arr[4] = "Vc = " + getVoltageText(volts[2]);
	}
	// we have to just assume current will flow either way, even though that
	// might cause singular matrix errors
	boolean getConnection(int n1, int n2) {
	    if (n1 == 2 || n2 == 2)
		return false;
	    return true;
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Normally closed",
					   (flags & FLAG_INVERT) != 0);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		flags = (ei.checkbox.getState()) ?
		    (flags | FLAG_INVERT) :
		    (flags & ~FLAG_INVERT);
	}
    }

    class AnalogSwitch2Elm extends AnalogSwitchElm {
	public AnalogSwitch2Elm(int xx, int yy) {
	    super(xx, yy);
	}
	public AnalogSwitch2Elm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}

	final int openhs = 16;
	Point swposts[], swpoles[], ctlPoint;
	void setPoints() {
	    super.setPoints();
	    calcLeads(32);
	    swposts = newPointArray(2);
	    swpoles = newPointArray(2);
	    interpPoint2(lead1,  lead2,  swpoles[0], swpoles[1], 1, openhs);
	    interpPoint2(point1, point2, swposts[0], swposts[1], 1, openhs);
	    ctlPoint = interpPoint(point1, point2, .5, openhs);
	}
	int getPostCount() { return 4; }

	void draw(Graphics g) {
	    setBbox(point1, point2, openhs);

	    // draw first lead
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);

	    // draw second lead
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, swpoles[0], swposts[0]);
	    
	    // draw third lead
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, swpoles[1], swposts[1]);

	    // draw switch
	    g.setColor(lightGrayColor);
	    int position = (open) ? 1 : 0;
	    drawThickLine(g, lead1, swpoles[position]);
	    
	    updateDotCount();
	    drawDots(g, point1, lead1, curcount);
	    drawDots(g, swpoles[position], swposts[position], curcount);
	    drawPosts(g);
	}
	
	Point getPost(int n) {
	    return (n == 0) ? point1 : (n == 3) ? ctlPoint : swposts[n-1];
	}
	int getDumpType() { return 160; }

	void calculateCurrent() {
	    if (open)
		current = (volts[0]-volts[2])/resistance;
	    else
		current = (volts[0]-volts[1])/resistance;
	}
	
	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	    stampNonLinear(nodes[2]);
	}
	void doStep() {
	    open = (volts[3] < 2.5);
	    if ((flags & FLAG_INVERT) != 0)
		open = !open;
	    if (open) {
		stampResistor(nodes[0], nodes[2], resistance);
		stampResistor(nodes[0], nodes[1], 1e10);
	    } else {
		stampResistor(nodes[0], nodes[1], resistance);
		stampResistor(nodes[0], nodes[2], 1e10);
	    }
	}
	
	boolean getConnection(int n1, int n2) {
	    if (n1 == 3 || n2 == 3)
		return false;
	    return true;
	}
	void getInfo(String arr[]) {
	    arr[0] = "analog switch (DPST)";
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	}
    }

    class SweepElm extends CircuitElm {
	double maxV, maxF, minF, sweepTime, frequency;
	final int FLAG_LOG = 1;
	final int FLAG_BIDIR = 2;
	
	public SweepElm(int xx, int yy) {
	    super(xx, yy);
	    minF = 20; maxF = 4000;
	    maxV = 5;
	    sweepTime = .1;
	    flags = FLAG_BIDIR;
	    reset();
	}
	public SweepElm(int xa, int ya, int xb, int yb, int f, StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    minF = new Double(st.nextToken()).doubleValue();
	    maxF = new Double(st.nextToken()).doubleValue();
	    maxV = new Double(st.nextToken()).doubleValue();
	    sweepTime = new Double(st.nextToken()).doubleValue();
	    reset();
	}
	int getDumpType() { return 170; }
	int getPostCount() { return 1; }
	final int circleSize = 17;

	String dump() {
	    return super.dump() + " " + minF + " " + maxF + " " + maxV + " " +
		sweepTime;
	}
	void setPoints() {
	    super.setPoints();
	    lead1 = interpPoint(point1, point2, 1-circleSize/dn);
	}
	void draw(Graphics g) {
	    setBbox(point1, point2, circleSize);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, point1, lead1);
	    g.setColor(this == mouseElm ? selectColor : Color.gray);
	    setPowerColor(g, false);
	    int xc = point2.x; int yc = point2.y;
	    drawThickCircle(g, xc, yc, circleSize);
	    int wl = 8;
	    adjustBbox(xc-circleSize, yc-circleSize,
		       xc+circleSize, yc+circleSize);
	    int i;
	    int xl = 10;
	    int ox = -1, oy = -1;
	    long tm = System.currentTimeMillis();
	    //double w = (this == mouseElm ? 3 : 2);
	    tm %= 2000;
	    if (tm > 1000)
		tm = 2000-tm;
	    double w = 1+tm*.002;
	    if (!stoppedCheck.getState())
		w = 1+2*(frequency-minF)/(maxF-minF);
	    for (i = -xl; i <= xl; i++) {
		int yy = yc+(int) (.95*Math.sin(i*pi*w/xl)*wl);
		if (ox != -1)
		    drawThickLine(g, ox, oy, xc+i, yy);
		ox = xc+i; oy = yy;
	    }
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(frequency, "Hz");
		if (dx == 0 || dy == 0)
		    drawValues(g, s, circleSize);
	    }
	    
	    drawPosts(g);
	    curcount = updateDotCount(-current, curcount);
	    if (dragElm != this)
		drawDots(g, point1, lead1, curcount);
	}
	
	void stamp() {
	    stampVoltageSource(0, nodes[0], voltSource);
	}
	double fadd, fmul, freqTime;
	int dir = 1;
	void setParams() {
	    if (frequency < minF || frequency > maxF) {
		frequency = minF;
		freqTime = 0;
		dir = 1;
	    }
	    if ((flags & FLAG_LOG) == 0) {
		fadd = dir*timeStep*(maxF-minF)/sweepTime;
		fmul = 1;
	    } else {
		fadd = 0;
		fmul = Math.pow(maxF/minF, dir*timeStep/sweepTime);
	    }
	}
	void reset() {
	    frequency = minF;
	    freqTime = 0;
	    dir = 1;
	    setParams();
	}
	double v;
	void startIteration() {
	    v = Math.sin(freqTime)*maxV;
	    freqTime += frequency*2*pi*timeStep;
	    frequency = frequency*fmul+fadd;
	    if (frequency >= maxF && dir == 1) {
		if ((flags & FLAG_BIDIR) != 0) {
		    fadd = -fadd;
		    fmul = 1/fmul;
		    dir = -1;
		} else
		    frequency = minF;
	    }
	    if (frequency <= minF && dir == -1) {
		fadd = -fadd;
		fmul = 1/fmul;
		dir = 1;
	    }
	}
	void doStep() {
	    updateVoltageSource(0, nodes[0], voltSource, v);
	}
	
	double getVoltageDiff() { return volts[0]; }
	int getVoltageSourceCount() { return 1; }
	boolean hasGroundConnection(int n1) { return true; }
	void getInfo(String arr[]) {
	    arr[0] = "sweep " + (((flags & FLAG_LOG) == 0) ? "(linear)" : "(log)");
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	    arr[2] = "V = " + getVoltageText(volts[0]);
	    arr[3] = "f = " + getUnitText(frequency, "Hz");
	    arr[4] = "range = " + getUnitText(minF, "Hz") + " .. " +
		getUnitText(maxF, "Hz");
	    arr[5] = "time = " + getUnitText(sweepTime, "s");
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Min Frequency (Hz)", minF, 0, 0);
	    if (n == 1)
		return new EditInfo("Max Frequency (Hz)", maxF, 0, 0);
	    if (n == 2)
		return new EditInfo("Sweep Time (s)", sweepTime, 0, 0);
	    if (n == 3) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Logarithmic", (flags & FLAG_LOG) != 0);
		return ei;
	    }
	    if (n == 4)
		return new EditInfo("Max Voltage", maxV, 0, 0);
	    if (n == 5) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Bidirectional", (flags & FLAG_BIDIR) != 0);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		minF = ei.value;
	    if (n == 1)
		maxF = ei.value;
	    if (n == 2)
		sweepTime = ei.value;
	    if (n == 3) {
		flags &= ~FLAG_LOG;
		if (ei.checkbox.getState())
		    flags |= FLAG_LOG;
	    }
	    if (n == 4)
		maxV = ei.value;
	    if (n == 5) {
		flags &= ~FLAG_BIDIR;
		if (ei.checkbox.getState())
		    flags |= FLAG_BIDIR;
	    }
	    setParams();
	}
    }
    
    class TransLineElm extends CircuitElm {
	double delay, imped;
	double voltageL[], voltageR[];
	int lenSteps, ptr, width;
	public TransLineElm(int xx, int yy) {
	    super(xx, yy);
	    delay = 1000*timeStep;
	    imped = 75;
	    noDiagonal = true;
	    reset();
	}
	public TransLineElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    delay = new Double(st.nextToken()).doubleValue();
	    imped = new Double(st.nextToken()).doubleValue();
	    width = new Integer(st.nextToken()).intValue();
	    // next slot is for resistance (losses), which is not implemented
	    st.nextToken();
	    noDiagonal = true;
	    reset();
	}
	int getDumpType() { return 171; }
	int getPostCount() { return 4; }
	int getInternalNodeCount() { return 2; }
	String dump() {
	    return super.dump() + " " + delay + " " + imped + " " + width + " " + 0.;
	}
	void drag(int xx, int yy) {
	    xx = snapGrid(xx);
	    yy = snapGrid(yy);
	    int w1 = max(gridSize, abs(yy-y));
	    int w2 = max(gridSize, abs(xx-x));
	    if (w1 > w2) {
		xx = x;
		width = w2;
	    } else {
		yy = y;
		width = w1;
	    }
	    x2 = xx; y2 = yy;
	    setPoints();
	}
	
	Point posts[], inner[];
	
	void reset() {
	    if (timeStep == 0)
		return;
	    lenSteps = (int) (delay/timeStep);
	    System.out.println(lenSteps + " steps");
	    if (lenSteps > 100000)
		voltageL = voltageR = null;
	    else {
		voltageL = new double[lenSteps];
		voltageR = new double[lenSteps];
	    }
	    ptr = 0;
	    super.reset();
	}
	void setPoints() {
	    super.setPoints();
	    int ds = (dy == 0) ? sign(dx) : -sign(dy);
	    Point p3 = interpPoint(point1, point2, 0, -width*ds);
	    Point p4 = interpPoint(point1, point2, 1, -width*ds);
	    int sep = gridSize/2;
	    Point p5 = interpPoint(point1, point2, 0, -(width/2-sep)*ds);
	    Point p6 = interpPoint(point1, point2, 1, -(width/2-sep)*ds);
	    Point p7 = interpPoint(point1, point2, 0, -(width/2+sep)*ds);
	    Point p8 = interpPoint(point1, point2, 1, -(width/2+sep)*ds);
	    
	    // we number the posts like this because we want the lower-numbered
	    // points to be on the bottom, so that if some of them are unconnected
	    // (which is often true) then the bottom ones will get automatically
	    // attached to ground.
	    posts = new Point[] { p3, p4, point1, point2 };
	    inner = new Point[] { p7, p8, p5, p6 };
	}
	void draw(Graphics g) {
	    setBbox(posts[0], posts[3], 0);
	    int segments = (int) (dn/2);
	    int ix0 = ptr-1+lenSteps;
	    double segf = 1./segments;
	    int i;
	    g.setColor(Color.darkGray);
	    g.fillRect(inner[2].x, inner[2].y,
		       inner[1].x-inner[2].x+2, inner[1].y-inner[2].y+2);
	    for (i = 0; i != 4; i++) {
		setVoltageColor(g, volts[i]);
		drawThickLine(g, posts[i], inner[i]);
	    }
	    if (voltageL != null) {
		for (i = 0; i != segments; i++) {
		    int ix1 = (ix0-lenSteps*i/segments) % lenSteps;
		    int ix2 = (ix0-lenSteps*(segments-1-i)/segments) % lenSteps;
		    double v = (voltageL[ix1]+voltageR[ix2])/2;
		    setVoltageColor(g, v);
		    interpPoint(inner[0], inner[1], ps1, i*segf);
		    interpPoint(inner[2], inner[3], ps2, i*segf);
		    g.drawLine(ps1.x, ps1.y, ps2.x, ps2.y);
		    interpPoint(inner[2], inner[3], ps1, (i+1)*segf);
		    drawThickLine(g, ps1, ps2);
		}
	    }
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, inner[0], inner[1]);
	    drawPosts(g);

	    curCount1 = updateDotCount(-current1, curCount1);
	    curCount2 = updateDotCount(current2, curCount2);
	    if (dragElm != this) {
		drawDots(g, posts[0], inner[0], curCount1);
		drawDots(g, posts[2], inner[2], -curCount1);
		drawDots(g, posts[1], inner[1], -curCount2);
		drawDots(g, posts[3], inner[3], curCount2);
	    }
	}

	int voltSource1, voltSource2;
	double current1, current2, curCount1, curCount2;
	void setVoltageSource(int n, int v) {
	    if (n == 0)
		voltSource1 = v;
	    else
		voltSource2 = v;
	}
	void setCurrent(int v, double c) {
	    if (v == voltSource1)
		current1 = c;
	    else
		current2 = c;
	}
	
	void stamp() {
	    stampVoltageSource(nodes[4], nodes[0], voltSource1);
	    stampVoltageSource(nodes[5], nodes[1], voltSource2);
	    stampResistor(nodes[2], nodes[4], imped);
	    stampResistor(nodes[3], nodes[5], imped);
	}

	void startIteration() {
	    // calculate voltages, currents sent over wire
	    if (voltageL == null) {
		stop("Transmission line delay too large!", this);
		return;
	    }
	    voltageL[ptr] = volts[2]-volts[0] + imped*current1;
	    voltageR[ptr] = volts[3]-volts[1] + imped*current2;
	    /*System.out.println("sending fwd  " + currentL[ptr] + " " + current1);
	      System.out.println("sending back " + currentR[ptr] + " " + current2);*/
	    //System.out.println("sending back " + voltageR[ptr]);
	    ptr = (ptr+1) % lenSteps;
	}
	void doStep() {
	    if (voltageL == null) {
		stop("Transmission line delay too large!", this);
		return;
	    }
	    updateVoltageSource(nodes[4], nodes[0], voltSource1, -voltageR[ptr]);
	    updateVoltageSource(nodes[5], nodes[1], voltSource2, -voltageL[ptr]);
	    if (Math.abs(volts[0]) > 1e-5 || Math.abs(volts[1]) > 1e-5) {
		stop("Need to ground transmission line!", this);
		return;
	    }
	}

	Point getPost(int n) {
	    return posts[n];
	}
	
	//double getVoltageDiff() { return volts[0]; }
	int getVoltageSourceCount() { return 2; }
	boolean hasGroundConnection(int n1) { return false; }
	boolean getConnection(int n1, int n2) {
	    return false;
	    /*if (comparePair(n1, n2, 0, 1))
		return true;
	    if (comparePair(n1, n2, 2, 3))
		return true;
		return false;*/
	}
	void getInfo(String arr[]) {
	    arr[0] = "transmission line";
	    arr[1] = getUnitText(imped, ohmString);
	    arr[2] = "length = " + getUnitText(2.9979e8*delay, "m");
	    arr[3] = "delay = " + getUnitText(delay, "s");
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Delay (ns)", delay*1e9, 0, 0);
	    if (n == 1)
		return new EditInfo("Impedance (ohms)", imped, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0) {
		delay = ei.value*1e-9;
		reset();
	    }
	    if (n == 1) {
		imped = ei.value;
		reset();
	    }
	}
    }

    class MemristorElm extends CircuitElm {
	double r_on, r_off, dopeWidth, totalWidth, mobility, resistance;
	public MemristorElm(int xx, int yy) {
	    super(xx, yy);
	    r_on = 100;
	    r_off = 160*r_on;
	    dopeWidth = 0;
	    totalWidth = 10e-9;
	    mobility = 1e-10;
	    resistance = 100;
	}
	public MemristorElm(int xa, int ya, int xb, int yb, int f,
		    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    r_on = new Double(st.nextToken()).doubleValue();
	    r_off = new Double(st.nextToken()).doubleValue();
	    dopeWidth = new Double(st.nextToken()).doubleValue();
	    totalWidth = new Double(st.nextToken()).doubleValue();
	    mobility = new Double(st.nextToken()).doubleValue();
	    resistance = 100;
	}
	int getDumpType() { return 'm'; }
	String dump() {
	    return super.dump() + " " + r_on + " " + r_off + " " + dopeWidth + " " +
		totalWidth + " " + mobility;
	}

	Point ps3, ps4;
	void setPoints() {
	    super.setPoints();
	    calcLeads(32);
	    ps3 = new Point();
	    ps4 = new Point();
	}
	
	void draw(Graphics g) {
	    int segments = 6;
	    int i;
	    int ox = 0;
	    double v1 = volts[0];
	    double v2 = volts[1];
	    int hs = 2+(int) (8*(1-dopeWidth/totalWidth));
	    setBbox(point1, point2, hs);
	    draw2Leads(g);
	    setPowerColor(g, true);
	    double segf = 1./segments;

	    // draw zigzag
	    for (i = 0; i <= segments; i++) {
		int nx = (i & 1) == 0 ? 1 : -1;
		if (i == segments)
		    nx = 0;
		double v = v1+(v2-v1)*i/segments;
		setVoltageColor(g, v);
		interpPoint(lead1, lead2, ps1, i*segf, hs*ox);
		interpPoint(lead1, lead2, ps2, i*segf, hs*nx);
		drawThickLine(g, ps1, ps2);
		if (i == segments)
		    break;
		interpPoint(lead1, lead2, ps1, (i+1)*segf, hs*nx);
		drawThickLine(g, ps1, ps2);
		ox = nx;
	    }
	    
	    doDots(g);
	    drawPosts(g);
	}
    
	boolean nonLinear() { return true; }
	void calculateCurrent() {
	    current = (volts[0]-volts[1])/resistance;
	}
	void reset() {
	    dopeWidth = 0;
	}
	void startIteration() {
	    double wd = dopeWidth/totalWidth;
	    dopeWidth += timeStep*mobility*r_on*current/totalWidth;
	    if (dopeWidth < 0)
		dopeWidth = 0;
	    if (dopeWidth > totalWidth)
		dopeWidth = totalWidth;
	    resistance = r_on * wd + r_off * (1-wd);
	}
	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	}
	void doStep() {
	    stampResistor(nodes[0], nodes[1], resistance);
	}
	void getInfo(String arr[]) {
	    arr[0] = "memristor";
	    getBasicInfo(arr);
	    arr[3] = "R = " + getUnitText(resistance, ohmString);
	    arr[4] = "P = " + getUnitText(getPower(), "W");
	}
	double getScopeValue(int x) {
	    return (x == 2) ? resistance : (x == 1) ? getPower() : getVoltageDiff();
	}
	String getScopeUnits(int x) {
	    return (x == 2) ? ohmString : (x == 1) ? "W" : "V";
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Max Resistance (ohms)", r_on, 0, 0);
	    if (n == 1)
		return new EditInfo("Min Resistance (ohms)", r_off, 0, 0);
	    if (n == 2)
		return new EditInfo("Width of Doped Region (nm)", dopeWidth*1e9, 0, 0);
	    if (n == 3)
		return new EditInfo("Total Width (nm)", totalWidth*1e9, 0, 0);
	    if (n == 4)
		return new EditInfo("Mobility (nm^2/(s*V))", mobility*1e12, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		r_on = ei.value;
	    if (n == 1)
		r_off = ei.value;
	    if (n == 2)
		dopeWidth = ei.value*1e-9;
	    if (n == 3)
		totalWidth = ei.value*1e-9;
	    if (n == 4)
		mobility = ei.value*1e-12;
	}
    }
    
    // factors a matrix into upper and lower triangular matrices by
    // gaussian elimination.  On entry, a[0..n-1][0..n-1] is the
    // matrix to be factored.  ipvt[] returns an integer vector of pivot
    // indices, used in the lu_solve() routine.
    boolean lu_factor(double a[][], int n, int ipvt[]) {
	double scaleFactors[];
	int i,j,k;

	scaleFactors = new double[n];
	
        // divide each row by its largest element, keeping track of the
	// scaling factors
	for (i = 0; i != n; i++) { 
	    double largest = 0;
	    for (j = 0; j != n; j++) {
		double x = Math.abs(a[i][j]);
		if (x > largest)
		    largest = x;
	    }
	    // if all zeros, it's a singular matrix
	    if (largest == 0)
		return false;
	    scaleFactors[i] = 1.0/largest;
	}
	
        // use Crout's method; loop through the columns
	for (j = 0; j != n; j++) {
	    
	    // calculate upper triangular elements for this column
	    for (i = 0; i != j; i++) {
		double q = a[i][j];
		for (k = 0; k != i; k++)
		    q -= a[i][k]*a[k][j];
		a[i][j] = q;
	    }

	    // calculate lower triangular elements for this column
	    double largest = 0;
	    int largestRow = -1;
	    for (i = j; i != n; i++) {
		double q = a[i][j];
		for (k = 0; k != j; k++)
		    q -= a[i][k]*a[k][j];
		a[i][j] = q;
		double x = Math.abs(q);
		if (x >= largest) {
		    largest = x;
		    largestRow = i;
		}
	    }
	    
	    // pivoting
	    if (j != largestRow) {
		double x;
		for (k = 0; k != n; k++) {
		    x = a[largestRow][k];
		    a[largestRow][k] = a[j][k];
		    a[j][k] = x;
		}
		scaleFactors[largestRow] = scaleFactors[j];
	    }

	    // keep track of row interchanges
	    ipvt[j] = largestRow;

	    // avoid zeros
	    if (a[j][j] == 0.0) {
		System.out.println("avoided zero");
		a[j][j]=1e-18;
	    }

	    if (j != n-1) {
		double mult = 1.0/a[j][j];
		for (i = j+1; i != n; i++)
		    a[i][j] *= mult;
	    }
	}
	return true;
    }

    // Solves the set of n linear equations using a LU factorization
    // previously performed by lu_factor.  On input, b[0..n-1] is the right
    // hand side of the equations, and on output, contains the solution.
    void lu_solve(double a[][], int n, int ipvt[], double b[]) {
	int i;

	// find first nonzero b element
	for (i = 0; i != n; i++) {
	    int row = ipvt[i];

	    double swap = b[row];
	    b[row] = b[i];
	    b[i] = swap;
	    if (swap != 0)
		break;
	}
	
	int bi = i++;
	for (; i < n; i++) {
	    int row = ipvt[i];
	    int j;
	    double tot = b[row];
	    
	    b[row] = b[i];
	    // forward substitution using the lower triangular matrix
	    for (j = bi; j < i; j++)
		tot -= a[i][j]*b[j];
	    b[i] = tot;
	}
	for (i = n-1; i >= 0; i--) {
	    double tot = b[i];
	    
	    // back-substitution using the upper triangular matrix
	    int j;
	    for (j = i+1; j != n; j++)
		tot -= a[i][j]*b[j];
	    b[i] = tot/a[i][i];
	}
    }

    class EditDialogLayout implements LayoutManager {
	public EditDialogLayout() {}
	public void addLayoutComponent(String name, Component c) {}
	public void removeLayoutComponent(Component c) {}
	public Dimension preferredLayoutSize(Container target) {
	    return new Dimension(500, 500);
	}
	public Dimension minimumLayoutSize(Container target) {
	    return new Dimension(100,100);
	}
	public void layoutContainer(Container target) {
	    Insets insets = target.insets();
	    int targetw = target.size().width - insets.left - insets.right;
	    int targeth = target.size().height - (insets.top+insets.bottom);
	    int i;
	    int h = insets.top;
	    int pw = 300;
	    int x = 0;
	    for (i = 0; i < target.getComponentCount(); i++) {
		Component m = target.getComponent(i);
		boolean newline = true;
		if (m.isVisible()) {
		    Dimension d = m.getPreferredSize();
		    if (pw < d.width)
			pw = d.width;
		    if (m instanceof Scrollbar) {
			h += 10;
			d.width = targetw-x;
		    }
		    if (m instanceof Choice && d.width > targetw)
			d.width = targetw-x;
		    if (m instanceof Label) {
			Dimension d2 =
			    target.getComponent(i+1).getPreferredSize();
			if (d.height < d2.height)
			    d.height = d2.height;
			h += d.height/5;
			newline = false;
		    }
		    if (m instanceof Button) {
			if (x == 0)
			    h += 20;
			if (i != target.getComponentCount()-1)
			    newline = false;
		    }
		    m.move(insets.left+x, h);
		    m.resize(d.width, d.height);
		    if (newline) {
			h += d.height;
			x = 0;
		    } else
			x += d.width;
		}
	    }
	    if (target.size().height < h)
		target.resize(pw + insets.right, h + insets.bottom);
	}
    };

    class EditInfo {
	EditInfo(String n, double val, double mn, double mx) {
	    name = n;
	    value = val;
	    if (minval == 0 && maxval == 0 && val > 0) {
		minval = 1e10;
		while (minval > val/100)
		    minval /= 10.;
		maxval = minval * 1000;
	    } else {
		minval = mn;
		maxval = mx;
	    }
	}
	String name, text;
	double value, minval, maxval;
	TextField textf;
	Scrollbar bar;
	Choice choice;
	Checkbox checkbox;
	boolean newDialog;
    }
    
    class EditDialog extends Dialog
	             implements AdjustmentListener, ActionListener,
                                ItemListener {
	CircuitElm elm;
	CircuitFrame cframe;
	Button applyButton, okButton;
	EditInfo einfos[];
	int einfocount;
	final int barmax = 1000;
	
	EditDialog(CircuitElm ce, CircuitFrame f) {
	    super(f, "Edit Component", false);
	    cframe = f;
	    elm = ce;
	    setLayout(new EditDialogLayout());
	    einfos = new EditInfo[10];
	    int i;
	    for (i = 0; ; i++) {
		einfos[i] = elm.getEditInfo(i);
		if (einfos[i] == null)
		    break;
		EditInfo ei = einfos[i];
		add(new Label(ei.name));
		if (ei.choice != null) {
		    add(ei.choice);
		    ei.choice.addItemListener(this);
		} else if (ei.checkbox != null) {
		    add(ei.checkbox);
		    ei.checkbox.addItemListener(this);
		} else {
		    add(ei.textf =
			new TextField(noCommaFormat.format(ei.value), 10));
		    if (ei.text != null)
			ei.textf.setText(ei.text);
		    ei.textf.addActionListener(this);
		    if (ei.text == null) {
			add(ei.bar = new Scrollbar(Scrollbar.HORIZONTAL,
						   50, 10, 0, barmax+2));
			setBar(ei);
			ei.bar.addAdjustmentListener(this);
		    }
		}
	    }
	    einfocount = i;
	    add(applyButton = new Button("Apply"));
	    applyButton.addActionListener(this);
	    add(okButton = new Button("OK"));
	    okButton.addActionListener(this);
	    Point x = main.getLocationOnScreen();
	    Dimension d = getSize();
	    setLocation(x.x + (winSize.width-d.width)/2,
			x.y + (winSize.height-d.height)/2);
	}

	void apply() {
	    int i;
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (ei.textf == null)
		    continue;
		if (ei.text == null) {
		    try {
			double d = noCommaFormat.parse(ei.textf.getText()).doubleValue();
			ei.value = d;
		    } catch (Exception ex) { /* ignored */ }
		}
		elm.setEditValue(i, ei);
		if (ei.text == null)
		    setBar(ei);
	    }
	    cframe.needAnalyze();
	}
	
	public void actionPerformed(ActionEvent e) {
	    int i;
	    Object src = e.getSource();
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (src == ei.textf) {
		    if (ei.text == null) {
			try {
			    double d = noCommaFormat.parse(ei.textf.getText()).doubleValue();
			    ei.value = d;
			} catch (Exception ex) { /* ignored */ }
		    }
		    elm.setEditValue(i, ei);
		    if (ei.text == null)
			setBar(ei);
		    cframe.needAnalyze();
		}
	    }
	    if (e.getSource() == okButton) {
		apply();
		main.requestFocus();
		setVisible(false);
		editDialog = null;
	    }
	    if (e.getSource() == applyButton)
		apply();
	}
	
	public void adjustmentValueChanged(AdjustmentEvent e) {
	    Object src = e.getSource();
	    int i;
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (ei.bar == src) {
		    double v = ei.bar.getValue() / 1000.;
		    if (v < 0)
			v = 0;
		    if (v > 1)
			v = 1;
		    ei.value = (ei.maxval-ei.minval)*v + ei.minval;
		    if (ei.maxval-ei.minval > 100)
			ei.value = Math.round(ei.value);
		    else
			ei.value = Math.round(ei.value*100)/100.;
 		    elm.setEditValue(i, ei);
		    ei.textf.setText(noCommaFormat.format(ei.value));
		    cframe.needAnalyze();
		}
	    }
	}

	public void itemStateChanged(ItemEvent e) {
	    Object src = e.getItemSelectable();
	    int i;
	    boolean changed = false;
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (ei.choice == src || ei.checkbox == src) {
		    elm.setEditValue(i, ei);
		    if (ei.newDialog)
			changed = true;
		    cframe.needAnalyze();
		}
	    }
	    if (changed) {
		setVisible(false);
		editDialog = new EditDialog(elm, cframe);
		editDialog.show();
	    }
	}
	
	public boolean handleEvent(Event ev) {
	    if (ev.id == Event.WINDOW_DESTROY) {
		main.requestFocus();
		setVisible(false);
		editDialog = null;
		return true;
	    }
	    return super.handleEvent(ev);
	}

	void setBar(EditInfo ei) {
	    int x = (int) (barmax*(ei.value-ei.minval)/(ei.maxval-ei.minval));
	    ei.bar.setValue(x);
	}
    }

    class ImportDialogLayout implements LayoutManager {
	public ImportDialogLayout() {}
	public void addLayoutComponent(String name, Component c) {}
	public void removeLayoutComponent(Component c) {}
	public Dimension preferredLayoutSize(Container target) {
	    return new Dimension(500, 500);
	}
	public Dimension minimumLayoutSize(Container target) {
	    return new Dimension(100,100);
	}
	public void layoutContainer(Container target) {
	    Insets insets = target.insets();
	    int targetw = target.size().width - insets.left - insets.right;
	    int targeth = target.size().height - (insets.top+insets.bottom);
	    int i;
	    int pw = 300;
	    if (target.getComponentCount() == 0)
		return;
	    Component cl = target.getComponent(target.getComponentCount()-1);
	    Dimension dl = cl.getPreferredSize();
	    target.getComponent(0).move(insets.left, insets.top);
	    int cw = target.size().width - insets.left - insets.right;
	    int ch = target.size().height - insets.top - insets.bottom -
		dl.height;
	    target.getComponent(0).resize(cw, ch);
	    int h = ch + insets.top;
	    int x = 0;
	    for (i = 1; i < target.getComponentCount(); i++) {
		Component m = target.getComponent(i);
		if (m.isVisible()) {
		    Dimension d = m.getPreferredSize();
		    m.move(insets.left+x, h);
		    m.resize(d.width, d.height);
		    x += d.width;
		}
	    }
	}
    };

    class ImportDialog extends Dialog implements ActionListener {
	CircuitFrame cframe;
	Button importButton, closeButton;
	TextArea text;
	
	ImportDialog(CircuitFrame f, String str) {
	    super(f, (str.length() > 0) ? "Export" : "Import", false);
	    cframe = f;
	    setLayout(new ImportDialogLayout());
	    add(text = new TextArea(str, 10, 60, TextArea.SCROLLBARS_BOTH));
	    add(importButton = new Button("Import"));
	    importButton.addActionListener(this);
	    add(closeButton = new Button("Close"));
	    closeButton.addActionListener(this);
	    Point x = main.getLocationOnScreen();
	    resize(400, 300);
	    Dimension d = getSize();
	    setLocation(x.x + (winSize.width-d.width)/2,
			x.y + (winSize.height-d.height)/2);
	    show();
	    if (str.length() > 0)
		text.selectAll();
	}

	public void actionPerformed(ActionEvent e) {
	    int i;
	    Object src = e.getSource();
	    if (src == importButton) {
		cframe.readSetup(text.getText());
		setVisible(false);
	    }
	    if (src == closeButton)
		setVisible(false);
	}
	
	public boolean handleEvent(Event ev) {
	    if (ev.id == Event.WINDOW_DESTROY) {
		main.requestFocus();
		setVisible(false);
		editDialog = null;
		return true;
	    }
	    return super.handleEvent(ev);
	}
    }
    
}
