/*
 * 01/09/2002 - 20:43:57
 *
 * JDictionary.java -
 * Copyright (C) 2002 Csaba Kertsz
 * kcsaba@jdictionary.info
 * www.jdictionary.info
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package info.jdictionary;

import info.jdictionary.pluginstuff.PluginManager;
import info.jdictionary.pluginstuff.PluginDownloader;
import info.jdictionary.pluginstuff.PluginStatusChanger;
import info.jdictionary.gui.UpgradeAgent;
import info.jdictionary.gui.AboutPanel;
import info.jdictionary.gui.NoPluginsPanel;
import info.jdictionary.gui.AllPluginsDeactivatedPanel;
import java.io.File;
import java.util.ResourceBundle;
import java.util.Locale;
import javax.swing.JFrame;
import javax.swing.JPanel;
import java.net.URL;
import java.net.Authenticator;
import java.net.PasswordAuthentication;


public class JDictionary {

    private static ResourceBundle JDictionaryResources;
    private static JDictionary jDictionary;

    private Prefs prefs;
    private PluginManager pluginManager;
    private PluginDownloader pluginDownloader;
    private PluginStatusChanger pluginStatusChanger;
    private NewsAgent newsAgent;
    private UpgradeAgent upgradeAgent;
    private Preferences preferences;
    private JDictionaryFrame frame;
    private DefaultFrameAssembler assembler;
    private NewJDictionaryVersionChecker newJDictionaryVersionChecker = new NewJDictionaryVersionChecker();
    private NewsChecker newsChecker = new NewsChecker(this);
    private String baseDir;


    public JDictionary() {
        prefs = loadPrefs();
        initProxy();
        frame = new JDictionaryFrame();
        pluginManager = new PluginManager(this);
        assembler = new DefaultFrameAssembler(this, frame);
        pluginManager.scanPlugins();
        if(prefs.CheckingForUpgrade)
            newJDictionaryVersionChecker.start(); //checking for new jdictionary version
        if(prefs.CheckingForNews)
            newsChecker.start(); //checking for news
    }


    public JPanel getSplashScreen() {
        return new AboutPanel();
    }


    public JPanel getNoPluginInstalledScreen() {
        return new NoPluginsPanel();
    }


    public JPanel getAllPluginsDeactivatedScreen() {
        return new AllPluginsDeactivatedPanel();
    }


    public PluginStatusChanger getPluginStatusChanger() {
        if(pluginStatusChanger == null) {
            pluginStatusChanger = new PluginStatusChanger(this);
            Thread pluginStatusChangerThread = new Thread(pluginStatusChanger);
            pluginStatusChangerThread.start();
        }
        return pluginStatusChanger;
    }


    public PluginDownloader getPluginDownloader() {
        if(pluginDownloader == null || pluginDownloader.isSuccess() == false) {
            pluginDownloader = new PluginDownloader(this);
            Thread pluginDownloaderThread = new Thread(pluginDownloader);
            pluginDownloaderThread.start();
        }
        return pluginDownloader;
    }


    public NewsAgent getNewsAgent() {
        if(newsAgent == null || newsAgent.isSuccess() == false) {
            newsAgent = new NewsAgent(this);
            Thread newsAgentThread = new Thread(newsAgent);
            newsAgentThread.start();
        }
        return newsAgent;
    }


    public UpgradeAgent getUpgradeAgent() {
        if(upgradeAgent == null || upgradeAgent.isSuccess() == false)
            upgradeAgent = new UpgradeAgent(this);
        return upgradeAgent;
    }


    public Preferences getPreferencesScreen() {
        if(preferences == null)
            preferences = new Preferences(this);
        return preferences;
    }


    public NewJDictionaryVersionChecker getNewJDictionaryVersionChecker() {
        return newJDictionaryVersionChecker;
    }


    public NewsChecker getNewsChecker() {
        return newsChecker;
    }


    public DefaultFrameAssembler getAssembler() {
        return assembler;
    }


    public void closeJDictionary() {
        savePrefs();
        pluginManager.removeAllPlugins();
        System.exit(0);
    }


    public void savePrefs() {
        assembler.savePrefs();
        //other items in prefs have already been adjusted
        prefs.lastSelectedPlugin = pluginManager.getSelectedPlugin();
        prefs.lastSelectedSubPluginName = pluginManager.getSelectedSubPlugin();
        Prefs.savePrefs(prefs, getSavedDir() + getFileSeparator() + getString("SavedPrefsFileName"));
    }


    public Prefs loadPrefs() {
        return Prefs.loadPrefs(getSavedDir() + getFileSeparator() + getString("SavedPrefsFileName"));
    }


    public PluginManager getPluginManager() {
        return pluginManager;
    }


    public Prefs getPrefs() {
        return prefs;
    }


    public String getBaseDir() {
        if(baseDir != null)
            return baseDir;
        URL url = getClass().getResource("/" + getString("ResourcesDirName"));
        String temp = url.toString();
        StringBuffer s = new StringBuffer();
        for(int i = 0; i < temp.length(); i++) {
            if(temp.charAt(i) != '%')
                s.append(temp.charAt(i));
            else {
                s.append(" ");
                i = i + 2;
            }
        }
        int e = s.toString().indexOf("jdictionary.jar");
        if(e < 10)
            return System.getProperty("user.dir");
        baseDir = s.toString().substring(9, e-1);
        return baseDir;
    }


    public String getPluginDirPath() {
        File dir = new File(getSavedDir() + getFileSeparator() + getString("PluginDirName"));
        if(!dir.exists() || !dir.isDirectory())
            dir.mkdir();
        return dir.getAbsolutePath();
    }


    public String getGlobalPluginDirPath() {
         File dir = null;
         try {
             dir = new File(getBaseDir() + getFileSeparator() + getString("PluginDirName"));
             if(!dir.exists() || !dir.isDirectory())
                 if(!dir.mkdir())
                     return null;
         }
         catch(java.lang.Exception e) {return null;}
         return dir.getAbsolutePath();
     }


//////////static methods/////////////////////////////////


     public static void main(String[] args) {
         jDictionary = new JDictionary();
    }


    public static JDictionary getJDictionary() {
        return jDictionary;
    }


    public static String getString(String key) {
        if (JDictionaryResources == null)
            JDictionaryResources = ResourceBundle.getBundle("info.jdictionary.resource.JDictionaryResources",new Locale("en","en"));
        return JDictionaryResources.getString(key);
    }


    public static float getJDictionaryVersion() {
        return Float.parseFloat(getString("JDictionaryVersion"));
    }


    public static String getUserHome() {
        return System.getProperty("user.home");
    }


    public static String getFileSeparator() {
        return System.getProperty("file.separator");
    }


    public static String getLanguage() {
        return System.getProperty("user.language");
    }


    public static String getSavedDir() {
        File dir = new File(getUserHome() + getFileSeparator() + getString("SavedDirName"));
        if(!dir.exists() || !dir.isDirectory())
            dir.mkdir();
        return dir.getAbsolutePath();
    }


    public void initProxy() {
        // cleaning up proxy settings
        System.getProperties().remove("proxySet");
        System.getProperties().remove("proxyHost");
        System.getProperties().remove("proxyPort");
        System.getProperties().remove("socksProxySet");
        System.getProperties().remove("socksProxyHost");
        System.getProperties().remove("socksProxyPort");
        Authenticator.setDefault(null);

        if (prefs.usingHttpProxy) {
            //set proxy host
            String host = prefs.httpProxyServer;
            if (host == null)
                return;
            System.setProperty("proxyHost", host);

            // set proxy port
            String port = prefs.httpProxyPort;
            if (port != null && port.length() > 0)
                System.setProperty("proxyPort", port);
            else
                System.setProperty("proxyPort", "80"); //http proxy default
            System.setProperty("proxySet", "true");
            // set proxy authentication
            if(prefs.usingHttpProxyLogin) {
                String username = prefs.httpProxyLogin;
                char[] password = prefs.httpProxyPassword;

                if(password == null)
                    password = new char[0];

                if(username == null || username.length() == 0) {
                    Authenticator.setDefault(new ProxyAuthenticator(null));
                }
                else {
                    PasswordAuthentication passwordAuthentication = new PasswordAuthentication(username, password);
                    Authenticator.setDefault(new ProxyAuthenticator(passwordAuthentication));
                }
            }
        }

        if(prefs.usingSocksProxy) {
        //set socks proxy host
           String host = prefs.socksProxyServer;
           if (host == null)
               return;
           System.getProperties().setProperty("socksProxyHost", host);

           // set socks proxy port
           String port = prefs.socksProxyPort;
           if (port != null && port.length() > 0)
               System.getProperties().setProperty("socksProxyPort", port);
           else
               System.getProperties().setProperty("socksProxyPort", "1080");  //socks proxy default
           System.getProperties().setProperty("socksProxySet", "true");
           // set proxy authentication
           if(prefs.usingSocksProxyLogin) {
               String username = prefs.socksProxyLogin;
               char[] password = prefs.socksProxyPassword;

               if(password == null)
                   password = new char[0];

               if(username == null || username.length() == 0) {
                   Authenticator.setDefault(new ProxyAuthenticator(null));
               }
               else {
                   PasswordAuthentication passwordAuthentication = new PasswordAuthentication(username, password);
                   Authenticator.setDefault(new ProxyAuthenticator(passwordAuthentication));
               }
           }
        }
    }
}