/*
 * 01/09/2002 - 20:43:57
 *
 * PluginManager.java -
 * Copyright (C) 2002 Csaba Kertsz
 * kcsaba@jdictionary.info
 * www.jdictionar.info
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


package info.jdictionary.pluginstuff;

import info.jdictionary.JDictionary;
import info.jdictionary.Plugin;
import info.jdictionary.Prefs;
import java.util.HashMap;
import java.util.Vector;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Enumeration;
import java.io.File;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;
import info.jdictionary.events.PluginStructureChangeEvent;
import info.jdictionary.events.PluginFilesChangeEvent;
import info.jdictionary.listeners.PluginStructureChangeListener;
import info.jdictionary.listeners.PluginFilesChangeListener;
import info.jdictionary.events.PluginSelectionEvent;
import info.jdictionary.listeners.PluginSelectionListener;
import info.jdictionary.events.PluginScanFinishedEvent;
import info.jdictionary.listeners.PluginScanFinishedListener;


public class PluginManager {

    HashMap plugins = new HashMap();
    File[] pluginFiles;
    JDictionary jDictionary;
    PluginInfoSheet selectedPlugin;
    String selectedSubPlugin;
    Vector pluginStructureChangeListeners = new Vector();
    Vector pluginFilesChangeListeners = new Vector();
    Vector pluginSelectionListeners = new Vector();
    Vector pluginScanFinishedListeners = new Vector();
    ScanThread scanThread;


    public PluginManager(JDictionary jDictionary) {
        this.jDictionary = jDictionary;
    }


    void pushPlugin(PluginInfoSheet infoSheet, Plugin plugin) {
        if (!plugins.containsKey(infoSheet))
            plugins.put(infoSheet, new ArrayList());
        ((ArrayList)plugins.get(infoSheet)).add(plugin);
    }


    public ArrayList getPlugin(PluginInfoSheet infoSheet) {
        if (plugins.containsKey(infoSheet))
            return (ArrayList) plugins.get(infoSheet);
        return null;
    }


    void removePlugin(PluginInfoSheet infoSheet) {
        Iterator it = getPlugin(infoSheet).iterator();
        while (it.hasNext())
            ((Plugin)it.next()).stop();
        plugins.remove(infoSheet);
        if(selectedPlugin == infoSheet)
            this.selectPlugin(null);
        notifyPluginRemovedListeners(infoSheet);
    }


    public void removeAllPlugins() {
        Object[] sheets = getSheets();
        int n = 0;
        while (n < sheets.length)
            removePlugin((PluginInfoSheet) sheets[n++]);
    }


    public Object[] getSheets() {
        return plugins.keySet().toArray();
    }


    public int getNumberOfLoadedPlugins() {
        return plugins.size();
    }


    public int getNumberOfPluginFiles() {
        return getPluginFiles().length;
    }


    public void checkPluginDirForChanges() {
        File[] fileList = getPluginFiles();
        File[] knownPluginFiles = pluginFiles;

            if(knownPluginFiles == null && fileList != null)
                notifyPluginFilesChangedListeners(fileList);
            else
                if(knownPluginFiles.length != fileList.length)
                    notifyPluginFilesChangedListeners(fileList);
                else
                    for(int i = 0; i < knownPluginFiles.length; i++) {
                        int j;
                        for(j = 0; j < fileList.length; j++)
                            if(fileList[j].equals(knownPluginFiles[i]))
                                break;
                        if(j == fileList.length) {
                            notifyPluginFilesChangedListeners(fileList);
                            break;
                        }
                    }
        pluginFiles = fileList;
    }


    private File[] getGlobalPluginFiles() {
     String globalPluginDirPath = jDictionary.getGlobalPluginDirPath();

     if(globalPluginDirPath == null)
         return new File[0];

     String[] globalPlugins = new File(globalPluginDirPath).list();
        if(globalPlugins == null) //@GDX
            globalPlugins = new String[0];
        File[] files = new File[globalPlugins.length];
        for(int i = 0; i < globalPlugins.length; i++)
            files[i] = new File(globalPluginDirPath + jDictionary.getFileSeparator() + globalPlugins[i]);
        return files;
    }


    private File[] getLocalPluginFiles() {
        String localPluginDirPath = jDictionary.getPluginDirPath();

        if(localPluginDirPath == null)
            return new File[0];

        String[] localPlugins = new File(localPluginDirPath).list();

        if(localPlugins == null)
            localPlugins = new String[0];
        File[] files = new File[localPlugins.length];
        for(int i = 0; i < localPlugins.length; i++)
            files[i] = new File(localPluginDirPath + jDictionary.getFileSeparator() + localPlugins[i]);
        return files;
    }


    private File[] getAvailablePluginFiles() {
        File[] localPlugins = getLocalPluginFiles();
        File[] globalPlugins = getGlobalPluginFiles();
        File[] pluginFiles = new File[globalPlugins.length + localPlugins.length];
        int i;
        for(i = 0; i < globalPlugins.length; i++)
            pluginFiles[i] = globalPlugins[i];
        for(int j = 0; j < localPlugins.length; j++)
            pluginFiles[i++] = localPlugins[j];
        return pluginFiles;
    }


    public File[] getPluginFiles() {
        File[] allPluginFiles = getAvailablePluginFiles();
        ArrayList pluginFiles = new ArrayList();

        for(int i = 0; i < allPluginFiles.length; i++) {
            int j;
            if(isThereNewerPluginFileAvailable(allPluginFiles[i].getName()))
                continue;
            for(j = 0; j < pluginFiles.size(); j++) {
                if(((File)pluginFiles.get(j)).getName().equals(allPluginFiles[i].getName()))
                    break;
            }
            if(j >= pluginFiles.size())
                pluginFiles.add(allPluginFiles[i]);
        }

        File[] result = new File[pluginFiles.size()];
        for(int i = 0; i < result.length; i++) {
            result[i] = (File)pluginFiles.get(i);
        }
        return result;
    }


    //this method is a bit deprecated
    public boolean isItInstalled(PluginInfoSheet infoSheet) {
        Iterator sheetsIterator = plugins.keySet().iterator();
        while (sheetsIterator.hasNext()) {
            PluginInfoSheet tempSheet = (PluginInfoSheet)sheetsIterator.next();
            if (infoSheet.getFileName().equals(tempSheet.getFileName()))
                if (extractVersionNumberFromFileName(infoSheet.getFileName()) == extractVersionNumberFromFileName(tempSheet.getFileName()))
                    return true;
        }
        return false;
    }


    public float getNewestVersion(PluginInfoSheet infoSheet) {
        Iterator sheetsIterator = plugins.keySet().iterator();
        float newestVersion = 0.0f;
        while (sheetsIterator.hasNext()) {
            PluginInfoSheet tempSheet = (PluginInfoSheet) sheetsIterator.next();
            if (infoSheet.getFileName().equals(tempSheet.getFileName()))
                if (Float.parseFloat(tempSheet.getVersion()) > newestVersion)
                    newestVersion = Float.parseFloat(tempSheet.getVersion());
        }
        return newestVersion;
    }


  public float extractVersionNumberFromFileName(String fileName) {
        int s = fileName.lastIndexOf("-");
        int e = fileName.lastIndexOf(".jar");
        if(s > -1 && (s < e))
            return Float.parseFloat(new String(fileName.substring(s + 1, e)).replace('_', '.'));
        else
            return 0.0f;
      }


    public String getNameWithoutVersion(String fileName) {
        int s = fileName.lastIndexOf("-");
        if(s > -1)
            return fileName.substring(0, s);
        s = fileName.lastIndexOf(".jar");
        if(s > -1)
            return fileName.substring(0, s);
        return fileName;
    }


    public boolean isPluginFileLoadedWithName(String fileName) {
        Object[] sheets = getSheets();
        for (int n = 0; n < sheets.length; n++) {
            String name = ((PluginInfoSheet)sheets[n]).getFileName();
            if (name.equals(fileName))
                return true;
        }
        return false;
    }


    public boolean isPluginFileLoaded(File file) {
        Object[] sheets = getSheets();
        for (int n = 0; n < sheets.length; n++) {
            File tempFile = ((PluginInfoSheet)sheets[n]).getFile();
            if (file.equals(tempFile))
                return true;
        }
        return false;
    }


    public boolean isItActive(String fileName) {
        String name = getNameWithoutVersion(fileName);
        Iterator it = jDictionary.getPrefs().InActivePlugins.iterator();
        while (it.hasNext())
            if(((String)it.next()).equals(name))
                return false;
        return true;
    }


    public void selectPlugin(PluginInfoSheet sheet) {
        selectPlugin(sheet, null);
    }


    public void selectPlugin(PluginInfoSheet sheet, String subPluginName) {
        ArrayList plugin = getPlugin(sheet);

        if(sheet == null && subPluginName == null) {
            selectedPlugin = null;
            selectedSubPlugin = null;
            notifyPluginSelectedListeners();
            return;
        }

        if((sheet == selectedPlugin && subPluginName == selectedSubPlugin) || (sheet == selectedPlugin && subPluginName == null))
            return;
        if(plugin != null)
            selectedPlugin = sheet;
        else {
            selectedPlugin = null;
            selectedSubPlugin = null;
            notifyPluginSelectedListeners();
            return;
        }
        for(int i = 0; i < plugin.size(); i++) {
            if(plugin.get(i).toString().equals(subPluginName)) {
                selectedSubPlugin = subPluginName;
                notifyPluginSelectedListeners();
                return;
            }
        }
        selectedSubPlugin = null;
        notifyPluginSelectedListeners();
    }


    public PluginInfoSheet getSelectedPlugin() {
        return selectedPlugin;
    }


    public String getSelectedSubPlugin() {
        return selectedSubPlugin;
    }


    public void setActive(String fileName) {
        String name = getNameWithoutVersion(fileName);
        jDictionary.getPrefs().InActivePlugins.remove(name);
    }


    public void setInActive(String fileName) {
        String name = getNameWithoutVersion(fileName);
        jDictionary.getPrefs().InActivePlugins.add(name);
    }


    public void cleanUpInactivePluginList() {
        Iterator inactivePlugins = jDictionary.getPrefs().InActivePlugins.iterator();
        while(inactivePlugins.hasNext()) {
            if(!isItAvailable(inactivePlugins.next().toString(), false))
                inactivePlugins.remove();
        }
    }


    public boolean isThereNewerPluginFileAvailable(String fileName) {
        File[] pluginFiles = getAvailablePluginFiles();
        float version = extractVersionNumberFromFileName(fileName);
        for(int i = 0; i < pluginFiles.length; i++) {
            if(getNameWithoutVersion(pluginFiles[i].getName()).equals(getNameWithoutVersion(fileName))) {
                float tempVersion = extractVersionNumberFromFileName(pluginFiles[i].getName());
                if (tempVersion > version)
                    return true;
            }
        }
        return false;
    }


      public boolean isThereNewerGlobalPluginFileAvailable(String fileName) {
        File[] pluginFiles = getGlobalPluginFiles();
        float version = extractVersionNumberFromFileName(fileName);
        for(int i = 0; i < pluginFiles.length; i++) {
            if(getNameWithoutVersion(pluginFiles[i].getName()).equals(getNameWithoutVersion(fileName))) {
                float tempVersion = extractVersionNumberFromFileName(pluginFiles[i].getName());
                if (tempVersion > version)
                    return true;
            }
        }
        return false;
    }


    public boolean isThereNewerLocalPluginFileAvailable(String fileName) {
        File[] pluginFiles = getLocalPluginFiles();
        float version = extractVersionNumberFromFileName(fileName);
        for(int i = 0; i < pluginFiles.length; i++) {
            if(getNameWithoutVersion(pluginFiles[i].getName()).equals(getNameWithoutVersion(fileName))) {
                float tempVersion = extractVersionNumberFromFileName(pluginFiles[i].getName());
                if (tempVersion > version)
                    return true;
            }
        }
        return false;
    }


    public boolean isItAvailable(String fileName, boolean careWithVersion) {
            File[] files = getPluginFiles();
            for(int i = 0; i < files.length; i++)
                if(careWithVersion) {
                    if(files[i].getName().equals(fileName))
                        return true;
                }
                else
                    if(getNameWithoutVersion(files[i].getName()).equals(getNameWithoutVersion(fileName)))
                        return true;
            return false;
    }


    public boolean isExist(File pluginFile) {
        File[] pluginFiles = getPluginFiles();
        for(int i = 0; i < pluginFiles.length; i++)
            if(pluginFiles[i].equals(pluginFile))
                return true;
        return false;
    }


    public void scanPlugins() {
        new Thread() {
            public void run() {
                while(scanThread != null && scanThread.isAlive()) {
                    try {
                        Thread.sleep(100);
                    }
                    catch(java.lang.InterruptedException e) {}
                }
                scanThread = new ScanThread();
                scanThread.start();
            }
        }.start();
    }


    void cleanupPlugins() {
        Object[] sheets = getSheets();
        for (int n = 0; n < sheets.length; n++) {
            PluginInfoSheet sheet = (PluginInfoSheet)sheets[n];
            File pluginFile = sheet.getFile();
            if (!isExist(pluginFile) || isItActive(pluginFile.getName()) == false) {
                removePlugin(sheet);
            }
        }
    }


    long getSize(ZipFile zipFile) {
        File f = new File(zipFile.getName());
        return f.length();
    }


    void notifyPluginAddedListeners(PluginInfoSheet sheet) {
        Vector tmpList;
        PluginStructureChangeEvent event = new PluginStructureChangeEvent(this, sheet);
        synchronized(this) {
            tmpList = (Vector)pluginStructureChangeListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginStructureChangeListener)tmpList.elementAt(i)).pluginAdded(event);
        }
    }


    void notifyPluginRemovedListeners(PluginInfoSheet sheet) {
        Vector tmpList;
        PluginStructureChangeEvent event = new PluginStructureChangeEvent(this, sheet);
        synchronized(this) {
            tmpList = (Vector)pluginStructureChangeListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginStructureChangeListener)tmpList.elementAt(i)).pluginRemoved(event);
        }
    }


    void notifyPluginFilesChangedListeners(File[] fileList) {
        cleanUpInactivePluginList();
        Vector tmpList;
        PluginFilesChangeEvent event = new PluginFilesChangeEvent(this, fileList);
        synchronized(this) {
            tmpList = (Vector)pluginFilesChangeListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginFilesChangeListener)tmpList.elementAt(i)).pluginFilesChanged(event);
        }
    }


    void notifyPluginSelectedListeners() {
        Vector tmpList;
        PluginSelectionEvent event = new PluginSelectionEvent(this, selectedPlugin, selectedSubPlugin);
        synchronized(this) {
            tmpList = (Vector)pluginSelectionListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginSelectionListener)tmpList.elementAt(i)).pluginSelected(event);
        }
    }


    void notifyPluginScanFinishedListeners() {
        Vector tmpList;
        PluginScanFinishedEvent event = new PluginScanFinishedEvent(this, plugins);
        synchronized(this) {
            tmpList = (Vector)pluginScanFinishedListeners.clone();
        }
        for(int i = 0; i < tmpList.size(); i++) {
            ((PluginScanFinishedListener)tmpList.elementAt(i)).pluginScanFinished(event);
        }
    }


    public synchronized void addPluginStructureChangeListener(PluginStructureChangeListener listener) {
        pluginStructureChangeListeners.addElement(listener);
    }


    public synchronized void removePluginStructureChangeListener(PluginStructureChangeListener listener) {
        pluginStructureChangeListeners.removeElement(listener);
    }


    public synchronized void addPluginFilesChangeListener(PluginFilesChangeListener listener) {
        pluginFilesChangeListeners.addElement(listener);
    }


    public synchronized void removePluginFilesChangeListener(PluginFilesChangeListener listener) {
        pluginFilesChangeListeners.removeElement(listener);
    }


    public synchronized void addPluginSelectionListener(PluginSelectionListener listener) {
        pluginSelectionListeners.addElement(listener);
    }


    public synchronized void removePluginSelectionListener(PluginSelectionListener listener) {
        pluginSelectionListeners.removeElement(listener);
    }


    public synchronized void addPluginScanFinishedListener(PluginScanFinishedListener listener) {
        pluginScanFinishedListeners.addElement(listener);
    }


    public synchronized void removePluginScanFinishedListener(PluginScanFinishedListener listener) {
        pluginScanFinishedListeners.removeElement(listener);
    }


    class ScanThread extends Thread {

        public void run() {
            cleanupPlugins();
            checkPluginDirForChanges();
            boolean loadLastUsed = true;
            int i = 0;
            File[] baseDirFiles = getPluginFiles();
            //this will run through the plugin files, and asks
            //the PluginManager, whether the forthcoming plugin file
            //has been loaded or not. If not, this will do the loading
            //and plugin registering process.
            while (i < baseDirFiles.length) {
                File fileToLoad = baseDirFiles[i];
                try {
                    //trying to load the last used plugin first
                    if(loadLastUsed) {
                        loadLastUsed = false;
                        Prefs prefs = jDictionary.getPrefs();
                        if(prefs.lastSelectedPlugin != null && isItAvailable(prefs.lastSelectedPlugin.getFileName(), true) && isItActive(prefs.lastSelectedPlugin.getFileName())) {
                            fileToLoad = new File(jDictionary.getGlobalPluginDirPath() + jDictionary.getFileSeparator() + jDictionary.getPrefs().lastSelectedPlugin.getFileName());
                            if(!fileToLoad.exists())
                                fileToLoad = new File(jDictionary.getPluginDirPath() + jDictionary.getFileSeparator() + jDictionary.getPrefs().lastSelectedPlugin.getFileName());
                            i = -1;
                        }
                    }

                    //skipping the files which belong to inactive plugins.
                    if(isItActive(fileToLoad.getName()) == false) {
                        i++;
                        continue;
                    }

                    //if plugin file fileToLoad is already loaded then skipping this file.
                    if (isPluginFileLoaded(fileToLoad)) {
                        System.out.println("Plugin " + fileToLoad.getName() + " is already loaded");
                        i++;
                        continue;
                    }
                    ZipFile jarFile = null;
                    try {
                        jarFile = new ZipFile(fileToLoad);
                    }
                    catch(java.util.zip.ZipException e) {
                        System.out.println("The plugin file named: " + fileToLoad.getName() + " is corrupted. Trying to delete...");
                        try {
                            fileToLoad.delete();
                        }
                        catch(java.lang.Exception ex) {}
                        i++;
                        continue;
                    }
                    System.out.println("\nscanPlugins(): Processing file: " + jarFile.getName());
                    if (jarFile != null) {
                        System.out.println("Loading InfoSheet from file named: " + jarFile.getName());
                        final PluginInfoSheet sheet = PluginInfoSheet.CreateInfoSheet(fileToLoad);
                        if (sheet == null) { //skipping file if no infosheet file was fount in it.
                            System.out.println("Unable to load InfoSheet from file named: " + jarFile.getName() + " Skipping...");
                            i++;
                            try {
                                jarFile.close();
                            }
                            catch (java.io.IOException ex) {}
                            continue;
                        }
                        if(Float.parseFloat(sheet.getMinJDictionaryVersion()) > JDictionary.getJDictionaryVersion()) {
                            System.out.println("Unable to load plugin named " + sheet.getName() + " (JDictionary version " + sheet.getMinJDictionaryVersion() + " or higher needed) Skipping...");
                            i++;
                            try {
                                jarFile.close();
                            }
                            catch (java.io.IOException ex) {}
                            continue;
                        }
                        sheet.setFileName(fileToLoad.getName()); //corecting fileName
                        sheet.setSize(Long.toString(fileToLoad.length())); //finding out and setting filesize
                        PluginLoader pluginLoader = new PluginLoader(jarFile.getName());
                        Enumeration entries = jarFile.entries();
                        while (entries.hasMoreElements()) {
                            String entryName = ((ZipEntry) entries.nextElement()).getName();
                            if (entryName.endsWith("Plugin.class")) {
                                //System.out.println("scanPlugins(): I want the PluginLoader to load the class named: " + entryName);
                                Class pluginClass = pluginLoader.loadClass(entryName);
                                final Object plugin = pluginClass.newInstance();
                                if (plugin instanceof Plugin) {
                                    pushPlugin(sheet, (Plugin) plugin);
                                }
                            }
                        }
                        notifyPluginAddedListeners(sheet);
                    }
                    i++;
                    jarFile.close();
                }
                catch(Exception e) {
                    System.out.println("scanPlugins(): An error occurred while loading plugin named: " + fileToLoad.getName() + " Skipping...");
                    e.printStackTrace();
                    i++;
                }
            }
            notifyPluginScanFinishedListeners();
        }
    }
}


    class DirFilter implements java.io.FilenameFilter {
        String afn;
        DirFilter(String afn) {
            this.afn = afn;
        }

        public boolean accept(File dir, String name) {
            String f = new File(name).getName();
            //return f.indexOf(afn) != -1;
            return f.endsWith(afn);
        }
}