/*
 * 01/28/2009
 *
 * PHPTokenMaker.java - Generates tokens for PHP syntax highlighting.
 * Copyright (C) 2009 Robert Futrell
 * robert_futrell at users.sourceforge.net
 * http://fifesoft.com/rsyntaxtextarea
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */
package org.fife.ui.rsyntaxtextarea.modes;

import java.io.*;
import javax.swing.text.Segment;

import org.fife.ui.rsyntaxtextarea.*;


/**
 * Scanner for PHP files.
 *
 * This implementation was created using
 * <a href="http://www.jflex.de/">JFlex</a> 1.4.1; however, the generated file
 * was modified for performance.  Memory allocation needs to be almost
 * completely removed to be competitive with the handwritten lexers (subclasses
 * of <code>AbstractTokenMaker</code>, so this class has been modified so that
 * Strings are never allocated (via yytext()), and the scanner never has to
 * worry about refilling its buffer (needlessly copying chars around).
 * We can achieve this because RText always scans exactly 1 line of tokens at a
 * time, and hands the scanner this line as an array of characters (a Segment
 * really).  Since tokens contain pointers to char arrays instead of Strings
 * holding their contents, there is no need for allocating new memory for
 * Strings.<p>
 *
 * The actual algorithm generated for scanning has, of course, not been
 * modified.<p>
 *
 * If you wish to regenerate this file yourself, keep in mind the following:
 * <ul>
 *   <li>The generated PHPTokenMaker.java</code> file will contain two
 *       definitions of both <code>zzRefill</code> and <code>yyreset</code>.
 *       You should hand-delete the second of each definition (the ones
 *       generated by the lexer), as these generated methods modify the input
 *       buffer, which we'll never have to do.</li>
 *   <li>You should also change the declaration/definition of zzBuffer to NOT
 *       be initialized.  This is a needless memory allocation for us since we
 *       will be pointing the array somewhere else anyway.</li>
 *   <li>You should NOT call <code>yylex()</code> on the generated scanner
 *       directly; rather, you should use <code>getTokenList</code> as you would
 *       with any other <code>TokenMaker</code> instance.</li>
 * </ul>
 *
 * @author Robert Futrell
 * @version 0.4
 *
 */
%%

%public
%class PHPTokenMaker
%extends AbstractMarkupTokenMaker
%unicode
%type org.fife.ui.rsyntaxtextarea.Token


%{

	/**
	 * Token type specific to HTMLTokenMaker; this signals that the user has
	 * ended a line with an unclosed HTML tag; thus a new line is beginning
	 * still inside of the tag.
	 */
	public static final int INTERNAL_INTAG					= -1;

	/**
	 * Token type specific to HTMLTokenMaker; this signals that the user has
	 * ended a line with an unclosed <code>&lt;script&gt;</code> tag.
	 */
	public static final int INTERNAL_INTAG_SCRIPT			= -2;

	/**
	 * Token type specifying we're in a double-qouted attribute in a
	 * script tag.
	 */
	public static final int INTERNAL_ATTR_DOUBLE_QUOTE_SCRIPT = -3;

	/**
	 * Token type specifying we're in a single-qouted attribute in a
	 * script tag.
	 */
	public static final int INTERNAL_ATTR_SINGLE_QUOTE_SCRIPT = -4;

	/**
	 * Token type specifying we're in JavaScript.
	 */
	public static final int INTERNAL_IN_JS					= -5;

	/**
	 * Token type specifying we're in a JavaScript multiline comment.
	 */
	public static final int INTERNAL_IN_JS_MLC				= -6;

	/**
	 * Token type specifying we're in PHP.
	 */
	public static final int INTERNAL_IN_PHP					= -7;

	/**
	 * Token type specifying we're in a PHP multiline comment.
	 */
	public static final int INTERNAL_IN_PHP_MLC				= -8;

	/**
	 * Token type specifying we're in a PHP multiline string.
	 */
	public static final int INTERNAL_PHP_STRING				= -9;

	/**
	 * Token type specifying we're in a PHP multiline char.
	 */
	public static final int INTERNAL_PHP_CHAR				= -10;

	/**
	 * Whether closing markup tags are automatically completed for PHP.
	 */
	private static boolean completeCloseTags;


	/**
	 * Constructor.  This must be here because JFlex does not generate a
	 * no-parameter constructor.
	 */
	public PHPTokenMaker() {
		super();
	}


	/**
	 * Adds the token specified to the current linked list of tokens as an
	 * "end token;" that is, at <code>zzMarkedPos</code>.
	 *
	 * @param tokenType The token's type.
	 */
	private void addEndToken(int tokenType) {
		addToken(zzMarkedPos,zzMarkedPos, tokenType);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param tokenType The token's type.
	 */
	private void addToken(int tokenType) {
		addToken(zzStartRead, zzMarkedPos-1, tokenType);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param tokenType The token's type.
	 */
	private void addToken(int start, int end, int tokenType) {
		int so = start + offsetShift;
		addToken(zzBuffer, start,end, tokenType, so);
	}


	/**
	 * Adds the token specified to the current linked list of tokens.
	 *
	 * @param array The character array.
	 * @param start The starting offset in the array.
	 * @param end The ending offset in the array.
	 * @param tokenType The token's type.
	 * @param startOffset The offset in the document at which this token
	 *                    occurs.
	 */
	public void addToken(char[] array, int start, int end, int tokenType, int startOffset) {
		super.addToken(array, start,end, tokenType, startOffset);
		zzStartRead = zzMarkedPos;
	}


	/**
	 * Sets whether markup close tags should be completed.  You might not want
	 * this to be the case, since some tags in standard HTML aren't usually
	 * closed.
	 *
	 * @return Whether closing markup tags are completed.
	 * @see #setCompleteCloseTags(boolean)
	 */
	public boolean getCompleteCloseTags() {
		return completeCloseTags;
	}


	/**
	 * {@inheritDoc}
	 */
	public boolean getMarkOccurrencesOfTokenType(int type) {
		return false;
	}


	/**
	 * Returns the first token in the linked list of tokens generated
	 * from <code>text</code>.  This method must be implemented by
	 * subclasses so they can correctly implement syntax highlighting.
	 *
	 * @param text The text from which to get tokens.
	 * @param initialTokenType The token type we should start with.
	 * @param startOffset The offset into the document at which
	 *        <code>text</code> starts.
	 * @return The first <code>Token</code> in a linked list representing
	 *         the syntax highlighted text.
	 */
	public Token getTokenList(Segment text, int initialTokenType, int startOffset) {

		resetTokenList();
		this.offsetShift = -text.offset + startOffset;

		// Start off in the proper state.
		int state = Token.NULL;
		switch (initialTokenType) {
			case Token.COMMENT_MULTILINE:
				state = COMMENT;
				start = text.offset;
				break;
			case Token.VARIABLE:
				state = DTD;
				start = text.offset;
				break;
			case INTERNAL_INTAG:
				state = INTAG;
				start = text.offset;
				break;
			case INTERNAL_INTAG_SCRIPT:
				state = INTAG_SCRIPT;
				start = text.offset;
				break;
			case Token.LITERAL_STRING_DOUBLE_QUOTE:
				state = INATTR_DOUBLE;
				start = text.offset;
				break;
			case Token.LITERAL_CHAR:
				state = INATTR_SINGLE;
				start = text.offset;
				break;
			case INTERNAL_ATTR_DOUBLE_QUOTE_SCRIPT:
				state = INATTR_DOUBLE_SCRIPT;
				start = text.offset;
				break;
			case INTERNAL_ATTR_SINGLE_QUOTE_SCRIPT:
				state = INATTR_SINGLE_SCRIPT;
				start = text.offset;
				break;
			case INTERNAL_IN_JS:
				state = JAVASCRIPT;
				start = text.offset;
				break;
			case INTERNAL_IN_JS_MLC:
				state = JS_MLC;
				start = text.offset;
				break;
			case INTERNAL_IN_PHP:
				state = PHP;
				start = text.offset;
				break;
			case INTERNAL_IN_PHP_MLC:
				state = PHP_MLC;
				start = text.offset;
				break;
			case INTERNAL_PHP_STRING:
				state = PHP_STRING;
				start = text.offset;
				break;
			case INTERNAL_PHP_CHAR:
				state = PHP_CHAR;
				start = text.offset;
				break;
			default:
				state = Token.NULL;
		}

		s = text;
		try {
			yyreset(zzReader);
			yybegin(state);
			return yylex();
		} catch (IOException ioe) {
			ioe.printStackTrace();
			return new DefaultToken();
		}

	}


	/**
	 * Sets whether markup close tags should be completed.  You might not want
	 * this to be the case, since some tags in standard HTML aren't usually
	 * closed.
	 *
	 * @param complete Whether closing markup tags are completed.
	 * @see #getCompleteCloseTags()
	 */
	public static void setCompleteCloseTags(boolean complete) {
		completeCloseTags = complete;
	}


	/**
	 * Refills the input buffer.
	 *
	 * @return      <code>true</code> if EOF was reached, otherwise
	 *              <code>false</code>.
	 */
	private boolean zzRefill() {
		return zzCurrentPos>=s.offset+s.count;
	}


	/**
	 * Resets the scanner to read from a new input stream.
	 * Does not close the old reader.
	 *
	 * All internal variables are reset, the old input stream 
	 * <b>cannot</b> be reused (internal buffer is discarded and lost).
	 * Lexical state is set to <tt>YY_INITIAL</tt>.
	 *
	 * @param reader   the new input stream 
	 */
	public final void yyreset(java.io.Reader reader) {
		// 's' has been updated.
		zzBuffer = s.array;
		/*
		 * We replaced the line below with the two below it because zzRefill
		 * no longer "refills" the buffer (since the way we do it, it's always
		 * "full" the first time through, since it points to the segment's
		 * array).  So, we assign zzEndRead here.
		 */
		//zzStartRead = zzEndRead = s.offset;
		zzStartRead = s.offset;
		zzEndRead = zzStartRead + s.count - 1;
		zzCurrentPos = zzMarkedPos = zzPushbackPos = s.offset;
		zzLexicalState = YYINITIAL;
		zzReader = reader;
		zzAtBOL  = true;
		zzAtEOF  = false;
	}


%}

// HTML-specific stuff.
Whitespace			= ([ \t\f]+)
LineTerminator			= ([\n])
Identifier			= ([^ \t\n<&]+)
AmperItem				= ([&][^; \t]*[;]?)
InTagIdentifier		= ([^ \t\n\"\'/=>]+)
EndScriptTag			= ("</" [sS][cC][rR][iI][pP][tT] ">")


// JavaScript stuff.
Letter							= [A-Za-z]
NonzeroDigit						= [1-9]
Digit							= ("0"|{NonzeroDigit})
HexDigit							= ({Digit}|[A-Fa-f])
OctalDigit						= ([0-7])
AnyCharacterButApostropheOrBackSlash	= ([^\\'])
AnyCharacterButDoubleQuoteOrBackSlash	= ([^\\\"\n])
EscapedSourceCharacter				= ("u"{HexDigit}{HexDigit}{HexDigit}{HexDigit})
Escape							= ("\\"(([btnfr\"'\\])|([0123]{OctalDigit}?{OctalDigit}?)|({OctalDigit}{OctalDigit}?)|{EscapedSourceCharacter}))
NonSeparator						= ([^\t\f\r\n\ \(\)\{\}\[\]\;\,\.\=\>\<\!\~\?\:\+\-\*\/\&\|\^\%\"\']|"#"|"\\")
IdentifierStart					= ({Letter}|"_"|"$")
IdentifierPart						= ({IdentifierStart}|{Digit}|("\\"{EscapedSourceCharacter}))
JS_UnclosedCharLiteral		= ("'"({AnyCharacterButApostropheOrBackSlash}|{Escape}))
JS_CharLiteral				= ({JS_UnclosedCharLiteral}"'")
JS_UnclosedErrorCharLiteral	= ([\'][^\'\n]*)
JS_ErrorCharLiteral			= (([\'][\'])|{JS_UnclosedErrorCharLiteral}[\'])
JS_UnclosedStringLiteral		= ([\"]({AnyCharacterButDoubleQuoteOrBackSlash}|{Escape})*)
JS_StringLiteral			= ({JS_UnclosedStringLiteral}[\"])
JS_UnclosedErrorStringLiteral	= ([\"]([^\"\n]|"\\\"")*)
JS_ErrorStringLiteral		= ({JS_UnclosedErrorStringLiteral}[\"])
JS_MLCBegin				= "/*"
JS_MLCEnd					= "*/"
JS_LineCommentBegin			= "//"
JS_IntegerHelper1			= (({NonzeroDigit}{Digit}*)|"0")
JS_IntegerHelper2			= ("0"(([xX]{HexDigit}+)|({OctalDigit}*)))
JS_IntegerLiteral			= ({JS_IntegerHelper1}[lL]?)
JS_HexLiteral				= ({JS_IntegerHelper2}[lL]?)
JS_FloatHelper1			= ([fFdD]?)
JS_FloatHelper2			= ([eE][+-]?{Digit}+{JS_FloatHelper1})
JS_FloatLiteral1			= ({Digit}+"."({JS_FloatHelper1}|{JS_FloatHelper2}|{Digit}+({JS_FloatHelper1}|{JS_FloatHelper2})))
JS_FloatLiteral2			= ("."{Digit}+({JS_FloatHelper1}|{JS_FloatHelper2}))
JS_FloatLiteral3			= ({Digit}+{JS_FloatHelper2})
JS_FloatLiteral			= ({JS_FloatLiteral1}|{JS_FloatLiteral2}|{JS_FloatLiteral3}|({Digit}+[fFdD]))
JS_ErrorNumberFormat		= (({JS_IntegerLiteral}|{JS_HexLiteral}|{JS_FloatLiteral}){NonSeparator}+)
JS_Separator				= ([\(\)\{\}\[\]\]])
JS_Separator2				= ([\;,.])
JS_NonAssignmentOperator		= ("+"|"-"|"<="|"^"|"++"|"<"|"*"|">="|"%"|"--"|">"|"/"|"!="|"?"|">>"|"!"|"&"|"=="|":"|">>"|"~"|"|"|"&&"|">>>")
JS_AssignmentOperator		= ("="|"-="|"*="|"/="|"|="|"&="|"^="|"+="|"%="|"<<="|">>="|">>>=")
JS_Operator				= ({JS_NonAssignmentOperator}|{JS_AssignmentOperator})
JS_Identifier				= ({IdentifierStart}{IdentifierPart}*)
JS_ErrorIdentifier			= ({NonSeparator}+)
BooleanLiteral				= ("true"|"false")


// PHP stuff (most PHP stuff is shared with JS for simplicity)
LetterOrUnderscore			= ({Letter}|[_])
LetterOrUnderscoreOrDigit	= ({LetterOrUnderscore}|{Digit})
PHP_Variable				= ("$"{LetterOrUnderscore}{LetterOrUnderscoreOrDigit}*)
PHP_LineCommentBegin		= ("//"|[#])

%state COMMENT
%state DTD
%state INTAG
%state INTAG_CHECK_TAG_NAME
%state INATTR_DOUBLE
%state INATTR_SINGLE
%state INTAG_SCRIPT
%state INATTR_DOUBLE_SCRIPT
%state INATTR_SINGLE_SCRIPT
%state JAVASCRIPT
%state JS_MLC
%state PHP
%state PHP_MLC
%state PHP_STRING
%state PHP_CHAR


%%

<YYINITIAL> {
	"<!--"					{ start = zzMarkedPos-4; yybegin(COMMENT); }
	"<script"					{
							  addToken(zzStartRead,zzStartRead, Token.MARKUP_TAG_DELIMITER);
							  addToken(zzMarkedPos-6,zzMarkedPos-1, Token.MARKUP_TAG_NAME);
							  start = zzMarkedPos; yybegin(INTAG_SCRIPT);
							}
	"<!"						{ start = zzMarkedPos-2; yybegin(DTD); }
	"<?php"						{ addToken(Token.SEPARATOR); yybegin(PHP); }
	"<?"						{ addToken(Token.SEPARATOR); yybegin(PHP); }
	"<"({Letter}|{Digit})+		{
									int count = yylength();
									addToken(zzStartRead,zzStartRead, Token.MARKUP_TAG_DELIMITER);
									zzMarkedPos -= (count-1); //yypushback(count-1);
									yybegin(INTAG_CHECK_TAG_NAME);
								}
	"</"({Letter}|{Digit})+		{
									int count = yylength();
									addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
									zzMarkedPos -= (count-2); //yypushback(count-2);
									yybegin(INTAG_CHECK_TAG_NAME);
								}
	"<"							{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(INTAG); }
	"</"						{ addToken(Token.MARKUP_TAG_DELIMITER); yybegin(INTAG); }
	{LineTerminator}			{ addNullToken(); return firstToken; }
	{Identifier}				{ addToken(Token.IDENTIFIER); } // Catches everything.
	{AmperItem}				{ addToken(Token.DATA_TYPE); }
	{Whitespace}				{ addToken(Token.WHITESPACE); }
	<<EOF>>					{ addNullToken(); return firstToken; }
}

<COMMENT> {
	[^\n\-]+					{}
	{LineTerminator}			{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); return firstToken; }
	"-->"					{ yybegin(YYINITIAL); addToken(start,zzStartRead+2, Token.COMMENT_MULTILINE); }
	"-"						{}
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); return firstToken; }
}

<DTD> {
	[^\n>]+					{}
	{LineTerminator}			{ addToken(start,zzStartRead-1, Token.VARIABLE); return firstToken; }
	">"						{ yybegin(YYINITIAL); addToken(start,zzStartRead, Token.VARIABLE); }
	<<EOF>>					{ addToken(start,zzStartRead-1, Token.VARIABLE); return firstToken; }
}

<INTAG_CHECK_TAG_NAME> {
	[Aa] |
	[aA][bB][bB][rR] |
	[aA][cC][rR][oO][nN][yY][mM] |
	[aA][dD][dD][rR][eE][sS][sS] |
	[aA][pP][pP][lL][eE][tT] |
	[aA][rR][eE][aA] |
	[aA][rR][tT][iI][cC][lL][eE] |
	[aA][sS][iI][dD][eE] |
	[aA][uU][dD][iI][oO] |
	[bB] |
	[bB][aA][sS][eE] |
	[bB][aA][sS][eE][fF][oO][nN][tT] |
	[bB][dD][oO] |
	[bB][gG][sS][oO][uU][nN][dD] |
	[bB][iI][gG] |
	[bB][lL][iI][nN][kK] |
	[bB][lL][oO][cC][kK][qQ][uU][oO][tT][eE] |
	[bB][oO][dD][yY] |
	[bB][rR] |
	[bB][uU][tT][tT][oO][nN] |
	[cC][aA][nN][vV][aA][sS] |
	[cC][aA][pP][tT][iI][oO][nN] |
	[cC][eE][nN][tT][eE][rR] |
	[cC][iI][tT][eE] |
	[cC][oO][dD][eE] |
	[cC][oO][lL] |
	[cC][oO][lL][gG][rR][oO][uU][pP] |
	[cC][oO][mM][mM][aA][nN][dD] |
	[cC][oO][mM][mM][eE][nN][tT] |
	[dD][dD] |
	[dD][aA][tT][aA][gG][rR][iI][dD] |
	[dD][aA][tT][aA][lL][iI][sS][tT] |
	[dD][aA][tT][aA][tT][eE][mM][pP][lL][aA][tT][eE] |
	[dD][eE][lL] |
	[dD][eE][tT][aA][iI][lL][sS] |
	[dD][fF][nN] |
	[dD][iI][aA][lL][oO][gG] |
	[dD][iI][rR] |
	[dD][iI][vV] |
	[dD][lL] |
	[dD][tT] |
	[eE][mM] |
	[eE][mM][bB][eE][dD] |
	[eE][vV][eE][nN][tT][sS][oO][uU][rR][cC][eE] |
	[fF][iI][eE][lL][dD][sS][eE][tT] |
	[fF][iI][gG][uU][rR][eE] |
	[fF][oO][nN][tT] |
	[fF][oO][oO][tT][eE][rR] |
	[fF][oO][rR][mM] |
	[fF][rR][aA][mM][eE] |
	[fF][rR][aA][mM][eE][sS][eE][tT] |
	[hH][123456] |
	[hH][eE][aA][dD] |
	[hH][eE][aA][dD][eE][rR] |
	[hH][rR] |
	[hH][tT][mM][lL] |
	[iI] |
	[iI][fF][rR][aA][mM][eE] |
	[iI][lL][aA][yY][eE][rR] |
	[iI][mM][gG] |
	[iI][nN][pP][uU][tT] |
	[iI][nN][sS] |
	[iI][sS][iI][nN][dD][eE][xX] |
	[kK][bB][dD] |
	[kK][eE][yY][gG][eE][nN] |
	[lL][aA][bB][eE][lL] |
	[lL][aA][yY][eE][rR] |
	[lL][eE][gG][eE][nN][dD] |
	[lL][iI] |
	[lL][iI][nN][kK] |
	[mM][aA][pP] |
	[mM][aA][rR][kK] |
	[mM][aA][rR][qQ][uU][eE][eE] |
	[mM][eE][nN][uU] |
	[mM][eE][tT][aA] |
	[mM][eE][tT][eE][rR] |
	[mM][uU][lL][tT][iI][cC][oO][lL] |
	[nN][aA][vV] |
	[nN][eE][sS][tT] |
	[nN][oO][bB][rR] |
	[nN][oO][eE][mM][bB][eE][dD] |
	[nN][oO][fF][rR][aA][mM][eE][sS] |
	[nN][oO][lL][aA][yY][eE][rR] |
	[nN][oO][sS][cC][rR][iI][pP][tT] |
	[oO][bB][jJ][eE][cC][tT] |
	[oO][lL] |
	[oO][pP][tT][gG][rR][oO][uU][pP] |
	[oO][pP][tT][iI][oO][nN] |
	[oO][uU][tT][pP][uU][tT] |
	[pP] |
	[pP][aA][rR][aA][mM] |
	[pP][lL][aA][iI][nN][tT][eE][xX][tT] |
	[pP][rR][eE] |
	[pP][rR][oO][gG][rR][eE][sS][sS] |
	[qQ] |
	[rR][uU][lL][eE] |
	[sS] |
	[sS][aA][mM][pP] |
	[sS][cC][rR][iI][pP][tT] |
	[sS][eE][cC][tT][iI][oO][nN] |
	[sS][eE][lL][eE][cC][tT] |
	[sS][eE][rR][vV][eE][rR] |
	[sS][mM][aA][lL][lL] |
	[sS][oO][uU][rR][cC][eE] |
	[sS][pP][aA][cC][eE][rR] |
	[sS][pP][aA][nN] |
	[sS][tT][rR][iI][kK][eE] |
	[sS][tT][rR][oO][nN][gG] |
	[sS][tT][yY][lL][eE] |
	[sS][uU][bB] |
	[sS][uU][pP] |
	[tT][aA][bB][lL][eE] |
	[tT][bB][oO][dD][yY] |
	[tT][dD] |
	[tT][eE][xX][tT][aA][rR][eE][aA] |
	[tT][fF][oO][oO][tT] |
	[tT][hH] |
	[tT][hH][eE][aA][dD] |
	[tT][iI][mM][eE] |
	[tT][iI][tT][lL][eE] |
	[tT][rR] |
	[tT][tT] |
	[uU] |
	[uU][lL] |
	[vV][aA][rR] |
	[vV][iI][dD][eE][oO]    { addToken(Token.MARKUP_TAG_NAME); }
	{InTagIdentifier}		{ /* A non-recognized HTML tag name */ yypushback(yylength()); yybegin(INTAG); }
	.						{ /* Shouldn't happen */ yypushback(1); yybegin(INTAG); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG); return firstToken; }
}

<INTAG> {
	"/"						{ addToken(Token.MARKUP_TAG_DELIMITER); }
	{InTagIdentifier}			{ addToken(Token.MARKUP_TAG_ATTRIBUTE); }
	{Whitespace}				{ addToken(Token.WHITESPACE); }
	"="						{ addToken(Token.OPERATOR); }
	"/>"						{ yybegin(YYINITIAL); addToken(Token.MARKUP_TAG_DELIMITER); }
	">"						{ yybegin(YYINITIAL); addToken(Token.MARKUP_TAG_DELIMITER); }
	[\"]						{ start = zzMarkedPos-1; yybegin(INATTR_DOUBLE); }
	[\']						{ start = zzMarkedPos-1; yybegin(INATTR_SINGLE); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG); return firstToken; }
}

<INATTR_DOUBLE> {
	[^\"]*						{}
	[\"]						{ yybegin(INTAG); addToken(start,zzStartRead, Token.LITERAL_STRING_DOUBLE_QUOTE); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.LITERAL_STRING_DOUBLE_QUOTE); return firstToken; }
}

<INATTR_SINGLE> {
	[^\']*						{}
	[\']						{ yybegin(INTAG); addToken(start,zzStartRead, Token.LITERAL_CHAR); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.LITERAL_CHAR); return firstToken; }
}

<INTAG_SCRIPT> {
	{InTagIdentifier}			{ addToken(Token.IDENTIFIER); }
	"/>"					{	addToken(Token.MARKUP_TAG_DELIMITER); yybegin(YYINITIAL); }
	"/"						{ addToken(Token.MARKUP_TAG_DELIMITER); } // Won't appear in valid HTML.
	{Whitespace}				{ addToken(Token.WHITESPACE); }
	"="						{ addToken(Token.OPERATOR); }
	">"						{ yybegin(JAVASCRIPT); addToken(Token.MARKUP_TAG_DELIMITER); }
	[\"]						{ start = zzMarkedPos-1; yybegin(INATTR_DOUBLE_SCRIPT); }
	[\']						{ start = zzMarkedPos-1; yybegin(INATTR_SINGLE_SCRIPT); }
	<<EOF>>					{ addToken(zzMarkedPos,zzMarkedPos, INTERNAL_INTAG_SCRIPT); return firstToken; }
}

<INATTR_DOUBLE_SCRIPT> {
	[^\"]*						{}
	[\"]						{ yybegin(INTAG_SCRIPT); addToken(start,zzStartRead, Token.LITERAL_STRING_DOUBLE_QUOTE); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.LITERAL_STRING_DOUBLE_QUOTE); addEndToken(INTERNAL_ATTR_DOUBLE_QUOTE_SCRIPT); return firstToken; }
}

<INATTR_SINGLE_SCRIPT> {
	[^\']*						{}
	[\']						{ yybegin(INTAG_SCRIPT); addToken(start,zzStartRead, Token.LITERAL_CHAR); }
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.LITERAL_STRING_DOUBLE_QUOTE); addEndToken(INTERNAL_ATTR_SINGLE_QUOTE_SCRIPT); return firstToken; }
}

<JAVASCRIPT> {

	{EndScriptTag}					{
								  yybegin(YYINITIAL);
								  addToken(zzStartRead,zzStartRead+1, Token.MARKUP_TAG_DELIMITER);
								  addToken(zzMarkedPos-7,zzMarkedPos-2, Token.MARKUP_TAG_NAME);
								  addToken(zzMarkedPos-1,zzMarkedPos-1, Token.MARKUP_TAG_DELIMITER);
								}

	// ECMA keywords.
	"break" |
	"continue" |
	"delete" |
	"else" |
	"for" |
	"function" |
	"if" |
	"in" |
	"new" |
	"return" |
	"this" |
	"typeof" |
	"var" |
	"void" |
	"while" |
	"with"						{ addToken(Token.RESERVED_WORD); }

	// Reserved (but not yet used) ECMA keywords.
	"abstract"					{ addToken(Token.RESERVED_WORD); }
	"boolean"						{ addToken(Token.DATA_TYPE); }
	"byte"						{ addToken(Token.DATA_TYPE); }
	"case"						{ addToken(Token.RESERVED_WORD); }
	"catch"						{ addToken(Token.RESERVED_WORD); }
	"char"						{ addToken(Token.DATA_TYPE); }
	"class"						{ addToken(Token.RESERVED_WORD); }
	"const"						{ addToken(Token.RESERVED_WORD); }
	"debugger"					{ addToken(Token.RESERVED_WORD); }
	"default"						{ addToken(Token.RESERVED_WORD); }
	"do"							{ addToken(Token.RESERVED_WORD); }
	"double"						{ addToken(Token.DATA_TYPE); }
	"enum"						{ addToken(Token.RESERVED_WORD); }
	"export"						{ addToken(Token.RESERVED_WORD); }
	"extends"						{ addToken(Token.RESERVED_WORD); }
	"final"						{ addToken(Token.RESERVED_WORD); }
	"finally"						{ addToken(Token.RESERVED_WORD); }
	"float"						{ addToken(Token.DATA_TYPE); }
	"goto"						{ addToken(Token.RESERVED_WORD); }
	"implements"					{ addToken(Token.RESERVED_WORD); }
	"import"						{ addToken(Token.RESERVED_WORD); }
	"instanceof"					{ addToken(Token.RESERVED_WORD); }
	"int"						{ addToken(Token.DATA_TYPE); }
	"interface"					{ addToken(Token.RESERVED_WORD); }
	"long"						{ addToken(Token.DATA_TYPE); }
	"native"						{ addToken(Token.RESERVED_WORD); }
	"package"						{ addToken(Token.RESERVED_WORD); }
	"private"						{ addToken(Token.RESERVED_WORD); }
	"protected"					{ addToken(Token.RESERVED_WORD); }
	"public"						{ addToken(Token.RESERVED_WORD); }
	"short"						{ addToken(Token.DATA_TYPE); }
	"static"						{ addToken(Token.RESERVED_WORD); }
	"super"						{ addToken(Token.RESERVED_WORD); }
	"switch"						{ addToken(Token.RESERVED_WORD); }
	"synchronized"					{ addToken(Token.RESERVED_WORD); }
	"throw"						{ addToken(Token.RESERVED_WORD); }
	"throws"						{ addToken(Token.RESERVED_WORD); }
	"transient"					{ addToken(Token.RESERVED_WORD); }
	"try"						{ addToken(Token.RESERVED_WORD); }
	"volatile"					{ addToken(Token.RESERVED_WORD); }
	"null"						{ addToken(Token.RESERVED_WORD); }

	// Literals.
	{BooleanLiteral}			{ addToken(Token.LITERAL_BOOLEAN); }
	"NaN"						{ addToken(Token.RESERVED_WORD); }
	"Infinity"					{ addToken(Token.RESERVED_WORD); }

	// Functions.
	"eval"						{ addToken(Token.FUNCTION); }
	"parseInt"					{ addToken(Token.FUNCTION); }
	"parseFloat"					{ addToken(Token.FUNCTION); }
	"escape"						{ addToken(Token.FUNCTION); }
	"unescape"					{ addToken(Token.FUNCTION); }
	"isNaN"						{ addToken(Token.FUNCTION); }
	"isFinite"					{ addToken(Token.FUNCTION); }

	{LineTerminator}				{ addEndToken(INTERNAL_IN_JS); return firstToken; }
	{JS_Identifier}				{ addToken(Token.IDENTIFIER); }
	{Whitespace}					{ addToken(Token.WHITESPACE); }

	/* String/Character literals. */
	{JS_CharLiteral}				{ addToken(Token.LITERAL_CHAR); }
	{JS_UnclosedCharLiteral}			{ addToken(Token.ERROR_CHAR); }
	{JS_UnclosedErrorCharLiteral}		{ addToken(Token.ERROR_CHAR); addEndToken(INTERNAL_IN_JS); return firstToken; }
	{JS_ErrorCharLiteral}			{ addToken(Token.ERROR_CHAR); }
	{JS_StringLiteral}				{ addToken(Token.LITERAL_STRING_DOUBLE_QUOTE); }
	{JS_UnclosedStringLiteral}		{ addToken(Token.ERROR_STRING_DOUBLE); addEndToken(INTERNAL_IN_JS); return firstToken; }
	{JS_UnclosedErrorStringLiteral}	{ addToken(Token.ERROR_STRING_DOUBLE); addEndToken(INTERNAL_IN_JS); return firstToken; }
	{JS_ErrorStringLiteral}			{ addToken(Token.ERROR_STRING_DOUBLE); }

	/* Comment literals. */
	"/**/"						{ addToken(Token.COMMENT_MULTILINE); }
	{JS_MLCBegin}					{ start = zzMarkedPos-2; yybegin(JS_MLC); }
	{JS_LineCommentBegin}.*			{ addToken(Token.COMMENT_EOL); addEndToken(INTERNAL_IN_JS); return firstToken; }

	/* Separators. */
	{JS_Separator}					{ addToken(Token.SEPARATOR); }
	{JS_Separator2}				{ addToken(Token.IDENTIFIER); }

	/* Operators. */
	{JS_Operator}					{ addToken(Token.OPERATOR); }

	/* Numbers */
	{JS_IntegerLiteral}				{ addToken(Token.LITERAL_NUMBER_DECIMAL_INT); }
	{JS_HexLiteral}				{ addToken(Token.LITERAL_NUMBER_HEXADECIMAL); }
	{JS_FloatLiteral}				{ addToken(Token.LITERAL_NUMBER_FLOAT); }
	{JS_ErrorNumberFormat}			{ addToken(Token.ERROR_NUMBER_FORMAT); }

	{JS_ErrorIdentifier}			{ addToken(Token.ERROR_IDENTIFIER); }

	/* Ended with a line not in a string or comment. */
	<<EOF>>						{ addEndToken(INTERNAL_IN_JS); return firstToken; }

	/* Catch any other (unhandled) characters and flag them as bad. */
	.							{ addToken(Token.ERROR_IDENTIFIER); }

}

<JS_MLC> {
	// JavaScript MLC's.  This state is essentially Java's MLC state.
	[^\n\*]+						{}
	\n							{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_IN_JS_MLC); return firstToken; }
	{JS_MLCEnd}					{ yybegin(JAVASCRIPT); addToken(start,zzStartRead+1, Token.COMMENT_MULTILINE); }
	\*							{}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_IN_JS_MLC); return firstToken; }
}


<PHP> {

	"?>"						{ addToken(Token.SEPARATOR); yybegin(YYINITIAL); }

	/* Keywords */
	"__CLASS__" |
	"__DIR__" |
	"__FILE__" |
	"__FUNCTION__" |
	"__METHOD__" |
	"__NAMESPACE__" |
	"abstract" |
	"and" |
	"array" |
	"as" |
	"break" |
	"case" |
	"catch" |
	"cfunction" |
	"class" |
	"clone" |
	"const" |
	"continue" |
	"declare" |
	"default" |
	"die" |
	"do" |
	"echo" |
	"else" |
	"elseif" |
	"empty" |
	"enddeclare" |
	"endfor" |
	"endforeach" |
	"endif" |
	"endswitch" |
	"endwhile" |
	"eval" |
	"exit" |
	"extends" |
	"final" |
	"for" |
	"foreach" |
	"function" |
	"global" |
	"goto" |
	"if" |
	"implements" |
	"include" |
	"include_once" |
	"interface" |
	"instanceof" |
	"isset" |
	"list" |
	"namespace" |
	"new" |
	"old_function" |
	"or" |
	"print" |
	"private" |
	"protected" |
	"public" |
	"require" |
	"require_once" |
	"return" |
	"static" |
	"switch" |
	"throw" |
	"try" |
	"unset" |
	"use" |
	"var" |
	"while" |
	"xor" |

	"parent" |
	"self" |
	"stdClass"					{ addToken(Token.RESERVED_WORD); }

	/* Functions */
	"__call" |
	"__construct" |
	"__getfunctions" |
	"__getlastrequest" |
	"__getlastresponse" |
	"__gettypes" |
	"__tostring" |
	"abs" |
	"acos" |
	"acosh" |
	"add" |
	("add"("_namespace"|"_root"|"action"|"color"|"cslashes"|"entry"|"fill"|"function"|"shape"|"slashes"|"string")) |
	"aggregate" |
	"aggregate_info" |
	"aggregate_methods" |
	"aggregate_methods_by_list" |
	"aggregate_methods_by_regexp" |
	"aggregate_properties" |
	"aggregate_properties_by_list" |
	"aggregate_properties_by_regexp" |
	"aggregation_info" |
	"align" |
	("apd_"("breakpoint"|"callstack"|"clunk"|"continue"|"croak"|"dump_function_table"|"dump_persistent_resources"|"dump_regular_resources"|"echo"|"get_active_symbols"|"set_pprof_trace"|"set_session"|"set_session_trace"|"set_socket_session_trace")) |
	"append" |
	"append_child" |
	"append_sibling" |
	"appendchild" |
	"appenddata" |
	("array_"("change_key_case"|"chunk"|"combine"|"count_values"|"diff"|"diff_assoc"|"diff_key"|"diff_uassoc"|"diff_ukey"|"fill"|"filter"|"flip"|"intersect"|"intersect_assoc"|"intersect_key"|"intersect_uassoc"|"intersect_ukey"|"key_exists"|"keys"|"map"|"merge"|"merge_recursive"|"multisort"|"pad"|"pop"|"push"|"rand"|"reduce"|"reverse"|"search"|"shift"|"slice"|"splice"|"sum"|"udiff"|"udiff_assoc"|"udiff_uassoc"|"uintersect"|"uintersect_assoc"|"uintersect_uassoc"|"unique"|"unshift"|"values"|"walk"|"walk_recursive")) |
	"arsort" |
	"ascii2ebcdic" |
	"asin" |
	"asinh" |
	"asort" |
	"assert" |
	"assert_options" |
	"assign" |
	"assignelem" |
	"asxml" |
	"atan" |
	"atan2" |
	"atanh" |
	"attreditable" |
	"attributes" |
	"base64_decode" |
	"base64_encode" |
	"base_convert" |
	"basename" |
	"bcadd" |
	"bccomp" |
	"bcdiv" |
	"bcmod" |
	"bcmul" |
	"bcpow" |
	"bcpowmod" |
	"bcscale" |
	"bcsqrt" |
	"bcsub" |
	"begintransaction" |
	"bin2hex" |
	"bind_textdomain_codeset" |
	"bindcolumn" |
	"bindec" |
	"bindparam" |
	"bindtextdomain" |
	"bzclose" |
	"bzcompress" |
	"bzdecompress" |
	"bzerrno" |
	"bzerror" |
	"bzerrstr" |
	"bzflush" |
	"bzopen" |
	"bzread" |
	"bzwrite" |
	"cal_days_in_month" |
	"cal_from_jd" |
	"cal_info" |
	"cal_to_jd" |
	"call_user_func" |
	"call_user_func_array" |
	"call_user_method" |
	"call_user_method_array" |
	("ccvs_"("add"|"auth"|"command"|"count"|"delete"|"done"|"init"|"lookup"|"new"|"report"|"return"|"reverse"|"sale"|"status"|"textvalue"|"void")) |
	"ceil" |
	"chdir" |
	"checkdate" |
	"checkdnsrr" |
	"checkin" |
	"checkout" |
	"chgrp" |
	"child_nodes" |
	"children" |
	"chmod" |
	"chop" |
	"chown" |
	"chr" |
	"chroot" |
	"chunk_split" |
	"class_exists" |
	"class_implements" |
	"class_parents" |
	"classkit_import" |
	"classkit_method_add" |
	"classkit_method_copy" |
	"classkit_method_redefine" |
	"classkit_method_remove" |
	"classkit_method_rename" |
	"clearstatcache" |
	"clone_node" |
	"clonenode" |
	"close" |
	"closedir" |
	"closelog" |
	"com" |
	"commit" |
	"compact" |
	"connect" |
	"connection_aborted" |
	"connection_status" |
	"connection_timeout" |
	"constant" |
	"content" |
	"convert_cyr_string" |
	"convert_uudecode" |
	"convert_uuencode" |
	"copy" |
	"cos" |
	"cosh" |
	"count" |
	"count_chars" |
	"crack_check" |
	"crack_closedict" |
	"crack_getlastmessage" |
	"crack_opendict" |
	"crc32" |
	("create"("_attribute"|"_cdata_section"|"_comment"|"_element"|"_element_ns"|"_entity_reference"|"_function"|"_processing_instruction"|"_text_node"|"attribute"|"attributens"|"cdatasection"|"comment"|"document"|"documentfragment"|"documenttype"|"element"|"elementns"|"entityreference"|"processinginstruction"|"textnode")) |
	"crypt" |
	("curl_"("close"|"copy_handle"|"errno"|"error"|"exec"|"getinfo"|"init"|"multi_add_handle"|"multi_close"|"multi_exec"|"multi_getcontent"|"multi_info_read"|"multi_init"|"multi_remove_handle"|"multi_select"|"setopt"|"version")) |
	"current" |
	"cybercash_base64_decode" |
	"cybercash_base64_encode" |
	"cybercash_decr" |
	"cybercash_encr" |
	"cyrus_authenticate" |
	"cyrus_bind" |
	"cyrus_close" |
	"cyrus_connect" |
	"cyrus_query" |
	"cyrus_unbind" |
	"data" |
	"date" |
	"date_sunrise" |
	"date_sunset" |
	"dblist" |
	"dbmclose" |
	"dbmdelete" |
	"dbmexists" |
	"dbmfetch" |
	"dbmfirstkey" |
	"dbminsert" |
	"dbmnextkey" |
	"dbmopen" |
	"dbmreplace" |
	"dbstat" |
	"dcgettext" |
	"dcngettext" |
	"dcstat" |
	"deaggregate" |
	"debug_backtrace" |
	"debug_print_backtrace" |
	"debug_zval_dump" |
	"debugger_off" |
	"debugger_on" |
	"decbin" |
	"dechex" |
	"decoct" |
	"decrement" |
	"define" |
	"define_syslog_variables" |
	"defined" |
	"deg2rad" |
	"delete" |
	"deletedata" |
	"description" |
	"dgettext" |
	("dio_"("close"|"fcntl"|"open"|"read"|"seek"|"stat"|"tcsetattr"|"truncate"|"write")) |
	"dir" |
	"dirname" |
	"disk_free_space" |
	"disk_total_space" |
	"diskfreespace" |
	"dl" |
	"dngettext" |
	"dns_check_record" |
	"dns_get_mx" |
	"dns_get_record" |
	"doctype" |
	"document_element" |
	"dom_import_simplexml" |
	("domxml_"("new_doc"|"open_file"|"open_mem"|"version"|"xmltree"|"xslt_stylesheet"|"xslt_stylesheet_doc"|"xslt_stylesheet_file")) |
	"dotnet" |
	"dotnet_load" |
	"doubleval" |
	"drawcurve" |
	"drawcurveto" |
	"drawline" |
	"drawlineto" |
	"dstanchors" |
	"dstofsrcanchors" |
	"dump_file" |
	"dump_mem" |
	"dump_node" |
	"each" |
	"easter_date" |
	"easter_days" |
	"ebcdic2ascii" |
	"end" |
	"entities" |
	"eof" |
	"erase" |
	"ereg" |
	"ereg_replace" |
	"eregi" |
	"eregi_replace" |
	"error_log" |
	"error_reporting" |
	"errorcode" |
	"errorinfo" |
	"escapeshellarg" |
	"escapeshellcmd" |
	"exec" |
	"execute" |
	"exif_imagetype" |
	"exif_read_data" |
	"exif_tagname" |
	"exif_thumbnail" |
	"exp" |
	"explode" |
	"expm1" |
	"export" |
	"extension_loaded" |
	"extract" |
	"ezmlm_hash" |
	"fclose" |
	"feof" |
	"fetch" |
	"fetchall" |
	"fetchsingle" |
	"fflush" |
	"fgetc" |
	"fgetcsv" |
	"fgets" |
	"fgetss" |
	"file" |
	("file"("_exists"|"_get_contents"|"_put_contents"|"atime"|"ctime"|"group"|"inode"|"mtime"|"owner"|"perms"|"pro"|"pro_fieldcount"|"pro_fieldname"|"pro_fieldtype"|"pro_fieldwidth"|"pro_retrieve"|"pro_rowcount"|"size"|"type")) |
	"find" |
	"first_child" |
	"floatval" |
	"flock" |
	"floor" |
	"flush" |
	"fmod" |
	"fnmatch" |
	"fopen" |
	"fpassthru" |
	"fprintf" |
	"fputcsv" |
	"fputs" |
	"fread" |
	"free" |
	"frenchtojd" |
	"fribidi_log2vis" |
	"fscanf" |
	"fseek" |
	"fsockopen" |
	"fstat" |
	"ftell" |
	"ftok" |
	("ftp_"("alloc"|"cdup"|"chdir"|"chmod"|"close"|"connect"|"delete"|"exec"|"fget"|"fput"|"get"|"get_option"|"login"|"mdtm"|"mkdir"|"nb_continue"|"nb_fget"|"nb_fput"|"nb_get"|"nb_put"|"nlist"|"pasv"|"put"|"pwd"|"quit"|"raw"|"rawlist"|"rename"|"rmdir"|"set_option"|"site"|"size"|"ssl_connect"|"systype")) |
	"ftruncate" |
	"ftstat" |
	"func_get_arg" |
	"func_get_args" |
	"func_num_args" |
	"function_exists" |
	"fwrite" |
	"gd_info" |
	"get" |
	("get"("_attr"|"_attribute"|"_attribute_node"|"_browser"|"_cfg_var"|"_class"|"_class_methods"|"_class_vars"|"_content"|"_current_user"|"_declared_classes"|"_declared_interfaces"|"_defined_constants"|"_defined_functions"|"_defined_vars"|"_element_by_id"|"_elements_by_tagname"|"_extension_funcs"|"_headers"|"_html_translation_table"|"_include_path"|"_included_files"|"_loaded_extensions"|"_magic_quotes_gpc"|"_magic_quotes_runtime"|"_meta_tags"|"_nodes"|"_object_vars"|"_parent_class"|"_required_files"|"_resource_type"|"allheaders"|"atime"|"attr"|"attribute"|"attributenode"|"attributenodens"|"attributens"|"buffering"|"children"|"crc"|"ctime"|"cwd"|"date"|"depth"|"elem"|"elementbyid"|"elementsbytagname"|"elementsbytagnamens"|"env"|"filename"|"filetime"|"functions"|"group"|"height"|"hostbyaddr"|"hostbyname"|"hostbynamel"|"hostos"|"imagesize"|"inneriterator"|"inode"|"iterator"|"lastmod"|"method"|"mtime"|"mxrr"|"mygid"|"myinode"|"mypid"|"myuid"|"name"|"nameditem"|"nameditemns"|"opt"|"owner"|"packedsize"|"path"|"pathname"|"perms"|"position"|"protobyname"|"protobynumber"|"randmax"|"rusage"|"servbyname"|"servbyport"|"shape1"|"shape2"|"size"|"stats"|"subiterator"|"text"|"timeofday"|"type"|"unpackedsize"|"version"|"width")) |
	"glob" |
	"gmdate" |
	"gmmktime" |
	("gmp_"("abs"|"add"|"and"|"clrbit"|"cmp"|"com"|"div"|"div_q"|"div_qr"|"div_r"|"divexact"|"fact"|"gcd"|"gcdext"|"hamdist"|"init"|"intval"|"invert"|"jacobi"|"legendre"|"mod"|"mul"|"neg"|"or"|"perfect_square"|"popcount"|"pow"|"powm"|"prob_prime"|"random"|"scan0"|"scan1"|"setbit"|"sign"|"sqrt"|"sqrtrem"|"strval"|"sub"|"xor")) |
	"gmstrftime" |
	"gregoriantojd" |
	("gz"("close"|"compress"|"deflate"|"encode"|"eof"|"file"|"getc"|"gets"|"getss"|"inflate"|"open"|"passthru"|"puts"|"read"|"rewind"|"seek"|"tell"|"uncompress"|"write")) |
	"handle" |
	("has"("_attribute"|"_attributes"|"_child_nodes"|"attribute"|"attributens"|"attributes"|"childnodes"|"children"|"feature"|"next"|"siblings")) |
	"header" |
	"headers_list" |
	"headers_sent" |
	"hebrev" |
	"hebrevc" |
	"hexdec" |
	"highlight_file" |
	"highlight_string" |
	"html_dump_mem" |
	"html_entity_decode" |
	"htmlentities" |
	"htmlspecialchars" |
	"http_build_query" |
	("hw_"("array2objrec"|"changeobject"|"children"|"childrenobj"|"close"|"connect"|"connection_info"|"cp"|"deleteobject"|"docbyanchor"|"docbyanchorobj"|"document_attributes"|"document_bodytag"|"document_content"|"document_setcontent"|"document_size"|"dummy"|"edittext"|"error"|"errormsg"|"free_document"|"getanchors"|"getanchorsobj"|"getandlock"|"getchildcoll"|"getchildcollobj"|"getchilddoccoll"|"getchilddoccollobj"|"getobject"|"getobjectbyquery"|"getobjectbyquerycoll"|"getobjectbyquerycollobj"|"getobjectbyqueryobj"|"getparents"|"getparentsobj"|"getrellink"|"getremote"|"getremotechildren"|"getsrcbydestobj"|"gettext"|"getusername"|"identify"|"incollections"|"info"|"inscoll"|"insdoc"|"insertanchors"|"insertdocument"|"insertobject"|"mapid"|"modifyobject"|"mv"|"new_document"|"objrec2array"|"output_document"|"pconnect"|"pipedocument"|"root"|"setlinkroot"|"stat"|"unlock"|"who")) |
	"hwapi_hgcsp" |
	"hwstat" |
	"hypot" |
	("ibase_"("add_user"|"affected_rows"|"backup"|"blob_add"|"blob_cancel"|"blob_close"|"blob_create"|"blob_echo"|"blob_get"|"blob_import"|"blob_info"|"blob_open"|"close"|"commit"|"commit_ret"|"connect"|"db_info"|"delete_user"|"drop_db"|"errcode"|"errmsg"|"execute"|"fetch_assoc"|"fetch_object"|"fetch_row"|"field_info"|"free_event_handler"|"free_query"|"free_result"|"gen_id"|"maintain_db"|"modify_user"|"name_result"|"num_fields"|"num_params"|"param_info"|"pconnect"|"prepare"|"query"|"restore"|"rollback"|"rollback_ret"|"server_info"|"service_attach"|"service_detach"|"set_event_handler"|"timefmt"|"trans"|"wait_event")) |
	"iconv" |
	("iconv_"("get_encoding"|"mime_decode"|"mime_decode_headers"|"mime_encode"|"set_encoding"|"strlen"|"strpos"|"strrpos"|"substr")) |
	"identify" |
	"ignore_user_abort" |
	("image"("2wbmp"|"_type_to_extension"|"_type_to_mime_type"|"alphablending"|"antialias"|"arc"|"char"|"charup"|"colorallocate"|"colorallocatealpha"|"colorat"|"colorclosest"|"colorclosestalpha"|"colorclosesthwb"|"colordeallocate"|"colorexact"|"colorexactalpha"|"colormatch"|"colorresolve"|"colorresolvealpha"|"colorset"|"colorsforindex"|"colorstotal"|"colortransparent"|"copy"|"copymerge"|"copymergegray"|"copyresampled"|"copyresized"|"create"|"createfromgd"|"createfromgd2"|"createfromgd2part"|"createfromgif"|"createfromjpeg"|"createfrompng"|"createfromstring"|"createfromwbmp"|"createfromxbm"|"createfromxpm"|"createtruecolor"|"dashedline"|"destroy"|"ellipse"|"fill"|"filledarc"|"filledellipse"|"filledpolygon"|"filledrectangle"|"filltoborder"|"filter"|"fontheight"|"fontwidth"|"ftbbox"|"fttext"|"gammacorrect"|"gd"|"gd2"|"gif"|"interlace"|"istruecolor"|"jpeg"|"layereffect"|"line"|"loadfont"|"palettecopy"|"png"|"polygon"|"psbbox"|"pscopyfont"|"psencodefont"|"psextendfont"|"psfreefont"|"psloadfont"|"psslantfont"|"pstext"|"rectangle"|"rotate"|"savealpha"|"setbrush"|"setpixel"|"setstyle"|"setthickness"|"settile"|"string"|"stringup"|"sx"|"sy"|"truecolortopalette"|"ttfbbox"|"ttftext"|"types"|"wbmp"|"xbm")) |
	("imap_"("8bit"|"alerts"|"append"|"base64"|"binary"|"body"|"bodystruct"|"check"|"clearflag_full"|"close"|"createmailbox"|"delete"|"deletemailbox"|"errors"|"expunge"|"fetch_overview"|"fetchbody"|"fetchheader"|"fetchstructure"|"get_quota"|"get_quotaroot"|"getacl"|"getmailboxes"|"getsubscribed"|"header"|"headerinfo"|"headers"|"last_error"|"list"|"listmailbox"|"listscan"|"listsubscribed"|"lsub"|"mail"|"mail_compose"|"mail_copy"|"mail_move"|"mailboxmsginfo"|"mime_header_decode"|"msgno"|"num_msg"|"num_recent"|"open"|"ping"|"qprint"|"renamemailbox"|"reopen"|"rfc822_parse_adrlist"|"rfc822_parse_headers"|"rfc822_write_address"|"scanmailbox"|"search"|"set_quota"|"setacl"|"setflag_full"|"sort"|"status"|"subscribe"|"thread"|"timeout"|"uid"|"undelete"|"unsubscribe"|"utf7_decode"|"utf7_encode"|"utf8")) |
	"implode" |
	"import" |
	"import_request_variables" |
	"importnode" |
	"in_array" |
	"increment" |
	"inet_ntop" |
	"inet_pton" |
	"info" |
	"ini_alter" |
	"ini_get" |
	"ini_get_all" |
	"ini_restore" |
	"ini_set" |
	"insert" |
	"insert_before" |
	"insertanchor" |
	"insertbefore" |
	"insertcollection" |
	"insertdata" |
	"insertdocument" |
	"interface_exists" |
	"internal_subset" |
	"intval" |
	"ip2long" |
	"iptcembed" |
	"iptcparse" |
	("is_"("a"|"array"|"blank_node"|"bool"|"callable"|"dir"|"double"|"executable"|"file"|"finite"|"float"|"infinite"|"int"|"integer"|"link"|"long"|"nan"|"null"|"numeric"|"object"|"readable"|"real"|"resource"|"scalar"|"soap_fault"|"string"|"subclass_of"|"uploaded_file"|"writable"|"writeable")) |
	("is"("asp"|"comment"|"dir"|"dot"|"executable"|"file"|"html"|"id"|"jste"|"link"|"php"|"readable"|"samenode"|"supported"|"text"|"whitespaceinelementcontent"|"writable"|"xhtml"|"xml")) |
	"item" |
	"iterator_count" |
	"iterator_to_array" |
	"java_last_exception_clear" |
	"java_last_exception_get" |
	"jddayofweek" |
	"jdmonthname" |
	"jdtofrench" |
	"jdtogregorian" |
	"jdtojewish" |
	"jdtojulian" |
	"jdtounix" |
	"jewishtojd" |
	"join" |
	"jpeg2wbmp" |
	"juliantojd" |
	"key" |
	"krsort" |
	"ksort" |
	"langdepvalue" |
	"last_child" |
	"lastinsertid" |
	"lcg_value" |
	("ldap_"("8859_to_t61"|"add"|"bind"|"close"|"compare"|"connect"|"count_entries"|"delete"|"dn2ufn"|"err2str"|"errno"|"error"|"explode_dn"|"first_attribute"|"first_entry"|"first_reference"|"free_result"|"get_attributes"|"get_dn"|"get_entries"|"get_option"|"get_values"|"get_values_len"|"list"|"mod_add"|"mod_del"|"mod_replace"|"modify"|"next_attribute"|"next_entry"|"next_reference"|"parse_reference"|"parse_result"|"read"|"rename"|"sasl_bind"|"search"|"set_option"|"set_rebind_proc"|"sort"|"start_tls"|"t61_to_8859"|"unbind")) |
	"levenshtein" |
	"link" |
	"linkinfo" |
	"load" |
	"loadhtml" |
	"loadhtmlfile" |
	"loadxml" |
	"localeconv" |
	"localtime" |
	"lock" |
	"log" |
	"log10" |
	"log1p" |
	"long2ip" |
	"lookupnamespaceuri" |
	"lookupprefix" |
	"lstat" |
	"ltrim" |
	"lzf_compress" |
	"lzf_decompress" |
	"lzf_optimized_for" |
	"mail" |
	"main" |
	"max" |
	("mb_"("convert_case"|"convert_encoding"|"convert_kana"|"convert_variables"|"decode_mimeheader"|"decode_numericentity"|"detect_encoding"|"detect_order"|"encode_mimeheader"|"encode_numericentity"|"ereg"|"ereg_match"|"ereg_replace"|"ereg_search"|"ereg_search_getpos"|"ereg_search_getregs"|"ereg_search_init"|"ereg_search_pos"|"ereg_search_regs"|"ereg_search_setpos"|"eregi"|"eregi_replace"|"get_info"|"http_input"|"http_output"|"internal_encoding"|"language"|"list_encodings"|"output_handler"|"parse_str"|"preferred_mime_name"|"regex_encoding"|"regex_set_options"|"send_mail"|"split"|"strcut"|"strimwidth"|"strlen"|"strpos"|"strrpos"|"strtolower"|"strtoupper"|"strwidth"|"substitute_character"|"substr"|"substr_count")) |
	("mcal_"("append_event"|"close"|"create_calendar"|"date_compare"|"date_valid"|"day_of_week"|"day_of_year"|"days_in_month"|"delete_calendar"|"delete_event"|"event_add_attribute"|"event_init"|"event_set_alarm"|"event_set_category"|"event_set_class"|"event_set_description"|"event_set_end"|"event_set_recur_daily"|"event_set_recur_monthly_mday"|"event_set_recur_monthly_wday"|"event_set_recur_none"|"event_set_recur_weekly"|"event_set_recur_yearly"|"event_set_start"|"event_set_title"|"expunge"|"fetch_current_stream_event"|"fetch_event"|"is_leap_year"|"list_alarms"|"list_events"|"next_recurrence"|"open"|"popen"|"rename_calendar"|"reopen"|"snooze"|"store_event"|"time_valid"|"week_of_year")) |
	("mcrypt_"("cbc"|"cfb"|"create_iv"|"decrypt"|"ecb"|"enc_get_algorithms_name"|"enc_get_block_size"|"enc_get_iv_size"|"enc_get_key_size"|"enc_get_modes_name"|"enc_get_supported_key_sizes"|"enc_is_block_algorithm"|"enc_is_block_algorithm_mode"|"enc_is_block_mode"|"enc_self_test"|"encrypt"|"generic"|"generic_deinit"|"generic_end"|"generic_init"|"get_block_size"|"get_cipher_name"|"get_iv_size"|"get_key_size"|"list_algorithms"|"list_modes"|"module_close"|"module_get_algo_block_size"|"module_get_algo_key_size"|"module_get_supported_key_sizes"|"module_is_block_algorithm"|"module_is_block_algorithm_mode"|"module_is_block_mode"|"module_open"|"module_self_test"|"ofb")) |
	"md5" |
	"md5_file" |
	"mdecrypt_generic" |
	"memcache_debug" |
	"memory_get_usage" |
	"metaphone" |
	"method_exists" |
	"mhash" |
	"mhash_count" |
	"mhash_get_block_size" |
	"mhash_get_hash_name" |
	"mhash_keygen_s2k" |
	"microtime" |
	"mime_content_type" |
	"mimetype" |
	"min" |
	"ming_setcubicthreshold" |
	"ming_setscale" |
	"ming_useswfversion" |
	"mkdir" |
	"mktime" |
	"money_format" |
	"move" |
	"move_uploaded_file" |
	"movepen" |
	"movepento" |
	"moveto" |
	("msession_"("connect"|"count"|"create"|"destroy"|"disconnect"|"find"|"get"|"get_array"|"get_data"|"inc"|"list"|"listvar"|"lock"|"plugin"|"randstr"|"set"|"set_array"|"set_data"|"timeout"|"uniq"|"unlock")) |
	"msg_get_queue" |
	"msg_receive" |
	"msg_remove_queue" |
	"msg_send" |
	"msg_set_queue" |
	"msg_stat_queue" |
	"msql" |
	"mt_getrandmax" |
	"mt_rand" |
	"mt_srand" |
	"name" |
	"natcasesort" |
	"natsort" |
	"next" |
	"next_sibling" |
	"nextframe" |
	"ngettext" |
	"nl2br" |
	"nl_langinfo" |
	"node_name" |
	"node_type" |
	"node_value" |
	"normalize" |
	"notations" |
	("notes_"("body"|"copy_db"|"create_db"|"create_note"|"drop_db"|"find_note"|"header_info"|"list_msgs"|"mark_read"|"mark_unread"|"nav_create"|"search"|"unread"|"version")) |
	"nsapi_request_headers" |
	"nsapi_response_headers" |
	"nsapi_virtual" |
	"number_format" |
	("ob_"("clean"|"end_clean"|"end_flush"|"flush"|"get_clean"|"get_contents"|"get_flush"|"get_length"|"get_level"|"get_status"|"gzhandler"|"iconv_handler"|"implicit_flush"|"list_handlers"|"start"|"tidyhandler")) |
	"object" |
	"objectbyanchor" |
	("oci"("_bind_by_name"|"_cancel"|"_close"|"_commit"|"_connect"|"_define_by_name"|"_error"|"_execute"|"_fetch"|"_fetch_all"|"_fetch_array"|"_fetch_assoc"|"_fetch_object"|"_fetch_row"|"_field_is_null"|"_field_name"|"_field_precision"|"_field_scale"|"_field_size"|"_field_type"|"_field_type_raw"|"_free_statement"|"_internal_debug"|"_lob_copy"|"_lob_is_equal"|"_new_collection"|"_new_connect"|"_new_cursor"|"_new_descriptor"|"_num_fields"|"_num_rows"|"_parse"|"_password_change"|"_pconnect"|"_result"|"_rollback"|"_server_version"|"_set_prefetch"|"_statement_type"|"bindbyname"|"cancel"|"closelob"|"collappend"|"collassign"|"collassignelem"|"collgetelem"|"collmax"|"collsize"|"colltrim"|"columnisnull"|"columnname"|"columnprecision"|"columnscale"|"columnsize"|"columntype"|"columntyperaw"|"commit"|"definebyname"|"error"|"execute"|"fetch"|"fetchinto"|"fetchstatement"|"freecollection"|"freecursor"|"freedesc"|"freestatement"|"internaldebug"|"loadlob"|"logoff"|"logon"|"newcollection"|"newcursor"|"newdescriptor"|"nlogon"|"numcols"|"parse"|"plogon"|"result"|"rollback"|"rowcount"|"savelob"|"savelobfile"|"serverversion"|"setprefetch"|"statementtype"|"writelobtofile"|"writetemporarylob")) |
	"octdec" |
	("odbc_"("autocommit"|"binmode"|"close"|"close_all"|"columnprivileges"|"columns"|"commit"|"connect"|"cursor"|"data_source"|"do"|"error"|"errormsg"|"exec"|"execute"|"fetch_array"|"fetch_into"|"fetch_object"|"fetch_row"|"field_len"|"field_name"|"field_num"|"field_precision"|"field_scale"|"field_type"|"foreignkeys"|"free_result"|"gettypeinfo"|"longreadlen"|"next_result"|"num_fields"|"num_rows"|"pconnect"|"prepare"|"primarykeys"|"procedurecolumns"|"procedures"|"result"|"result_all"|"rollback"|"setoption"|"specialcolumns"|"statistics"|"tableprivileges"|"tables")) |
	"offsetexists" |
	"offsetget" |
	"offsetset" |
	"offsetunset" |
	"opendir" |
	"openlog" |
	("openssl_"("csr_export"|"csr_export_to_file"|"csr_new"|"csr_sign"|"error_string"|"free_key"|"get_privatekey"|"get_publickey"|"open"|"pkcs7_decrypt"|"pkcs7_encrypt"|"pkcs7_sign"|"pkcs7_verify"|"pkey_export"|"pkey_export_to_file"|"pkey_get_private"|"pkey_get_public"|"pkey_new"|"private_decrypt"|"private_encrypt"|"public_decrypt"|"public_encrypt"|"seal"|"sign"|"verify"|"x509_check_private_key"|"x509_checkpurpose"|"x509_export"|"x509_export_to_file"|"x509_free"|"x509_parse"|"x509_read")) |
	("ora_"("bind"|"close"|"columnname"|"columnsize"|"columntype"|"commit"|"commitoff"|"commiton"|"do"|"error"|"errorcode"|"exec"|"fetch"|"fetch_into"|"getcolumn"|"logoff"|"logon"|"numcols"|"numrows"|"open"|"parse"|"plogon"|"rollback")) |
	"ord" |
	"output" |
	"output_add_rewrite_var" |
	"output_reset_rewrite_vars" |
	"overload" |
	"override_function" |
	"owner_document" |
	"pack" |
	"parent_node" |
	"parents" |
	"parse_ini_file" |
	"parse_str" |
	"parse_url" |
	"parsekit_compile_file" |
	"parsekit_compile_string" |
	"parsekit_func_arginfo" |
	"passthru" |
	"pathinfo" |
	"pclose" |
	("pcntl_"("alarm"|"exec"|"fork"|"getpriority"|"setpriority"|"signal"|"wait"|"waitpid"|"wexitstatus"|"wifexited"|"wifsignaled"|"wifstopped"|"wstopsig"|"wtermsig")) |
	"pconnect" |
	("pdf_"("add_annotation"|"add_bookmark"|"add_launchlink"|"add_locallink"|"add_note"|"add_outline"|"add_pdflink"|"add_thumbnail"|"add_weblink"|"arc"|"arcn"|"attach_file"|"begin_page"|"begin_pattern"|"begin_template"|"circle"|"clip"|"close"|"close_image"|"close_pdi"|"close_pdi_page"|"closepath"|"closepath_fill_stroke"|"closepath_stroke"|"concat"|"continue_text"|"curveto"|"delete"|"end_page"|"end_pattern"|"end_template"|"endpath"|"fill"|"fill_stroke"|"findfont"|"fit_pdi_page"|"get_buffer"|"get_font"|"get_fontname"|"get_fontsize"|"get_image_height"|"get_image_width"|"get_majorversion"|"get_minorversion"|"get_parameter"|"get_pdi_parameter"|"get_pdi_value"|"get_value"|"initgraphics"|"lineto"|"load_font"|"makespotcolor"|"moveto"|"new"|"open"|"open_ccitt"|"open_file"|"open_gif"|"open_image"|"open_image_file"|"open_jpeg"|"open_memory_image"|"open_pdi"|"open_pdi_page"|"open_png"|"open_tiff"|"place_image"|"place_pdi_page"|"rect"|"restore"|"rotate"|"save"|"scale"|"set_border_color"|"set_border_dash"|"set_border_style"|"set_char_spacing"|"set_duration"|"set_font"|"set_horiz_scaling"|"set_info"|"set_info_author"|"set_info_creator"|"set_info_keywords"|"set_info_subject"|"set_info_title"|"set_leading"|"set_parameter"|"set_text_matrix"|"set_text_pos"|"set_text_rendering"|"set_text_rise"|"set_value"|"set_word_spacing"|"setcolor"|"setdash"|"setflat"|"setfont"|"setgray"|"setgray_fill"|"setgray_stroke"|"setlinecap"|"setlinejoin"|"setlinewidth"|"setmatrix"|"setmiterlimit"|"setpolydash"|"setrgbcolor"|"setrgbcolor_fill"|"setrgbcolor_stroke"|"show"|"show_boxed"|"show_xy"|"skew"|"stringwidth"|"stroke"|"translate")) |
	"pfpro_cleanup" |
	"pfpro_init" |
	"pfpro_process" |
	"pfpro_process_raw" |
	"pfpro_version" |
	"pfsockopen" |
	("pg_"("affected_rows"|"cancel_query"|"client_encoding"|"close"|"connect"|"connection_busy"|"connection_reset"|"connection_status"|"convert"|"copy_from"|"copy_to"|"dbname"|"delete"|"end_copy"|"escape_bytea"|"escape_string"|"fetch_all"|"fetch_array"|"fetch_assoc"|"fetch_object"|"fetch_result"|"fetch_row"|"field_is_null"|"field_name"|"field_num"|"field_prtlen"|"field_size"|"field_type"|"free_result"|"get_notify"|"get_pid"|"get_result"|"host"|"insert"|"last_error"|"last_notice"|"last_oid"|"lo_close"|"lo_create"|"lo_export"|"lo_import"|"lo_open"|"lo_read"|"lo_read_all"|"lo_seek"|"lo_tell"|"lo_unlink"|"lo_write"|"meta_data"|"num_fields"|"num_rows"|"options"|"parameter_status"|"pconnect"|"ping"|"port"|"put_line"|"query"|"result_error"|"result_seek"|"result_status"|"select"|"send_query"|"set_client_encoding"|"trace"|"tty"|"unescape_bytea"|"untrace"|"update"|"version")) |
	"php_check_syntax" |
	"php_ini_scanned_files" |
	"php_logo_guid" |
	"php_sapi_name" |
	"php_strip_whitespace" |
	"php_uname" |
	"phpcredits" |
	"phpinfo" |
	"phpversion" |
	"pi" |
	"png2wbmp" |
	"popen" |
	"pos" |
	("posix_"("ctermid"|"get_last_error"|"getcwd"|"getegid"|"geteuid"|"getgid"|"getgrgid"|"getgrnam"|"getgroups"|"getlogin"|"getpgid"|"getpgrp"|"getpid"|"getppid"|"getpwnam"|"getpwuid"|"getrlimit"|"getsid"|"getuid"|"isatty"|"kill"|"mkfifo"|"setegid"|"seteuid"|"setgid"|"setpgid"|"setsid"|"setuid"|"strerror"|"times"|"ttyname"|"uname")) |
	"pow" |
	"prefix" |
	"preg_grep" |
	"preg_match" |
	"preg_match_all" |
	"preg_quote" |
	"preg_replace" |
	"preg_replace_callback" |
	"preg_split" |
	"prepare" |
	"prev" |
	"previous_sibling" |
	"print_r" |
	("printer_"("abort"|"close"|"create_brush"|"create_dc"|"create_font"|"create_pen"|"delete_brush"|"delete_dc"|"delete_font"|"delete_pen"|"draw_bmp"|"draw_chord"|"draw_elipse"|"draw_line"|"draw_pie"|"draw_rectangle"|"draw_roundrect"|"draw_text"|"end_doc"|"end_page"|"get_option"|"list"|"logical_fontheight"|"open"|"select_brush"|"select_font"|"select_pen"|"set_option"|"start_doc"|"start_page"|"write")) |
	"printf" |
	"proc_close" |
	"proc_get_status" |
	"proc_nice" |
	"proc_open" |
	"proc_terminate" |
	"process" |
	"public_id" |
	"putenv" |
	"qdom_error" |
	"qdom_tree" |
	"query" |
	"quoted_printable_decode" |
	"quotemeta" |
	"rad2deg" |
	"rand" |
	"range" |
	"rar_close" |
	"rar_entry_get" |
	"rar_list" |
	"rar_open" |
	"rawurldecode" |
	"rawurlencode" |
	"read" |
	"read_exif_data" |
	"readdir" |
	"readfile" |
	"readgzfile" |
	"readline" |
	("readline_"("add_history"|"callback_handler_install"|"callback_handler_remove"|"callback_read_char"|"clear_history"|"completion_function"|"info"|"list_history"|"on_new_line"|"read_history"|"redisplay"|"write_history")) |
	"readlink" |
	"realpath" |
	"reason" |
	"recode" |
	"recode_file" |
	"recode_string" |
	"register_shutdown_function" |
	"register_tick_function" |
	"registernamespace" |
	"relaxngvalidate" |
	"relaxngvalidatesource" |
	"remove" |
	"remove_attribute" |
	"remove_child" |
	"removeattribute" |
	"removeattributenode" |
	"removeattributens" |
	"removechild" |
	"rename" |
	"rename_function" |
	"replace" |
	"replace_child" |
	"replace_node" |
	"replacechild" |
	"replacedata" |
	"reset" |
	"restore_error_handler" |
	"restore_exception_handler" |
	"restore_include_path" |
	"result_dump_file" |
	"result_dump_mem" |
	"rewind" |
	"rewinddir" |
	"rmdir" |
	"rollback" |
	"rotate" |
	"rotateto" |
	"round" |
	"rowcount" |
	"rsort" |
	"rtrim" |
	"save" |
	"savehtml" |
	"savehtmlfile" |
	"savexml" |
	"scale" |
	"scaleto" |
	"scandir" |
	"schemavalidate" |
	"schemavalidatesource" |
	"seek" |
	"sem_acquire" |
	"sem_get" |
	"sem_release" |
	"sem_remove" |
	"serialize" |
	("sesam_"("affected_rows"|"commit"|"connect"|"diagnostic"|"disconnect"|"errormsg"|"execimm"|"fetch_array"|"fetch_result"|"fetch_row"|"field_array"|"field_name"|"free_result"|"num_fields"|"query"|"rollback"|"seek_row"|"sesam_settransaction")) |
	("session_"("cache_expire"|"cache_limiter"|"commit"|"decode"|"destroy"|"encode"|"get_cookie_params"|"id"|"is_registered"|"module_name"|"name"|"regenerate_id"|"register"|"save_path"|"set_cookie_params"|"set_save_handler"|"start"|"unregister"|"unset"|"write_close")) |
	"set" |
	("set"("_attribute"|"_content"|"_error_handler"|"_exception_handler"|"_file_buffer"|"_include_path"|"_magic_quotes_runtime"|"_name"|"_namespace"|"_time_limit"|"action"|"attribute"|"attributenode"|"attributenodens"|"attributens"|"background"|"bounds"|"buffering"|"class"|"color"|"commitedversion"|"cookie"|"depth"|"dimension"|"down"|"font"|"frames"|"height"|"hit"|"indentation"|"leftfill"|"leftmargin"|"line"|"linespacing"|"locale"|"margins"|"name"|"over"|"persistence"|"rate"|"ratio"|"rawcookie"|"rightfill"|"rightmargin"|"spacing"|"type"|"up")) |
	"sha1" |
	"sha1_file" |
	"shell_exec" |
	"shm_attach" |
	"shm_detach" |
	"shm_get_var" |
	"shm_put_var" |
	"shm_remove" |
	"shm_remove_var" |
	"shmop_close" |
	"shmop_delete" |
	"shmop_open" |
	"shmop_read" |
	"shmop_size" |
	"shmop_write" |
	"show_source" |
	"shuffle" |
	"similar_text" |
	"simplexml_import_dom" |
	"simplexml_load_file" |
	"simplexml_load_string" |
	"sin" |
	"sinh" |
	"size" |
	"sizeof" |
	"skewx" |
	"skewxto" |
	"skewy" |
	"skewyto" |
	"sleep" |
	("socket_"("accept"|"bind"|"clear_error"|"close"|"connect"|"create"|"create_listen"|"create_pair"|"get_option"|"get_status"|"getpeername"|"getsockname"|"last_error"|"listen"|"read"|"recv"|"recvfrom"|"select"|"send"|"sendto"|"set_block"|"set_blocking"|"set_nonblock"|"set_option"|"set_timeout"|"shutdown"|"strerror"|"write")) |
	"sort" |
	"soundex" |
	"specified" |
	"spl_classes" |
	"split" |
	"spliti" |
	"splittext" |
	"sprintf" |
	"sql_regcase" |
	"sqrt" |
	"srand" |
	"srcanchors" |
	"srcsofdst" |
	"sscanf" |
	"stat" |
	("str_"("ireplace"|"pad"|"repeat"|"replace"|"rot13"|"shuffle"|"split"|"word_count")) |
	"strcasecmp" |
	"strchr" |
	"strcmp" |
	"strcoll" |
	"strcspn" |
	("stream_"("context_create"|"context_get_default"|"context_get_options"|"context_set_option"|"context_set_params"|"copy_to_stream"|"filter_append"|"filter_prepend"|"filter_register"|"filter_remove"|"get_contents"|"get_filters"|"get_line"|"get_meta_data"|"get_transports"|"get_wrappers"|"register_wrapper"|"select"|"set_blocking"|"set_timeout"|"set_write_buffer"|"socket_accept"|"socket_client"|"socket_enable_crypto"|"socket_get_name"|"socket_recvfrom"|"socket_sendto"|"socket_server"|"wrapper_register"|"wrapper_restore"|"wrapper_unregister")) |
	("str"("eammp3"|"ftime"|"ip_tags"|"ipcslashes"|"ipos"|"ipslashes"|"istr"|"len"|"natcasecmp"|"natcmp"|"ncasecmp"|"ncmp"|"pbrk"|"pos"|"ptime"|"rchr"|"rev"|"ripos"|"rpos"|"spn"|"str"|"tok"|"tolower"|"totime"|"toupper"|"tr"|"val")) |
	"substr" |
	"substr_compare" |
	"substr_count" |
	"substr_replace" |
	"substringdata" |
	("swf_"("actiongeturl"|"actiongotoframe"|"actiongotolabel"|"actionnextframe"|"actionplay"|"actionprevframe"|"actionsettarget"|"actionstop"|"actiontogglequality"|"actionwaitforframe"|"addbuttonrecord"|"addcolor"|"closefile"|"definebitmap"|"definefont"|"defineline"|"definepoly"|"definerect"|"definetext"|"endbutton"|"enddoaction"|"endshape"|"endsymbol"|"fontsize"|"fontslant"|"fonttracking"|"getbitmapinfo"|"getfontinfo"|"getframe"|"labelframe"|"lookat"|"modifyobject"|"mulcolor"|"nextid"|"oncondition"|"openfile"|"ortho"|"ortho2"|"perspective"|"placeobject"|"polarview"|"popmatrix"|"posround"|"pushmatrix"|"removeobject"|"rotate"|"scale"|"setfont"|"setframe"|"shapearc"|"shapecurveto"|"shapecurveto3"|"shapefillbitmapclip"|"shapefillbitmaptile"|"shapefilloff"|"shapefillsolid"|"shapelinesolid"|"shapelineto"|"shapemoveto"|"showframe"|"startbutton"|"startdoaction"|"startshape"|"startsymbol"|"textwidth"|"translate"|"viewport")) |
	"swfbutton_keypress" |
	"symlink" |
	"syslog" |
	"system" |
	"system_id" |
	"tagname" |
	"tan" |
	"tanh" |
	"target" |
	"tcpwrap_check" |
	"tell" |
	"tempnam" |
	"textdomain" |
	("tidy_"("access_count"|"clean_repair"|"config_count"|"diagnose"|"error_count"|"get_body"|"get_config"|"get_error_buffer"|"get_head"|"get_html"|"get_html_ver"|"get_output"|"get_release"|"get_root"|"get_status"|"getopt"|"is_xhtml"|"is_xml"|"load_config"|"parse_file"|"parse_string"|"repair_file"|"repair_string"|"reset_config"|"save_config"|"set_encoding"|"setopt"|"warning_count")) |
	"time" |
	"time_nanosleep" |
	"title" |
	"tmpfile" |
	"token_get_all" |
	"token_name" |
	"touch" |
	"trigger_error" |
	"trim" |
	"truncate" |
	"type" |
	"uasort" |
	"ucfirst" |
	"ucwords" |
	"uksort" |
	"umask" |
	"uniqid" |
	"unixtojd" |
	"unlink" |
	"unlink_node" |
	"unlock" |
	"unpack" |
	"unregister_tick_function" |
	"unserialize" |
	"urldecode" |
	"urlencode" |
	"user" |
	"user_error" |
	"userlist" |
	"usleep" |
	"usort" |
	"utf8_decode" |
	"utf8_encode" |
	"valid" |
	"validate" |
	"value" |
	"values" |
	"var_dump" |
	"var_export" |
	("variant_"("abs"|"add"|"and"|"cast"|"cat"|"cmp"|"date_from_timestamp"|"date_to_timestamp"|"div"|"eqv"|"fix"|"get_type"|"idiv"|"imp"|"int"|"mod"|"mul"|"neg"|"not"|"or"|"pow"|"round"|"set"|"set_type"|"sub"|"xor")) |
	"version_compare" |
	"vfprintf" |
	"virtual" |
	("vpopmail_"("add_alias_domain"|"add_alias_domain_ex"|"add_domain"|"add_domain_ex"|"add_user"|"alias_add"|"alias_del"|"alias_del_domain"|"alias_get"|"alias_get_all"|"auth_user"|"del_domain"|"del_domain_ex"|"del_user"|"error"|"passwd"|"set_user_quota")) |
	"vprintf" |
	"vsprintf" |
	"w32api_deftype" |
	"w32api_init_dtype" |
	"w32api_invoke_function" |
	"w32api_register_function" |
	"w32api_set_call_method" |
	"wddx_add_vars" |
	"wddx_deserialize" |
	"wddx_packet_end" |
	"wddx_packet_start" |
	"wddx_serialize_value" |
	"wddx_serialize_vars" |
	"wordwrap" |
	"write" |
	"writetemporary" |
	"xattr_get" |
	"xattr_list" |
	"xattr_remove" |
	"xattr_set" |
	"xattr_supported" |
	"xinclude" |
	("xml"("_error_string"|"_get_current_byte_index"|"_get_current_column_number"|"_get_current_line_number"|"_get_error_code"|"_parse"|"_parse_into_struct"|"_parser_create"|"_parser_create_ns"|"_parser_free"|"_parser_get_option"|"_parser_set_option"|"_set_character_data_handler"|"_set_default_handler"|"_set_element_handler"|"_set_end_namespace_decl_handler"|"_set_external_entity_ref_handler"|"_set_notation_decl_handler"|"_set_object"|"_set_processing_instruction_handler"|"_set_start_namespace_decl_handler"|"_set_unparsed_entity_decl_handler"|"rpc_decode"|"rpc_decode_request"|"rpc_encode"|"rpc_encode_request"|"rpc_get_type"|"rpc_is_fault"|"rpc_parse_method_descriptions"|"rpc_server_add_introspection_data"|"rpc_server_call_method"|"rpc_server_create"|"rpc_server_destroy"|"rpc_server_register_introspection_callback"|"rpc_server_register_method"|"rpc_set_type")) |
	"xpath" |
	"xpath_eval" |
	"xpath_eval_expression" |
	"xpath_new_context" |
	"xptr_eval" |
	"xptr_new_context" |
	("xsl_xsltprocessor_"("get_parameter"|"has_exslt_support"|"import_stylesheet"|"register_php_functions"|"remove_parameter"|"set_parameter"|"transform_to_doc"|"transform_to_uri"|"transform_to_xml"|"xslt_backend_info")) |
	("xslt_"("backend_name"|"backend_version"|"create"|"errno"|"error"|"free"|"getopt"|"process"|"set_base"|"set_encoding"|"set_error_handler"|"set_log"|"set_object"|"set_sax_handler"|"set_sax_handlers"|"set_scheme_handler"|"set_scheme_handlers"|"setopt")) |
	("yp_"("all"|"cat"|"err_string"|"errno"|"first"|"get_default_domain"|"master"|"match"|"next"|"yp_order")) |
	"zend_logo_guid" |
	"zend_version" |
	("zip_"("close"|"entry_close"|"entry_compressedsize"|"entry_compressionmethod"|"entry_filesize"|"entry_name"|"entry_open"|"entry_read"|"open"|"read")) |
	"zlib_get_coding_type" |

	/* Function aliases */
	"apache_request_headers" |
	"apache_response_headers" |
	"attr_get" |
	"attr_set" |
	"autocommit" |
	"bind_param" |
	"bind_result" |
	"bzclose" |
	"bzflush" |
	"bzwrite" |
	"change_user" |
	"character_set_name" |
	"checkdnsrr" |
	"chop" |
	"client_encoding" |
	"close" |
	"commit" |
	"connect" |
	"data_seek" |
	"debug" |
	"disable_reads_from_master" |
	"disable_rpl_parse" |
	"diskfreespace" |
	"doubleval" |
	"dump_debug_info" |
	"enable_reads_from_master" |
	"enable_rpl_parse" |
	"escape_string" |
	"execute" |
	"fbsql" |
	"fbsql_tablename" |
	"fetch" |
	"fetch_array" |
	"fetch_assoc" |
	"fetch_field" |
	"fetch_field_direct" |
	"fetch_fields" |
	"fetch_object" |
	"fetch_row" |
	"field_count" |
	"field_seek" |
	"fputs" |
	"free" |
	"free_result" |
	"ftp_quit" |
	"get_client_info" |
	"get_required_files" |
	"get_server_info" |
	"getallheaders" |
	"getmxrr" |
	"gmp_div" |
	("gz"("close"|"eof"|"getc"|"gets"|"getss"|"passthru"|"puts"|"read"|"rewind"|"seek"|"tell"|"gzwrite")) |
	"imap_create" |
	"imap_fetchtext" |
	"imap_header" |
	"imap_listmailbox" |
	"imap_listsubscribed" |
	"imap_rename" |
	"ini_alter" |
	"init" |
	"is_double" |
	"is_int" |
	"is_integer" |
	"is_real" |
	"is_writeable" |
	"join" |
	"key_exists" |
	"kill" |
	"ldap_close" |
	"ldap_modify" |
	"magic_quotes_runtime" |
	"master_query" |
	"ming_keypress" |
	"ming_setcubicthreshold" |
	"ming_setscale" |
	"ming_useconstants" |
	"ming_useswfversion" |
	"more_results" |
	"next_result" |
	"num_rows" |
	("oci"("_free_cursor"|"bindbyname"|"cancel"|"collappend"|"collassignelem"|"collgetelem"|"collmax"|"collsize"|"colltrim"|"columnisnull"|"columnname"|"columnprecision"|"columnscale"|"columnsize"|"columntype"|"columntyperaw"|"commit"|"definebyname"|"error"|"execute"|"fetch"|"fetchstatement"|"freecollection"|"freecursor"|"freedesc"|"freestatement"|"internaldebug"|"loadlob"|"logoff"|"logon"|"newcollection"|"newcursor"|"newdescriptor"|"nlogon"|"numcols"|"parse"|"passwordchange"|"plogon"|"result"|"rollback"|"rowcount"|"savelob"|"savelobfile"|"serverversion"|"setprefetch"|"statementtype"|"writelobtofile")) |
	"odbc_do" |
	"odbc_field_precision" |
	"openssl_free_key" |
	"openssl_get_privatekey" |
	"openssl_get_publickey" |
	"options" |
	("pg_"("clientencoding"|"cmdtuples"|"errormessage"|"exec"|"fieldisnull"|"fieldname"|"fieldnum"|"fieldprtlen"|"fieldsize"|"fieldtype"|"freeresult"|"getlastoid"|"loclose"|"locreate"|"loexport"|"loimport"|"loopen"|"loread"|"loreadall"|"lounlink"|"lowrite"|"numfields"|"numrows"|"result"|"setclientencoding")) |
	"ping" |
	"pos" |
	"posix_errno" |
	"prepare" |
	"query" |
	"read_exif_data" |
	"real_connect" |
	"real_escape_string" |
	"real_query" |
	"recode" |
	"reset" |
	"result_metadata" |
	"rollback" |
	"rpl_parse_enabled" |
	"rpl_probe" |
	"rpl_query_type" |
	"select_db" |
	"send_long_data" |
	"session_commit" |
	"set_file_buffer" |
	"set_local_infile_default" |
	"set_local_infile_handler" |
	"set_opt" |
	"show_source" |
	"sizeof" |
	"slave_query" |
	"snmpwalkoid" |
	"socket_get_status" |
	"socket_getopt" |
	"socket_set_blocking" |
	"socket_set_timeout" |
	"socket_setopt" |
	"sqlite_fetch_string" |
	"sqlite_has_more" |
	"ssl_set" |
	"stat" |
	"stmt" |
	"stmt_init" |
	"store_result" |
	"strchr" |
	"stream_register_wrapper" |
	"thread_safe" |
	"use_result" |
	"user_error" |
	("velocis_"("autocommit"|"close"|"commit"|"connect"|"exec"|"fetch"|"fieldname"|"fieldnum"|"freeresult"|"off_autocommit"|"result"|"rollback")) |
	"virtual"							{ addToken(Token.FUNCTION); }

	{BooleanLiteral}			{ addToken(Token.LITERAL_BOOLEAN); }
	{PHP_Variable}				{ addToken(Token.VARIABLE); }

	{LineTerminator}				{ addEndToken(INTERNAL_IN_PHP); return firstToken; }
	{JS_Identifier}				{ addToken(Token.IDENTIFIER); }
	{Whitespace}					{ addToken(Token.WHITESPACE); }

	/* String/Character literals. */
	[\"]							{ start = zzMarkedPos-1; yybegin(PHP_STRING); }
	[\']							{ start = zzMarkedPos-1; yybegin(PHP_CHAR); }

	/* Comment literals. */
	"/**/"						{ addToken(Token.COMMENT_MULTILINE); }
	{JS_MLCBegin}					{ start = zzMarkedPos-2; yybegin(PHP_MLC); }
	{PHP_LineCommentBegin}.*			{ addToken(Token.COMMENT_EOL); addEndToken(INTERNAL_IN_PHP); return firstToken; }

	/* Separators. */
	{JS_Separator}					{ addToken(Token.SEPARATOR); }
	{JS_Separator2}				{ addToken(Token.IDENTIFIER); }

	/* Operators. */
	{JS_Operator}					{ addToken(Token.OPERATOR); }

	/* Numbers */
	{JS_IntegerLiteral}				{ addToken(Token.LITERAL_NUMBER_DECIMAL_INT); }
	{JS_HexLiteral}				{ addToken(Token.LITERAL_NUMBER_HEXADECIMAL); }
	{JS_FloatLiteral}				{ addToken(Token.LITERAL_NUMBER_FLOAT); }
	{JS_ErrorNumberFormat}			{ addToken(Token.ERROR_NUMBER_FORMAT); }

	{JS_ErrorIdentifier}			{ addToken(Token.ERROR_IDENTIFIER); }

	/* Ended with a line not in a string or comment. */
	<<EOF>>						{ addEndToken(INTERNAL_IN_PHP); return firstToken; }

	/* Catch any other (unhandled) characters and assume they are okay. */
	.							{ addToken(Token.IDENTIFIER); }

}


<PHP_MLC> {
	// PHP MLC's.  This state is essentially Java's MLC state.
	[^\n\*]+						{}
	\n							{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_IN_PHP_MLC); return firstToken; }
	{JS_MLCEnd}					{ yybegin(PHP); addToken(start,zzStartRead+1, Token.COMMENT_MULTILINE); }
	\*							{}
	<<EOF>>						{ addToken(start,zzStartRead-1, Token.COMMENT_MULTILINE); addEndToken(INTERNAL_IN_PHP_MLC); return firstToken; }
}


<PHP_STRING> {
	[^\n\\\$\"]+		{}
	\n					{ addToken(start,zzStartRead-1, Token.LITERAL_STRING_DOUBLE_QUOTE); addEndToken(INTERNAL_PHP_STRING); return firstToken; }
	\\.?				{ /* Skip escaped chars. */ }
	{PHP_Variable}		{ int temp=zzStartRead; addToken(start,zzStartRead-1, Token.LITERAL_STRING_DOUBLE_QUOTE); addToken(temp,zzMarkedPos-1, Token.VARIABLE); start = zzMarkedPos; }
	"$"					{}
	\"					{ yybegin(PHP); addToken(start,zzStartRead, Token.LITERAL_STRING_DOUBLE_QUOTE); }
	<<EOF>>				{ addToken(start,zzStartRead-1, Token.LITERAL_STRING_DOUBLE_QUOTE); addEndToken(INTERNAL_PHP_STRING); return firstToken; }
}


<PHP_CHAR> {
	[^\n\\\']+			{}
	\\.?				{ /* Skip escaped single quotes only, but this should still work. */ }
	\n					{ addToken(start,zzStartRead-1, Token.LITERAL_CHAR); addEndToken(INTERNAL_PHP_CHAR); return firstToken; }
	\'					{ yybegin(PHP); addToken(start,zzStartRead, Token.LITERAL_CHAR); }
	<<EOF>>				{ addToken(start,zzStartRead-1, Token.LITERAL_CHAR); addEndToken(INTERNAL_PHP_CHAR); return firstToken; }
}

