/* $Id: MsqlBlob.java,v 2.1 1999/03/03 03:27:06 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.sql.Blob;
import java.sql.SQLException;

/**
 * Implements the JDBC 2.0 <CODE>Blob</CODE> interface.  Under mSQL,
 * there are no binary data types.  There is therefore no optimization in
 * using <CODE>Blob</CODE> objects.  This implementation exists simply to make
 * it easier to write platform independent database code.
 * <BR>
 * Last modified $Date: 1999/03/03 03:27:06 $
 * @version $Revision: 2.1 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlBlob implements Blob {
    /**
     * The internal representation of the binary object.
     */
    private byte[]   data   = null;
    /**
     * The size of the binary object in bytes.
     */
    private long     length = -1;

    /**
     * Constructs a new <CODE>MsqlBlob</CODE> from the specified byte array.
     * @param b the array of bytes this <CODE>Blob</CODE> represents
     */
    public MsqlBlob(byte[] b) {
	super();
	data = new byte[b.length];
	System.arraycopy(b, 0, data, 0, b.length);
	length = b.length;
    }

    /**
     * Constructs a new <CODE>MsqlBlob</CODE> from a binary stream.
     * @param is the stream from which this <CODE>Blob</CODE> will be
     * created
     * @throws java.sql.SQLException a database error occurred
     */
    public MsqlBlob(InputStream is) throws SQLException {
	super();
	try {
	    ByteArrayOutputStream baos = new ByteArrayOutputStream(10240);
	    BufferedInputStream bis = new BufferedInputStream(is);
	    byte[] tmp = new byte[10240];
	    int sz = bis.read(tmp, 0, 10240);

	    while( sz != -1 ) {
		baos.write(tmp, 0, sz);
		tmp = new byte[10240];
		sz = bis.read(tmp, 0, 10240);
	    }
	    data = baos.toByteArray();
	    length = data.length;
	}
	catch( IOException e ) {
	    throw new MsqlException(e);
	}
    }

    /**
     * @return the <CODE>Blob</CODE> as an <CODE>InputStream</CODE>
     * @throws java.sql.SQLException could not construct the input stream
     */
    public InputStream getBinaryStream() throws SQLException {
	return new ByteArrayInputStream(data);
    }

    /**
     * Provides a byte array of data from this <CODE>Blob</CODE>
     * starting at the specified start position within this <CODE>Blob</CODE>
     * and having the specified length.  Note that the first argument is
     * a <CODE>long</CODE> as required by the JDBC 2.0 specification.
     * Because mSQL has no binary types and mSQL does some fudging via
     * base 64 encoding to support binary types, you will never end up with
     * a <CODE>Blob</CODE> in mSQL whose start position is greater than
     * <CODE>Integer.MAX_VALUE</CODE>.
     * @param pos the start position 
     * @param len the length of the desired return array
     * @return an array of the bytes of this <CODE>Blob</CODE> from the
     * specified start position up to the specified length
     * @throws java.sql.SQLException the position argument was greater
     * than the size of the <CODE>Blob</CODE>.
     */
    public byte[] getBytes(long pos, int len) throws SQLException {
	byte[] tmp;

	if( pos >= length ) {
	    throw new MsqlException("Invalid start position.");
	}
	if( len > length - pos ) {
	    len = (int)(length - pos);
	}
	tmp = new byte[len];
	System.arraycopy(data, (int)pos, tmp, 0, len);
	return tmp;
    }

    /**
     * @return the number of bytes in this <CODE>Blob</CODE>
     * @throws java.sql.SQLException this is never thrown in mSQL-JDBC
     */
    public long length() throws SQLException {
	return length;
    }

    /**
     * Provides the position of the first occurrence of the specified pattern
     * after the specified start position.
     * @param pattern a <CODE>Blob</CODE> representing the desired pattern
     * @param start the index from which to start the search measured
     * from 1
     * @return the index of the first occurrence of the specified pattern
     * or -1 if the pattern does not occur
     * @throws java.sql.SQLException this is never thrown in mSQL-JDBC
     */
    public long position(Blob pattern, long start) throws SQLException {
	return position(pattern.getBytes(0, (int)pattern.length()), start);
    }

    /**
     * Provides the position of the first occurrence of the specified pattern
     * after the specified start position.
     * @param pattern a byte array representing the desired pattern
     * @param start the index from which to start the search measured
     * from 1
     * @return the index of the first occurrence of the specified pattern
     * or -1 if the pattern does not occur
     * @throws java.sql.SQLException this is never thrown in mSQL-JDBC
     */
    public long position(byte[] pattern, long start) throws SQLException {
	// for some dumbass reason, JDBC says the start position is 1
	start--;
	for(int i=(int)start; i<data.length; i++) {
	    if( pattern.length > (data.length-i) ) {
		break;
	    }
	    if( data[i] == pattern[0] ) {
		boolean match = true;
		
		for(int j=1; j<pattern.length; j++) {
		    if( data[i+j] != pattern[j] ) {
			match = false;
			break;
		    }
		}
		if( match ) {
		    return i+1;
		}
	    }
	}
	return -1;
    }
}
