/* $Id: ImaginaryRowSet.java,v 1.1 1999/03/03 06:00:21 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Map;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.sql.RowSet;
import javax.sql.RowSetListener;

/**
 * Database-independent implementation of the JDBC Standard Extension
 * <CODE>RowSet</CODE> interface. This class encapsulates database
 * access in a JavaBeans component.
 * <BR>
 * Last modified $Date: 1999/03/03 06:00:21 $
 * @version $Revision: 1.1 $
 * @author George Reese (borg@imaginary.com)
 */
public class ImaginaryRowSet implements RowSet {
    private String             command              = null;
    private Connection         connection           = null;
    private ResultSet          currentResults       = null;
    private PreparedStatement  currentStatement     = null;
    private DataSource         dataSource           = null;
    private String             dataSourceName       = null;
    private boolean            escapeProcessing     = true;
    private RowSetEventSupport eventSupport         = null;
    private int                fetchDirection       = ResultSet.FETCH_FORWARD;
    private int                fetchSize            = 0;
    private int                maxFieldSize         = 0;
    private int                maxRows              = 0;
    private String             password             = null;
    private int                queryTimeout         = 0;
    private boolean            readOnly             = false;
    private int                resultSetConcurrency = ResultSet.CONCUR_READ_ONLY;
    private int                resultSetType        = ResultSet.TYPE_FORWARD_ONLY;
    private int                transactionIsolation = Connection.TRANSACTION_NONE;
    private Map                typeMap              = null;
    private String             url                  = null;
    private String             username             = null;
    
    public ImaginaryRowSet() {
	super();
	eventSupport = new RowSetEventSupport(this);
    }
    
    public ImaginaryRowSet(DataSource ds) {
	super();
	dataSource = ds;
	eventSupport = new RowSetEventSupport(this);
    }

    public ImaginaryRowSet(int type, int concur) {
	super();
	resultSetConcurrency = concur;
	resultSetType = type;
	eventSupport = new RowSetEventSupport(this);
    }
    
    public ImaginaryRowSet(DataSource ds, int type, int concur) {
	super();
	dataSource = ds;
	resultSetType = type;
	resultSetConcurrency = concur;
	eventSupport = new RowSetEventSupport(this);
    }	

    public synchronized boolean absolute(int row) throws SQLException {
	boolean val, same;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	same = (row == getRow());
	val = currentResults.absolute(row);
	if( !same ) {
	    eventSupport.cursorMoved();
	}
	return val;
    }

    public synchronized void addRowSetListener(RowSetListener listener) {
	eventSupport.addListener(listener);
    }

    public synchronized void afterLast() throws SQLException {
	boolean same;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	same = isAfterLast();
	currentResults.afterLast();
	if( !same ) {
	    eventSupport.cursorMoved();
	}
    }

    public synchronized void beforeFirst() throws SQLException {
	boolean same;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	same = isBeforeFirst();
	currentResults.beforeFirst();
	if( !same ) {
	    eventSupport.cursorMoved();
	}
    }

    public synchronized void cancelRowUpdates() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.cancelRowUpdates();
    }

    public void clearParameters() throws SQLException {
	if( currentStatement == null ) {
	    throw new SQLException("Row set references no statement.");
	}
	currentStatement.clearParameters();
    }

    public synchronized void clearWarnings() throws SQLException {
	if( connection != null ) {
	    connection.clearWarnings();
	}
	if( currentResults != null ) {
	    currentResults.clearWarnings();
	}
	if( currentStatement != null ) {
	    currentStatement.clearWarnings();
	}
    }

    public synchronized void close() throws SQLException {
	SQLException err = null;

	// the if( err != null ) bit is repeated to make for easier
	// to read stack traces
	if( connection != null ) {
	    if( !connection.getAutoCommit() ) {
		try {
		    connection.commit();
		}
		catch( SQLException e ) {
		    try { connection.rollback(); }
		    catch( SQLException ick ) { ick.printStackTrace(); }
		    throw e;
		}
	    }
	    try { connection.close(); }
	    catch( SQLException e ) { err = e; }
	    connection = null;
	    currentStatement = null;
	    currentResults = null;
	    if( err != null ) {
		throw err;
	    }
	}
	else if( currentStatement != null ) {
	    try { currentStatement.close(); }
	    catch( SQLException e ) { err = e; }
	    currentStatement = null;
	    currentResults = null;
	    if( err != null ) {
		throw err;
	    }
	}
	else if( currentResults != null ) {
	    try { currentResults.close(); }
	    catch( SQLException e ) { err = e; }
	    currentResults = null;
	    if( err != null ) {
		throw err;
	    }
	}
    }

    public synchronized void deleteRow() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.deleteRow();
	eventSupport.rowChanged();
    }

    public synchronized void execute() throws SQLException {
	if( currentStatement == null ) {
	    throw new SQLException("No current statement to execute.");
	}
	currentResults = currentStatement.executeQuery();
	eventSupport.rowSetChanged();
    }

    public synchronized int findColumn(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.findColumn(col);
    }

    public synchronized boolean first() throws SQLException {
	boolean val, same;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	same = isFirst();
	val = currentResults.first();
	if( !same ) {
	    eventSupport.cursorMoved();
	}
	return val;
    }

    public synchronized Array getArray(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getArray(col);
    }

    public synchronized Array getArray(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getArray(col);
    }

    /**
     * @param col the number of the desired column
     * @return the specified col as an ASCII stream
     * @throws java.sql.SQLException a database error occurred
     */
    public synchronized InputStream getAsciiStream(int col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getAsciiStream(col);
    }

    /**
     * @param col the name of the desired column
     * @return the specified col as an ASCII stream
     * @throws java.sql.SQLException a database error occurred
     */
    public synchronized InputStream getAsciiStream(String col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getAsciiStream(col);
    }

    public synchronized BigDecimal getBigDecimal(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBigDecimal(col);
    }

    public synchronized BigDecimal getBigDecimal(String col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBigDecimal(col);
    }

    /**
     * @param col the number of the desired column
     * @param scale the desired <CODE>BigDecimal</CODE> scale
     * @throws java.sql.SQLException a database error occurred
     * @deprecated
     */
    public synchronized BigDecimal getBigDecimal(int col, int scale)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBigDecimal(col);
    }

    /**
     * @param col the name of the desired column
     * @param scale the desired <CODE>BigDecimal</CODE> scale
     * @throws java.sql.SQLException a database error occurred
     * @deprecated
     */
    public synchronized BigDecimal getBigDecimal(String col, int scale)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBigDecimal(col);
    }

    public synchronized InputStream getBinaryStream(int col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBinaryStream(col);
    }

    public synchronized InputStream getBinaryStream(String col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBinaryStream(col);
    }

    public synchronized Blob getBlob(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBlob(col);
    }

    public synchronized Blob getBlob(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBlob(col);
    }

    public synchronized boolean getBoolean(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBoolean(col);
    }

    public synchronized boolean getBoolean(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBoolean(col);
    }

    public synchronized byte getByte(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getByte(col);
    }

    public synchronized byte getByte(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getByte(col);
    }

    public synchronized byte[] getBytes(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBytes(col);
    }

    public synchronized byte[] getBytes(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getBytes(col);
    }

    public synchronized Reader getCharacterStream(int col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getCharacterStream(col);
    }

    public synchronized Reader getCharacterStream(String col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getCharacterStream(col);
    }

    public synchronized Clob getClob(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getClob(col);
    }

    public synchronized Clob getClob(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getClob(col);
    }
    
    public synchronized String getCommand() {
	return command;
    }

    public synchronized int getConcurrency() throws SQLException {
	return resultSetConcurrency;
    }

    public synchronized String getCursorName() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getCursorName();
    }

    private DataSource getDataSource() {
	try {
	    Context ctx = new InitialContext();
	    
	    return (DataSource)ctx.lookup(dataSourceName);
	}
	catch( NamingException e ) {
	    // cannot do much else
	    e.printStackTrace();
	    return null;
	}
    }
    
    public synchronized String getDataSourceName() {
	return dataSourceName;
    }

    public synchronized Date getDate(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getDate(col);
    }

    public synchronized Date getDate(int col, Calendar cal)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getDate(col, cal);
    }
    
    public synchronized Date getDate(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getDate(col);
    }

    public synchronized Date getDate(String col, Calendar cal)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getDate(col, cal);
    }

    public synchronized double getDouble(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getDouble(col);
    }

    public synchronized double getDouble(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getDouble(col);
    }

    public synchronized boolean getEscapeProcessing() throws SQLException {
	return escapeProcessing;
    }
    
    public synchronized int getFetchDirection() throws SQLException {
	return fetchDirection;
    }

    public synchronized int getFetchSize() throws SQLException {
	return fetchSize;
    }

    public synchronized float getFloat(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getFloat(col);
    }

    public synchronized float getFloat(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getFloat(col);
    }

    public synchronized int getInt(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getInt(col);
    }

    public synchronized int getInt(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getInt(col);
    }

    public synchronized long getLong(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getLong(col);
    }

    public synchronized long getLong(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getLong(col);
    }

    public synchronized int getMaxFieldSize() throws SQLException {
	return maxFieldSize;
    }

    public synchronized int getMaxRows() throws SQLException {
	return maxRows;
    }

    public synchronized ResultSetMetaData getMetaData() throws SQLException {
	return null;
    }

    public synchronized Object getObject(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getObject(col);
    }

    public synchronized Object getObject(int col, Map map)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getObject(col, map);
    }

    public synchronized Object getObject(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getObject(col);
    }

    public synchronized Object getObject(String col, Map map)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getObject(col, map);
    }

    public synchronized String getPassword() {
	return password;
    }

    public synchronized int getQueryTimeout() throws SQLException {
	return queryTimeout;
    }

    public synchronized Ref getRef(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getRef(col);
    }

    public synchronized Ref getRef(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getRef(col);
    }

    public synchronized int getRow() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getRow();
    }

    public synchronized short getShort(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getShort(col);
    }

    public synchronized short getShort(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getShort(col);
    }

    public synchronized Statement getStatement() throws SQLException {
	return currentStatement;
    }

    public synchronized String getString(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getString(col);
    }

    public synchronized String getString(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getString(col);
    }

    public synchronized Time getTime(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTime(col);
    }

    public synchronized Time getTime(int col, Calendar cal)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTime(col, cal);
    }
    
    public synchronized Time getTime(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTime(col);
    }

    public synchronized Time getTime(String col, Calendar cal)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTime(col, cal);
    }

    public synchronized Timestamp getTimestamp(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTimestamp(col);
    }

    public synchronized Timestamp getTimestamp(int col, Calendar cal)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTimestamp(col, cal);
    }

    public synchronized Timestamp getTimestamp(String col)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTimestamp(col);
    }

    public synchronized Timestamp getTimestamp(String col, Calendar cal)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getTimestamp(col, cal);
    }

    public int getTransactionIsolation() {
	return transactionIsolation;
    }

    public synchronized int getType() throws SQLException {
	return resultSetType;
    }
    
    public synchronized Map getTypeMap() throws SQLException {
	return typeMap;
    }

    /**
     * @param col the number of the desired column
     * @return the specified col as a Unicode stream
     * @throws java.sql.SQLException a database error occurred
     * @deprecated use <CODE>getCharacteerStream()</CODE>
     */
    public InputStream getUnicodeStream(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getUnicodeStream(col);
    }

    /**
     * @param col the name of the desired column
     * @return the specified col as a Unicode stream
     * @throws java.sql.SQLException a database error occurred
     * @deprecated use <CODE>getCharacteerStream()</CODE>
     */
    public InputStream getUnicodeStream(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.getUnicodeStream(col);
    }
    
    public synchronized String getUrl() throws SQLException {
	return url;
    }

    public synchronized String getUsername() {
	return username;
    }

    public synchronized SQLWarning getWarnings() throws SQLException {
	SQLWarning warning = null;

	if( connection == null ) {
	    return null;
	}
	if( currentResults != null ) {
	    warning = currentResults.getWarnings();
	}
	if( currentStatement != null ) {
	    SQLWarning tmp = currentStatement.getWarnings();

	    if( tmp != null ) {
		if( warning == null ) {
		    warning = tmp;
		}
		else {
		    warning.setNextWarning(tmp);
		}
	    }
	}
	if( connection != null ) {
	    SQLWarning tmp = connection.getWarnings();

	    if( tmp != null ) {
		if( warning == null ) {
		    warning = tmp;
		}
		else {
		    warning.setNextWarning(tmp);
		}
	    }
	}
	return warning;
    }

    public synchronized void insertRow() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.insertRow();
	eventSupport.rowChanged();
    }

    public synchronized boolean isAfterLast() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.isAfterLast();
    }

    public synchronized boolean isBeforeFirst() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.isBeforeFirst();
    }

    public synchronized boolean isFirst() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.isFirst();
    }

    public synchronized boolean isLast() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.isLast();
    }
    
    public synchronized boolean isReadOnly() {
	return readOnly;
    }

    public synchronized boolean last() throws SQLException {
	boolean val, same;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	same = isLast();
	val = currentResults.last();
	if( !same ) {
	    eventSupport.cursorMoved();
	}
	return val;
    }

    public synchronized void moveToCurrentRow() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.moveToCurrentRow();
	eventSupport.cursorMoved();
    }

    public synchronized void moveToInsertRow() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.moveToInsertRow();
	eventSupport.cursorMoved();
    }

    public synchronized boolean next() throws SQLException {
	boolean val;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	val = currentResults.next();
	eventSupport.cursorMoved();
	return val;
    }

    public synchronized boolean previous() throws SQLException {
	boolean val;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	val = currentResults.previous();
	eventSupport.cursorMoved();
	return val;
    }

    public synchronized void refreshRow() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.refreshRow();
    }

    public synchronized boolean relative(int rows) throws SQLException {
	boolean val;
	
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	val = currentResults.relative(rows);
	eventSupport.cursorMoved();
	return val;
    }

    public synchronized void removeRowSetListener(RowSetListener rsl) {
	eventSupport.removeListener(rsl);
    }
    
    public synchronized boolean rowDeleted() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.rowDeleted();
    }

    public synchronized boolean rowInserted() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.rowInserted();
    }

    public synchronized boolean rowUpdated() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.rowUpdated();
    }

    public synchronized void setArray(int col, Array arr) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setArray(col, arr);
    }
    
    public synchronized void setAsciiStream(int col, InputStream is, int len)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setAsciiStream(col, is, len);
    }

    public synchronized void setBigDecimal(int col, BigDecimal d)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setBigDecimal(col, d);	
    }

    public synchronized void setBinaryStream(int col, InputStream is, int len)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setBinaryStream(col, is, len);
    }

    public synchronized void setBlob(int col, Blob b) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setBlob(col, b);
    }

    public synchronized void setBoolean(int col, boolean b)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setBoolean(col, b);
    }

    public synchronized void setByte(int col, byte b) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setByte(col, b);
    }

    public synchronized void setBytes(int col, byte[] bts)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setBytes(col, bts);
    }

    public synchronized void setCharacterStream(int col, Reader r, int len)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setCharacterStream(col, r, len);
    }

    public synchronized void setClob(int col, Clob clob) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setClob(col, clob);
    }

    public synchronized void setCommand(String cmd) throws SQLException {
	if( currentStatement != null ) {
	    throw new SQLException("Cannot reset command while a " +
				    " statement is being processed.");
	}
	command = cmd;
    }

    public synchronized void setConcurrency(int concur) throws SQLException {
	resultSetConcurrency = concur;
    }

    public synchronized void setDataSourceName(String nom)
    throws SQLException {
	if( connection != null ) {
	    throw new SQLException("Cannot change DSN with an open " +
				    "connection.");
	}
	dataSourceName = nom;
	dataSource = getDataSource();
    }

    public synchronized void setDate(int col, Date d) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setDate(col, d);
    }

    public synchronized void setDate(int col, Date d, Calendar cal)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setDate(col, d, cal);
    }

    public synchronized void setDouble(int col, double d) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setDouble(col, d);
    }

    public synchronized void setEscapeProcessing(boolean ep)
    throws SQLException {
	if( currentStatement == null ) {
	    currentStatement.setEscapeProcessing(ep);
	}
	escapeProcessing = ep;
    }

    private void prepareStatement() throws SQLException {
	if( connection == null ) {
	    if( dataSource == null ) {
		connection = DriverManager.getConnection(url, username,
							 password);
	    }
	    else {
		if( username == null ) {
		    connection = dataSource.getConnection();
		}
		else {
		    connection = dataSource.getConnection(username, password);
		}
	    }
	    connection.setReadOnly(readOnly);
	    connection.setTransactionIsolation(transactionIsolation);
	    if( typeMap != null ) {
		connection.setTypeMap(typeMap);
	    }
	}
	currentStatement = connection.prepareStatement(command,
						       resultSetType,
						       resultSetConcurrency);
	//	currentStatement.setEscapeProcessing(escapeProcessing);
	currentStatement.setFetchDirection(fetchDirection);
	currentStatement.setFetchSize(fetchSize);
	currentStatement.setMaxFieldSize(maxFieldSize);
	currentStatement.setMaxRows(maxRows);
	currentStatement.setQueryTimeout(queryTimeout);
    }
    
    public synchronized void setFetchDirection(int dir) throws SQLException {
	if( currentStatement != null ) {
	    currentStatement.setFetchDirection(dir);
	}
	fetchDirection = dir;
    }

    public synchronized void setFetchSize(int rows) throws SQLException {
	if( currentStatement != null ) {
	    currentStatement.setFetchSize(rows);
	}
	fetchSize = rows;
    }
    
    public synchronized void setFloat(int col, float f) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setFloat(col, f);
    }

    public synchronized void setInt(int col, int x) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setInt(col, x);
    }

    public synchronized void setLong(int col, long l) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setLong(col, l);
    }

    public synchronized void setMaxFieldSize(int sz) throws SQLException {
	if( currentStatement != null ) {
	    currentStatement.setMaxFieldSize(sz);
	}
	maxFieldSize = sz;
    }

    public synchronized void setMaxRows(int rows) throws SQLException {
	if( currentStatement != null ) {
	    currentStatement.setMaxRows(rows);
	}
	maxRows = rows;
    }

    public synchronized void setNull(int col, int type) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setNull(col, type);
    }

    public synchronized void setNull(int col, int type, String tn)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setNull(col, type);
    }

    public synchronized void setObject(int col, Object ob)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setObject(col, ob);
    }

    public synchronized void setObject(int col, Object ob, int type)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setObject(col, ob, type);
    }

    public synchronized void setObject(int col, Object ob, int type, int scale)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setObject(col, ob, type, scale);
    }

    public synchronized void setPassword(String pw) throws SQLException {
	password = pw;
    }

    public synchronized void setQueryTimeout(int seconds) throws SQLException {
	if( currentStatement != null ) {
	    currentStatement.setQueryTimeout(seconds);
	}
	queryTimeout = seconds;
    }

    public synchronized void setReadOnly(boolean ro) throws SQLException {
	if( connection != null ) {
	    connection.setReadOnly(ro);
	}
	readOnly = ro;
    }

    public synchronized void setRef(int col, Ref ref) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setRef(col, ref);
    }

    public synchronized void setShort(int col, short x) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setShort(col, x);
    }

    public synchronized void setString(int col, String str)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setString(col, str);
    }

    public synchronized void setTime(int col, Time t) throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setTime(col, t);
    }

    public synchronized void setTime(int col, Time t, Calendar cal)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setTime(col, t, cal);
    }

    public synchronized void setTimestamp(int col, Timestamp ts)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setTimestamp(col, ts);
    }

    public synchronized void setTimestamp(int col, Timestamp ts, Calendar cal)
    throws SQLException {
	if( currentStatement == null ) {
	    prepareStatement();
	}
	currentStatement.setTimestamp(col, ts, cal);
    }

    public synchronized void setTransactionIsolation(int lvl)
    throws SQLException {
	if( connection != null ) {
	    connection.setTransactionIsolation(lvl);
	}
	transactionIsolation = lvl;
    }

    public synchronized void setType(int type) throws SQLException {
	resultSetType = type;
    }

    public synchronized void setTypeMap(Map map) throws SQLException {
	if( connection != null ) {
	    connection.setTypeMap(map);
	}
	typeMap = map;
    }

    public synchronized void setUrl(String u) throws SQLException {
	url = u;
    }

    public synchronized void setUsername(String uid) throws SQLException {
	username = uid;
    }

    public synchronized void updateAsciiStream(int col, InputStream is,
					       int len)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateAsciiStream(col, is, len);
    }

    public synchronized void updateAsciiStream(String col, InputStream is,
					       int len)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateAsciiStream(col, is, len);
    }
    
    public synchronized void updateBigDecimal(int col, BigDecimal d)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBigDecimal(col, d);
    }

    public synchronized void updateBigDecimal(String col, BigDecimal d)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBigDecimal(col, d);
    }

    public synchronized void updateBinaryStream(int col, InputStream is,
						int len)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBinaryStream(col, is, len);
    }

    public synchronized void updateBinaryStream(String col, InputStream is,
						int len)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBinaryStream(col, is, len);
    }

    public synchronized void updateBoolean(int col, boolean b)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBoolean(col, b);
    }

    public synchronized void updateBoolean(String col, boolean b)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBoolean(col, b);
    }

    public synchronized void updateByte(int col, byte b) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateByte(col, b);
    }

    public synchronized void updateByte(String col, byte b)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateByte(col, b);
    }

    public synchronized void updateBytes(int col, byte[] bts)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBytes(col, bts);
    }

    public synchronized void updateBytes(String col, byte[] bts)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateBytes(col, bts);
    }

    public synchronized void updateCharacterStream(int col, Reader r, int len)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateCharacterStream(col, r, len);
    }

    public synchronized void updateCharacterStream(String col, Reader r,
						   int len)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateCharacterStream(col, r, len);
    }

    public synchronized void updateDate(int col, Date d) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateDate(col, d);
    }

    public synchronized void updateDate(String col, Date d)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateDate(col, d);
    }

    public synchronized void updateDouble(int col, double d)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateDouble(col, d);
    }

    public synchronized void updateDouble(String col, double d)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateDouble(col, d);
    }

    public synchronized void updateFloat(int col, float f)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateFloat(col, f);
    }

    public synchronized void updateFloat(String col, float f)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateFloat(col, f);
    }

    public synchronized void updateInt(int col, int x) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateInt(col, x);
    }

    public synchronized void updateInt(String col, int x) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateInt(col, x);
    }

    public synchronized void updateLong(int col, long l) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateLong(col, l);
    }

    public synchronized void updateLong(String col, long l)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateLong(col, l);
    }

    public synchronized void updateNull(int col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateNull(col);
    }

    public synchronized void updateNull(String col) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateNull(col);
    }

    public synchronized void updateObject(int col, Object ob)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateObject(col, ob);
    }

    public synchronized void updateObject(String col, Object ob)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateObject(col, ob);
    }

    public synchronized void updateObject(int col, Object ob, int sc)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateObject(col, ob, sc);
    }

    public synchronized void updateObject(String col, Object ob, int sc)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateObject(col, ob, sc);
    }

    public synchronized void updateRow() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateRow();
    }

    public synchronized void updateShort(int col, short x)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateShort(col, x);
    }

    public synchronized void updateShort(String col, short x)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateShort(col, x);
    }

    public synchronized void updateString(int col, String str)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateString(col, str);
    }

    public synchronized void updateString(String col, String str)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateString(col, str);
    }

    public synchronized void updateTime(int col, Time t) throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateTime(col, t);
    }

    public synchronized void updateTime(String col, Time t)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateTime(col, t);
    }

    public synchronized void updateTimestamp(int col, Timestamp ts)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateTimestamp(col, ts);
    }

    public synchronized void updateTimestamp(String col, Timestamp ts)
    throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	currentResults.updateTimestamp(col, ts);
    }

    public synchronized boolean wasNull() throws SQLException {
	if( currentResults == null ) {
	    throw new SQLException("Row set references no results.");
	}
	return currentResults.wasNull();
    }
}

