// --------------------------------------------------------------------------
// The jSyncManager Project -- Source File.
// Copyright (c) 1998 - 2003 Brad BARCLAY <bbarclay@jsyncmanager.org>
// --------------------------------------------------------------------------
// OSI Certified Open Source Software
// --------------------------------------------------------------------------
//
// This library is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published 
// by the Free Software Foundation; either version 2.1 of the License, or 
// (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public 
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// --------------------------------------------------------------------------
// $Id: SerialInterface.java,v 1.2 2003/11/20 17:50:58 yaztromo Exp $
// --------------------------------------------------------------------------

package org.jSyncManager.JSerial;

import org.jSyncManager.JSerial.SerialPeer;
import java.io.*;

// ==========================================================================

/** A class for accessing serial port functionality via Java.
  * This class provides serial port functionality to Java programs,
  * allowing them to set various port parameters, read and write to
  * the port, and grab and release the port.  The serial functionality
  * is presented as a set of standard Java InputStream/OutputStreams,
  * making it easily interchangable with entities such as java.net.Socket.
  * @author Brad BARCLAY &lt;bbarclay@jsyncmanager.org&gt;
  * <br>Last modified by: $Author: yaztromo $ on $Date: 2003/11/20 17:50:58 $.
  * @version $Revision: 1.2 $
  */
public class SerialInterface {
   // The following statements define useful static values.
   /** Used to specify the use of no parity.    */
   public static final int NOPARITY=20;

   /** Used to specify the use of odd parity.   */
   public static final int ODDPARITY=21;

   /** Used to specify the use of even parity.  */
   public static final int EVENPARITY=22;

   /** Used to denote the use of mark parity.   */
   public static final int MARKPARITY=23;

   /** Used to denote the use of space parity.  */
   public static final int SPACEPARITY=24;

   // The following statement define private instance variables and values
   private SerialPeer peer;                  // Used to store a handle to the peer object
   private InputStream in;                   // Used for our data input stream
   private OutputStream out;                 // Used for our data output stream

// ==========================================================================

   /** Starts a new serial interface using the given port identifier at the default speed (9600bps)
     * using 8 data bits, no parity, and 1 stop bit and starts off with the port in "grabbed" mode.
     * @param port the identifier of the serial port to open
     * @exception SerialInitializationException thrown if the port cannot be opened.
     * @exception SerialSpeedException thrown if the given speed is invalid or unsupported.
     * @exception SerialDataBitException thrown if the given data bit size is invalid.
     * @exception SerialParityException thrown if the given parity is invalid.
     * @exception SerialStopBitException thrown if the current stop-bit size is invalid.
     * @exception SerialGrabException thrown if <B>status</B> is true, and the port is currently not available.
     */
	public SerialInterface(byte port) 
      throws SerialInitializationException, SerialSpeedException, SerialDataBitException,
      SerialParityException, SerialStopBitException, SerialGrabException {

      this(port, 9600, 8, NOPARITY, 1, true);
   } // end-constructor

// --------------------------------------------------------------------------

   /** Starts a new serial interface using the given port, at the given speed,
     * using 8 data bits, no parity, and 1 stop bit and starts off with the port in "grabbed" mode.
     * @param port the identifier of the serial port to open
     * @param speed the speed at which to initialize the port
     * @exception SerialInitializationException thrown if the port cannot be opened.
     * @exception SerialSpeedException thrown if the given speed is invalid or unsupported.
     * @exception SerialDataBitException thrown if the given data bit size is invalid.
     * @exception SerialParityException thrown if the given parity is invalid.
     * @exception SerialStopBitException thrown if the current stop-bit size is invalid.
     * @exception SerialGrabException thrown if <B>status</B> is true, and the port is currently not available.
     */
	public SerialInterface(byte port, int speed) 
      throws SerialInitializationException, SerialSpeedException, SerialDataBitException,
      SerialParityException, SerialStopBitException, SerialGrabException {

      this(port, speed, 8, NOPARITY, 1, true);
   } // end-constructor

// --------------------------------------------------------------------------

   /** Starts a new serial interface using the given port at the given speed,
     * using the given values for data bits, parity and stop bits.
     * @param port the identifier of the serial port to open
     * @param speed the speed at which to initialize the port
     * @param databits the number of data bits per byte to transmit/receive
     * @param parity the parity type for a single byte
     * @param stopbits the number of stop bits in a single byte
     * @param status the desired "grab" status.
     * If true the port is grabbed during construction, otherwise everything is prepared,
     * but the port is left in the non-"grabbed" state.
     * @exception SerialInitializationException thrown if the port cannot be opened.
     * @exception SerialSpeedException thrown if the given speed is invalid or unsupported.
     * @exception SerialDataBitException thrown if the given data bit size is invalid.
     * @exception SerialParityException thrown if the given parity is invalid.
     * @exception SerialStopBitException thrown if the current stop-bit size is invalid.
     * @exception SerialGrabException thrown if <B>status</B> is true, and the port is currently not available.
     */
	public SerialInterface(byte port, int speed, int databits, 
                          int parity, int stopbits, boolean status) 
      throws SerialInitializationException, SerialSpeedException, SerialDataBitException,
      SerialParityException, SerialStopBitException, SerialGrabException {

      // Note that no special exception handling is needed, as the peer class will
      // throw the exception, and this constructor will simply propogate it automatically.

      peer=new SerialPeer(port);
      
      if (status) {
         peer.grabPort();
      } /* endif */

      peer.setSpeed(speed);
      peer.setDataBits(databits);
      peer.setParity(parity-20);
      peer.setStopBits(stopbits-1);

      // With the interface (hopefully) configured, we'll now setup our I/O streams.

      in=new InputStream() {
         // This is an anonymous inner class to define the read() method
         public int read() throws IOException {
            // The method required to read a byte from the serial peer
            return peer.getChar();
         } /* end-inner-method */
      }; /* end-innerclass */

      out=new OutputStream() {
         // This is an anonymous inner class to define the write(int b) method
         public void write(int b) throws IOException {
            // This is the method required to write a byte to the serial peer
            peer.putChar(b);
         } /* end-inner-method */
      }; /* end-innerclass */

   } // end-constructor

// --------------------------------------------------------------------------

   /** This method returns the current input stream.
     * @return the serial input stream
     */
   public InputStream getInputStream() {
      return in;
   } // end-method

// --------------------------------------------------------------------------

   /** This method returns the current output stream.
     * @return the serial output stream
     */
   public OutputStream getOutputStream() {
      return out;
   } // end-method

// --------------------------------------------------------------------------

   /** Used to get the name of the platform to which the native code portion is written.
     * @return the name of the operating system/environment the peer was written for
     */
	public static String getPlatformName() {
      return SerialPeer.getPlatformName();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to get the version number of the native code peer class.
     * @return the version number of the peer interface implementation
     */
	public static String getPeerVersion() {
      return SerialPeer.getVersion();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to get the current serial port id number.
     * @return the current serial port id number.
     */
	public byte getCurrentPort() {
      return peer.getPort();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to get the current serial speed.
     * @return the current serial speed
     */
	public int getCurrentSpeed() {
      return peer.getSpeed();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to set the current serial speed.
     * @param speed the desired serial communications speed.
     * @exception SerialSpeedException thrown if the given speed is invalid or unsupported.
     */
	public void setCurrentSpeed(int speed) throws SerialSpeedException {
      peer.setSpeed(speed);
   } // end-method

// --------------------------------------------------------------------------

   /** Used to get the current data bit size.
     * @return the current number of data bits in a single transmission byte
     */
	public int getCurrentDatabits() {
      return peer.getDataBits();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to set the current serial data bit size.
     * @param databits the desired number of data bits per transmission byte.
     * @exception SerialDataBitException thrown if the given data bit size is invalid.
     */
	public void setCurrentDatabits(int databits) throws SerialDataBitException {
      peer.setDataBits(databits);
   } // end-method
      

// --------------------------------------------------------------------------

   /** Returns the current serial interface parity.
     * @return the parity type (ie: EVENPARITY, ODDPARITY, NOPARITY, MARKPARITY).
     */
	public int getCurrentParity() {
      return peer.getParity();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to set the current serial parity.
     * @param parity the desired parity type (ie: EVENPARITY, ODDPARITY, NOPARITY, MARKPARITY).
     * @exception SerialParityException thrown if the given parity is invalid.
     */
	public void setCurrentParity(int parity) throws SerialParityException {
      peer.setParity(parity);
   } // end-method

// --------------------------------------------------------------------------

   /** Used to get the current stop bit size.
     * @return the current number of stop bits per byte.
     */
	public int getCurrentStopbits() {
      return peer.getStopBits();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to set the current serial stop bit size.
     * @param stopbits the desired number of stop bits ber transmission byte
     * @exception SerialStopBitException thrown if the given stop bit size is invalid.
     */
	public void setCurrentStopbits(int stopbits) throws SerialStopBitException {
      peer.setStopBits(stopbits);
   } // end-method

// --------------------------------------------------------------------------

   /** Used to get ahold of the handle to the Serial Peer class.
     * Note that end programmers should never use this, however it is provided for completeness,
     * and mirrors the getPeer() method in java.awt.Component.
     * @return the current instance of the SerialPeer class
     */
	public SerialPeer getPeer() {
      return peer;
   } // end-method

// --------------------------------------------------------------------------

   /** Releases the port back to the OS's use, but keeps the peer and streams alive.
     * This method will release the serial port for use by other applications.  This method should be
     * called by the finalize() method, but is also available for calling by application developers
     * for flexibility.
     * @exception SerialReleaseException thrown if there is a problem releasing the current port
     */
	public void releasePort() throws SerialReleaseException {
      peer.releasePort();
   } // end-method

// --------------------------------------------------------------------------

   /** This method will grab a released port.
     * If the serial port has been released back to the OS's use, this method will
     * pick it up again if it's available.  Otherwise a SerialGrabException is generated.
     * @exception SerialGrabException thrown if the port is not available, or is already in the "grabbed" state.
     */
	public void grabPort() throws SerialGrabException {
      peer.grabPort();
   } // end-method

// --------------------------------------------------------------------------

   /** Used to check the status of the port assigned to this instance of the SerialInterface.
     * This method will return true if the serial interface is released from 
     * its port, and false if otherwise.
     * @return true if the port has been released, false otherwise.
     */
	public boolean isReleased() {
      return peer.portStatus();
   } // end-method

// --------------------------------------------------------------------------

   /** Finalizes this object.
     * @exception Throwable any Throwable that occurs during finalization.
     */
	protected void finalize() throws Throwable {
      // Cleans up after ourselves - overrides java.lang.Object.finalize(), allowing us
      // to nicely release the port back to the OS before the object is destroyed.

      try {
         releasePort();
      } /* endtry */
      catch (SerialReleaseException e) {
         // Don't bother to do anything - this should only occur if the port is already
         // in the released state anyway.
      } /* endcatch */

   } // end-method

// --------------------------------------------------------------------------

   /** A method to set the state of the RTS/CTS handshaking.
     * Sets the state of RTS/CTS hardware handshaking.
     * @param state if <B>true</B>, enables RTS/CTS hardware handshaking.
     * @exception SerialHandshakingException thrown if the serial driver is unable to change modes.
     */
   public void setRTSCTS(boolean state) throws SerialHandshakingException {
      peer.setRTSCTS(state);
   } // end-method

// --------------------------------------------------------------------------

   /** A method to set the state of the DSR/DTR handshaking.
     * Sets the state of DSR/DTR hardware handshaking.
     * @param state if <B>true</B>, enables hardware handshaking.
     * @exception SerialHandshakingException thrown if the serial driver is unable to change modes.
     */
   public void setDSRDTR(boolean state) throws SerialHandshakingException {
      peer.setDSRDTR(state);
   } // end-method

// --------------------------------------------------------------------------

   /** A method to set the state of the XON/XOFF handshaking.
     * Sets the state of XON/XOFF software handshaking.  Note that this
     * function will set both the transmit and receive to use XON/XOFF control.
     * @param state if <B>true</B>, enables software handshaking.
     * @exception SerialHandshakingException thrown if the serial driver is unable to change modes.
     */
   public void setXONXOFF(boolean state) throws SerialHandshakingException {
      peer.setXONXOFF(state);
   } // end-method

// --------------------------------------------------------------------------

   /** Transmits a break signal.
     * @param length how long, in milliseconds, to transmit the break signal.
     */
   public void transmitBreak(long length) {
      peer.startBreak();
      synchronized(this) {
         try {
            wait(length);
         } /* endtry */
         catch (InterruptedException e) {
            // Do nothing
         } /* endcatch */
      }
      peer.stopBreak();
   } // end-method

// --------------------------------------------------------------------------

   /** Queries the state of the RTS/CTS handshaking.
     * @return <B>true</B> if RTS/CTS is enabled.
     */
   public boolean getRTSCTSState() {
      return peer.getRTSCTSState();
   } // end-method

// --------------------------------------------------------------------------

   /** Queries the state of the DSR/DTR handshaking.
     * @return <B>true</B> if DSR/DTR is enabled.
     */
   public boolean getDSRDTRState() {
      return peer.getDSRDTRState();
   } // end-method

// --------------------------------------------------------------------------

   /** Queries the state of the XON/XOFF handshaking.
     * @return <B>true</B> if XON/XOFF is enabled.
     */
   public boolean getXONXOFFState() {
      return peer.getXONXOFFState();
   } // end-method

// ==========================================================================

} // end-class

