(* ==================================================================== *)
(*                                                                      *)
(*  Main Module for the Gardens Point Component Pascal Compiler.        *)
(*      Copyright (c) John Gough 1999, 2000.                            *)
(*      This module was extensively modified from the driver            *)
(*      automatically produced by the M2 version of COCO/R, using       *)
(*      the CPascal.atg grammar used for the JVM version of GPCP.       *)
(*                                                                      *)
(* ==================================================================== *)

MODULE CPascal;
(* This is an example of a rudimentary main module for use with COCO/R.
   The auxiliary modules <Grammar>S (scanner) and <Grammar>P (parser)
   are assumed to have been constructed with COCO/R compiler generator. *)

  IMPORT
        GPCPcopyright,
        Symbols,
        RTS,
        FileNames,
        IdDesc,
        Error,
        Console,
        ProgArgs,
        G := CompState,
        CPascalP,
        S := CPascalS,
        CPascalErrors,
        SymFileRW,
        NameHash,
        Visitor,
        Builtin,
        GPText,
        Target,
        T := GPTextFiles;

(* ==================================================================== *)
(*                         Option Setting                               *)
(* ==================================================================== *)

    PROCEDURE ResetOptions*;
    BEGIN
      G.InitOptions;
    END ResetOptions;

    (* -------------------------- *)

    PROCEDURE Message*(IN msg : ARRAY OF CHAR);
    BEGIN
      G.Message(msg);
    END Message;

    (* -------------------------- *)

    PROCEDURE DoOption*(IN opt : ARRAY OF CHAR);
    BEGIN
      G.ParseOption(opt);
    END DoOption;

    (* -------------------------- *)

    PROCEDURE CondMsg(IN msg : ARRAY OF CHAR);
    BEGIN
      IF G.verbose THEN G.Message(msg) END;
    END CondMsg;

(* ==================================================================== *)
(*                    Calling the Compiler                              *)
(* ==================================================================== *)

    PROCEDURE Finalize*;
      VAR a : ARRAY 16 OF CHAR;
          b : ARRAY 256 OF CHAR;
    BEGIN
      IF CPascalErrors.forVisualStudio OR
         CPascalErrors.xmlErrors THEN RETURN END;
      b := "<" + CompState.modNam + ">";
      IF CPascalS.errors = 0 THEN
        b := (b + " No errors");
      ELSIF CPascalS.errors = 1 THEN
        b := (b + " There was one error");
      ELSE
        GPText.IntToStr(CPascalS.errors, a);
        b := (b + " There were " + a + " errors");
      END;
      IF CPascalS.warnings = 1 THEN
        b := (b + ", and one warning");
      ELSIF CPascalS.warnings > 1 THEN
        GPText.IntToStr(CPascalS.warnings, a);
        b := (b + ", and " + a + " warnings");
      END;
      G.Message(b);
    END Finalize;

(* ==================================================================== *)

    PROCEDURE FixListing*;
      VAR doList : BOOLEAN;
          events : INTEGER;
    BEGIN
      doList := (G.listLevel > CPascalS.listNever);
      events := CPascalS.errors;
      IF G.warning THEN INC(events, CPascalS.warnings) END;
      IF (events > 0) OR
         (G.listLevel = CPascalS.listAlways) THEN
        CPascalS.lst := GPTextFiles.createFile(G.lstNam);
        IF CPascalS.lst # NIL THEN
          CPascalErrors.PrintListing(doList);
          GPTextFiles.CloseFile(CPascalS.lst);
          CPascalS.lst := NIL;
        ELSE
          G.Message("cannot create file <" + G.lstNam + ">");
          IF events > 0 THEN CPascalErrors.PrintListing(FALSE) END;
        END;
      END;
      CPascalErrors.ResetErrorList();
    END FixListing;

(* ==================================================================== *)

    PROCEDURE Compile*(IN nam : ARRAY OF CHAR; OUT retVal : INTEGER);
    BEGIN
      G.CheckOptionsOK;
      retVal := 0;
      G.totalS := RTS.GetMillis();
      CPascalS.src := GPTextFiles.findLocal(nam);
      IF CPascalS.src = NIL THEN
        G.Message("cannot open local file <" + nam + ">");
      ELSE
        NameHash.InitNameHash(G.hashSize);
        G.InitCompState(nam);
        Builtin.RebindBuiltins();
        Target.Select(G.thisMod, G.target);
        Target.Init();
        CondMsg("Starting Parse");
        CPascalP.Parse;   (* do the compilation here *)
        G.parseE := RTS.GetMillis();
        IF CPascalS.errors = 0 THEN
          CondMsg("Doing statement attribution");
          G.thisMod.StatementAttribution(Visitor.newImplementedCheck());
          IF (CPascalS.errors = 0) & G.extras THEN
            CondMsg("Doing type erasure");
            G.thisMod.TypeErasure(Visitor.newTypeEraser());
          END;
          IF CPascalS.errors = 0 THEN
            CondMsg("Doing dataflow analysis");
            G.thisMod.DataflowAttribution();
            G.attrib := RTS.GetMillis();
            IF CPascalS.errors = 0 THEN
              IF G.doSym THEN
                CondMsg("Emitting symbol file");
                SymFileRW.EmitSymfile(G.thisMod);
                G.symEnd := RTS.GetMillis();
                IF G.doAsm THEN
                  IF G.isForeign() THEN
                    CondMsg("Foreign module: no code output");
                  ELSE
                    CondMsg("Emitting assembler");
                    Target.Emit();
                    G.asmEnd := RTS.GetMillis();
                    IF G.doCode THEN Target.Assemble() END;
                  END;
                END;
              END;
            END;
          END;
        END;
        IF CPascalS.errors # 0 THEN retVal := 2 END;
        G.totalE := RTS.GetMillis();
        FixListing;
        Finalize;
        IF G.doStats THEN G.Report END;
      END;
    RESCUE (sysX)
      retVal := 2;
      G.Message("<< COMPILER PANIC >>");
      S.SemError.RepSt1(299, RTS.getStr(sysX), 1, 1);
     (* 
      *  If an exception is raised during listing, FixListing will
      *  be called twice. Avoid an attempted sharing violation...
      *)
      IF CPascalS.lst # NIL THEN 
      GPTextFiles.CloseFile(CPascalS.lst);
      G.Message(RTS.getStr(sysX));
      CPascalS.lst := NIL;
      ELSE
        FixListing;
      END;
      Finalize;
    END Compile;

(* ==================================================================== *)
(*                       Main Argument Loop                             *)
(* ==================================================================== *)

BEGIN
  G.InitOptions;
  CPascalErrors.Init;
  Builtin.InitBuiltins;
END CPascal.

(* ==================================================================== *)
