/**
 * Title:        Comedia Utils
 * Description:  Project contains some general purpose non-visual beans.
 * Beans do not require any special libraies.
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.util.xml;

import java.io.*;
import org.comedia.util.scanner.*;

/**
 * Presents XML document which can be read and written from/to stream.
 * It parses the stream and build a tree structure with CXmlElement objects.
 */
public class CXmlDocument {
  /**
   * The root element object.
   */
  private CXmlElement rootElement = null;

  /**
   * The XML lexical scanner.
   */
  private CXmlScanner scanner = new CXmlScanner();

  /**
   * Constructs this class with default parameters.
   */
  public CXmlDocument() {
    this(new CDefaultXmlElement("Root"));
  }

  /**
   * Constructs this class and assignes root XML element.
   * @param rootElement a root XML element.
   */
  public CXmlDocument(CXmlElement rootElement) {
    this.rootElement = rootElement;
  }

  /**
   * Reads from input stream, parses and build a XML tree structure.
   * @param in an input stream to read this XML document.
   */
  public void readDocument(Reader in) throws IOException, CXmlException {
    BufferedReader reader = new BufferedReader(in);
    String buffer = "";
    String line;
    while ((line = reader.readLine()) != null)
      buffer += line + "\n";
    rootElement.clear();
    scanner.setBuffer(buffer);
    scanner.gotoNextToken();
    rootElement.parseChildElements(scanner);
  }

  /**
   * Writes from output stream build XML tree structure of this document.
   * @param out an output stream to write this XML document.
   */
  public void writeDocument(Writer out) throws IOException {
    rootElement.writeChildElements(out, -2);
    out.flush();
  }

  /**
   * Gets a root XML element which starts the document sructure.
   * @result a XML document root element.
   */
  public CXmlElement getRootElement() {
    return rootElement;
  }

  /**
   * The main function for test purposes.
   */
  public static void main(String[] args) {
    CXmlDocument xml = new CXmlDocument();
    try {
      xml.readDocument(new FileReader("d:\\example.xml"));
      xml.writeDocument(new FileWriter("d:\\result.xml"));
    }
    catch (Exception e) {
      System.out.println("Oopps.. File access error.");
    }
  }
}