/**
 * Title:        Comedia Beans
 * Description:  Digital Clock Bean
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.beans;

import java.awt.*;
import javax.swing.*;
import java.util.*;
import java.text.*;

/**
 * Visual bean which presents simple digital clock.
 * This bean can be resized, it uses font size to fits time string to
 * the draw rectangle.
 * <p>
 * <center><img src="CDigitalClock.gif"></center>
 */
public class CDigitalClock extends JComponent {
  private ClockThread clockThread = null;
  private String delimiter = ":";
  private boolean showDate = true;

  /**
   * Default bean constructor.
   */
  public CDigitalClock() {
    clockThread = new ClockThread();
    clockThread.start();
  }

  /**
   * Gets ShowDate property.
   * @result ShowDate property value.
   */
  public boolean isShowDate() {
    return showDate;
  }

  /**
   * Sets DhowDate property.
   * @param showDate new property value.
   */
  public void setShowDate(boolean showDate) {
    this.showDate = showDate;
    repaint();
  }

  /**
   * Finds a font size which better fits the specified message to given sizes.
   * @param g current graphical context.
   * @param s string message.
   * @param w width of the rectangle.
   * @param h height of the rectangle.
   */
  private Font findFont(Graphics g, String s, int w, int h) {
    int size = 6;
    Font font = new Font(this.getFont().getName(), this.getFont().getStyle(), size);
    while (true) {
      FontMetrics m = g.getFontMetrics(font);
      if (m.getHeight() > h || m.stringWidth(s) > w)
        break;
      size++;
      font = new Font(this.getFont().getName(), this.getFont().getStyle(), size);
    }
    return font;
  }

  /**
   * Paints current time on the screen.
   * It uses current foreground and background colors fro drawing.
   * @param g current graphical context.
   */
  public void paint(Graphics g) {
    String strTime, strDate;
    Calendar calendar = new GregorianCalendar();
    SimpleDateFormat format = new SimpleDateFormat();
    double timeRate = (showDate) ? 0.5 : 0.8;

    // Find suitable font and position
    Dimension d = this.getSize();
    g.clearRect(0, 0, d.width, d.height);

    // Draw current time
    format.applyPattern("HH" + delimiter + "mm" /*+ delimiter + "ss"*/);
    strTime = format.format(calendar.getTime());
    Font font = findFont(g, strTime, (int)(d.getWidth() * 0.8),
      (int)(d.getHeight() * timeRate));
    FontMetrics m = g.getFontMetrics(font);
    int x = (d.width - m.stringWidth(strTime)) / 2;
    int y = (int) ((d.getHeight() * timeRate - m.getHeight()) / 2
      + d.getHeight() * 0.1) + m.getAscent();
    g.setFont(font);
    g.drawString(strTime, x, y);

    // Draw current date
    if (showDate) {
      format.applyPattern("EEEE, dd MMMM yyyy");
      strDate = format.format(calendar.getTime());
      font = findFont(g, strDate, (int)(d.getWidth() * 0.8),
        (int)(d.getHeight() * 0.3));
      m = g.getFontMetrics(font);
      x = (d.width - m.stringWidth(strDate)) / 2;
      y = (int) ((d.getHeight() * 0.3 - m.getHeight()) / 2
        + d.getHeight() * 0.6) + m.getAscent();
      g.setFont(font);
      g.drawString(strDate, x, y);
    }
  }

  /**
   * Shows this bean in separate application frame on the screen.
   * It allows to use this bean as stand along application.
   */
  public static void main(String[] args) {
    CDigitalClock digitalClock = new CDigitalClock();
    JFrame frame = new JFrame("Digital Clock");
//    digitalClock.setShowDate(false);
    frame.setContentPane(digitalClock);
    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    frame.setSize(300, 300);
    frame.setLocation(300, 300);
    frame.show();
  }

  /**
   * Thread for display current time each 0.5 second.
   */
  private class ClockThread extends Thread {
    /**
     * Makes all work in the thread.
     */
    public void run() {
      while (true) {
        try {
          delimiter = (delimiter != ":") ? ":" : " ";
          repaint();
          sleep(500);
        }
        catch (InterruptedException e){}
      }
    }
  }
}