/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.text;

import java.awt.*;
import javax.swing.*;
import javax.swing.text.*;
import org.comedia.util.scanner.*;

/**
 * Presents an abstract highlighter for CSyntaxEditor. This highlighter is used
 * to highlight a text while editing according mappings for different program
 * languages.
 */
public abstract class CAbstractHighlighter {
  /**
   * The whitespace attribute.
   */
  protected MutableAttributeSet whiteSpaceAttr = new SimpleAttributeSet();

  /**
   * The identificator attribute.
   */
  protected MutableAttributeSet identAttr = new SimpleAttributeSet();

  /**
   * The comment attribute.
   */
  protected MutableAttributeSet commentAttr = new SimpleAttributeSet();

  /**
   * The delimiters attribute.
   */
  protected MutableAttributeSet delimAttr = new SimpleAttributeSet();

  /**
   * The keyword attribute.
   */
  protected MutableAttributeSet keywordAttr = new SimpleAttributeSet();

  /**
   * The string attribute.
   */
  protected MutableAttributeSet stringAttr = new SimpleAttributeSet();

  /**
   * The number attribute.
   */
  protected MutableAttributeSet numberAttr = new SimpleAttributeSet();

  /**
   * The language syntax analizer.
   */
  protected CScanner scanner = new CScanner();

  /**
   * Construct this class with default parameters.
   */
  public CAbstractHighlighter() {
    whiteSpaceAttr.addAttribute(StyleConstants.NameAttribute, "White");
    identAttr.addAttribute(StyleConstants.NameAttribute, "Indent");
    commentAttr.addAttribute(StyleConstants.NameAttribute, "Comment");
    delimAttr.addAttribute(StyleConstants.NameAttribute, "Delim");
    keywordAttr.addAttribute(StyleConstants.NameAttribute, "Keyword");
    stringAttr.addAttribute(StyleConstants.NameAttribute, "String");
    numberAttr.addAttribute(StyleConstants.NameAttribute, "Number");

    StyleConstants.setForeground(whiteSpaceAttr, Color.black);
    StyleConstants.setForeground(identAttr, Color.black);
    StyleConstants.setForeground(commentAttr, new Color(0, 130, 0));
    StyleConstants.setItalic(commentAttr, true);
    StyleConstants.setForeground(delimAttr, Color.black);
    StyleConstants.setForeground(keywordAttr, new Color(0, 0, 132));
    StyleConstants.setBold(keywordAttr, true);
    StyleConstants.setForeground(stringAttr, Color.blue);
//    StyleConstants.setItalic(stringAttr, true);
    StyleConstants.setForeground(numberAttr, Color.blue);
    scanner.setShowComment(true);
    scanner.setShowEol(true);
    scanner.setShowString(true);
    scanner.setShowKeyword(true);
    scanner.setShowType(true);
    scanner.setShowSpace(true);
  }

  /**
   * Gets an attribute for whitespaces.
   * @result an attribute for whitespaces.
   */
  public MutableAttributeSet getWhiteSpaceAttribute() {
    return whiteSpaceAttr;
  }

  /**
   * Sets an attribute for whitespaces.
   * @param attrs an attribute for whitespaces.
   */
  public void setWhiteSpaceAttribute(MutableAttributeSet attrs) {
    whiteSpaceAttr = attrs;
  }

  /**
   * Gets an attribute for identifiers.
   * @result an attribute for identifiers.
   */
  public MutableAttributeSet getIdentAttribute() {
    return identAttr;
  }

  /**
   * Sets an attribute for identifiers.
   * @param attrs an attribute for identifiers.
   */
  public void setIdentAttribute(MutableAttributeSet attrs) {
    identAttr = attrs;
  }

  /**
   * Gets an attribute for comments.
   * @result an attribute for comments.
   */
  public MutableAttributeSet getCommentAttribute() {
    return commentAttr;
  }

  /**
   * Sets an attribute for comments.
   * @param attrs an attribute for comments.
   */
  public void setCommentAttribute(MutableAttributeSet attrs) {
    commentAttr = attrs;
  }

  /**
   * Gets an attribute for delimiters.
   * @result an attribute for delimiters.
   */
  public MutableAttributeSet getDelimAttribute() {
    return delimAttr;
  }

  /**
   * Sets an attribute for delimiters.
   * @param attrs an attribute for delimiters.
   */
  public void setDelimAttribute(MutableAttributeSet attrs) {
    delimAttr = attrs;
  }

  /**
   * Gets an attribute for keywords.
   * @result an attribute for keywords.
   */
  public MutableAttributeSet getKeywordAttribute() {
    return keywordAttr;
  }

  /**
   * Sets an attribute for keywords.
   * @param attrs an attribute for keywords.
   */
  public void setKeywordAttribute(MutableAttributeSet attrs) {
    keywordAttr = attrs;
  }

  /**
   * Gets an attribute for strings.
   * @result an attribute for strings.
   */
  public MutableAttributeSet getStringAttribute() {
    return stringAttr;
  }

  /**
   * Sets an attribute for strings.
   * @param attrs an attribute for strings.
   */
  public void setStringAttribute(MutableAttributeSet attrs) {
    stringAttr = attrs;
  }

  /**
   * Gets an attribute for numbers.
   * @result an attribute for numbers.
   */
  public MutableAttributeSet getNumberAttribute() {
    return numberAttr;
  }

  /**
   * Sets an attribute for numbers.
   * @param attrs an attribute for numbers.
   */
  public void setNumberAttribute(MutableAttributeSet attrs) {
    numberAttr = attrs;
  }

//  String tk = "";

  /**
   * Sets a buffer string.
   * @param buffer a new buffer string.
   */
  public void setBuffer(String buffer) {
    scanner.setBuffer(buffer);
//    tk = "";
  }

  /**
   * Goes to the next token in the buffer.
   * @result a token string.
   */
  public String gotoNextToken() {
    scanner.gotoNextToken();
    return scanner.getToken();
/*
    tk = "";
    do {
      scanner.gotoNextToken();
      tk += scanner.getToken();
    } while (scanner.getTokenType() != scanner.EOF
      && scanner.getTokenType() != scanner.COMMENT
      && scanner.getNextTokenType() != scanner.COMMENT);
    return tk;
*/
  }

  /**
   * Gets the current token value.
   * result the current token value.
   */
  public String getToken() {
    return scanner.getToken();
//    return tk;
  }

  /**
   * Gets the attribute for the current token value.
   * @param an attribute for the current token value.
   */
  public AttributeSet getAttribute() {
    int tokenType = scanner.getTokenType();
//    int tokenType = scanner.IDENT;
    if (scanner.getTokenType() == scanner.COMMENT)
      tokenType = scanner.COMMENT;

    if (tokenType == scanner.KEYWORD || tokenType == scanner.TYPE
      || tokenType == scanner.BOOL)
      return keywordAttr;
    if (tokenType == scanner.COMMENT)
      return commentAttr;
    if (tokenType == scanner.IDENT)
      return identAttr;
    if (tokenType == scanner.UNKNOWN || tokenType == scanner.SPACE
      || tokenType == scanner.LF || tokenType == scanner.EOL
      || tokenType == scanner.EOF)
      return whiteSpaceAttr;
    if (tokenType == scanner.STRING)
      return stringAttr;
    if (tokenType == scanner.INT || tokenType == scanner.FLOAT)
      return numberAttr;
    return delimAttr;
  }

  /**
   * Locates a last unclosed multiline comment before specified position.
   * @param text the text to search in.
   * @param pos the last position
   * @param startSymbol comment starting symbol.
   * @param endSymbol comment end symbol.
   * @result a position of last unclosed comment or -1 otherwise.
   */
  protected int locateUnclosedComment(String text, int pos, String startSymbol,
    String endSymbol) {

    int startPos = text.lastIndexOf(startSymbol, pos);
    int endPos = text.lastIndexOf(endSymbol, pos);
    return (startPos > endPos) ? startPos : -1;
  }

  /**
   * Locates a last unclosed multiline comment before specified position.
   * @param text the text to search in.
   * @param pos the last position
   * @result a position of last unclosed comment or -1 otherwise.
   */
  public abstract int locateUnclosedComment(String text, int pos);

  /**
   * Check if current is a multiline comment.
   * @result <code>TRUE</code> if current token is a multiline comment
   * and <code>FALSE</code> otherwise.
   */
  public boolean isMultilineComment() {
    return false;
  }
}