/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import javax.swing.*;
import java.awt.event.*;
import java.util.*;

/**
 * Implements a ComboBox which displays a list of predefined colors and
 * allows to choose one.
 * <p>
 * <img src="CColorComboBox.gif">
 * <p>
 * Usage examples:
 * <pre>
 * JFrame frame = new JFrame("Comedia Editor Test");
 *
 * CColorComboBox ctrl = new CColorComboBox();
 * ctrl.setSelectedColor(new Color(25, 150, 100));
 *
 * frame.getContentPane().add(ctrl);
 * </pre>
 */
public class CColorComboBox extends JComboBox {
  /**
   * The model which contains all predefined colors.
   */
  protected final CImageListModel model = new CImageListModel();

  /**
   * Constructs this combo box with default properties.
   */
  public CColorComboBox() {
    this.setModel(model);
    this.setRenderer(new CImageCellRenderer());
    this.setSelectedIndex(0);
  }

  /**
   * Gets the selected color.
   * @result the selected color.
   */
  public Color getSelectedColor() {
    return model.getColor(this.getSelectedIndex());
  }

  /**
   * Sets the nearest color from the list to the  specified color.
   * @param color a color to set in the combo box.
   */
  public void setSelectedColor(Color color) {
    this.setSelectedIndex(model.getColorIndex(color));
  }

  /**
   * Provides a model for this Color ComboBox.
   */
  private static class CImageListModel extends DefaultComboBoxModel {
    private static final Color color[] = {
      Color.black, new Color(132, 0, 0), new Color(0, 130, 0),
      new Color(132, 130, 0), new Color(0, 0, 132), new Color(132, 0, 132),
      new Color(0, 130, 132), Color.gray, Color.lightGray, Color.red,
      Color.green, Color.yellow, Color.blue, Color.magenta, Color.cyan,
      Color.white};

    private static final String label [] = {
      "Black", "Maroon", "Green", "Olive", "Navy", "Purple", "Teal", "Gray",
      "Silver", "Red", "Lime", "Yellow", "Blue", "Fuchsia", "Aqua", "White"};

    public CImageListModel () {
      Icon icon;
      for (int i = 0, n = label.length; i < n; i++) {
        icon = new CSquareIcon (color[i]);
        Hashtable result = new Hashtable();
        result.put ("label", label[i]);
        result.put ("icon",  icon);
        addElement(result);
      }
    }

    public Color getColor(int index) {
      return color[index];
    }

    public int getColorIndex(Color c) {
      int result = 0;
      int max = 1000;

      for(int i = 0; i < color.length; i++) {
        Color current = color[i];
        int n = Math.abs(current.getRed() - c.getRed()) +
          Math.abs(current.getGreen() - c.getGreen()) +
          Math.abs(current.getBlue() - c.getBlue());
        if (n < max) {
          max = n;
          result = i;
        }
      }
      return result;
    }
  }

  /**
   * Implements the Color ComboBox renderer.
   */
  private static class CImageCellRenderer implements ListCellRenderer {
    private boolean focused = false;
    private JLabel renderer;

    public CImageCellRenderer() {
      renderer = new JLabel();
      renderer.setOpaque (true);
    }

    public Component getListCellRendererComponent(JList list, Object value,
      int index, boolean isSelected, boolean cellHasFocus) {

      if (value == null) {
        renderer.setText("");
        renderer.setIcon(null);
      } else {
        Hashtable h = (Hashtable) value;
        renderer.setText((String)h.get("label"));
        renderer.setIcon((Icon)h.get("icon"));
      }

      renderer.setBackground (isSelected ? SystemColor.textHighlight :
        SystemColor.text);
      renderer.setForeground (isSelected ? SystemColor.textHighlightText :
        SystemColor.textText);

      return renderer;
    }
  }

  /**
   * Presents a rectangle icon filled with solid color.
   */
  private static class CSquareIcon implements Icon {
    private Color color;

    public CSquareIcon (Color color) {
      this.color = color;
    }

    public void paintIcon (Component c, Graphics g, int x, int y) {
      g.setColor(color);
      g.fillRect (x + 2, y, getIconWidth() - 2, getIconHeight());
      g.setColor(Color.black);
      g.drawRect(x + 2, y, getIconWidth() - 2, getIconHeight());
    }

    public int getIconWidth() {
      return 25;
    }

    public int getIconHeight() {
      return 11;
    }
  }

  /**
   * Runs a standalone application for test purposes.
   * @param args command line arguments.
   */
  public static void main(String[] args) {
    try {
//      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
    }
    catch(Exception e) {}

    JFrame frame = new JFrame("Comedia Editor Test");
    CColorComboBox ctrl = new CColorComboBox();
    ctrl.setSelectedColor(new Color(25, 150, 100));
    frame.getContentPane().setLayout(new FlowLayout());
    frame.getContentPane().add(ctrl);

    frame.setLocation(100, 100);
    frame.setSize(300, 300);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
  }
}