/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import javax.swing.*;
import java.awt.*;

/**
 * Presents an icon with transparent encoded image.
 * The class contains a bunch of predefined icons and can draw disabled icons.
 * <p><img src="CEncodedIcon.gif">
 * <p>Usage examples:
 * <pre>
 * panel.setLayout(new GridLayout(3, 3));
 * panel.add(new JButton(CEncodedIcon.BIG_RIGHT_ARROW));
 * panel.add(new JButton(CEncodedIcon.MANY_SIGN));
 * panel.add(new JButton(CEncodedIcon.ONE_SIGN));
 * panel.add(new JButton(CEncodedIcon.PEN_SIGN));
 * panel.add(new JButton(CEncodedIcon.SMALL_LEFT_ARROW));
 * panel.add(new JButton(CEncodedIcon.SMALL_RIGHT_ARROW));
 * panel.add(new JButton(CEncodedIcon.STAR_SIGN));
 *
 * CEncodedIcon icon = new CEncodedIcon(new int[] {0x60, 0x60, 0xf0, 0x60, 0x60},
 *   6, 5);
 * icon.setColor(Color.blue);
 * panel.add(new JButton(icon));
 * </pre>
 */
public class CEncodedIcon implements Icon {
  /**
   * The first sign.
   */
  public final static CEncodedIcon FIRST_SIGN =
    new CEncodedIcon(
      new int[] {0xc1, 0xc3, 0xc7, 0xcf, 0xc7, 0xc3, 0xc1}, 8, 7);

  /**
   * The prior sign.
   */
  public final static CEncodedIcon PRIOR_SIGN =
    new CEncodedIcon(
      new int[] {0x10, 0x30, 0x70, 0xf0, 0x70, 0x30, 0x10}, 4, 7);

  /**
   * The last sign.
   */
  public final static CEncodedIcon LAST_SIGN =
    new CEncodedIcon(
      new int[] {0x83, 0xc3, 0xe3, 0xf3, 0xe3, 0xc3, 0x83}, 8, 7);

  /**
   * The last new sign.
   */
  public final static CEncodedIcon LAST_NEW_SIGN =
    new CEncodedIcon(
      new int[] {0x80, 0x40, 0xc1, 0x50, 0xe0, 0xe0, 0xf3, 0xf8, 0xe0, 0xe0,
      0xc1, 0x50, 0x80, 0x40}, 13, 7);

  /**
   * The next sign.
   */
  public final static CEncodedIcon NEXT_SIGN =
    new CEncodedIcon(
      new int[] {0x80, 0xc0, 0xe0, 0xf0, 0xe0, 0xc0, 0x80}, 4, 7);

  /**
   * The big right arrow sign.
   */
  public final static CEncodedIcon BIG_RIGHT_ARROW =
    new CEncodedIcon(
      new int[] {0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xf0, 0xe0, 0xc0, 0x80}, 5, 9);

  /**
   * The big down arrow sign.
   */
  public final static CEncodedIcon SMALL_DOWN_ARROW =
    new CEncodedIcon(
      new int[] {/*0xfe,*/ 0x7c, 0x38, 0x10}, 7, 3);

  /**
   * The big up arrow sign.
   */
  public final static CEncodedIcon SMALL_UP_ARROW =
    new CEncodedIcon(
      new int[] {0x10, 0x38, 0x7c/*, 0xfe*/}, 7, 3);

  /**
   * The star sign.
   */
  public final static CEncodedIcon STAR_SIGN =
    new CEncodedIcon(new int[] {0x10, 0x54, 0x38, 0xfe, 0x38, 0x54, 0x10}, 7, 7);

  /**
   * The writing pen sign.
   */
  public final static CEncodedIcon PEN_SIGN =
    new CEncodedIcon(new int[] {0x01, 0xc0, 0x01, 0x20, 0x02, 0xc0, 0x02, 0x40,
      0x04, 0x80, 0x04, 0x80, 0x09, 0x00, 0x09, 0x00, 0x0e, 0x00, 0x0c, 0x00,
      0xa8, 0x00}, 11, 11);

  /**
   * The small left arrow sign.
   */
  public final static CEncodedIcon SMALL_LEFT_ARROW =
    new CEncodedIcon(new int[] {0x18, 0x38, 0x78, 0xf8, 0x78, 0x38, 0x18}, 5, 7);

  /**
   * The small right arrow sign.
   */
  public final static CEncodedIcon SMALL_RIGHT_ARROW =
    new CEncodedIcon(new int[]{0xc0, 0xe0, 0xf0, 0xf8, 0xf0, 0xe0, 0xc0}, 5, 7);

  /**
   * The one unit sign.
   */
  public final static CEncodedIcon ONE_SIGN =
    new CEncodedIcon(new int[] {0x60, 0xe0, 0x60, 0x60, 0x60, 0xf0}, 4, 6);

  /**
   * The many units sign.
   */
  public final static CEncodedIcon MANY_SIGN =
    new CEncodedIcon(
      new int[] {0x79, 0xe0, 0xc6, 0x30, 0xc6, 0x30, 0xc6, 0x30, 0x79, 0xe0},
      12, 5);

  /**
   * The encoded points of image.
   */
  private int[] image = null;

  /**
   * The width of this image.
   */
  private int width = 0;

  /**
   * The height of this image.
   */
  private int height = 0;

  /**
   * The color of this image.
   */
  private Color color = Color.black;

  /**
   * The disabled flag.
   */
  private boolean enabled = true;

  /**
   * Constructs this image and assignes main properties.
   */
  public CEncodedIcon(int[] image, int width, int height) {
    this.image = image;
    this.width = width;
    this.height = height;
  }

  /**
   * Paints the current image in the screen.
   * @param c a component to paint.
   * @param g a current graphical context.
   * @param x x coordinate of image.
   * @param y y coordinate of image.
   */
  public void paintIcon(Component c, Graphics g, int x, int y) {
    if (image != null) {
      if (enabled) {
        g.setColor(color);
        paint(g, x, y);
      } else {
        g.setColor(Color.white);
        paint(g, x + 1, y + 1);
        g.setColor(Color.gray);
        paint(g, x, y);
      }
    }
  }

  /**
   * Paints the image somewhere.
   * @param g a current graphical context.
   * @param x x coordinate of image.
   * @param y y coordinate of image.
   */
  private void paint(Graphics g, int x, int y) {
    int bytes = width / 8 + ((width % 8 != 0) ? 1: 0);
    for (int i = 0; i < bytes; i++) {
      for (int j = 0; j < height; j++) {
        int p = image[i + j * bytes];
        for (int k = 0; k < 8; k++)
          if (((p << k) & 0x80) != 0)
            g.drawLine(x + i * 8 + k, y + j, x + i * 8 + k, y + j);
      }
    }
  }

  /**
   * Gets an image binary array of this icon.
   * @param an image binary array of this icon.
   */
  public int[] getImageArray() {
    return image;
  }

  /**
   * Gets the current image width.
   * @result a width of this image.
   */
  public int getIconWidth() {
    return width;
  }

  /**
   * Gets the current image height.
   * @result a height of this image.
   */
  public int getIconHeight() {
    return height;
  }

  /**
   * Gets the current icon color.
   * @result the current icon color.
   */
  public Color getColor() {
    return color;
  }

  /**
   * Sets a new icon color.
   * @param color a new icon color.
   */
  public void setColor(Color color) {
    this.color = color;
  }

  /**
   * Checks the current enabled status.
   * @result the current enabled status.
   */
  public boolean isEnabled() {
    return enabled;
  }

  /**
   * Sets a new enabled status.
   * @param color a new enabled status.
   */
  public void setEnabled(boolean enabled) {
    this.enabled = enabled;
  }

  /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    JFrame frame = new JFrame("Comedia CEncodedIcon Test");
    JPanel panel = new JPanel();

    panel.setLayout(new GridLayout(3, 4));
    panel.add(new JButton(CEncodedIcon.SMALL_UP_ARROW));
    panel.add(new JButton(CEncodedIcon.BIG_RIGHT_ARROW));
    panel.add(new JButton(CEncodedIcon.SMALL_DOWN_ARROW));
    panel.add(new JButton(CEncodedIcon.MANY_SIGN));
    panel.add(new JButton(CEncodedIcon.ONE_SIGN));
    panel.add(new JButton(CEncodedIcon.PEN_SIGN));
    panel.add(new JButton(CEncodedIcon.SMALL_LEFT_ARROW));
    panel.add(new JButton(CEncodedIcon.SMALL_RIGHT_ARROW));
    panel.add(new JButton(CEncodedIcon.STAR_SIGN));
    panel.add(new JButton(CEncodedIcon.FIRST_SIGN));
    panel.add(new JButton(CEncodedIcon.PRIOR_SIGN));
    panel.add(new JButton(CEncodedIcon.NEXT_SIGN));
    panel.add(new JButton(CEncodedIcon.LAST_SIGN));
    panel.add(new JButton(CEncodedIcon.LAST_NEW_SIGN));
    frame.getContentPane().add(panel, BorderLayout.CENTER);

    frame.setLocation(100, 100);
    frame.setSize(300, 300);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
  }
}