/**
 * Title:        Comedia Beans
 * Description:  Outlook Like Bar
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.ArrayList;

/**
 * Implements Outlook like bar with sweep animation.
 * <p>
 * <center><img src="COutlookBar.gif"></center>
 * Example of the usage:
 * <pre>
 * // Creates outlook bar
 * COutlookBar bar = new COutlookBar();
 *
 * // Initializes outlook bar content
 * bar.setAnimationSpeed(50);
 * bar.addPanel("Panel", new JPanel());
 * bar.addPanel("Button", new JButton("xxx"));
 * bar.addPanel("Label", new JLabel("xxx"));
 * bar.addPanel("Text", new JTextPane());
 * bar.addPanel("Auto Panel");
 * bar.removePanel(0);
 * </pre>
 */
public class COutlookBar extends JComponent implements ActionListener {
  /**
   * The list of contained items.
   */
  private ArrayList items = new ArrayList();

  /**
   * The current selected item.
   */
  private int currentIndex = -1;

  /**
   * The speed of sweep animation.
   */
  private int animationSpeed = 20;

  /**
   * Constructs this bar with default parameters.
   */
  public COutlookBar() {
    this.setLayout(new BorderLayout());
  }

  /**
   * Performs actions from user inteface and local timers.
   * @param e an object which describes occured event.
   */
  public void actionPerformed(ActionEvent e) {
    if (e.getSource() instanceof JButton) {
      int index = getPanelIndex(((JButton) e.getSource()).getText());
      if (currentIndex >= 0 && currentIndex != index)
        sweepBar(index);
    }
  }

  /**
   * Gets the current animation speed.
   * @result the current speed of sweep animation in ms.
   */
  public int getAnimationSpeed() {
    return animationSpeed;
  }

  /**
   * Sets the new animation speed.
   * @param animationSpeed the new speed of sweep animation in ms.
   */
  public void setAnimationSpeed(int animationSpeed) {
    this.animationSpeed = animationSpeed;
  }

  /**
   * Gets selected panel in this bar.
   * @result an index of current selected panel.
   */
  public int getSelectedIndex() {
    return this.currentIndex;
  }

  /**
   * Sets current selected item of this bar.
   * @param index an index of new selected panel.
   */
  public void setSelectedIndex(int index) {
    if (currentIndex != index) {
      currentIndex = index;
      updateContent();
    }
  }

  /**
   * Gets an index of panel with specified name.
   * @param the name of a existed panel.
   * @result an index of panel with specified name.
   */
  public int getPanelIndex(String name) {
    for (int i = 0; i < items.size(); i++) {
      COutlookBarItem current = (COutlookBarItem) items.get(i);
      if (current.getButton().getText().equals(name))
        return i;
    }
    return -1;
  }

  /**
   * Gets an index of panel with specified component.
   * @param component the component of a existed panel.
   * @result an index of panel with specified component.
   */
  public int getPanelIndex(Component component) {
    for (int i = 0; i < items.size(); i++) {
      COutlookBarItem current = (COutlookBarItem) items.get(i);
      if (current.getComponent() == component)
        return i;
    }
    return -1;
  }

  /**
   * Gets a count of current available panels.
   * @result a count of available panels.
   */
  public int getPanelCount() {
    return items.size();
  }

  /**
   * Gets a panel with specified index.
   * @param an index of a existed panel.
   * @result a panel with specified index.
   */
  public Component getPanel(int index) {
    return ((COutlookBarItem) items.get(index)).getComponent();
  }

  /**
   * Gets a panel with specified name.
   * @param the name of a existed panel.
   * @result a panel with specified name.
   */
  public Component getPanel(String name) {
    for (int i = 0; i < items.size(); i++) {
      COutlookBarItem current = (COutlookBarItem) items.get(i);
      if (current.getButton().getText().equals(name))
        return current.getComponent();
    }
    return null;
  }

  /**
   * Gets a panel title with specified panel index.
   * @param an index of a existed panel.
   * @result a title of panel with specified index.
   */
  public String getPanelTitle(int index) {
    return ((COutlookBarItem) items.get(index)).getButton().getText();
  }

  /**
   * Adds a panel with specified name.
   * @param name a name of new panel.
   * @result a JPanel component of new panel.
   */
  public Component addPanel(String name) {
    return addPanel(name, new JPanel());
  }

  /**
   * Adds a panel with specified name.
   * @param name a name of new panel.
   * @param component a component of new panel.
   * @result a JPanel component of new panel.
   */
  public Component addPanel(String name, Component component) {
    COutlookBarItem item = new COutlookBarItem(name, component);

    items.add(item);
    item.getButton().addActionListener(this);
    if (currentIndex < 0) currentIndex = 0;
    updateContent();

    return component;
  }

  /**
   * Removes panel with specified index.
   * @param index an index of removing panel.
   */
  public void removePanel(int index) {
    if (currentIndex >= index) currentIndex--;
    if (currentIndex < 0 && items.size() > 0)
      currentIndex = 0;
    items.remove(index);
    updateContent();
  }

  /**
   * Removes panel with specified name.
   * @param name a name of removing panel.
   */
  public void removePanel(String name) {
    removePanel(getPanelIndex(name));
  }

  /**
   * Removes panel with specified component.
   * @param component a component of removing panel.
   */
  public void removePanel(Component component) {
    removePanel(getPanelIndex(component));
  }

  /**
   * Updates content of this bar.
   */
  private void updateContent() {
    super.removeAll();

    if (items.size() >= 0 && currentIndex >= 0 && currentIndex < items.size()) {
      JPanel topPanel = null;
      JPanel bottomPanel = null;
      int topItems = currentIndex + 1;
      int bottomItems = items.size() - topItems;

      if (topItems > 0) {
        topPanel = new JPanel();
        topPanel.setLayout(new GridLayout(topItems, 1));
        super.add(topPanel, BorderLayout.NORTH);
      } else
        topPanel = null;

      if (bottomItems > 0) {
        bottomPanel = new JPanel();
        bottomPanel.setLayout(new GridLayout(bottomItems, 1));
        super.add(bottomPanel, BorderLayout.SOUTH);
      }

      super.add(((COutlookBarItem) items.get(currentIndex)).getComponent(),
        BorderLayout.CENTER);

      for (int i = 0; i < items.size(); i++) {
        COutlookBarItem current = (COutlookBarItem) items.get(i);

        if (i <= currentIndex)
          topPanel.add(current.getButton());
        else
          bottomPanel.add(current.getButton());
      }
    }

    this.revalidate();
    this.repaint();
  }

  /**
   * Adds a new component to this bar.
   * @param name a name of associated group
   * @param component a new component.
   */
/*
  public Component add(String name, Component component) {
    return addPanel(name, component);
  }
*/

  /**
   * Adds a new component to this bar.
   * @param component a new component.
   */
/*
  public Component add(Component component) {
    int n = 1;
    for (int i = 0; i < items.size(); i++) {
      COutlookBarItem current = (COutlookBarItem) items.get(i);
      if (current.getButton().getText().equals("Panel " + n)) {
        n++;
        i = 0;
      }
    }
    return addPanel("Panel " + n, component);
  }
*/

  /**
   * Removes a component from this bar.
   * @param component a component to remove.
   */
/*
  public void remove(Component component) {
    removePanel(component);
  }
*/

  /**
   * Removes a component from this bar.
   * @param index an index of component to remove.
   */
/*
  public void remove(int index) {
    removePanel(index);
  }
*/

  /**
   * Updates Look&Feel of user interface.
   */
  public void updateUI() {
    for (int i = 0; i < items.size(); i++) {
      COutlookBarItem current = (COutlookBarItem) items.get(i);
      SwingUtilities.updateComponentTreeUI(current.getButton());
      if (current.getComponent() instanceof JComponent)
        SwingUtilities.updateComponentTreeUI(
          (JComponent) current.getComponent());
    }
    super.updateUI();
  }

  /**
   * Starts to sweep current this OutlookBar content. It adds the
   * specialized sweep panel and creates the sweep timer.
   */
  private void sweepBar(int index) {
    super.removeAll();

    JPanel topPanel = new JPanel();
    JPanel centralPanel = new JPanel();
    JPanel bottomPanel = new JPanel();

    for (int i = 0; i < items.size(); i++) {
      COutlookBarItem current = (COutlookBarItem) items.get(i);
      JButton button = new JButton(current.getButton().getText());
      if (i <= index && i <= currentIndex)
        topPanel.add(button);
      else if (i > index && i > currentIndex)
        bottomPanel.add(button);
      else
        centralPanel.add(button);
    }

    if (topPanel.getComponentCount() > 0) {
      topPanel.setLayout(new GridLayout(topPanel.getComponentCount(), 1));
      super.add(topPanel, BorderLayout.NORTH);
    }

    if (bottomPanel.getComponentCount() > 0) {
      bottomPanel.setLayout(new GridLayout(bottomPanel.getComponentCount(), 1));
      super.add(bottomPanel, BorderLayout.SOUTH);
    }

    if (centralPanel.getComponentCount() > 0) {
      centralPanel.setLayout(new GridLayout(centralPanel.getComponentCount(), 1));
    }

    Component first = ((COutlookBarItem) items.get(currentIndex)).getComponent();
    Component second = ((COutlookBarItem) items.get(index)).getComponent();
    CSweepPanel sweep = new CSweepPanel();
    sweep.reset(first, second, centralPanel, currentIndex - index);
    super.add(sweep, BorderLayout.CENTER);
    this.revalidate();
    this.repaint();

    currentIndex = index;

    CSweepThread thread = new CSweepThread(sweep);
    try {
      thread.start();
    }
    catch (Exception e) {}
  }

  /**
   * Impements Panel which performs sweep operation in OutlookBar.
   */
  private class CSweepPanel extends JPanel {
    private int y = 0;
    private int dir = 0;
    private JPanel topPanel = new JPanel();
    private JPanel centerPanel = new JPanel();
    private JPanel bottomPanel = new JPanel();
    private JPanel movePanel = new JPanel();

    public CSweepPanel() {
      this.setLayout(new BorderLayout());
      movePanel.setLayout(new BorderLayout());
      topPanel.setLayout(new BorderLayout());
      centerPanel.setLayout(new BorderLayout());
      bottomPanel.setLayout(new BorderLayout());
    }

    public void reset(Component first, Component second, Component central,
      int dir) {
      y = 0;

      this.dir = dir;
      this.removeAll();


      topPanel.removeAll();
      bottomPanel.removeAll();
      centerPanel.removeAll();
      centerPanel.add(central);

      if (dir >= 0) {
        topPanel.add(second);
        bottomPanel.add(first);

        movePanel.add(topPanel, BorderLayout.CENTER);
        movePanel.add(centerPanel, BorderLayout.SOUTH);

        this.add(movePanel, BorderLayout.NORTH);
        this.add(bottomPanel, BorderLayout.CENTER);
      } else {
        topPanel.add(first);
        bottomPanel.add(second);

        movePanel.add(bottomPanel, BorderLayout.CENTER);
        movePanel.add(centerPanel, BorderLayout.NORTH);

        this.add(movePanel, BorderLayout.SOUTH);
        this.add(topPanel, BorderLayout.CENTER);
      }

      topPanel.setPreferredSize(new Dimension(0, 0));
      bottomPanel.setPreferredSize(new Dimension(0, 0));
    }

    public boolean step(int dy) {
      y += dy;
      if (dir >= 0)
        topPanel.setPreferredSize(new Dimension(0, y));
      else
        bottomPanel.setPreferredSize(new Dimension(0, y));
      int ph = this.getSize().height - centerPanel.getPreferredSize().height;
      return y+dy < ph && ph != 0;
    }
  }

  /**
   * Implements a thread which manages the sweep process.
   */
  private class CSweepThread extends Thread {
    private CSweepPanel sweep = null;

    public CSweepThread(CSweepPanel sweep) {
      this.sweep = sweep;
    }

    public void run() {
      while (sweep.step(20)) {
        sweep.revalidate();
        sweep.repaint();
        try {
          sleep(animationSpeed);
        }
        catch (Exception e) {}
      }
      updateContent();
    }
  }

  /**
   * Presents OutlookBar item with all related properties.
   */
  private class COutlookBarItem {
    private JButton button = null;
    private Component component = null;

    public COutlookBarItem(String name, Component component) {
      this.button = new JButton(name);
      this.component = component;
    }

    public JButton getButton() {
      return button;
    }

    public Component getComponent() {
      return component;
    }
  }

  /**
   * Runs this outlook bar as standalone application for test purposes.
   * @param args arguments of command line.
   */
  public static void main(String[] args) {
    try {
//      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
//      UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
    }
    catch(Exception e) {}

    // Creates outlook bar
    COutlookBar bar = new COutlookBar();
    // Initializes outlook bar content
    bar.setAnimationSpeed(50);
    bar.addPanel("Panel", new JPanel());
    bar.addPanel("Button", new JButton("xxx"));
    bar.addPanel("Label", new JLabel("xxx"));
    bar.addPanel("Text", new JTextPane());
    bar.addPanel("Auto Panel");
    bar.removePanel(0);

    // Constructs outlined frame
    JFrame frame = new JFrame("Outlook Bar");
    frame.getContentPane().add(bar);
    frame.setSize(305, 320);
    frame.setLocation(200, 200);
    frame.setDefaultCloseOperation(3);
    frame.show();
/*
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      SwingUtilities.updateComponentTreeUI(bar);
    }
    catch(Exception e) {}
*/
  }
}
