/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * ColumnLayout.java
 *
 * Creator:
 * 10.02.2005 11:43:21 Pecar
 *
 * Maintainer:
 * 10.02.2005 11:43:21 Pecar
 *
 * Last Modification:
 * $Id: ColumnLayout.java,v 1.8 2005/09/07 14:42:04 pecar Exp $
 *
 * Copyright (c) 2005 ABACUS Research AG, All Rights Reserved
 */
package ch.abacus.lib.ui.layout;

import java.awt.*;
import java.util.ArrayList;

public class ColumnLayout implements LayoutManager2 {
    public static final int NO_VARIABLE_COLUMN = -1;
    public static final String FULL_LINE = "line";

    private int mHgap;
    private int mVgap;
    private final int mColumnCount;
    private final int mVariableColumn;
    private final ArrayList<LineItem> mComponentLines = new ArrayList<LineItem>();
    private Point mCurrentPoint;
    private Insets mInsets = new Insets(0,0,0,0);

    class LineItem {
        private Component mSingleComponent;
        private ArrayList<Component> mComponents;

        public LineItem(Component singleComponent) {
            mSingleComponent = singleComponent;
        }
        public LineItem() {
        }

        boolean isSingle() { return mSingleComponent != null; }
        Component get(int index) {
            if(index >= mComponents.size())
                return null;
            return mComponents.get(index);
        }
        Component get() {
            return mSingleComponent;
        }
        void set(int index, Component component) {
            if(mSingleComponent != null || mComponents == null)
                mComponents = new ArrayList<Component>();
            mSingleComponent = null;
            for(int i=0; i<=index; i++)
                mComponents.add(null);
            mComponents.set(index, component);
        }
        void set(Component component) {
            mComponents = null;
            mSingleComponent = component;
        }
        void remove(Component component) {
            if(mSingleComponent == component)
                mSingleComponent = null;
            if(mComponents != null) {
                for(int i=mComponents.size() - 1; i>=0; i--) {
                    Component c = mComponents.get(i);
                    if(c == component)
                        mComponents.remove(i);
                }
            }
        }
        public int size() {
            assert mSingleComponent == null;
            return mComponents != null ? mComponents.size() : 0;
        }
        Component[] getComponentArray() {
            return mSingleComponent != null ? new Component[] { mSingleComponent } : mComponents != null ? mComponents.toArray(new Component[mComponents.size()]) : new Component[] {};
        }
    }

    public ColumnLayout(int columnCount) {
        this(columnCount, columnCount-1, 20, 5);
    }
    public ColumnLayout(int columnCount, int hgap, int vgap) {
        this(columnCount, columnCount-1, hgap, vgap);
    }
    public ColumnLayout(int columnCount, int variableColumn, int hgap, int vgap) {
        if(columnCount < 1)
            throw new IllegalArgumentException("columncount must be greater than zero");
        if(variableColumn < -1 || variableColumn >= columnCount)
            throw new IllegalArgumentException("variable column must be a valid column or -1");
        mColumnCount = columnCount;
        mVariableColumn = variableColumn;
        mHgap = hgap;
        mVgap = vgap;
        mCurrentPoint = new Point(0,0);
    }
    public int getHgap() { return mHgap; }
    public void setHgap(int hgap) { mHgap = hgap; }
    public int getVgap() { return mVgap; }
    public void setVgap(int vgap) { mVgap = vgap; }
    public int getColumnCount() { return mColumnCount; }
    public int getVariableColumn() { return mVariableColumn; }
    public Insets getInsets() { return mInsets; }
    public void setInsets(int top, int left, int bottom, int right) {
        mInsets.top = top;
        mInsets.left = left;
        mInsets.bottom = bottom;
        mInsets.right = right;
    }
    public void addLayoutComponent(Component comp, Object constraints) {
        synchronized(comp.getTreeLock()) {
            if((constraints == null) || (constraints instanceof String)) {
                addLayoutComponent((String) constraints, comp);
            } else {
                throw new IllegalArgumentException("cannot add to layout: constraint must be a string (or null)");
            }
        }
    }
    private Component getComponent(Point point) {
        if(point.y >= mComponentLines.size())
            return null;
        LineItem line = mComponentLines.get(point.y);
        if(line == null)
            return null;
        return line.isSingle() ? line.get() : line.get(point.x);
    }
    public void addLayoutComponent(String name, Component comp) {
        synchronized(comp.getTreeLock()) {
            if(name == null) {
                for(int i=mComponentLines.size(); i<=mCurrentPoint.y; i++)
                    mComponentLines.add(null); //fill the list
                LineItem line = mComponentLines.get(mCurrentPoint.y);
                if(line == null) {
                    line = new LineItem();
                    mComponentLines.set(mCurrentPoint.y, line);
                }
                line.set(mCurrentPoint.x, comp);
                do {
                    mCurrentPoint.x++;
                    if(mCurrentPoint.x >= mColumnCount) {
                        mCurrentPoint.x = 0;
                        mCurrentPoint.y++;
                    }
                } while(getComponent(mCurrentPoint) != null);
            } else if(FULL_LINE.equals(name)) {
                if(mCurrentPoint.x != 0)
                    mCurrentPoint.y++;
                mCurrentPoint.x = 0;
                for(int i=mComponentLines.size(); i<=mCurrentPoint.y; i++) {
                    mComponentLines.add(null);
                }
                mComponentLines.set(mCurrentPoint.y++, new LineItem(comp));
                while(getComponent(mCurrentPoint) != null) {
                    mCurrentPoint.x++;
                    if(mCurrentPoint.x >= mColumnCount) {
                        mCurrentPoint.x = 0;
                        mCurrentPoint.y++;
                    }
                }
            } else {
                int pos = name.indexOf(',');
                int line;
                int column = -1;
                if(pos == -1) {
                    try {
                        line = Integer.parseInt(name);
                    } catch(NumberFormatException nfe) {
                        throw new IllegalArgumentException("cannot add to layout: " + name + " cannot be interpreted as linenumber use a number");
                    }
                } else {
                    try {
                        column = Integer.parseInt(name.substring(0,pos));
                        line = Integer.parseInt(name.substring(pos+1));
                    } catch(NumberFormatException nfe) {
                        throw new IllegalArgumentException("cannot add to layout: " + name + " cannot be interpreted as coordinate, specify something like 1,2");
                    }
                }
                if(line < 0)
                    throw new IllegalArgumentException("cannot add to layout at line " + line + ": line cannot be negative");
                for(int i=mComponentLines.size(); i<=line; i++) {
                    mComponentLines.add(null);
                }
                if(column == -1) {
                    mComponentLines.set(line, new LineItem(comp));
                } else {
                    if(column < 0)
                        throw new IllegalArgumentException("cannot add to layout at column " + column + ": column cannot be negative");
                    if(column >= mColumnCount)
                        throw new IllegalArgumentException("cannot add to layout at column " + column + ": maximum column is " + (mColumnCount-1) + " ");

                    LineItem lineItem = mComponentLines.get(line);
                    if(lineItem == null) {
                        lineItem = new LineItem(comp);
                        mComponentLines.set(line, lineItem);
                    }
                    lineItem.set(column, comp);
                }
            }
        }
    }
    public void removeLayoutComponent(Component comp) {
        synchronized(comp.getTreeLock()) {
            for(LineItem line : mComponentLines) {
                if(line != null)
                    line.remove(comp);
            }
        }
    }
    private boolean isVariableColumn(int ix) {
        return ix == mVariableColumn;
    }
    public boolean hasVariableColumn() {
        return mVariableColumn != -1;
    }
    public Dimension minimumLayoutSize(Container target) {
        synchronized(target.getTreeLock()) {
            int[] widths = new int[mColumnCount];
            int[] heights = new int[mComponentLines.size()];
            int fullLineWidth = 0;

            for(int i=0; i < mComponentLines.size(); i++) {
                LineItem line = mComponentLines.get(i);
                if(line == null)
                    continue;
                if(!line.isSingle()) {
                    for (int ix = 0; ix < Math.min(line.size(), mColumnCount); ix++) {
                        Component c = line.get(ix);
                        if(c != null) {
                            Dimension d = c.getMinimumSize();
                            if (d.width > widths[ix])
                                widths[ix] = d.width;
                            if(d.height > heights[i])
                                heights[i] = d.height;
                        }
                    }
                } else {
                    Dimension d = line.get().getMinimumSize();
                    if(d.width > fullLineWidth)
                        fullLineWidth = d.width;
                    if(d.height > heights[i])
                        heights[i] = d.height;
                }
            }

            Dimension dim = new Dimension(0,0);

            for(int i=0; i<widths.length; i++) {
                dim.width += widths[i];
                if(i != 0)
                    dim.width += mHgap;
            }
            if(fullLineWidth > dim.width)
                dim.width = fullLineWidth;

            for(int i=0; i<heights.length; i++) {
                dim.height += heights[i];
                if(i != 0)
                    dim.height += mVgap;
            }

            Insets insets = target.getInsets();
            dim.width += insets.left + insets.right + mInsets.left + mInsets.right;
            dim.height += insets.top + insets.bottom + mInsets.top + mInsets.bottom;

            return dim;
        }
    }
    public Dimension preferredLayoutSize(Container target) {
        synchronized(target.getTreeLock()) {
            int[] widths = new int[mColumnCount];
            int[] heights = new int[mComponentLines.size()];
            int fullLineWidth = 0;

            for(int i=0; i < mComponentLines.size(); i++) {
                LineItem line = mComponentLines.get(i);
                if(line == null)
                    continue;
                if(!line.isSingle()) {
                    for (int ix = 0; ix < Math.min(line.size(), mColumnCount); ix++) {
                        Component c = line.get(ix);
                        if(c != null) {
                            Dimension d = c.getPreferredSize();
                            if (d.width > widths[ix])
                                widths[ix] = d.width;
                            if(d.height > heights[i])
                                heights[i] = d.height;
                        }
                    }
                } else {
                    Dimension d = line.get().getPreferredSize();
                    if(d.width > fullLineWidth)
                        fullLineWidth = d.width;
                    if(d.height > heights[i])
                        heights[i] = d.height;
                }
            }

            Dimension dim = new Dimension(0,0);

            for(int i=0; i<widths.length; i++) {
                dim.width += widths[i];
                if(i != 0)
                    dim.width += mHgap;
            }
            if(fullLineWidth > dim.width)
                dim.width = fullLineWidth;

            for(int i=0; i<heights.length; i++) {
                dim.height += heights[i];
                if(i != 0)
                    dim.height += mVgap;
            }

            Insets insets = target.getInsets();
            dim.width += insets.left + insets.right + mInsets.left + mInsets.right;
            dim.height += insets.top + insets.bottom + mInsets.top + mInsets.bottom;

            return dim;
        }
    }
    public Dimension maximumLayoutSize(Container target) {
        synchronized(target.getTreeLock()) {
            int[] widths = new int[mColumnCount];
            int[] heights = new int[mComponentLines.size()];
            int fullLineWidth = 0;

            for(int i=0; i < mComponentLines.size(); i++) {
                LineItem line = mComponentLines.get(i);
                if(line == null)
                    continue;
                if(!line.isSingle()) {
                    for (int ix = 0; ix < Math.min(line.size(), mColumnCount); ix++) {
                        Component c = line.get(ix);
                        if(c != null) {
                            Dimension d = c.getMaximumSize();
                            if (d.width > widths[ix])
                                widths[ix] = d.width;
                            if(d.height > heights[i])
                                heights[i] = d.height;
                        }
                    }
                } else {
                    Dimension d = line.get().getMaximumSize();
                    if(d.width > fullLineWidth)
                        fullLineWidth = d.width;
                    if(d.height > heights[i])
                        heights[i] = d.height;
                }
            }

            Dimension dim = new Dimension(0,0);

            for(int i=0; i<widths.length; i++) {
                dim.width += widths[i];
                if(i != 0)
                    dim.width += mHgap;
            }
            if(fullLineWidth > dim.width)
                dim.width = fullLineWidth;

            for(int i=0; i<heights.length; i++) {
                dim.height += heights[i];
                if(i != 0)
                    dim.height += mVgap;
            }

            Insets insets = target.getInsets();
            dim.width += insets.left + insets.right + mInsets.left + mInsets.right;
            dim.height += insets.top + insets.bottom + mInsets.top + mInsets.bottom;

            return dim;
        }
    }
    public float getLayoutAlignmentX(Container parent) {
        return 0.5f;
    }
    public float getLayoutAlignmentY(Container parent) {
        return 0.5f;
    }
    public void invalidateLayout(Container target) {
        for(int i=mComponentLines.size() - 1; i >= 0; i--) {
            LineItem line = mComponentLines.get(i);
            if(line == null)
                mComponentLines.remove(i);
            else if(!line.isSingle()) {
                boolean doRemove = true;
                for(int ii=line.size() - 1; ii >= 0; ii--) {
                    if(line.get(ii) != null) {
                        if(ii == line.size() - 1) {
                            mCurrentPoint.y = i + 1;
                            mCurrentPoint.x = 0;
                        }  else {
                            mCurrentPoint.y = i;
                            mCurrentPoint.x = ii + 1;
                        }
                        doRemove = false;
                        break;
                    }
                }
                if(doRemove)
                    mComponentLines.remove(i);
                else
                    break;
            } else {
                mCurrentPoint.y = i;
                break;
            }
        }
    }
    public void layoutContainer(Container target) {
        synchronized(target.getTreeLock()) {
            final Insets insets = target.getInsets();

            int top = insets.top + mInsets.top;

            int[] widths = new int[mColumnCount];
            int[] heights = new int[mComponentLines.size()];

            for(int iy=0; iy<mComponentLines.size(); iy++) {
                LineItem line = mComponentLines.get(iy);
                if(line == null)
                    continue;
                if(!line.isSingle()) {
                    for(int ix=0; ix<Math.min(line.size(), mColumnCount); ix++) {
                        Component c = line.get(ix);
                        if(c != null) {
                            Dimension d = c.getPreferredSize();
                            if(!isVariableColumn(ix))
                                widths[ix] = Math.max(d.width, widths[ix]);
                            heights[iy] = Math.max(d.height, heights[iy]);
                        }
                    }
                } else {
                    Dimension d = line.get().getPreferredSize();
                    heights[iy] = d.height;
                }
            }

            int leftWidth = target.getWidth() - insets.right - insets.left - mInsets.right - mInsets.left;
            if(hasVariableColumn()) {
                for(int i=0; i<widths.length; i++) {
                    if(i != 0) leftWidth -= mHgap;
                    leftWidth -= widths[i];
                }
            }

            int left;
            for(int iy=0; iy<mComponentLines.size(); iy++) {
                left = insets.left + mInsets.left;
                LineItem line = mComponentLines.get(iy);
                if(line == null)
                    continue;
                if(!line.isSingle()) {
                    for(int ix=0; ix<Math.min(line.size(), mColumnCount); ix++) {
                        Component c = line.get(ix);
                        if(!isVariableColumn(ix)) {
                            if(c != null)
                                c.setBounds(left, top, widths[ix], heights[iy]);
                            left += widths[ix] + mHgap;
                        } else {
                            if(c != null)
                                c.setBounds(left, top, leftWidth, heights[iy]);
                            left += leftWidth + mHgap;
                        }
                    }
                } else  {
                    line.get().setBounds(left, top, target.getWidth() - insets.right - insets.left - mInsets.right - mInsets.left, heights[iy]);
                }
                top += heights[iy] + mVgap;
            }
        }
    }
    public String toString() {
        return getClass().getName() + "[columncount=" + mColumnCount + ",variablecolumn=" + mVariableColumn + ",hgap=" + mHgap + ",vgap=" + mVgap + "]";
    }
    /**
     * zero base line to remove; the line is only remove in the layout not in the container
     * @param line
     */
    public void removeLine(int line) {
        if(line >= 0 && line < mComponentLines.size())
            mComponentLines.remove(line);
        invalidateLayout(null);
    }
    public Component[] getLine(int line) {
        if(line >= 0 && line < mComponentLines.size()) {
            LineItem lineItem = mComponentLines.get(line);
            if(lineItem != null)
                return lineItem.getComponentArray();
        }
        return new Component[] {};
    }
    public int getLineCount() {
        return mComponentLines.size();
    }
}