/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.abaRenderer;

import ch.abacus.lib.ui.JAScrollPane;
import ch.abacus.lib.ui.layout.AnchoringLayoutManager;
import ch.abacus.lib.ui.renderer.common.*;
import ch.abacus.lib.util.CloseInterface;
import ch.abacus.lib.util.GlobalInterface;

import javax.swing.*;
import javax.swing.text.JTextComponent;
import java.awt.*;
import java.io.PrintStream;
import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;
import java.util.WeakHashMap;

/**
 * <p>Title: Abacus Java AbaRenderer</p>
 * <p>Description: This library renders java interfaces from XML descriptions</p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company:A Abacus Research</p>
 * @author Michael Gouker (Cagey Logic Technologies)
 * @version 0.0.1 - Test version for prototype.  Runs output from IDE.
 *
 *  Revision 1 5/1/2003 - added insets for frames in metaobject and auto embed scrollpanes.
 *
 */

/**
 * class: AbaRenderer - This class loads a user interface from a metaproject specification.
 *
 * Implementation: AbaMetaDataUser (for configuration manager, class loader, etc)
 *                 CloseInterface  (for instance management on exit)
 */


public class AbaRenderer implements AbaMetaDataUser, CloseInterface, AbaRendererInterface {

    /**
     *
     theXMLDocumentName String - this is the xml document where the project is stored.
     */
    public String theXMLDocumentName = "";

    public byte[] theProzBuffer = null;
    public String theProzBufferClassName = null;
    /**
     attachedToDesigner boolean - set to true if run internally from the design cockpit.
     */
    public boolean attachedToDesigner = false;
    /**
     // sBufferedXML String - this is a buffer used to load xml from memory rather than from a document
     */
    public String sBufferedXML;
    /**
     // sBaseDir String - where the system is run from
     */
    public String sBaseDir = "";
    /**
     // sUserHome String - the user's home directory - default for output of default.conf file.
     */
    public String sUserHome = "";
    /**
     // theRenderingDestination Object - where the rendering will occur when not its own frame.
     */
    public Object theRenderingDestination;
    /**
     // SystemConfigurationPathName String - this is set to fix where the deployment files are.
     */
    public String SystemConfigurationPathName = "ch.abacus.lib.ui.renderer.deploy.";
    /**
     // SystemConfigurationDocumentName - String - this is the document that stores system configuration
     */
    public String SystemConfigurationDocumentName = "hammer-configuration.xml";
    /**
     // SystemMetadataDocumentName - String - this is the name of the metadata document that contains
     // definitions of classes and constants.
     */
    public String SystemMetadataDocumentName = "metadata.meta";
    /**
     // props - Properties - java properties stored locally in an object.
     */
    public Properties props = null;
    /**
     // theProject - MetaProject - the project being rendered
     */
    public MetaProject theProject = null;
    /**
     // theFileManager - HammerFileManager - system that finds documents using the class path.
     */
    public HammerFileManager theFileManager = new HammerFileManager(this);
    /**
     // theLogFile - HammmerLogFileAccess - system that writes out log messages to a file or console.
     */
    public HammerLogFileAccess theLogFile = null;
    /**
     // theConfigurationManager - HammerConfigurationManager - system that manages configuration information
     */
    public HammerConfigurationManager theConfigurationManager = null;
    /**
     // theClassLoader - HammerClassLoader - system that loads classes, resources, and files from the class path
     */
    public HammerClassLoader theClassLoader = null;
    /**
     // theSeparator - String - store separator so we don't have to keep looking for it.
     */
    public String theSeparator = "";
    /**
     // iVersion - int - the version of the renderer
     */
    public int iVersion = 0;
    /**
     // iRevision - int - the revision of the renderer
     */
    public int iRevision = 1;
    /**
     // iSubRevision - int - the sub revision of the renderer
     */
    public int iSubRevision = 1;
    /**
     // iBuild - int - the build number
     */
    public int iBuild = 1;
    /**
     // theLanguage - int - the current language
     */
    public int theLanguage = HammerLanguagePresentation.DEFAULT;
    /**
     // oldErr - PrintStream - used to write messages to standard out and restore to old driver.
     */
    public PrintStream oldErr = null;
    /**
     // theGlobalInterface - GlobalInterface - Abacus specific data.
     */
    public GlobalInterface theGlobalInterface = null;
    /**
     // NO_MODE - not defined
     */
    public static final int NO_MODE = 0;
    /**
     // DESIGN_MODE - rendering from designer
     */
    public static final int DESIGN_MODE = 1;
    /**
     // EXECUTION_MODE - rendering from app
     */
    public static final int EXECUTION_MODE = 2;
    /**
     // iInputType - int - whether the doc is coming from a jar, a separate file, or memory.
     */
    public int iInputType = 0;
    /**
     // theMetadataDispenser - MetadataDispenser - system to share metadata among rendering projects.
     */
    public MetadataDispenser theMetadataDispenser;

    /**
     * theRegisteredInterfaceObjects - a quick access list of all externally registered objects.
     */
    protected WeakHashMap theRegisteredInterfaceObjects = new WeakHashMap(0);
    private boolean bNovaState = false;
    private boolean bOpenSourceState = false;
    private boolean bLoaded = false;

    /**
     * bAddDefaultPaths - boolean - add default and user paths to class loader
     */

    public boolean bAddDefaultPaths = false;

    /**
     * getSystemMetadataDocumentName - returns the name of the metadata document that contains the information about classes, constants, etc
     *
     * @return String - the name of the metadata document.
     */


    public String getSystemMetadataDocumentName() {
        return SystemMetadataDocumentName;
    }

    /**
     * setSystemMetadataDocumentName - allows the name of the metadata document.  This allows the programmer
     * to make the renderer render a different set of classes than the standard classes in metadata.meta.
     *
     * @param SystemDocumentName (String) the name of the metadata document.
     */
    public void setSystemMetadataDocumentName(String SystemDocumentName) {
        this.SystemMetadataDocumentName = SystemDocumentName;
    }

    /**
     * getSystemConfigurationPathName - returns the path where the system will look for configuration files like metadata.meta.
     * @return String - the path of the metadata document.
     */
    public String getSystemConfigurationPathName() {
        return SystemConfigurationPathName;
    }

    /**
     * setSystemConfigurationPathName - sets the system configuration path where files like metadata.meta are found
     * @param systemConfigurationPathName - String - the new configuration path.
     */
    public void setSystemConfigurationPathName(String systemConfigurationPathName) {
        SystemConfigurationPathName = systemConfigurationPathName;
    }

    /**
     * getSystemConfigurationDocumentName - returns the name of the system configuration document which by default
     * is hammer-configuration.xml.
     * @return String - the name of the system configuration document.
     */
    public String getSystemConfigurationDocumentName() {
        return SystemConfigurationDocumentName;
    }

    /**
     * setSystemConfigurationDocumentName - allows the configuration document name to be set.
     * @param systemConfigurationDocumentName - String - the new configuration document name.
     */
    public void setSystemConfigurationDocumentName(String systemConfigurationDocumentName) {
        SystemConfigurationDocumentName = systemConfigurationDocumentName;
    }

    public boolean getGuruState() {
        return false;
    }

    public void showObject(MetaObject theBottomPanel) {
    }

    public void tellProgress(String s) {
    }

    /**
     * getProps - returns the property object.
     * @return Properties - the properties at initialization.
     */
    public Properties getProps() {
        return props;
    }

    /**
     * getUserHome - Gets the name of the user's home directory.
     * @return String - the name of the users home directory.
     */
    public String getUserHome() {
        return sUserHome;
    }

    /**
     * getBaseDir - Gets the directory from which the system is run.
     * @return String - the name of the base directory.
     */
    public String getBaseDir() {
        return sBaseDir;
    }

    /**
     * getVersionNumber - returns the version number.
     * @return int - the version number.
     */
    public int getVersionNumber() {
        return iVersion;
    }

    /**
     * getRevisionNumber - returns the revision number
     * @return int - the revision number.
     */
    public int getRevisionNumber() {
        return iRevision;
    }

    /**
     * getSubRevisionNumber - returns the sub-revision number
     * @return int - the sub-revision number.
     */
    public int getSubRevisionNumber() {
        return iSubRevision;
    }

    /**
     * getBuildNumber - returns the build number
     * @return int - the build number.
     */
    public int getBuildNumber() {
        return iBuild;
    }

    /**
     * getLogFile - returns the object that logs operations.
     * @return HammerLogFileAccess - the logging object.
     */
    public HammerLogFileAccess getLogFile() {
        return theLogFile;
    }

    /**
     * getFileManager - returns the object that finds files
     * @return HammerFileManager - the file manager
     */
    public HammerFileManager getFileManager() {
        return theFileManager;
    }

    /**
     * setFileManager - Allows the file manager to be changed.
     * @param theFileManager - HammerFileManager - the file manager that finds files.
     */
    public void setFileManager(HammerFileManager theFileManager) {
        this.theFileManager = theFileManager;
    }

    /**
     * getConfigurationManager
     * @return HammerConfigurationManager - the system that finds configuration information.
     */
    public HammerConfigurationManager getConfigurationManager() {
        return theConfigurationManager;
    }

    /**
     * setConfigurationManager - Allows the configuration manager to be set.
     * @param theConfigurationManager - the new configuration manager.
     */
    public void setConfigurationManager(HammerConfigurationManager theConfigurationManager) {
        this.theConfigurationManager = theConfigurationManager;
    }

    /**
     * getClassLoader - gets the object that searches class path to load resources
     * and classes into the renderer.
     * @return HammerClassLoader - the class loader.
     */
    public HammerClassLoader getClassLoader() {
        return theClassLoader;
    }

    /**
     * setClassLoader - sets the object that searches class path to load resources
     * and classes into the renderer.
     * @param theClassLoader - HammerClassLoader - the new class loader.
     */
    public void setClassLoader(HammerClassLoader theClassLoader) {
        this.theClassLoader = theClassLoader;
    }

    /**
     * setBufferedXML - allows the input buffer to be set to some text.  The
     * renderer can also load a project from memory.
     * @param sData - the buffer where the xml to render is stored.
     */
    public void setBufferedXML(String sData) {
        this.sBufferedXML = sData;
    }

    /**
     * getBufferedXML - returns a buffer to xml text
     * @return String - xml buffer for rendering.
     */
    public String getBufferedXML() {
        return sBufferedXML;
    }

    /**
     * setAddDefaultPaths - when true default paths are added to class path
     * (this is the default - you must set to false to not add them)
     * @param bAddDefaultPaths - boolean (true/false)
     */

    public void setAddDefaultPaths(boolean bAddDefaultPaths) {
        this.bAddDefaultPaths = bAddDefaultPaths;
    }

    /**
     * getAddDefaultPaths - returns whether default paths are added.
     * @return boolean true/false
     */

    public boolean getAddDefaultPaths() {
        return bAddDefaultPaths;
    }


    /**
     * init - set up the renderer to load a project.  Performs all necessary initialization.
     *
     * @param bAssociateWithController - if true, sets this instance to create a metadata access object.
     * @throws HammerException
     */
    public void init(boolean bAssociateWithController) throws HammerException {
        // Set up metadata for sharing.
        StaticRendererData theController =
                StaticRendererData.getStaticData();

        String sConfigurationPath = getSystemConfigurationPathName() + getSystemConfigurationDocumentName();
        // Store props and old system stderr.
        props = System.getProperties();
        theSeparator = props.getProperty("file.separator");
        oldErr = System.err;
        // Load the class loader before adding configuration class paths.
        ClassLoader theSavedClassLoader = setupClassLoader();
        // Sharing.
        if (bAssociateWithController) {
            if (theController==null) {
                theController = new StaticRendererData(this, sBaseDir, sUserHome,
                    theClassLoader, sConfigurationPath);
                theController.setSavedClassLoader(theSavedClassLoader);
                StaticRendererData.setStaticData(theController);
            }
            theMetadataDispenser = theController.getMetadataDispenser();
            theMetadataDispenser.add(this);
            theConfigurationManager = theController.getConfigurationManager();
        } else // no sharing.  This is for the design cockpit or the fake renderer (first time).
            theConfigurationManager = new HammerConfigurationManager(sBaseDir, sUserHome, theClassLoader, sConfigurationPath);

        if (theConfigurationManager.bConfigured == false)
            throw new HammerException(HammerException.NO_CONFIGURATION_MANAGER_AVAILABLE, "Cannot Load Hammer Configuration");
        // Set up logging.
        theLogFile = new HammerLogFileAccess(this);
        if (theLogFile.bConfigured == false)
            throw new HammerException(HammerException.NO_LOGFILE_AVAILABLE, "Cannot open or create hammer log file");
        // Process all flags in configuration document.
        boolean bConsoleLogging = theConfigurationManager.checkConfigurationSetting("ConsoleLogging");
        boolean bLogging = false;

        bNovaState = theConfigurationManager.checkConfigurationSetting("Nova");
        bOpenSourceState = theConfigurationManager.checkConfigurationSetting("OpenSource");
        if (bConsoleLogging) {
            theLogFile.setConsoleLogging(true);
            bLogging = theConfigurationManager.checkConfigurationSetting("RedirectErr");
            if (bLogging)
                System.setErr(theLogFile.getLogConsoleStream());
            bLogging = theConfigurationManager.checkConfigurationSetting("RedirectOut");
            if (bLogging)
                System.setOut(theLogFile.getLogConsoleStream());
        }

        bLogging = theConfigurationManager.checkConfigurationSetting("FileLogging");
        // Redirect err & out to file log if they are not already set to console.
        if (bLogging) {
            theLogFile.setFileLogging(true);
            if (bConsoleLogging == false) {
                bLogging = theConfigurationManager.checkConfigurationSetting("RedirectErr");
                if (bLogging)
                    System.setErr(theLogFile.getLogConsoleStream());
                bLogging = theConfigurationManager.checkConfigurationSetting("RedirectOut");
                if (bLogging)
                    System.setOut(theLogFile.getLogConsoleStream());
            }
        }


        // Extended logging are entries that are not just system.err and system.out.

        bLogging = theConfigurationManager.checkConfigurationSetting("ExtendedLogging");

        if (bLogging)
            theLogFile.setExtendedLogging(true);

        bLogging = theConfigurationManager.checkConfigurationSetting("NLSLogging");
        if (bLogging)
            theLogFile.setNLSLogging(true);

        bLogging = theConfigurationManager.checkConfigurationSetting("DedicatedLogging");
        if (bLogging)
            theLogFile.setDedicatedConsole(true);

        bLogging = theConfigurationManager.checkConfigurationSetting("DebuggingLogging");
        if (bLogging)
            theLogFile.setDebuggingLogging(true);

        bLogging = theConfigurationManager.checkConfigurationSetting("ClassLoaderLogging");
        if (bLogging)
            theLogFile.setClassLoaderLogging(true);

        bLogging = theConfigurationManager.checkConfigurationSetting("CacheManagerLogging");
        if (bLogging)
            theLogFile.setCacheManagerLogging(true);

        String sPlaf = theConfigurationManager.getConfigurationVariable("PLAF", "ch.abacus.lib.ui.plaf.AbacusLookAndFeel");

        // Set the look and feel to abacus lnf.
        try
        {
            new HammerLNF(this, sPlaf);
        }
        catch (UnsupportedLookAndFeelException e)
        {
            sPlaf = UIManager.getSystemLookAndFeelClassName();
        }
        catch (Exception e)
        {
            throw new HammerException(HammerException.BAD_LNF, "Cannot initialize LNF");
        }

        // Load basic nls documents.
        theConfigurationManager.loadNLSDocuments();
        // Set up class loader after configuration paths.
        setupClassLoader();
        // Make sure we remove this object when the applet exits.
        if (theGlobalInterface != null) {
            theGlobalInterface.addToShutdownCollector(this);
        }

    }

    public void close() {
//        System.out.println("Invoking close of AbaRenderer!");
        theProject = null;
        theRenderingDestination = null;
        if (theClassLoader != null)
            theClassLoader.setUser(null);
        theClassLoader = null;
        if (theLogFile != null) {
            theLogFile.setUser(null);
        }
        theLogFile = null;
        //#ALEX
        theRegisteredInterfaceObjects.clear();
        if (theMetadataDispenser != null) {
            theMetadataDispenser.remove(this);
            if (theMetadataDispenser.getCount() == 0) {
                Thread.currentThread().setContextClassLoader(StaticRendererData.getSavedClassLoader());
                StaticRendererData.removeMetadata();
                theMetadataDispenser = null;
                theConfigurationManager = null;
                StaticRendererData.setStaticData(null);
            }
        }

        if (theGlobalInterface != null) {
            theGlobalInterface.removeFromShutdownCollector(this);
            theGlobalInterface = null;
        }
    }


    /**
     * testRemove
     *
     * This method is just for testing that the metadata access is removed.
     *
     */
    public void testRemove() {
        if (theMetadataDispenser != null) {
            theMetadataDispenser.remove(this);
            if (theMetadataDispenser.getCount() == 0) {
                StaticRendererData.removeMetadata();
                theMetadataDispenser = null;
            }
        }
    }


    public AbaRenderer(MetaProject theProject, boolean attachedToDesigner, GlobalInterface theGlobalInterface) throws HammerException {
        this.theXMLDocumentName = theProject.sName;
        this.attachedToDesigner = attachedToDesigner;
        this.theGlobalInterface = theGlobalInterface;
        init(true);
        this.theProject = new MetaProject(AbaMetaDataUser.EXECUTION_MODE, theGlobalInterface, theClassLoader, theMetadataDispenser, theProject);
        //ALEX
    }

    public AbaRenderer(String theXMLDocumentName, boolean attachedToDesigner, GlobalInterface theGlobalInterface) throws HammerException {
        this.theXMLDocumentName = theXMLDocumentName;
        this.attachedToDesigner = attachedToDesigner;
        this.theGlobalInterface = theGlobalInterface;
        init(true);
    }

    public AbaRenderer(String _ClassName, byte[] theProz, boolean attachedToDesigner, GlobalInterface theGlobalInterface) throws HammerException {
        this.theProzBufferClassName = _ClassName;
        this.theProzBuffer = theProz;
        this.attachedToDesigner = attachedToDesigner;
        this.theGlobalInterface = theGlobalInterface;
        init(true);
    }

    public AbaRenderer(MetaProject theProject, boolean attachedToDesigner, GlobalInterface theGlobalInterface, int iInputType) throws HammerException {
        this.theXMLDocumentName = theProject.sName;
        this.attachedToDesigner = attachedToDesigner;
        this.theGlobalInterface = theGlobalInterface;
        init(true);
        this.theProject = new MetaProject(AbaMetaDataUser.EXECUTION_MODE, theGlobalInterface, theClassLoader, theMetadataDispenser, theProject);
        this.iInputType = iInputType;
        theProject.setInputType(iInputType);
    }

    public AbaRenderer(String theXMLParameter, boolean attachedToDesigner, GlobalInterface theGlobalInterface, int iInputType) throws HammerException {
        if (iInputType == MetaProject.FROM_STRING) {
            this.theXMLDocumentName = "None";
            setBufferedXML(theXMLParameter);
        } else
            this.theXMLDocumentName = theXMLParameter;
        this.attachedToDesigner = attachedToDesigner;
        this.theGlobalInterface = theGlobalInterface;
        this.iInputType = iInputType;
        init(true);
    }

    public AbaRenderer() throws HammerException {
        String theXMLParameter = null;
        attachedToDesigner = false;
        theGlobalInterface = null;
        init(false);
    }

    public AbaRenderer (boolean _LoadMetadata) throws HammerException {
        String theXMLParameter = null;
        attachedToDesigner = false;
        theGlobalInterface = null;
        init ( _LoadMetadata );
    }

    protected ClassLoader setupClassLoader() {
        String[] sPaths = null;
        if (theConfigurationManager != null)
            sPaths = theConfigurationManager.getConfigurationVariableSet("ClassPath");

        String sClassLoaderPath = null;
        String sShortClassPath = null;
        String sAuxiliaryPath = null;
        String sBaseClassPath = System.getProperty("java.class.path");
        String sUserHome = System.getProperty("user.home");

        if (sPaths != null) {
            for (int iAuxPath = 0; iAuxPath < sPaths.length; iAuxPath++) {
                if (iAuxPath > 0)
                    sAuxiliaryPath = sAuxiliaryPath + File.pathSeparator + sPaths[iAuxPath];
                else
                    sAuxiliaryPath = sPaths[iAuxPath];
            }
        }
        // If an aux class path is defined in the config file use it to look for classes.
        // Long class path is always used to find files.

        if (getAddDefaultPaths()) {  // adds home path and current to the class path
            sClassLoaderPath = "." + theSeparator;
            if (sUserHome != null)
                sClassLoaderPath = sClassLoaderPath + File.pathSeparator + sUserHome;
        } else
            sClassLoaderPath = "";
        if (sAuxiliaryPath != null)
            sClassLoaderPath = sClassLoaderPath + File.pathSeparator + sAuxiliaryPath;
        sShortClassPath = sClassLoaderPath;
        if (sBaseClassPath != null)
            sClassLoaderPath = sClassLoaderPath + File.pathSeparator + sBaseClassPath;
        if (sAuxiliaryPath == null)
            sShortClassPath = sClassLoaderPath;
        // Set up the final paths of the class loader.
        if (theClassLoader != null)
            theClassLoader.setPaths(sClassLoaderPath, sShortClassPath);
        else
            theClassLoader = new HammerClassLoader(this, sClassLoaderPath, sShortClassPath);
        ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(theClassLoader.getLoader());
        return oldClassLoader;
    }

    public int getInputType() {
        return iInputType;
    }

    public void setInputType(int iInputType) {
        this.iInputType = iInputType;
        if (theProject != null)
            theProject.setInputType(iInputType);
    }


    public boolean load() throws HammerException {
        // try to automatically set the language ...
        if (theGlobalInterface != null) {
            if (theGlobalInterface.getUser() != null) {
                if (theGlobalInterface.getUser().getLanguage() != null) {
                    setLanguage(theGlobalInterface.getUser().getLanguage().getId());
                }
            } else if (theGlobalInterface.getLanguage() != null) {
                setLanguage(theGlobalInterface.getLanguage().getId());
            }
        }

        theProject = new MetaProject(AbaMetaDataUser.EXECUTION_MODE, theGlobalInterface, theClassLoader, theMetadataDispenser);
        theProject.setMetaDataUser(this);
        String SystemDocumentPath = this.getSystemConfigurationPathName() + this.getSystemMetadataDocumentName();
        if (SystemDocumentPath == null)
            SystemDocumentPath = "metadata.meta";

        boolean bTest = theProject.loadMeta(SystemDocumentPath);
        if(bOpenSourceState==true)
            theProject.loadCustomMeta(SystemDocumentPath,"ch\\abacus\\components\\imported\\");
        if (bTest) {
            switch (iInputType) {
                case MetaProject.UNDEFINED_INPUT_TYPE:
                    bTest = false;
                    if (theProzBuffer != null)
                        bTest = theProject.loadProzBuffer(theProzBufferClassName, theProzBuffer);
                    if (sBufferedXML != null)
                        bTest = theProject.loadXML(sBufferedXML);
                    if (bTest == false)
                        bTest = theProject.loadResource(theXMLDocumentName);
                    if (bTest == false)
                        bTest = theProject.load(theXMLDocumentName);

                    if (bTest == false)
                        throw new HammerException(HammerException.CANNOT_OPEN_PROJ, "Project " + theXMLDocumentName + " cannot be found.");
                    break;
                case MetaProject.FROM_RESOURCE:
                    bTest = theProject.loadResource(theXMLDocumentName);
                    if (bTest == false)
                        throw new HammerException(HammerException.CANNOT_OPEN_PROJ, "Project " + theXMLDocumentName + " cannot be found in jars of classpath.");
                    break;
                case MetaProject.FROM_STRING:
                    if (sBufferedXML != null)
                        bTest = theProject.loadXML(sBufferedXML);
                    if (bTest == false)
                        throw new HammerException(HammerException.CANNOT_OPEN_PROJ, "Project cannot be loaded from string.");
                    break;
                case MetaProject.FROM_XML_DOCUMENT:
                    bTest = theProject.load(theXMLDocumentName);
                    if (bTest == false)
                        throw new HammerException(HammerException.CANNOT_OPEN_PROJ, "Project " + theXMLDocumentName + " cannot be found.");
                    break;
            }
            iInputType = theProject.getInputType(); // might change in loader.
            theProject.getNLS().Load(theProject);
        } else {
            throw new HammerException(HammerException.NO_METADATA_AVAILABLE, "Cannot load metadata.meta");
        }
        return bTest;
    }

    public void setLanguage(int iLanguage) {
        theLanguage = iLanguage;
    }

    public boolean renderCompiledClass() throws NoSuchMethodException, InstantiationException, InvocationTargetException, IllegalAccessException {
        RenderingProjectInterface theRenderingProject;
        theRenderingDestination = null;
        if (theProject.renderingClass == null)
            return false;
        Class cls = theProject.renderingClass;
        Object[] theObjects = new Object[1];
        Class[] theClasses = new Class[1];
        theObjects[0] = this;
        theClasses[0] = AbaRendererInterface.class;
        Constructor constr = cls.getConstructor(theClasses);
        theRenderingProject = (RenderingProjectInterface) constr.newInstance(theObjects);
        theRenderingDestination = theRenderingProject.getRenderingDestination();
        if (theRenderingDestination != null) {
            if (theRenderingDestination instanceof JFrame)
                ((JFrame) theRenderingDestination).requestFocus();
        }
        return true;
    }

    public boolean renderCompiledClass(JFrame theFrame) throws NoSuchMethodException, InstantiationException, InvocationTargetException, IllegalAccessException {
        RenderingProjectInterface theRenderingProject = null;
        if (theProject.renderingClass == null)
            return false;
        theRenderingDestination = theFrame;

        if (shouldApplyRendererFocusTraversalPolicy())
        RendererFocusTraversalPolicy.getInstance().init(theFrame, theGlobalInterface);

        Class cls = theProject.renderingClass;
        Object[] theObjects = new Object[2];
        Class[] theClasses = new Class[2];
        theObjects[0] = this;
        theClasses[0] = AbaRendererInterface.class;
        theObjects[1] = theFrame;
        theClasses[1] = JFrame.class;
        Constructor constr = cls.getConstructor(theClasses);
        theRenderingProject = (RenderingProjectInterface) constr.newInstance(theObjects);
        setFocus(theFrame);
        return true;
    }

    public boolean renderCompiledClass(JComponent theComponent) throws NoSuchMethodException, InstantiationException, InvocationTargetException, IllegalAccessException {
        RenderingProjectInterface theRenderingProject;
        if (theProject.renderingClass == null)
            return false;

        // We need to set the focus traversal policy of every intermediate container.
        Container theContainer = theComponent.getParent();
        while ((theContainer != null) && (theContainer instanceof JFrame == false) &&
                (theContainer instanceof JDialog == false)) {
            if (shouldApplyRendererFocusTraversalPolicy())
            theContainer.setFocusTraversalPolicy(RendererFocusTraversalPolicy.getInstance());
            theContainer = theContainer.getParent();
        }
        // And initialize the policy at the Frame.
        theRenderingDestination = theComponent;
        if (theContainer != null) {
            if (shouldApplyRendererFocusTraversalPolicy())
            theContainer.setFocusTraversalPolicy(RendererFocusTraversalPolicy.getInstance());
            if (theContainer instanceof JFrame) {
                if (shouldApplyRendererFocusTraversalPolicy())
                RendererFocusTraversalPolicy.getInstance().init((JFrame) theContainer, theGlobalInterface);
            }
            else if (theContainer instanceof JDialog) {
                if (shouldApplyRendererFocusTraversalPolicy())
                RendererFocusTraversalPolicy.getInstance().init((JDialog) theContainer, theGlobalInterface);
            }
            theRenderingDestination = theContainer;
        }
        // Now load the class.
        Class cls = theProject.renderingClass;
        Object[] theObjects = new Object[2];
        Class[] theClasses = new Class[2];
        theObjects[0] = this;
        theClasses[0] = AbaRendererInterface.class;
        theObjects[1] = theComponent;
        theClasses[1] = JComponent.class;
        Constructor constr = cls.getConstructor(theClasses);
        theRenderingProject = (RenderingProjectInterface) constr.newInstance(theObjects);
        setFocus(theContainer);
        return true;
    }

    public boolean renderInterface() {
        theProject.setMetaDataUser(this);
        theProject.setCurrentLanguage(theLanguage);
        if (iInputType == MetaProject.FROM_CLASS) {
            try {
                return renderCompiledClass();
            } catch (NoSuchMethodException e) {
                e.printStackTrace();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (InvocationTargetException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        } else {
            MetaObject theObject = theProject.getFirstObject();
            while (theObject != null) {
                theObject.renderObject();
                theObject = theObject.theNextObject;
            }
            if (theProject.getFirstObject() != null) {
                if (shouldApplyRendererFocusTraversalPolicy())
                RendererFocusTraversalPolicy.getInstance().init(theProject.getFirstObject(), theGlobalInterface);
                Object theSwingObject = theProject.getFirstObject().getSwingObject();
                if (theSwingObject instanceof JFrame)
                    ((JFrame) theSwingObject).requestFocus();
                theRenderingDestination = theSwingObject;
            }
        }
        return true;
    }

    public boolean renderInterface(JFrame theFrame) {
        theProject.setMetaDataUser(this);
        theProject.setCurrentLanguage(theLanguage);
        if (iInputType == MetaProject.FROM_CLASS) {
            try {
                return renderCompiledClass(theFrame);
            } catch (NoSuchMethodException e) {
                e.printStackTrace();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (InvocationTargetException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        } else {
            MetaObject theObject = theProject.getFirstObject();
            if (theProject.getFirstObject() != null) {
                theObject.renderObject(theFrame);
                if (shouldApplyRendererFocusTraversalPolicy())
                RendererFocusTraversalPolicy.getInstance().init(theProject.getFirstObject(), theGlobalInterface);
            }
        }
        theFrame.requestFocus();
        theRenderingDestination = theFrame;
        return true;
    }

    public boolean renderInterface(JComponent theContainer) {
        theProject.setMetaDataUser(this);
        theProject.setCurrentLanguage(theLanguage);
        if (iInputType == MetaProject.FROM_CLASS) {
            try {
                return renderCompiledClass(theContainer);
            } catch (NoSuchMethodException e) {
                e.printStackTrace();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (InvocationTargetException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        } else {
            MetaObject theObject = theProject.getFirstObject();
            if (theProject.getFirstObject() != null) {
                theObject.renderObject(theContainer);
                if (shouldApplyRendererFocusTraversalPolicy())
                RendererFocusTraversalPolicy.getInstance().init(theProject.getFirstObject(), theGlobalInterface);
                theContainer.requestFocus();
            }
        }
        theRenderingDestination = theContainer;
        return true;
    }

    public Object getObject(String sName) {
        // check interface list first.
        Object objMatch = this.theRegisteredInterfaceObjects.get(sName);
        if (objMatch != null)
            return objMatch;
        MetaObject theMatch = theProject.findObject(sName);
        if (theMatch != null) {
            return theMatch.getSwingObject();
        }
        return null;
    }

    public Map getObjects() {
        return this.theRegisteredInterfaceObjects;
    }

    // Implement AbaMetaDataUser interface.

    public int getMode() {
        return AbaMetaDataUser.DESIGN_MODE;
    }

    public void setMode(int iMode) {
        // do nothing.
    }

    public void updateDisplayLayer(MetaObject theObject, int iLayer) {
        String sName = ((AnchoringLayoutManager) theObject.theParentObject.theLayoutManager).getName(theObject.theVisualObject);
        // Only set up layers for things that can be added to an anchoring layout manager like
        // controls in tab pages, panels, frames and panels in tab pages and panels and frames.
        // Objects in scrollpanes and tab pages in tabbed panes are skipped.
        if (sName != null) {
            ((JLayeredPane) theObject.theParentObject.theVisualObject).setLayer(theObject.theVisualObject, iLayer);
            ((AnchoringLayoutManager) theObject.theParentObject.theLayoutManager).addLayoutComponent(sName, theObject.theVisualObject);
            ((AnchoringLayoutManager) theObject.theParentObject.theLayoutManager).setAnchoring(theObject.theVisualObject,
                    theObject.bLeftAnchoring, theObject.bRightAnchoring,
                    theObject.bTopAnchoring, theObject.bBottomAnchoring);
        }

    }

    public void processPlacement(MetaObject theObject) {
        Container theContainer = null;
        if (theObject.theParentObject.theVisualObject != null)
            theContainer = theObject.theParentObject.theVisualObject;
        else
            theContainer = theObject.theParentObject.theVisualFrame.getContentPane();
        // split pane support.
        if (theObject.theParentObject.isHorzSplit()) {
            if (theObject.theParentObject.getFirstChild().equals(theObject))
                ((JSplitPane) theContainer).setLeftComponent(theObject.theVisualObject);
            else {
                ((JSplitPane) theContainer).setRightComponent(theObject.theVisualObject);
                int iDividerLocation = 0;
                MetaPropertyValueEx theDividerLocation = theObject.theParentObject.getPropertyValue("DividerLocation", 0);
                if (theDividerLocation != null)
                    iDividerLocation = ((Integer) theDividerLocation.getNativeValue()).intValue();
                ((JSplitPane) theContainer).resetToPreferredSizes();
                ((JSplitPane) theContainer).setDividerLocation(iDividerLocation);
            }
            Dimension dimSize = theObject.theVisualObject.getSize();
            Point ptLocation = theObject.theVisualObject.getLocation();
            System.out.println("Size of panel inside splitpane: " + dimSize.toString());
            System.out.println("Location of panel inside splitpane: " + ptLocation.toString());
            return;
        }
        if (theObject.theParentObject.isVertSplit()) {
            if (theObject.theParentObject.getFirstChild().equals(theObject))
                ((JSplitPane) theContainer).setTopComponent(theObject.theVisualObject);
            else {
                ((JSplitPane) theContainer).setBottomComponent(theObject.theVisualObject);
                int iDividerLocation = 0;
                MetaPropertyValueEx theDividerLocation = theObject.theParentObject.getPropertyValue("DividerLocation", 0);
                if (theDividerLocation != null)
                    iDividerLocation = ((Integer) theDividerLocation.getNativeValue()).intValue();
                ((JSplitPane) theContainer).resetToPreferredSizes();
                ((JSplitPane) theContainer).setDividerLocation(iDividerLocation);
            }
            Dimension dimSize = theObject.theVisualObject.getSize();
            Point ptLocation = theObject.theVisualObject.getLocation();
            System.out.println("Size of panel inside splitpane: " + dimSize.toString());
            System.out.println("Location of panel inside splitpane: " + ptLocation.toString());
            return;
        }
        if (theObject.isVertSplit() || theObject.isHorzSplit()) {
            Dimension dimSize = theObject.theVisualObject.getSize();
            Point ptLocation = theObject.theVisualObject.getLocation();
            System.out.println("Size of splitpane: " + dimSize.toString());
            System.out.println("Location of splitpane: " + ptLocation.toString());
            dimSize.width = 0;
        }
        // Fix for radio buttons.
        if (theObject.theVisualObject instanceof AbstractButton) {
            if (theObject.theParentObject.isGroup()) {
                ButtonGroup theGroup = theObject.theParentObject.getButtonGroup();
                if (theGroup != null)
                    theGroup.add((AbstractButton) theObject.theVisualObject);
            }
        }
        // Make sure the scroll pane is null if autoembed is off.
        theObject.theScrollPane = null;
        AnchoringLayoutManager theParentLayout = (AnchoringLayoutManager) theObject.theParentObject.theLayoutManager;
        if (theParentLayout != null) {
            // Now do auto embed.
            if (theObject.bAutoEmbed) {
                theObject.theScrollPane = new JAScrollPane(theObject.theVisualObject); // change JScrollPane to JAScrollPane (lauchenauer)
                theObject.theScrollPane.setSize(theObject.dimSize);
                theObject.theScrollPane.setLocation(theObject.iXLocation, theObject.iYLocation);

                //Take the color from the component
                if (theObject.theVisualObject != null) {
                    theObject.theScrollPane.getViewport().setBackground(theObject.theVisualObject.getBackground());
                }

                String sComponentName = theParentLayout.getName(theObject.theScrollPane);
                theContainer.add(theObject.theScrollPane, theObject.getName());
                if (sComponentName != null)
                    theParentLayout.addLayoutComponent(sComponentName, theObject.theScrollPane);
                theParentLayout.setAnchoring(theObject.theScrollPane,
                        theObject.getAnchorLeft(), theObject.getAnchorRight(),
                        theObject.getAnchorTop(), theObject.getAnchorBottom());
            } else {
                String sComponentName = theParentLayout.getName(theObject.theVisualObject);
                theContainer.add(theObject.theVisualObject, theObject.getName());
                if (sComponentName != null)
                    theParentLayout.addLayoutComponent(sComponentName, theObject.theVisualObject);
                theParentLayout.setAnchoring(theObject.theVisualObject,
                        theObject.getAnchorLeft(), theObject.getAnchorRight(),
                        theObject.getAnchorTop(), theObject.getAnchorBottom());
            }
        }
    }

    public void processMenus ( MetaObject theMetaObject ) {
        if ( theMetaObject.theParentObject.theVisualObject instanceof JMenuBar ) {
            JMenuBar theMenuBar = ( JMenuBar ) theMetaObject.theParentObject.theVisualObject;
            theMenuBar.add ( theMetaObject.theVisualObject );
        }
    }

    public void processMenuItems ( MetaObject theMetaObject ) {
        if ( theMetaObject.theParentObject.theVisualObject instanceof JMenu ) {
            JMenu theMenu = ( JMenu ) theMetaObject.theParentObject.theVisualObject;
            theMenu.add ( theMetaObject.theVisualObject );
        }
    }

    public void processMenuSeparators ( MetaObject theMetaObject ) {
        if ( theMetaObject.theParentObject.theVisualObject instanceof JSeparator ) {
            JMenu theMenu = ( JMenu ) theMetaObject.theParentObject.theVisualObject;
            theMenu.add ( theMetaObject.theVisualObject );
        }
    }

    public void processToolbarItems(MetaObject theMetaObject)
    {
        if (theMetaObject.theParentObject.theVisualObject instanceof JToolBar) {
            JToolBar theToolBar = (JToolBar) theMetaObject.theParentObject.theVisualObject;
            theToolBar.add(theMetaObject.theVisualObject);
        }
    }

    public void processTabPages(MetaObject theMetaObject) {
        if (theMetaObject.theParentObject.theVisualObject instanceof JTabbedPane) {
            JTabbedPane theTabbedPane = (JTabbedPane) theMetaObject.theParentObject.theVisualObject;
            int iIndex = theTabbedPane.indexOfComponent(theMetaObject.theVisualObject);
            if (iIndex == -1) {
                iIndex = theMetaObject.theParentObject.getChildIndex(theMetaObject);
                MetaPropertyValueEx theTabTitle = theMetaObject.theParentObject.getPropertyValue("TabTitle", iIndex);
                if (theTabTitle == null)
                    theTabbedPane.addTab("tab", theMetaObject.theVisualObject);
                else
                    theTabbedPane.addTab(theTabTitle.getStringValue(), theMetaObject.theVisualObject);
            } else {
                int iTestIndex = theTabbedPane.getSelectedIndex();
                if (iTestIndex != iIndex)
                    theTabbedPane.setSelectedIndex(iIndex);
            }
        }
    }

    public void doNewVisualObject(MetaObject theObject) {
        if (theObject.theFabricatedObject instanceof JComponent)
            theObject.theVisualObject = (JComponent) theObject.theFabricatedObject;
        else if (theObject.theFabricatedObject instanceof JFrame)
            theObject.theVisualFrame = (JFrame) theObject.theFabricatedObject;
    }

    public boolean getAttachedToDesigner() {
        return this.attachedToDesigner;
    }

    // stubs for the interface
    public void processObjectNameChanged(MetaObject metaObject) {
    }

    public void propertyEditedRepaint(MetaObject metaObject) {
    }

    public void processCreateLayout(MetaObject metaObject) {
    }

    public void processDoFrameSizeboxes(MetaObject metaObject) {
    }

    public void processDeleteObject(MetaObject metaObject) {
    }

    public void processProjectChangedState(boolean bState) {
    }

    public MetaObject getSelectedContainer() {
        return null;
    }

    public MetaObject getSelectedObject() {
        return null;
    }

    // methods to allow info about a frame.
    public Dimension getBaseFrameSize () throws HammerException {
        if ( theRenderingDestination == null )
            throw new HammerException ( HammerException.NOT_RENDERED_TO_DESTINATION,
                                        "getBaseFrameSize has failed because the project has not been rendered." );
        if ( theRenderingDestination instanceof JFrame )
            return ( ( JFrame ) theRenderingDestination ).getSize ();
        else
            return ( ( JComponent ) theRenderingDestination ).getSize ();
    }

    public Color getBaseFrameColor () throws HammerException {
        if ( theRenderingDestination == null )
            throw new HammerException ( HammerException.NOT_RENDERED_TO_DESTINATION,
                                        "getBaseFrameColor has failed because the project has not been rendered." );
        if ( theRenderingDestination instanceof JFrame )
            return ( ( JFrame ) theRenderingDestination ).getBackground ();
        else
            return ( ( JComponent ) theRenderingDestination ).getBackground ();
    }

    public Dimension getBaseFrameMaxSize () throws HammerException {
        if ( theRenderingDestination == null )
            throw new HammerException ( HammerException.NOT_RENDERED_TO_DESTINATION,
                                        "getBaseFrameMaxSize has failed because the project has not been rendered." );
        if ( theRenderingDestination instanceof JFrame )
            return ( ( JFrame ) theRenderingDestination ).getMaximumSize ();
        else
            return ( ( JComponent ) theRenderingDestination ).getMaximumSize ();
    }

    public Dimension getBaseFrameMinSize () throws HammerException {
        if ( theRenderingDestination == null )
            throw new HammerException ( HammerException.NOT_RENDERED_TO_DESTINATION,
                                        "getBaseFrameMinSize has failed because the project has not been rendered." );
        if ( theRenderingDestination instanceof JFrame )
            return ( ( JFrame ) theRenderingDestination ).getMinimumSize ();
        else
            return ( ( JComponent ) theRenderingDestination ).getMinimumSize ();
    }

    public String getBaseFrameTitle () throws HammerException {
        if ( theRenderingDestination == null )
            throw new HammerException ( HammerException.NOT_RENDERED_TO_DESTINATION,
                                        "getBaseFrameTitle has failed because the project has not been rendered." );
        if ( theRenderingDestination instanceof JFrame )
            return ( ( JFrame ) theRenderingDestination ).getTitle ();
        else
            return null;
    }

    /**
     * Returns the MetaObject that represents the "Frame" in the designed project.<p>
     * This is useful when you are rendering to your own container, but want to access the properties of the
     * original Frame object.
     * @return the MetaObject representing the Original Frame
     * @throws HammerException - HammerException.PROJECT_NOT_LOADED if the project has not been loaded
     */
    public MetaObject getOriginalFrame() throws HammerException {
        if (this.bLoaded == false)
            throw new HammerException ( HammerException.PROJECT_NOT_LOADED,
                                        "getOriginalFrame has failed because the project has not been loaded." );

        return theProject.getFirstObject();
    }

    /**
     * Returns the Title property of the Frame as it was designed in the Design Cockpit.<p>
     * You would typically use this method if you are rendering to your own container, otherwise
     * you could use getBaseFrameTitle().
     * @return The value of the Title property.
     * @throws HammerException - HammerException.PROJECT_NOT_LOADED if the project has not been loaded
     */
    public String getOriginalFrameTitle () throws HammerException {
        if (this.bLoaded == false)
            throw new HammerException ( HammerException.PROJECT_NOT_LOADED,
                                        "getOriginalFrameTitle has failed because the project has not been loaded." );

        MetaPropertyValueEx titleValue = getOriginalFrame().getPropertyValue("Title", 0);

        if (titleValue != null)
            return titleValue.getStringValue();
        else
            return null;

    }

    /**
     * Returns the size of the original Frame object of the project.<p>
     * This is useful if you are rendering to your own container, such as a JPanel, and need to know the original
     * size that the project was designed in.
     * @return A Dimension of the original size.
     * @throws HammerException - HammerException.PROJECT_NOT_LOADED if the project has not been loaded
     */
    public Dimension getOriginalFrameSize () throws HammerException {
        if (this.bLoaded == false)
            throw new HammerException ( HammerException.PROJECT_NOT_LOADED,
                                        "getOriginalFrameSize has failed because the project has not been loaded." );

        MetaPropertyValueEx widthValue = getOriginalFrame().getPropertyValue("Size.Width", 0);
        MetaPropertyValueEx heightValue = getOriginalFrame().getPropertyValue("Size.Height", 0);

        int width = ((Integer)widthValue.getNativeValue()).intValue();
        int height = ((Integer)heightValue.getNativeValue()).intValue();

        return new Dimension(width, height);
    }

    private Hashtable buildMinimumPropertySet(Object objSwing) {
        if (objSwing == null)
            return null;
        // return size, location, class, name, bg/fgcolor, font, text/title and that's it.
        Hashtable properties = new Hashtable(0);
        Dimension dimSize = new Dimension(0, 0);
        Point ptLocation = new Point(0, 0);
        String sName = "Unknown";
        Font font = null;
        Color clrBackground = null;
        Color clrForeground = null;
        if (objSwing instanceof JFrame) {
            dimSize = ((JFrame) objSwing).getSize();
            ptLocation = ((JFrame) objSwing).getLocation();
            sName = ((JFrame) objSwing).getName();
            String sTitle = ((JFrame) objSwing).getTitle();
            properties.put("Title", sTitle);
            font = ((JFrame) objSwing).getFont();
            clrBackground = ((JFrame) objSwing).getBackground();
            clrForeground = ((JFrame) objSwing).getForeground();
        } else if (objSwing instanceof JComponent) {
            dimSize = ((JComponent) objSwing).getSize();
            ptLocation = ((JComponent) objSwing).getLocation();
            sName = ((JComponent) objSwing).getName();
            String sText = "";
            if (objSwing instanceof AbstractButton)
                sText = ((AbstractButton) objSwing).getText();
            else if (objSwing instanceof JTextComponent)
                sText = ((JTextComponent) objSwing).getText();
            else if (objSwing instanceof JLabel)
                sText = ((JLabel) objSwing).getText();
            properties.put("Text", sText);
            font = ((JComponent) objSwing).getFont();
            clrBackground = ((JComponent) objSwing).getBackground();
            clrForeground = ((JComponent) objSwing).getForeground();
        }
        properties.put("Width", new Integer(dimSize.width));
        properties.put("Height", new Integer(dimSize.width));
        properties.put("Left", new Integer(ptLocation.x));
        properties.put("Top", new Integer(ptLocation.y));
        properties.put("Name", sName);
        properties.put("Font", font);
        properties.put("Foreground", clrForeground);
        properties.put("Background", clrBackground);
        return properties;
    }

    public Object createInterfaceObject(String sObjectName, String sClassName) {
        Hashtable properties = null;
        MetaClass theClass = theProject.findClass(sClassName);

        // MHC -- a better & more sofisticated error handler and display is needed
        // but this will do for now.
        if(theClass==null)
        {
            JOptionPane.showMessageDialog(null,
                sClassName + " Class was not found",
                "Missing jar or wrong meatadata",
                JOptionPane.ERROR_MESSAGE);
        }

        MetaType theType = new MetaType(theClass.getMetadata().sClassName,
                theClass.getMetadata().sFullClassName,
                theClass.getMetadata().theFactoryName,
                theProject);
        theType.theFactoryName = theClass.getMetadata().theFactoryName;
        theType.theDesignProject = theProject;
        if ((theType.theFactoryName != null) && (theType.theFactoryName.length() > 0))
            properties = buildMinimumPropertySet(null);
        Object theFabricatedObject = theType.Create(sObjectName, this, properties);
        if (theFabricatedObject != null)
            theRegisteredInterfaceObjects.put(sObjectName, theFabricatedObject);
        return theFabricatedObject;
    }

    public void createInterfaceObjectEx(String sObjectName, Object newedObject) {
            if (newedObject != null)
                theRegisteredInterfaceObjects.put(sObjectName, newedObject);
     }

    public Object getInterfaceObject(String sObjectName) {
        Object objMatch = this.theRegisteredInterfaceObjects.get(sObjectName);
        if (objMatch != null)
            return objMatch;
        return null;
    }

    public Object createConstantObject(String sConstantName, String sConstantClass) {
        String sMatch = null;
        MetaConstantGroup theConstantTable = theProject.findConstantGroup(sConstantClass);
        if (theConstantTable != null) {
            sMatch = theConstantTable.get(sConstantName);
        } else {
            MetaConstantGroupCollection theConstantTableCollection =
                    theProject.findConstantGroupCollection(sConstantClass);
            if (theConstantTableCollection != null) {
                sMatch = theConstantTableCollection.get(sConstantName);
            }
        }
        if (sMatch != null) {
            Object objRetVal = null;
            try {
                objRetVal = MetaConstantGroup.resolve(sMatch, theMetadataDispenser, theClassLoader.getLoader());
            } catch (HammerException e) {
                e.printStackTrace();  //To change body of catch statement use Options | File Templates.
            }
            return objRetVal;
        }
        return null;
    }

    public String doNLSTranslation(String sValue) {
        String theResolvedValue =
                theProject.getNLS().getTranslatedValue
                ((String) sValue,
                        theProject.getCurrentLanguage().theLanguage, false);
        return theResolvedValue;
    }

    public String getCurrentLanguageValue(String[] values) {
        int iLangId = theProject.getCurrentLanguage().theLanguage;
        if (iLangId > values.length -1)
            iLangId = 0;
        return values[iLangId];
    }

    public char doNLSMnemonicTranslation(String sValue) {
        String theResolvedValue =
                theProject.getNLS().getTranslatedValue
                ((String) sValue,
                        theProject.getCurrentLanguage().theLanguage, true);
        if (theResolvedValue != null)
            return theResolvedValue.charAt(0);
        else
            return '\0';
    }

    public Object resolveConstantValue(String sValue) {
        try {
            return MetaConstantGroup.resolve(sValue, theMetadataDispenser, theClassLoader.getLoader());
        } catch (HammerException e) {
            e.printStackTrace();
            return null;
        }
    }

    public void addNLSDocument(String sName, String sKey) {
        theProject.getNLS().addDocument(sName, sKey);
    }

    public void addFocus(Object oParent, Object oChild) {
        if ((oParent == null) && (oChild instanceof JFrame)) {
            if (shouldApplyRendererFocusTraversalPolicy())
            RendererFocusTraversalPolicy.getInstance().init((JFrame) oChild, theGlobalInterface);
        }
        else {
            if (oParent == null)
                oParent = theRenderingDestination;
            if (shouldApplyRendererFocusTraversalPolicy()) {
            if (RendererFocusTraversalPolicy.getInstance().getTree() != null) {
                RendererFocusTraversalPolicy.getInstance().getTree().addFocus(oParent, oChild);
            }
        }
    }
    }

    public void setNovaState(boolean bState) {
        bNovaState = bState;
    }

    public void setOpenSourceState(boolean bState) {
        this.bOpenSourceState = bState;
    }

    public boolean getNovaState() {
        return bNovaState;
    }

    public boolean getOpenSourceState() {
        return bOpenSourceState;
    }

    public boolean getDropletsState() {
        return false;
    }

    public void setFocus() {
        setFocus(this.theRenderingDestination);
    }

    public void setFocus(Object oNewFocus) {
        if (shouldApplyRendererFocusTraversalPolicy() == false) return;

        if (RendererFocusTraversalPolicy.getInstance().getTree() == null) return;

        RendererFocusNode theNode = RendererFocusTraversalPolicy.getInstance().getTree().getMatchingNode(oNewFocus);
        if (oNewFocus != null) {
            KeyboardFocusManager focusManager =
                    KeyboardFocusManager.getCurrentKeyboardFocusManager();
            Window newFocus = focusManager.getFocusedWindow();
            if (theNode != null) {
                // frame may not be set correctly if we are rendering in a component.
                Component theFirst = theNode.first();
                // dispatch event to change focus
                theFirst.requestFocus();
            }
        }
    }

    public AbaRenderer makeFakeRenderer() throws HammerException {
        return new AbaRenderer();
    }

    public String getConfigurationPath() {
        return null;  //To change body of created methods use File | Settings | File Templates.
    }

    private boolean shouldApplyRendererFocusTraversalPolicy() {
        return (theGlobalInterface == null);
    }

}
