/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import java.lang.reflect.Constructor;

/**
 * <p>Title: uifactory</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2001</p>
 * <p>Company: Abacus Research</p>
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */

public class MetaPropertyValueEx {
    int iType = TYPE_UNDEFINED;
    Class objectClass = null;
    String[] sValues = new String[HammerLanguagePresentation.LAST_LANGUAGE + 1];
    static final int TYPE_UNDEFINED = 0;
    static final int TYPE_STRING = 1;
    static final int TYPE_INT = 2;
    static final int TYPE_DOUBLE = 3;
    static final int TYPE_FLOAT = 4;
    static final int TYPE_CHAR = 5;
    static final int TYPE_BYTE = 6;
    static final int TYPE_LONG = 7;
    static final int TYPE_BOOLEAN = 8;
    static final int TYPE_OBJECT = 9;
    MetaProject theDesignProject = null;
    boolean bAllowMultipleLanguages = false;
    boolean bResolveAsObject = false;
    boolean bMnemonic = false;  // Kludge to support mnemonic translations.

//    public boolean equals(MetaPropertyValueEx obj) {
//	    if ((obj.iType==iType) && (obj.bAllowMultipleLanguages==bAllowMultipleLanguages) &&
//                (obj.bResolveAsObject==bResolveAsObject)) {
//            if (sValues.length==obj.sValues.length) {
//                for (int i=0; i<this.sValues.length; i++) {
//                    if (sValues[i].equals(obj.sValues[i])==false)
//                        return false;
//                }
//                return true;
//            }
//        }
//        return false;
//    }

    public MetaPropertyValueEx(String sValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.theDesignProject = theProject;
        this.iType = TYPE_STRING;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setLocalString(sValue);
    }

    public MetaPropertyValueEx(int iValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.iType = TYPE_INT;
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setValue(iValue);
    }

    public MetaPropertyValueEx(double dValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.iType = TYPE_DOUBLE;
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setValue(dValue);
    }

    public MetaPropertyValueEx(float fValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.iType = TYPE_FLOAT;
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setValue(fValue);
    }

    public MetaPropertyValueEx(char cValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.iType = TYPE_CHAR;
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setValue(cValue);
    }

    public MetaPropertyValueEx(byte byteValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.iType = TYPE_BYTE;
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setValue(byteValue);
    }

    public MetaPropertyValueEx(long lValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.iType = TYPE_LONG;
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setValue(lValue);
    }

    public MetaPropertyValueEx(boolean bValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        this.iType = TYPE_BOOLEAN;
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setValue(bValue);
    }

    public MetaPropertyValueEx(String sType, String sValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        setType(sType);
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setLocalString(sValue);
    }

    public MetaPropertyValueEx(int iType, String sValue, MetaProject theProject, boolean bAllowMultipleLanguages) {
        setType(iType);
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        setLocalString(sValue);
    }

    public MetaPropertyValueEx(int iType, String[] sValues, MetaProject theProject, boolean bAllowMultipleLanguages) {
        setType(iType);
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        for (int i = 0; (i < this.sValues.length) && (i < sValues.length); i++) {
            if (sValues[i] == null) {
                String sNewValue = sValues[0];
                if (sNewValue != null)
                    this.sValues[i] = new String(sValues[0]);
                else
                    this.sValues[i] = "";
            } else
                this.sValues[i] = new String(sValues[i]);
        }
    }

    public MetaPropertyValueEx(String sType, String[] sValues, MetaProject theProject, boolean bAllowMultipleLanguages) {
        setType(translateType(sType));
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        String sDefaultValue = sValues[HammerLanguagePresentation.DEFAULT];
        for (int i = 0; (i < this.sValues.length) && (i < sValues.length); i++) {
            String sNewValue = sValues[i];
            if (sNewValue == null)
                sNewValue = sDefaultValue;
            if (sNewValue == null)
                sNewValue = "";
            this.sValues[i] = new String(sNewValue);
        }
    }

    public MetaPropertyValueEx(String sClassName, Object someValue, boolean bCast, MetaProject theProject, boolean bAllowMultipleLanguages) {
        int iTranslatedType = translateType(sClassName);
        this.theDesignProject = theProject;
        this.bAllowMultipleLanguages = bAllowMultipleLanguages;
        if (someValue instanceof String) {
            setValue(someValue);
            if (iTranslatedType == MetaPropertyValueEx.TYPE_STRING)
                setType(TYPE_STRING);
            else {
                setType(sClassName);
                iTranslatedType = this.iType;  // don't cast this!
//              iTranslatedType = TYPE_OBJECT;
            }
        } else if (someValue instanceof Integer) {
            setValue(((Integer) someValue).intValue());
            setType(TYPE_INT);
        } else if (someValue instanceof Double) {
            setValue(someValue.toString());
            setType(TYPE_DOUBLE);
        } else if (someValue instanceof Float) {
            setValue(someValue.toString());
            setType(TYPE_FLOAT);
        } else if (someValue instanceof Character) {
            setValue(someValue.toString());
            setType(TYPE_CHAR);
        } else if (someValue instanceof Byte) {
            setValue(someValue.toString());
            setType(TYPE_BYTE);
        } else if (someValue instanceof Long) {
            setValue(someValue.toString());
            setType(TYPE_LONG);
        } else if (someValue instanceof Boolean) {
            setValue(someValue.toString());
            setType(TYPE_BOOLEAN);
        }
        if (iTranslatedType == TYPE_UNDEFINED) {
            iTranslatedType = TYPE_STRING;
            setValue(someValue.toString());
        }
        if (bCast && (getType() != iTranslatedType))
            setType(iTranslatedType);
    }

    public static Class getTypeAsClass(int type) {
        switch (type) {
            case TYPE_STRING:
                return String.class;
            case TYPE_INT:
                return int.class;
            case TYPE_DOUBLE:
                return double.class;
            case TYPE_FLOAT:
                return float.class;
            case TYPE_CHAR:
                return char.class;
            case TYPE_BYTE:
                return byte.class;
            case TYPE_LONG:
                return long.class;
            case TYPE_BOOLEAN:
                return boolean.class;
        }
        return null;
    }

    public MetaPropertyValueEx Clone() {
        MetaPropertyValueEx retVal = new MetaPropertyValueEx(iType, this.sValues, theDesignProject, this.bAllowMultipleLanguages);
        retVal.bResolveAsObject = bResolveAsObject;
        retVal.bMnemonic = bMnemonic;
        return retVal;
    }

    /**
     * getLocalString - Returns the value out of the buffer for the correct language.
     * @return
     */
    public String getLocalString() {
        String sReturnValue = "";
        if (bAllowMultipleLanguages == false)
            sReturnValue = sValues[HammerLanguagePresentation.DEFAULT];
        else if ((theDesignProject != null) &&
                (sValues[theDesignProject.theCurrentLanguage.theLanguage] != null))
            sReturnValue = sValues[theDesignProject.theCurrentLanguage.theLanguage];
        else
            sReturnValue = sValues[HammerLanguagePresentation.DEFAULT];
        // AbaRenderer translates.
        if ((theDesignProject != null) &&
                (theDesignProject.iMode == AbaMetaDataUser.EXECUTION_MODE)) {
            return theDesignProject.getNLS().getTranslatedValue(sReturnValue,
                    theDesignProject.theCurrentLanguage.theLanguage, bMnemonic);
        } else
            return sReturnValue;
    }

    public void setLocalString(String sValue) {
        if ((iType == MetaPropertyValueEx.TYPE_BYTE) ||
                (iType == MetaPropertyValueEx.TYPE_INT) || (iType == MetaPropertyValueEx.TYPE_LONG)) {
            if (sValue.startsWith("$") == true) {
                sValue = "0x" + sValue.substring(1);
            }
            if (sValue.startsWith("0x") == true) {
                switch (iType) {
                    case MetaPropertyValueEx.TYPE_BYTE:
                        {
                            sValue = Byte.valueOf(sValue.substring(2), 16).toString();
                        }
                        break;
                    case MetaPropertyValueEx.TYPE_INT:
                        {
                            sValue = Integer.valueOf(sValue.substring(2), 16).toString();
                        }
                        break;
                    case MetaPropertyValueEx.TYPE_LONG:
                        {
                            sValue = Long.valueOf(sValue.substring(2), 16).toString();
                        }
                        break;
                }
            }
        }
        if (bAllowMultipleLanguages == false) {
            sValues[HammerLanguagePresentation.DEFAULT] = sValue;
            return;
        }
        sValues[theDesignProject.theCurrentLanguage.theLanguage] = sValue;
        if (theDesignProject.theCurrentLanguage.theLanguage == HammerLanguagePresentation.DEFAULT) {
            for (int i = 1; i <= HammerLanguagePresentation.LAST_LANGUAGE; i++) {
                if (sValues[i] == null)
                    sValues[i] = sValue;
            }
        }
    }

    public boolean getResolveAsObject() {
        return bResolveAsObject;
    }

    public void setResolveAsObject(boolean bResolveAsObject) {
        this.bResolveAsObject = bResolveAsObject;
    }


    public void setValue(Object oValue) {
        if (oValue instanceof String) {
            String sValue = (String) oValue;
            setLocalString(sValue);
        } else
            setLocalString(oValue.toString());
    }

    public void setValue(int iValue) {
        setLocalString(new Integer(iValue).toString());
    }

    public void setValue(double dValue) {
        setLocalString(new Double(dValue).toString());
    }

    public void setValue(float fValue) {
        setLocalString(new Float(fValue).toString());
    }

    public void setValue(char cValue) {
        int iValue = (int) cValue;
        setLocalString(new Integer(iValue).toString());
    }

    public void setValue(byte byteValue) {
        int iValue = (int) byteValue;
        setLocalString(new Integer(iValue).toString());
    }

    public void setValue(boolean bValue) {
        setLocalString(new Boolean(bValue).toString());
    }

    public void setValue(long lValue) {
        setLocalString(new Long(lValue).toString());
    }

    public String getLanguageDependentValue(int iLang) {
        if (sValues[iLang] == null)
            return sValues[HammerLanguagePresentation.DEFAULT];
        return sValues[iLang];
    }

    public int getLanguageCount() {
        int i = 0;
        for (i = 0; i < HammerLanguagePresentation.LAST_LANGUAGE; i++) {
            if (sValues[i] != null)
                i++;
        }
        if (i > 1)
            return HammerLanguagePresentation.LAST_LANGUAGE;
        else
            return 1;
    }

    public void setType(int type) {
        iType = type;
    }

    public void setType(String sTypeName) {
        iType = translateType(sTypeName);
        if (iType == TYPE_UNDEFINED) {
            try {
                objectClass = Class.forName(sTypeName);
                iType = MetaPropertyValueEx.TYPE_OBJECT;
            } catch (ClassNotFoundException e1) {
            }
        } else
            objectClass = null;
    }

    public int getType() {
        return iType;
    }

    static public Object getObjectForValue(int type, String value) {
        switch (type) {
            case TYPE_STRING:
                return new String(value);
            case TYPE_INT:
                return new Integer(value);
            case TYPE_DOUBLE:
                return new Double(value);
            case TYPE_FLOAT:
                return new Float(value);
            case TYPE_LONG:
                return new Long(value);
            case TYPE_BOOLEAN:
                return new Boolean(value);
            case TYPE_CHAR:
                return new Character(value.charAt(0));
        }
        return new String("Undefined");
    }

    public static String translateType(int type) {
        switch (type) {
            case TYPE_STRING:
                return new String("java.lang.String");
            case TYPE_INT:
                return new String("int");
            case TYPE_DOUBLE:
                return new String("double");
            case TYPE_FLOAT:
                return new String("float");
            case TYPE_CHAR:
                return new String("char");
            case TYPE_BYTE:
                return new String("byte");
            case TYPE_LONG:
                return new String("long");
            case TYPE_BOOLEAN:
                return new String("boolean");
        }
        return new String("Undefined");
    }

    public void setMnemonic() {
        bMnemonic = true;
    }

    public static int translateType(String sTypeName) {
        int type;
        if (sTypeName.equals("java.lang.String") || (sTypeName.equals("String")))
            type = TYPE_STRING;
        else if (sTypeName.equals("int"))
            type = TYPE_INT;
        else if (sTypeName.equals("double"))
            type = TYPE_DOUBLE;
        else if (sTypeName.equals("float"))
            type = TYPE_FLOAT;
        else if (sTypeName.equals("char"))
            type = TYPE_CHAR;
        else if (sTypeName.equals("byte"))
            type = TYPE_LONG;
        else if (sTypeName.equals("boolean"))
            type = TYPE_BOOLEAN;
        else
            type = TYPE_UNDEFINED;
        return type;
    }

    public int toInt() {
        int iRetVal = 0;
        try {
            iRetVal = new Integer(getLocalString()).intValue();
        } catch (NumberFormatException e1) {
        }
        return iRetVal;
    }

    public float toFloat() {
        float fRetVal = 0;
        try {
            fRetVal = new Float(getLocalString()).floatValue();
        } catch (NumberFormatException e1) {
        }
        return fRetVal;
    }

    public double toDouble() {
        double dRetVal = 0;
        try {
            dRetVal = new Double(getLocalString()).doubleValue();
        } catch (NumberFormatException e1) {
        }
        return dRetVal;
    }

    public String toChar() {
        return getLocalString();
    }

    public byte toByte() {
        int iRetVal = 0;
        try {
            iRetVal = new Integer(getLocalString()).intValue();
        } catch (NumberFormatException e1) {
        }
        return (byte) iRetVal;
    }

    public long toLong() {
        long lRetVal = 0;
        try {
            lRetVal = new Long(getLocalString()).longValue();
        } catch (NumberFormatException e1) {
        }
        return lRetVal;
    }

    public boolean toBoolean() {
        boolean bRetVal = false;
        try {
            bRetVal = new Boolean(getLocalString()).booleanValue();
        } catch (NumberFormatException e1) {
        }
        return bRetVal;
    }

    public String getStringValue() {
        return getLocalString();
    }

    public Object getNativeValue() {
        switch (iType) {
            case TYPE_STRING:
                return getLocalString();
            case TYPE_INT:
                {
                    int iValue = toInt();
                    return new Integer(iValue);
                }
            case TYPE_DOUBLE:
                {
                    double dValue = toDouble();
                    return new Double(dValue);
                }
            case TYPE_FLOAT:
                {
                    float fValue = toFloat();
                    return new Float(fValue);
                }
            case TYPE_CHAR:
                {
                    return new String(toChar());
                }
            case TYPE_BYTE:
                {
                    byte bValue = toByte();
                    return new Byte(bValue);
                }
            case TYPE_LONG:
                {
                    long lValue = toLong();
                    return new Long(lValue);
                }
            case TYPE_BOOLEAN:
                {
                    boolean bValue = toBoolean();
                    return new Boolean(bValue);
                }
            case TYPE_OBJECT:
                {
                    if (this.getResolveAsObject() == false) /// MHC
                    {
                        try {
                            Class[] theClasses = new Class[1];
                            Object[] theObjects = new Object[1];
                            String sValue = getLocalString();
                            if (sValue != null) {
                                theClasses[0] = sValue.getClass();
                                theObjects[0] = sValue;
                                Constructor constr = objectClass.getConstructor(theClasses);
                                return constr.newInstance(theObjects);
                            } else
                                return null;
                        } catch (Exception e1) {
                            return null;
                        }
                    } else {
                        try {
                            String sValue = getLocalString();
                            MetaObject obj = this.theDesignProject.findObject(sValue);
                            Object javaObject = obj.getSwingObject();
                            return javaObject;
                        } catch (Exception e1) {
                            return null;
                        }
                    }
                }
        }
        return null;
    }

    public boolean isMultipleLanguages() {
        return bAllowMultipleLanguages;
    }
}
