/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.ArrayList;

/**
 * Title:        uifactory
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Research
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */

/**
 * Generates the metadata document from a tree of design classes.
 */

public class MetadataDocumentGenerator {

    MetaClass theFirstClass;
    MetaClass theLastClass;
    String theMetaDocumentName = "";

    public MetadataDocumentGenerator(String sName, MetaClass objFirstClass, MetaClass objLastClass) {
        theFirstClass = objFirstClass;
        theLastClass = objLastClass;
        theMetaDocumentName = sName;
    }

    /**
     * Writes the attributes of a property to the XML document<P>
     * You <b>must</b> update this method if you add/delete/change attributes of properties
     *
     * @param propListMetadata the property
     * @param theParent the xml element
     * @return true if success, or false if an error occurred
     */
    public boolean doPropertyListLevel(MetaPropertyDescriptor propListMetadata, electric.xml.Element theParent) {
        while (propListMetadata != null) {
            electric.xml.Element theProp = new electric.xml.Element("Property");
            electric.xml.Element thePropName = new electric.xml.Element("Name");
            electric.xml.Element theDefaultConstantType = new electric.xml.Element("DefaultConstantType");
            electric.xml.Element theGet = new electric.xml.Element("Get");
            electric.xml.Element theSet = new electric.xml.Element("Set");
            electric.xml.Element theGenerate = new electric.xml.Element("Generate");
            electric.xml.Element theDataAwareness = new electric.xml.Element("DataAwareness");
            electric.xml.Element theArray = new electric.xml.Element("Array");
            electric.xml.Element theLanguages = new electric.xml.Element("Languages");
            electric.xml.Element theHelper = new electric.xml.Element("Helper");
            electric.xml.Element theVisible = new electric.xml.Element("Visible");
            electric.xml.Element theDoc = new electric.xml.Element("Doc");

            electric.xml.Element thePropType = new electric.xml.Element("Type");
            electric.xml.Element thePropClass = new electric.xml.Element("Class");
            electric.xml.Element theReadOnly = new electric.xml.Element("ReadOnly");
            electric.xml.Element theObjRef = new electric.xml.Element("ObjectReference");

            theParent.addChild(theProp);
            String sPropertyName = propListMetadata.getName();
            thePropName.addText(sPropertyName);
            theProp.addChild(thePropName);

            String sGetMethod = propListMetadata.getGetMethod();
            if (sGetMethod != null && sGetMethod.length() > 0) {
                theProp.addChild(theGet);
                theGet.addText(sGetMethod);
            }

            String sSetMethod = propListMetadata.getSetMethod();
            if (sSetMethod != null && sSetMethod.length() > 0) {
                theProp.addChild(theSet);
                theSet.addText(sSetMethod);
            }

            String sDefaultPropertyValue = propListMetadata.getDefaultValue();
            if ((sDefaultPropertyValue != null) && (sDefaultPropertyValue.trim().length() != 0)) {
                electric.xml.Element theDefaultValue = new electric.xml.Element("DefaultValue");
                theProp.addChild(theDefaultValue);
                theDefaultValue.addText(sDefaultPropertyValue);
            }

            String sDefaultConstantType = propListMetadata.getIncludeTypeConstants();
            if (sDefaultConstantType != null && sDefaultConstantType.length() > 0) {
                theDefaultConstantType.addText(sDefaultConstantType);
                theProp.addChild(theDefaultConstantType);
            }


            int iPropertyValueChoices = propListMetadata.getValueChoiceCount();
            for (int iValueChoice = 0; iValueChoice < iPropertyValueChoices; iValueChoice++) {
                electric.xml.Element theValueChoice = new electric.xml.Element("ValueChoice");
                theValueChoice.addText(propListMetadata.getValueChoiceAsString(iValueChoice));
                theProp.addElement(theValueChoice);
            }
            if (propListMetadata.bGenerate)
                theProp.addChild(theGenerate);
            if (propListMetadata.getDataAwareness())
                theProp.addChild(theDataAwareness);
            if (propListMetadata.bVisible)
                theProp.addChild(theVisible);
            if (propListMetadata.isReadOnly())
                theProp.addChild(theReadOnly);

            if (propListMetadata.bAllowMultipleValues)
                theProp.addChild(theArray);
            if (propListMetadata.bAllowMultipleLanguages)
                theProp.addChild(theLanguages);
            if (propListMetadata.bObjectReference)
                theProp.addChild(theObjRef);

            String sHelper = propListMetadata.getHelperClass();
            if (sHelper != null && sHelper.length() > 0) {
                theHelper.addText(sHelper);
                theProp.addChild(theHelper);
            }

            // Add cdata sections for doc and value.
            if ( (propListMetadata.getDoc() != null) && (propListMetadata.getDoc().length() != 0) ) {
                electric.xml.CData theDocText = new electric.xml.CData(propListMetadata.getDoc());
                theProp.addChild(theDocText);
            }
            // Add type or class.
            if (propListMetadata.getClassName().length() != 0) {
                thePropType.addText("Class");
                thePropClass.addText(propListMetadata.getClassName());
                theProp.addElement(thePropClass);
            } else if (propListMetadata.getTypeName().length() != 0)
                thePropType.addText(propListMetadata.getTypeName());
            else
                thePropType.addText("String");
            theProp.addElement(thePropType);


            if (propListMetadata.theFirstSubproperty != null) {
                doPropertyListLevel(propListMetadata.theFirstSubproperty, theProp);
            }
            propListMetadata = propListMetadata.getNextProperty();
        }
        return true;
    }

    public boolean addClass(electric.xml.Element theClassList, MetaClass theClass, MetadataDispenser theMetadataDispenser) {

        MetaClassDetail clsmd = theClass.theMetadata;

        if (clsmd != null) {
            electric.xml.Element thisClass = new electric.xml.Element(clsmd.sClassName);
            theClassList.addElement(thisClass);

            if (clsmd.sClassComment != null) {
                electric.xml.Element theClassCommentsElement = new electric.xml.Element("Comments");
                theClassCommentsElement.setText(clsmd.sClassComment);
                thisClass.addElement(theClassCommentsElement);
            }

            electric.xml.Element theFullClassName = new electric.xml.Element("FullClassName");
            theFullClassName.addText(clsmd.sFullClassName);
            thisClass.addElement(theFullClassName);
            // Superclass entry.
            if (clsmd.getSuperClassName() != null) {
                electric.xml.Element theSuper = new electric.xml.Element("Superclass");
                electric.xml.Element theClassName = new electric.xml.Element("Name");
                theSuper.addChild(theClassName);
                theClassName.addText(clsmd.getSuperClassName());
                if (clsmd.getFullPathOfSuperClass() != null) {
                    electric.xml.Element theFullPath = new electric.xml.Element("FullPath");
                    theSuper.addChild(theFullPath);
                    theFullPath.addText(clsmd.getFullPathOfSuperClass());
                }
                thisClass.addElement(theSuper);
            }

            if (clsmd.bHidden == true) {
                electric.xml.Element xmlHidden = new electric.xml.Element("Hidden");
                thisClass.addElement(xmlHidden);
            }

            if (clsmd.bSkipDecl == true) {
                electric.xml.Element theSkipDeclarations = new electric.xml.Element("SkipDeclarations");
                thisClass.addElement(theSkipDeclarations);
            }

            if (clsmd.theFactoryName != null) {
                electric.xml.Element xmlFactoryName = new electric.xml.Element("ClassFactory");
                xmlFactoryName.setText(clsmd.theFactoryName);
                thisClass.addElement(xmlFactoryName);
            }
            if (clsmd.bDefaultListeners == false) {
                electric.xml.Element theSkipDefaultListeners = new electric.xml.Element("SkipDefaultListeners");
                thisClass.addElement(theSkipDefaultListeners);
            }
            if (clsmd.containerType.getContainerType() != MetaContainerType.CONTAINER_NONE) {
                electric.xml.Element theContainerType = new electric.xml.Element("ContainerType");
                thisClass.addElement(theContainerType);
                theContainerType.setText(clsmd.containerType.getContainerTypeAsString());
            }
            if (clsmd.sCustomCodeGenerationMethod.trim().length() != 0) {
                electric.xml.Element theCustom = new electric.xml.Element("Custom");
                thisClass.addElement(theCustom);
                theCustom.setText(clsmd.sCustomCodeGenerationMethod);
            }
            if ((clsmd.sDisplayClass != null) && (clsmd.sDisplayClass.trim().length() != 0)) {
                electric.xml.Element theDisplayClass = new electric.xml.Element("Display");
                theDisplayClass.setText(clsmd.sDisplayClass);
                thisClass.addElement(theDisplayClass);
            }

            if (clsmd.bDesignBorders == true) {
                electric.xml.Element xmlDesignBorders = new electric.xml.Element("DesignBorders");
                thisClass.addElement(xmlDesignBorders);
            }

            // Visibility entry.
            if (clsmd.getVisibility() != 0) {
                electric.xml.Element theVisibility = new electric.xml.Element("Visibility");
                switch (clsmd.getVisibility()) {
                    case MetaClassDetail.VISIBILITY_PRIVATE:
                        theVisibility.addText("private");
                        break;
                    case MetaClassDetail.VISIBILITY_PROTECTED:
                        theVisibility.addText("protected");
                        break;
                    case MetaClassDetail.VISIBILITY_PUBLIC:
                        theVisibility.addText("public");
                        break;
                }
                thisClass.addChild(theVisibility);
            }
            // Listeners.
            MetaClassDetail theListener = clsmd.theFirstListener;
            while (theListener != null) {
                int iListener = 0;
                String sListenerName = theListener.sClassName;
                // Check if it is a default listener.
                int iDefaultListeners = theMetadataDispenser.theListenerCollection.iDefaultListeners;
                for (iListener = 0; iListener < iDefaultListeners; iListener++) {
                    if (sListenerName.equals(theMetadataDispenser.theListenerCollection.getDefaultListenerName(iListener)))
                        break;
                }
                if (iListener == iDefaultListeners) { // not found means it's not default, we need to write it.
                    electric.xml.Element theListenerElement = new electric.xml.Element("Listener");
                    theListenerElement.setText(sListenerName);
                    thisClass.addElement(theListenerElement);
                }
                theListener = theListener.theNextClass;
            }
            // Properties (get/set).
            MetaPropertyDescriptor propListMetadata = clsmd.theFirstProperty;
            doPropertyListLevel(propListMetadata, thisClass);  // add property list.
            // Member Data.
            MetaDataMember dataListMetadata = clsmd.theFirstData;
            while (dataListMetadata != null) {
                electric.xml.Element theMemberData = new electric.xml.Element("MemberData");
                electric.xml.Element theMemberName = new electric.xml.Element("Name");
                electric.xml.Element theDoc = new electric.xml.Element("Doc");
                electric.xml.Element theType = new electric.xml.Element("Type");
                electric.xml.Element theValue = new electric.xml.Element("Value");
                thisClass.addChild(theMemberData);
                theMemberData.addChild(theMemberName);
                theMemberData.addChild(theType);
                theMemberData.addChild(theValue);
                theMemberName.addText(dataListMetadata.getDataName());
                theType.addText(dataListMetadata.getTypeName());
                // Add cdata sections for doc and value.
                if ((dataListMetadata.getDoc() != null) && (dataListMetadata.getDoc().length() != 0)) {
                    electric.xml.CData theDocText = new electric.xml.CData(dataListMetadata.getDoc());
                    theMemberData.addChild(theDoc);
                    theDoc.addChild(theDocText);
                }
                theValue.addText(dataListMetadata.getValue());
                // Move to the next data member.
                dataListMetadata = dataListMetadata.getNextData();
            }
            // Methods.
            MetaMethod methodListMetadata = clsmd.theFirstMethod;
            while (methodListMetadata != null) {
                if (methodListMetadata.getCode().trim().length() > 0) {
                    electric.xml.Element theMethod = new electric.xml.Element("Method");
                    thisClass.addChild(theMethod);
                    electric.xml.Element theMethodName = new electric.xml.Element("Name");
                    theMethodName.addText(methodListMetadata.getMethodName());
                    theMethod.addElement(theMethodName);
                    electric.xml.Element theMethodCode = new electric.xml.Element("Code");
                    electric.xml.CData theMethodCodeText = new electric.xml.CData(methodListMetadata.getCode());
                    theMethodCode.addChild(theMethodCodeText);
                    theMethod.addElement(theMethodCode);
                    if ((methodListMetadata.getDoc() != null) && (methodListMetadata.getDoc().length() != 0)) {
                        electric.xml.Element theMethodDocumentation = new electric.xml.Element("Documentation");
                        electric.xml.CData theMethodDocumentationText = new electric.xml.CData(methodListMetadata.getDoc());
                        theMethodDocumentation.addChild(theMethodDocumentationText);
                        theMethod.addElement(theMethodDocumentation);
                    }
                }
                methodListMetadata = methodListMetadata.getNextMethod();
            }
            if (theClass.theFirstClass != null) {
                theClassList = new electric.xml.Element("Classes");
                thisClass.addElement(theClassList);
                theClass = theClass.theFirstClass;
                while (theClass != null) {
                    addClass(theClassList, theClass, theMetadataDispenser);
                    theClass = theClass.theNextSibling;
                }
            }
        }
        return true;
    }

    public boolean GenerateFromMetadata(MetadataDispenser theMetadataDispenser) {  // Outputs xml from progInfo of parser.

        try {
            MetaClass theClass;
            File fileOutput = new File(theMetaDocumentName);
            PrintWriter pwOutput = new PrintWriter(new FileWriter(fileOutput));

            electric.xml.Document theDocument = new electric.xml.Document();
            electric.xml.Element theRootElement = new electric.xml.Element("Metadata");
            // Add the root to the document.
            theDocument.addChild(theRootElement);

            // add the document comments
            String sComments = theMetadataDispenser.theMetadataDocument.sDocumentComments;
            if (sComments != null) {
                electric.xml.Element theDocumentCommentsElement = new electric.xml.Element("Comments");
                theDocumentCommentsElement.setText(theMetadataDispenser.theMetadataDocument.sDocumentComments);
                theRootElement.addChild(theDocumentCommentsElement);
            }

            // output the Version Info...
            electric.xml.Element theVersionInfo = new electric.xml.Element("Version");
            electric.xml.Element theVersionNumber = new electric.xml.Element("VersionNumber");
            electric.xml.Element theRevisionNumber = new electric.xml.Element("RevisionNumber");
            electric.xml.Element theSubrevisionNumber = new electric.xml.Element("SubrevisionNumber");
            electric.xml.Element theBuildNumber = new electric.xml.Element("BuildNumber");

            theVersionNumber.setText((new Integer(theMetadataDispenser.theMetaDataAdministrator.iVersionNumber).toString()));
            theRevisionNumber.setText((new Integer(theMetadataDispenser.theMetaDataAdministrator.iRevisionNumber).toString()));
            theSubrevisionNumber.setText((new Integer(theMetadataDispenser.theMetaDataAdministrator.iSubrevisionNumber).toString()));
            theBuildNumber.setText((new Integer(theMetadataDispenser.theMetaDataAdministrator.iBuildNumber).toString()));

            theVersionInfo.addElement(theVersionNumber);
            theVersionInfo.addElement(theRevisionNumber);
            theVersionInfo.addElement(theSubrevisionNumber);
            theVersionInfo.addElement(theBuildNumber);

            theRootElement.addElement(theVersionInfo);

            // output the DesignBorderDefinition...
            electric.xml.Element theDesignBorderDefinition = new electric.xml.Element("DesignBorderDefinition");
            theDesignBorderDefinition.setText(theMetadataDispenser.sDesignBorderDefinition);
            theRootElement.addElement(theDesignBorderDefinition);

            // output the ConstantCollections...
            addConstantCollections(theMetadataDispenser, theRootElement);

            // output the Constants...
            addContantGroups(theMetadataDispenser, theRootElement);

            // Process classes.
            if (theFirstClass != null) {
                electric.xml.Element theClassList = new electric.xml.Element("Classes");
                theRootElement.addElement(theClassList);
                theClass = theFirstClass;
                while (theClass != null) {
                    addClass(theClassList, theClass, theMetadataDispenser);
                    theClass = theClass.theNextSibling;
                }
            }
            // Write the document.
            theDocument.write(pwOutput);
            pwOutput.close();

        } catch (java.io.IOException e) {
            return false;
        }

        return true;
    }

    private void addContantGroups(MetadataDispenser theMetadataDispenser, electric.xml.Element theRootElement) {
        electric.xml.Element xmlConstants = new electric.xml.Element("Constants");

        MetaConstantGroup group;

        group = theMetadataDispenser.getFirstMetaConstantGroup();
        while (group != null) {
            String sGroup = group.sClassName;
            electric.xml.Element xmlGroup = new electric.xml.Element(sGroup);
            xmlConstants.addElement(xmlGroup);

            ArrayList constants = group.getConstantList();
            for (int iGroup = 0; iGroup < constants.size(); iGroup++) {
                String sConstantName = (String) group.getConstantList().get(iGroup);
                String sConstantValue = group.get((String) group.getConstantList().get(iGroup));
                electric.xml.Element xmlConstant = new electric.xml.Element(sConstantName);
                xmlConstant.setText(sConstantValue);
                xmlGroup.addElement(xmlConstant);
            }
            group = group.theNextSibling;
        }

        theRootElement.addElement(xmlConstants);
    }

    private void addConstantCollections(MetadataDispenser theMetadataDispenser, electric.xml.Element theRootElement) {
        MetaConstantGroupCollection collection;

        electric.xml.Element xmlConstantCollections = new electric.xml.Element("ConstantCollections");

        collection = theMetadataDispenser.getFirstMetaConstantGroupCollection();
        while (collection != null) {
            String sCollection = collection.sClassName;
            electric.xml.Element xmlCollection = new electric.xml.Element(sCollection);
            xmlConstantCollections.addElement(xmlCollection);

            // add the Groups to the Collection...
            //String[] groups = new String[collection.getGroupCount()];
            for (int i = 0; i < collection.getGroupCount(); i++) {
                electric.xml.Element xmlGroup = new electric.xml.Element(collection.getGroup(i).sClassName);
                xmlCollection.addElement(xmlGroup);
            }

            collection = collection.theNextSibling;
        }
        theRootElement.addElement(xmlConstantCollections);

    }

}

